<?php
/**
 * Holds rest functionality for Tutor LMS quiz.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Quiz
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Quiz;

use Throwable;
use TUTOR\Course_List;
use Tutor\Models\CourseModel;
use TUTOR\Quiz;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Class QuizActionRest
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Quiz
 */
class QuizActionRest extends QuizRest {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * QuizRest instance.
	 *
	 * @since 2.2.80
	 * @return QuizActionRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/start',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'start' ),
					'permission_callback' => array( $this, 'get_complete_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/complete',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'complete' ),
					'permission_callback' => array( $this, 'get_complete_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to lesson item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_complete_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the lesson `complete` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_quiz_start_permissions_check', $retval, $request );
	}

	/**
	 * Start quiz.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return WP_REST_Response
	 */
	public function start( $request ) {
		$quiz_id = (int) $request['id'];
		$user_id = get_current_user_id();
		$course  = CourseModel::get_course_by_quiz( $quiz_id );

		if ( ! $course ) {
			return rest_ensure_response( QuizError::instance()->course_not_found() );
		}

		if ( ! tutor_utils()->has_enrolled_content_access( 'quiz', $quiz_id, $user_id ) && ! Course_List::is_public( $course->ID ) ) {
			return rest_ensure_response( QuizError::instance()->not_enrolled_in_course() );
		}

		$quiz_attempt = Quiz::quiz_attempt( $course->ID, $quiz_id, $user_id );

		if ( $quiz_attempt ) {
			$get_quiz = new WP_REST_Request();

			$get_quiz->set_method( 'GET' );
			$get_quiz->set_param( 'id', $quiz_id );
			$get_quiz->set_route( $this->namespace . '/' . $this->rest_base . '/' . $quiz_id );

			$response = $this->get_item( $get_quiz );
		} else {
			$response = QuizError::instance()->quiz_start_failed();
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Complete quiz.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return WP_REST_Response
	 */
	public function complete( $request ) {
		$attempt_id = $request->get_param( 'attempt_id' );
		$attempt    = tutor_utils()->get_attempt( $attempt_id );

		if ( ! $attempt ) {
			return rest_ensure_response( QuizError::instance()->attempt_not_found() );
		}

		$course = CourseModel::get_course_by_quiz( $attempt->quiz_id );

		if ( ! $course ) {
			return rest_ensure_response( QuizError::instance()->course_not_found() );
		}

		$user_id = get_current_user_id();

		if ( ! tutor_utils()->has_enrolled_content_access( 'quiz', $attempt->quiz_id, $user_id ) && ! Course_List::is_public( $course->ID ) ) {
			return rest_ensure_response( QuizError::instance()->not_enrolled_in_course() );
		}

		$attempt_answers = ! empty( $request->get_param( 'attempt' ) ) ? tutor_sanitize_data( $request->get_param( 'attempt' ) ) : false;
		$attempt_answers = is_array( $attempt_answers ) ? $attempt_answers : array();

		if ( ! $attempt_answers ) {
			return rest_ensure_response( QuizError::instance()->invalid_quiz_answers() );
		}

		if ( $user_id !== (int) $attempt->user_id ) {
			return rest_ensure_response( QuizError::instance()->operation_not_allowed() );
		}

		try {
			Quiz::manage_attempt_answers( $attempt_answers, $attempt, $attempt_id, $course->ID, $user_id );

			$get_quiz = new WP_REST_Request();
			$get_quiz->set_method( 'GET' );
			$get_quiz->set_param( 'id', $attempt->quiz_id );
			$get_quiz->set_route( $this->namespace . '/' . $this->rest_base . '/' . $attempt->quiz_id );

			return rest_ensure_response( $this->get_item( $request ) );
		} catch ( Throwable $th ) {
			return rest_ensure_response( QuizError::instance()->quiz_complete_failed() );
		}
	}
}
