<?php
/**
 * Holds lesson rest functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Lesson
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Lesson;

use BuddyBossApp\Api\TutorLMS\V1\Core\TutorRestController;
use WP_Error;
use WP_Post;
use WP_Query;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Lesson rest class.
 */
class LessonsRest extends TutorRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Post type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'lesson';

	/**
	 * LessonsRest instance.
	 *
	 * @since 2.2.80
	 * @return LessonsRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 2.2.80
	 */
	public function __construct() {
		$this->rest_base = 'lessons';

		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to lesson items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_items_permissions_check( $request ) {
		/**
		 * Filter the lesson `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_lessons_permissions_check', true, $request );
	}

	/**
	 * Check if a given request has access to lesson item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_item_permissions_check( $request ) {
		/**
		 * Filter the lesson `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_lesson_permissions_check', true, $request );
	}

	/**
	 * Retrieve Lessons.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/lessons Get TutorLMS Lessons
	 * @apiName        GetTutorLessons
	 * @apiGroup       Tutor Lessons
	 * @apiDescription Retrieve Lessons
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure a result set includes specific IDs.
	 * @apiParam {String} [after] Limit results to those published after a given ISO8601 compliant date.
	 * @apiParam {String} [before] Limit results to those published before a given ISO8601 compliant date.
	 * @apiParam {Array} [author] Limit results to those assigned to specific authors.
	 * @apiParam {Array} [author_exclude] Ensure results to exclude those assigned to specific authors.
	 * @apiParam {String=asc,desc} [order=desc] Sort result set by given order.
	 * @apiParam {String=date,id,title,menu_order} [orderby=date] Sort result set by given field.
	 * @apiParam {Array} [parent] Limit results to those assigned to specific parent.
	 */
	public function get_items( $request ) {
		$registered = $this->get_collection_params();

		/**
		 * Filter the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$request = apply_filters( 'bbapp_tutor_get_lessons_request', $request );

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'author'         => 'author__in',
			'author_exclude' => 'author__not_in',
			'exclude'        => 'post__not_in',
			'include'        => 'post__in',
			'offset'         => 'offset',
			'order'          => 'order',
			'orderby'        => 'orderby',
			'page'           => 'paged',
			'parent'         => 'post_parent__in',
			'parent_exclude' => 'post_parent__not_in',
			'search'         => 's',
			'slug'           => 'post_name__in',
			'status'         => 'post_status',
			'per_page'       => 'posts_per_page',
		);

		/**
		 * For each known parameter which is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		// Check for & assign any parameters which require special handling or setting.
		$args['date_query'] = array();

		// Set before into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['before'], $request['before'] ) ) {
			$args['date_query'][0]['before'] = $request['before'];
		}

		// Set after into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['after'], $request['after'] ) ) {
			$args['date_query'][0]['after'] = $request['after'];
		}

		$args = $this->prepare_items_query( $args, $request );

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$args = apply_filters( 'bbapp_tutor_get_lessons_args', $args, $request );

		$args['post_type']      = $this->post_type;
		$posts_query            = new WP_Query();
		$leesons['posts']       = $posts_query->query( $args );
		$leesons['total_posts'] = $posts_query->found_posts;

		/**
		 * Fires list of Lesson is fetched via Query.
		 *
		 * @param array            $leesons Fetched lessons.
		 * @param WP_REST_Response $args    Query arguments.
		 * @param WP_REST_Request  $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$leesons = apply_filters( 'bbapp_tutor_get_lessons', $leesons, $args, $request );

		$retval = array();

		foreach ( $leesons['posts'] as $couese ) {
			if ( ! $this->check_read_permission( $couese ) ) {
				continue;
			}

			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $couese, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		$response = bbapp_tutor_response_add_total_headers( $response, $leesons['total_posts'], $args['posts_per_page'] );

		/**
		 * Fires after a list of lesson response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_lessons_items_response', $response, $request );

		return $response;
	}

	/**
	 * Retrieve Lesson.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/lessons/:id Get Tutor LMS Lesson
	 * @apiName        GetTutorLesson
	 * @apiGroup       Tutor Lessons
	 * @apiDescription Retrieve a single Lesson
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Lesson.
	 */
	public function get_item( $request ) {
		$lesson_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$lesson    = get_post( $lesson_id );

		if ( empty( $lesson ) || $this->post_type !== $lesson->post_type ) {
			return LessonsError::instance()->invalid_lesson_id();
		}

		/**
		 * Fire after Lesson is fetched via Query.
		 *
		 * @param array           $lesson    Fetched lesson.
		 * @param WP_REST_Request $lesson_id lesson id.
		 *
		 * @since 2.2.80
		 */
		$lesson = apply_filters( 'bbapp_tutor_get_lesson', $lesson, $lesson_id );

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $lesson, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a lesson response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_lesson_item_response', $response, $request );

		return $response;
	}

	/**
	 * Prepare a single post-output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$GLOBALS['post'] = $post; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited

		setup_postdata( $post );

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$schema  = $this->get_public_item_schema();

		// Base fields for every post.
		$data = array(
			'id'           => $post->ID,
			'title'        => array(
				'raw'      => $post->post_title,
				'rendered' => get_the_title( $post->ID ),
			),
			'content'      => array(
				'raw'      => bbapp_tutor_fix_relative_urls_protocol( $post->post_content ),
				'rendered' => bbapp_tutor_fix_relative_urls_protocol( apply_filters( 'the_content', $post->post_content ) ),
			),
			'date'         => mysql_to_rfc3339( $post->post_date ),
			'date_gmt'     => mysql_to_rfc3339( $post->post_date_gmt ),
			'modified'     => mysql_to_rfc3339( $post->post_modified ),
			'modified_gmt' => mysql_to_rfc3339( $post->post_modified_gmt ),
			'link'         => get_permalink( $post->ID ),
			'slug'         => $post->post_name,
			'author'       => (int) $post->post_author,
			'excerpt'      => array(
				'raw'      => bbapp_tutor_fix_relative_urls_protocol( $post->post_excerpt ),
				'rendered' => bbapp_tutor_fix_relative_urls_protocol( apply_filters( 'the_excerpt', $post->post_excerpt ) ),
			),
			'menu_order'   => (int) $post->menu_order,
			'status'       => $post->post_status,
		);

		/**
		 * Feature Media
		 */
		$post->featured_media            = $this->get_feature_media( $post );
		$data['featured_media']          = array();
		$data['featured_media']['small'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['small'] ) ) ? $post->featured_media['small'] : null;
		$data['featured_media']['large'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['large'] ) ) ? $post->featured_media['large'] : null;

		if ( ! empty( $schema['properties']['topic'] ) && in_array( $context, $schema['properties']['topic']['context'], true ) ) {
			$post->topic   = $this->get_topic_id( $post );
			$data['topic'] = (int) $post->topic;
		}

		if ( ! empty( $schema['properties']['course'] ) && in_array( $context, $schema['properties']['course']['context'], true ) ) {
			$post->course   = $this->get_course_id( $post );
			$data['course'] = (int) $post->course;
		}

		if ( ! empty( $schema['properties']['settings'] ) && in_array( $context, $schema['properties']['settings']['context'], true ) ) {
			$post->settings   = $this->get_settings( $post );
			$data['settings'] = $post->settings;
		}

		if ( ! empty( $schema['properties']['video'] ) && in_array( $context, $schema['properties']['video']['context'], true ) ) {
			$post->video   = $this->get_video( $post );
			$data['video'] = $post->video;
		}

		if ( ! empty( $schema['properties']['completed'] ) && in_array( $context, $schema['properties']['completed']['context'], true ) ) {
			$post->completed   = $this->is_completed( $post );
			$data['completed'] = (bool) $post->completed;
		}

		if ( ! empty( $schema['properties']['enrolled'] ) && in_array( $context, $schema['properties']['enrolled']['context'], true ) ) {
			$post->enrolled   = $this->is_enrolled( $post );
			$data['enrolled'] = (bool) $post->enrolled;
		}

		if ( ! empty( $schema['properties']['can_complete'] ) && in_array( $context, $schema['properties']['can_complete']['context'], true ) ) {
			$post->can_complete   = $this->get_can_complete( $post );
			$data['can_complete'] = $post->can_complete;
		}

		if ( ! empty( $schema['properties']['can_comment'] ) && in_array( $context, $schema['properties']['can_comment']['context'], true ) ) {
			$post->can_comment   = tutor_utils()->get_option( 'enable_comment_for_lesson' ) && comments_open();
			$data['can_comment'] = $post->can_comment;
		}

		if ( ! empty( $schema['properties']['has_attachment'] ) && in_array( $context, $schema['properties']['has_attachment']['context'], true ) ) {
			$post->has_attachment   = $this->has_attachment( $post );
			$data['has_attachment'] = $post->has_attachment;
		}

		if ( ! empty( $schema['properties']['attachments'] ) && in_array( $context, $schema['properties']['attachments']['context'], true ) ) {
			$post->attachments   = $this->get_attachments( $post );
			$data['attachments'] = $post->attachments;
		}

		if ( ! empty( $schema['properties']['error_message'] ) && in_array( $context, $schema['properties']['error_message']['context'], true ) ) {
			$post->error_message = $this->get_error_message( $post );
			if ( ! empty( $post->error_message ) ) {
				$error_code            = $post->error_message->get_error_code();
				$data['error_message'] = array(
					'code'    => $error_code,
					'message' => $post->error_message->get_error_message(),
					'data'    => $post->error_message->get_error_data( $error_code ),

				);
			} else {
				$data['error_message'] = array();
			}
		}

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filters lesson response.
		 *
		 * @param WP_REST_Response $response Lessons response.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 *
		 * @since 2.2.80
		*/
		return apply_filters( 'bbapp_tutor_rest_prepare_lesson', $response, $post, $request );
	}

	/**
	 * Prepare item query.
	 *
	 * @param array $prepared_args Item query arguments.
	 * @param null  $request       Request parameters.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function prepare_items_query( $prepared_args = array(), $request = null ) {
		$query_args = array();

		foreach ( $prepared_args as $key => $value ) {
			/**
			 * Filters the query_vars used in get_items() for the constructed query.
			 *
			 * The dynamic portion of the hook name, `$key`, refers to the query_var key.
			 *
			 * @param string $value The query_var value.
			 *
			 * @since 2.2.80
			 */
			$query_args[ $key ] = apply_filters( "rest_query_var-{$key}", $value ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		}

		$query_args['ignore_sticky_posts'] = true;

		// Map to proper WP_Query orderby param.
		if ( isset( $query_args['orderby'] ) && isset( $request['orderby'] ) ) {
			$orderby_mappings = array(
				'id'            => 'ID',
				'include'       => 'post__in',
				'slug'          => 'post_name',
				'include_slugs' => 'post_name__in',
			);

			if ( isset( $orderby_mappings[ $request['orderby'] ] ) ) {
				$query_args['orderby'] = $orderby_mappings[ $request['orderby'] ];
			}
		}

		return $query_args;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array $data item object.
	 *
	 * @since 2.2.80
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {
		$links = parent::prepare_links( $data );

		return $links;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author'] = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset']            = array(
			'description'       => __( 'Offset the result set by a specific number of items.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']['enum'][] = 'menu_order';

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent IDs.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent ID.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);

		$params['slug'] = array(
			'description'       => __( 'Limit result set to posts with a specific slug.', 'buddyboss-app' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['filter'] = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.', 'buddyboss-app' ),
		);

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'lesson',
			'type'       => 'object',
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'          => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content'        => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'date'           => array(
					'description' => __( 'The date the object was published, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date_gmt'       => array(
					'description' => __( 'The date the object was published, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'modified'       => array(
					'description' => __( 'The date the object was last modified, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'modified_gmt'   => array(
					'description' => __( 'The date the object was last modified, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'link'           => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'           => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'         => array(
					'description' => __( 'The ID for the author of the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'excerpt'        => array(
					'description' => __( 'The excerpt for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'featured_media' => array(
					'description' => __( 'Feature media object containing thumb and full URL of image.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'menu_order'     => array(
					'description' => __( 'The order of the object in relation to other object of its type.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
			),
		);

		$schema['properties']['topic'] = array(
			'description' => __( 'The Topic id for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['course'] = array(
			'description' => __( 'The Course id for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['video'] = array(
			'description' => __( 'video for the object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['completed'] = array(
			'description' => __( 'The object is completed by current user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['enrolled'] = array(
			'description' => __( 'The object is enrolled by current user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['settings'] = array(
			'description' => __( 'The object settings.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['can_complete'] = array(
			'description' => __( 'The object can complete by user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['can_comment'] = array(
			'description' => __( 'The object can comment by user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['has_attachment'] = array(
			'description' => __( 'The object has attachment or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['attachments'] = array(
			'description' => __( 'The object has attachment or not.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get lesson feature medias.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_feature_media( $post ) {
		$return = array(
			'large' => null,
			'small' => null,
		);
		$large  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'large' );
		$small  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'medium' );

		if ( isset( $large[0] ) ) {
			$return['large'] = $large[0];
		}
		if ( isset( $small[0] ) ) {
			$return['small'] = $small[0];
		}

		return $return;
	}

	/**
	 * Get a lesson topic id.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @since 2.2.80
	 * @return int
	 */
	protected function get_topic_id( $post ) {
		$topic_id = $post->post_parent;

		if ( ! isset( $topic_id ) ) {
			$topic_id = 0;
		}

		return $topic_id;
	}

	/**
	 * Get lesson course id.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @since 2.2.80
	 * @return int
	 */
	protected function get_course_id( $post ) {
		$course_id = tutor_utils()->get_course_id_by_subcontent( $post->ID );

		if ( ! isset( $course_id ) ) {
			$course_id = 0;
		}

		return $course_id;
	}

	/**
	 * Get lesson video.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_video( $post ) {
		return tutor_utils()->get_video( $post->ID );
	}

	/**
	 * This function we use in Course endpoint, so it should be public
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	public function is_completed( $post ) {
		return tutor_utils()->is_completed_lesson( $post->ID, get_current_user_id() ) !== false;
	}

	/**
	 * This function we use in Course is enrolled or not.
	 *
	 * @since 2.2.80
	 *
	 * @param WP_Post $post      Lesson post.
	 * @param int     $course_id Course id.
	 *
	 * @return bool
	 */
	private function is_enrolled( $post, $course_id = 0 ) {
		if ( empty( $course_id ) ) {
			$course_id = $this->get_course_id( $post );
		}

		$is_enrolled = tutor_utils()->is_enrolled( $course_id );

		return ! empty( $is_enrolled );
	}

	/**
	 * Current user complete course or not.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	private function get_can_complete( $post ) {
		$can_complete = false;

		if ( is_user_logged_in() ) {
			$can_complete = true;
		}

		return $can_complete;
	}

	/**
	 * Has attachment or not.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	protected function has_attachment( $post ) {
		return count( $this->get_attachments( $post ) ) > 0;
	}

	/**
	 * Get attachment or not.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_attachments( $post ) {
		return tutor_utils()->get_attachments( $post->ID );
	}

	/**
	 * Get lesson settings.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_settings( $post ) {
		return array();
	}

	/**
	 * Get an error message.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @since 2.2.80
	 * @return WP_Error|array
	 */
	public function get_error_message( $post ) {
		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_forbidden_context', __( 'Sorry, you are not allowed to access this resource.', 'buddyboss-app' ), array( 'status' => rest_authorization_required_code() ) );
		}

		return array();
	}
}
