<?php
/**
 * Holds lesson action rest functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Lesson
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Lesson;

use Tutor\Models\LessonModel;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Lesson action rest class.
 */
class LessonsActionRest extends LessonsRest {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * LessonsActionRest instance.
	 *
	 * @since 2.2.80
	 * @return LessonsActionRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/complete',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'complete' ),
					'permission_callback' => array( $this, 'get_complete_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to lesson item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 2.2.80
	 */
	public function get_complete_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = LessonsError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the lesson `complete` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_lesson_complete_permissions_check', $retval, $request );
	}

	/**
	 * Process Lesson completes.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_Error | WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/tutor/v1/lessons/:id/complete Process Tutor LMS Lesson Complete
	 * @apiName        PostTutorLessonComplete
	 * @apiGroup       Tutor Lessons
	 * @apiDescription Process Lesson Completes
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the lesson.
	 */
	public function complete( $request ) {
		$lesson_id = $request->get_param( 'id' );
		$user_id   = get_current_user_id();
		$lesson    = get_post( $lesson_id );

		if ( empty( $lesson ) || $this->post_type !== $lesson->post_type ) {
			return LessonsError::instance()->invalid_lesson_id();
		}

		$validated = apply_filters( 'tutor_validate_lesson_complete', true, $user_id, $lesson_id );

		if ( ! $validated ) {
			return rest_ensure_response( LessonsError::instance()->lesson_complete_validation_error() );
		}

		/**
		 * Fires before the lesson is marked as completed.
		 * This is Tutor LMS hook.
		 *
		 * @param int $lesson_id The lesson ID.
		 *
		 * @since 2.2.80
		 */
		do_action( 'tutor_lesson_completed_before', $lesson_id );

		LessonModel::mark_lesson_complete( $lesson_id );

		/**
		 * Fires after the lesson is marked as completed.
		 * This is Tutor LMS hook.
		 *
		 * @param int $lesson_id The lesson ID.
		 * @param int $user_id The user ID.
		 *
		 * @since 2.2.80
		 */
		do_action( 'tutor_lesson_completed_email_after', $lesson_id, $user_id );

		$update_meta = get_user_meta( $user_id, '_tutor_completed_lesson_id_' . $lesson_id, true );

		if ( $update_meta ) {
			return $this->get_item( $request );
		}

		return LessonsError::instance()->lesson_complete_error();
	}
}
