<?php
/**
 * Holds lesson rest functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Lesson
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Lesson;

use TUTOR\Lesson;
use WP_Comment;
use WP_Error;
use WP_Post;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Lesson rest class.
 */
class LessonsCommentRest extends LessonsRest {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * LessonsCommentRest instance.
	 *
	 * @since 2.2.80
	 * @return LessonsCommentRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/comments',
			array(
				array(
					'methods'             => 'GET',
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_permissions_check' ),
					'args'                => $this->get_comments_collection_params(),
				),
				array(
					'methods'             => 'POST',
					'callback'            => array( $this, 'create' ),
					'permission_callback' => array( $this, 'get_permissions_check' ),
					'args'                => $this->get_comments_collection_params(),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/comments/(?P<comment_id>[\d]+)',
			array(
				array(
					'methods'             => 'GET',
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_permissions_check' ),
					'args'                => $this->get_comments_collection_params(),
				),
			)
		);
	}

	/**
	 * Check if a given request has access to lesson items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the lesson comment `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_lessons_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve Lessons.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/lesson/:id/comments Get TutorLMS Lessons
	 * @apiName        GetTutorLessonComments
	 * @apiGroup       Tutor Lesson Comments
	 * @apiDescription Retrieve Lesson Comments
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [post_id] Lesson id to get the comments.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {Number} [parent] Limit results to those assigned to specific parent.
	 */
	public function get_items( $request ) {
		$registered = $this->get_comments_collection_params();
		$lesson_id  = $request->get_param( 'id' );
		$page       = $request->get_param( 'page' );
		$lesson     = get_post( $lesson_id );
		$per_page   = $request->get_param( 'per_page' );
		$parent     = $request->get_param( 'parent' );

		if ( empty( $lesson ) || $this->post_type !== $lesson->post_type ) {
			return rest_ensure_response( LessonsError::instance()->invalid_lesson_id() );
		}

		/**
		 * Filter the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$request = apply_filters( 'bbapp_tutor_get_lesson_comments_request', $request );

		$args = array(
			'post_id' => $lesson_id,
			'parent'  => $parent ? $parent : $registered['parent']['default'],
			'page'    => $page ? $page : $registered['page']['default'],
			'number'  => $per_page ? $per_page : $registered['per_page']['default'],
		);

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$args = apply_filters( 'bbapp_tutor_get_lesson_comments_args', $args, $request );

		$comments = Lesson::get_comments( $args );

		/**
		 * Fires list of Lesson is fetched via Query.
		 *
		 * @param array            $leesons Fetched lessons.
		 * @param WP_REST_Response $args    Query arguments.
		 * @param WP_REST_Request  $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$lesson_comments = apply_filters( 'bbapp_tutor_get_lesson_comments', $comments, $args, $request );

		$retval = array();

		foreach ( $lesson_comments as $lesson_comment ) {
			if ( ! $this->check_read_permission( $lesson_comment ) ) {
				continue;
			}

			$retval[] = $this->prepare_response_for_collection( $this->prepare_comment_item_for_response( $lesson_comment, $request ) );
		}

		$comment_count_args = array(
			'post_id' => $lesson_id,
			'parent'  => $parent ? $parent : $registered['parent']['default'],
			'count'   => true,
		);
		$response           = rest_ensure_response( $retval );
		$response           = bbapp_tutor_response_add_total_headers( $response, Lesson::get_comments( $comment_count_args ), $args['number'] );

		/**
		 * Fires after a list of lesson response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_lesson_comment_items_response', $response, $request );

		return $response;
	}

	/**
	 * Retrieve Lesson comment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/lesson/:id/comments/:comment_id Get TutorLMS Lesson comment
	 * @apiName        GetTutorLessonComment
	 * @apiGroup       Tutor Lesson Comments
	 * @apiDescription Retrieve Lesson Comment
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the lesson.
	 * @apiParam {Number} comment_id A unique numeric ID for the comment.
	 */
	public function get_item( $request ) {
		$lesson_id  = $request->get_param( 'id' );
		$comment_id = $request->get_param( 'comment_id' );
		$lesson     = get_post( $lesson_id );
		$comment    = get_comment( $comment_id );

		if ( empty( $lesson ) || $this->post_type !== $lesson->post_type ) {
			return rest_ensure_response( LessonsError::instance()->invalid_lesson_id() );
		}

		if ( empty( $comment ) || (int) $lesson_id !== (int) $comment->comment_post_ID ) {
			return rest_ensure_response( LessonsError::instance()->invalid_lesson_comment_id() );
		}

		if ( ! $this->check_read_permission( $comment ) ) {
			return rest_ensure_response( LessonsError::instance()->invalid_lesson_comment_id() );
		}

		$response = $this->prepare_response_for_collection( $this->prepare_comment_item_for_response( $comment, $request ) );

		/**
		 * Fires after a lesson response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_lesson_comment_item_response', $response, $request );

		return $response;
	}

	/**
	 * Process Lesson completes.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_Error | WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/tutor/v1/lessons/:id Process Tutor LMS Lesson Complete
	 * @apiName        PostTutorLessonComplete
	 * @apiGroup       Tutor Lessons
	 * @apiDescription Process Lesson Completes
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the lesson.
	 */
	public function create( $request ) {
		$lesson_id       = ( ! empty( $request->get_param( 'id' ) ) ) ? absint( $request->get_param( 'id' ) ) : 0;
		$lesson          = get_post( $lesson_id );
		$comment_content = $request->get_param( 'content' );
		$comment_parent  = ( ! empty( $request->get_param( 'parent' ) ) ) ? absint( $request->get_param( 'parent' ) ) : 0;

		if ( empty( $lesson ) || $this->post_type !== $lesson->post_type ) {
			return LessonsError::instance()->invalid_lesson_id();
		}

		$comment_data = array(
			'comment_content' => sanitize_textarea_field( $comment_content ),
			'comment_post_ID' => $lesson_id,
			'comment_parent'  => $comment_parent,
		);

		$comment_id = Lesson::create_comment( $comment_data );

		if ( false !== $comment_id ) {
			if ( ! empty( $comment_parent ) ) {
				do_action( 'tutor_reply_lesson_comment_thread', $comment_id, $comment_data );
			} else {
				do_action( 'tutor_new_comment_added', $comment_data );
			}

			// Get the comment.
			$request         = new WP_REST_Request();
			$send_comment_id = ( ! empty( $comment_parent ) ) ? $comment_parent : $comment_id;

			$request->set_method( 'GET' );
			$request->set_route( '/' . $this->namespace . '/' . $this->rest_base . '/' . $lesson_id . '/comments/' . $send_comment_id );
			$request->set_url_params(
				array(
					'id'         => $lesson_id,
					'comment_id' => $send_comment_id,
				)
			);

			return $this->get_item( $request );
		} else {
			return rest_ensure_response( LessonsError::instance()->lesson_comment_error() );
		}
	}

	/**
	 * Prepare a single comment output for response.
	 *
	 * @param WP_Comment      $comment Comment object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since [BBAAPVERSION]
	 * @return WP_REST_Response $data
	 */
	public function prepare_comment_item_for_response( $comment, $request ) {
		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';

		// Base fields for every post.
		$data = array(
			'id'            => $comment->comment_ID,
			'post_id'       => $comment->comment_post_ID,
			'author'        => $comment->comment_author,
			'author_email'  => $comment->comment_author_email,
			'author_url'    => $comment->comment_author_url,
			'author_avatar' => get_avatar_url( $comment->user_id ),
			'date'          => mysql_to_rfc3339( $comment->comment_date ),
			'date_gmt'      => mysql_to_rfc3339( $comment->comment_date_gmt ),
			'content'       => array(
				'raw'      => bbapp_tutor_fix_relative_urls_protocol( $comment->comment_content ),
				'rendered' => bbapp_tutor_fix_relative_urls_protocol( apply_filters( 'the_content', $comment->comment_content ) ),
			),
			'approved'      => (int) $comment->comment_approved,
			'agent'         => $comment->comment_agent,
			'type'          => $comment->comment_type,
			'parent'        => $comment->comment_parent,
			'user_id'       => $comment->user_id,
			'replies'       => $this->get_comment_replies( $comment, $request ),
		);

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filters lesson response.
		 *
		 * @param WP_REST_Response $response Lessons response.
		 * @param WP_Post          $comment  Post object.
		 * @param WP_REST_Request  $request  Request object.
		 */
		return apply_filters( 'bbapp_tutor_rest_prepare_lesson_comment', $response, $comment, $request );
	}

	/**
	 * Get the comment query params for collections of attachments.
	 *
	 * @since [BBAAPVERSION]
	 * @return array
	 */
	public function get_comments_collection_params() {
		$params = parent::get_collection_params();

		$params['id'] = array(
			'description'       => __( 'Lesson post id.', 'buddyboss-app' ),
			'type'              => 'integer',
			'validate_callback' => 'rest_validate_request_arg',
			'required'          => true,
		);

		$params['parent'] = array(
			'description'       => __( 'Parent comment id.', 'buddyboss-app' ),
			'type'              => 'integer',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['content'] = array(
			'description'       => __( 'Comment content.', 'buddyboss-app' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * Get comment replies.
	 *
	 * @param WP_Comment      $comment Comment object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since [BBAAPVERSION]
	 * @return array
	 */
	public function get_comment_replies( $comment, $request ) {
		$retval  = array();
		$replies = Lesson::get_comments(
			array(
				'post_id' => $comment->comment_post_ID,
				'parent'  => $comment->comment_ID,
			)
		);

		if ( ! empty( $replies ) ) {
			foreach ( $replies as $reply ) {
				$retval[] = $this->prepare_response_for_collection( $this->prepare_comment_item_for_response( $reply, $request ) );
			}
		}

		return $retval;
	}
}
