<?php
/**
 * Holds Questions rest functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Course
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Course\QuestionAnswer;

use BuddyBossApp\Api\TutorLMS\V1\Course\CoursesError;
use BuddyBossApp\Api\TutorLMS\V1\Course\CoursesRest;
use TUTOR\Q_and_A;
use WP_Comment;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Questions rest class.
 */
class QuestionsRest extends CoursesRest {
	/**
	 * Class isntance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Rest questions base.
	 *
	 * @var string
	 */
	protected string $rest_questions_base;

	/**
	 * QuestionsRest instance.
	 *
	 * @since 2.2.80
	 *
	 * @return QuestionsRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 2.2.80
	 */
	public function __construct() {
		parent::__construct();
		$this->rest_questions_base = 'questions';
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/' . $this->rest_questions_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'save_item' ),
					'permission_callback' => array( $this, 'save_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/' . $this->rest_questions_base . '/(?P<question_id>[\d]+)',
			array(
				'args'   => array(
					'question_id' => array(
						'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Prepare links.
	 *
	 * @param array $data Data.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function prepare_links( $data ) {
		$links = parent::prepare_links( $data );

		$links['self'] = array(
			'href' => rest_url( sprintf( '%s/%s/%s/%s/%d', $this->namespace, $this->rest_base, $data['course_id'], $this->rest_questions_base, $data['id'] ) ),
		);

		return $links;
	}

	/**
	 * Get the collection parameters.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = array(
			'page'     => array(
				'description'       => __( 'Current page of the collection.', 'buddyboss-app' ),
				'type'              => 'integer',
				'default'           => 1,
				'sanitize_callback' => 'absint',
			),
			'per_page' => array(
				'description'       => __( 'Maximum number of items to be returned in result set.', 'buddyboss-app' ),
				'type'              => 'integer',
				'default'           => 10,
				'sanitize_callback' => 'absint',
			),
		);

		$params['with_answers'] = array(
			'description' => __( 'Include answers.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'default'     => false,
		);

		/**
		 * Filter the collection parameters for the questions and answers.
		 *
		 * @param array $params Collection parameters.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_question_get_collection_params', $params );
	}

	/**
	 * Get the item schema.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function get_public_item_schema() {
		return array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'question',
			'type'       => 'object',
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'course_id'      => array(
					'description' => __( 'Course ID.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'content'        => array(
					'description' => __( 'Question content.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'user_id'        => array(
					'description' => __( 'User ID.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'display_name'   => array(
					'description' => __( 'Display name.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'user_email'     => array(
					'description' => __( 'User email.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'user_avatar'    => array(
					'description' => __( 'User avatar.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'answer_count'   => array(
					'description' => __( 'Answer count.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'created_at'     => array(
					'description' => __( 'Created at.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'created_at_gmt' => array(
					'description' => __( 'Created at GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'meta'           => array(
					'description' => __( 'Meta.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
				),
				'answers'        => array(
					'description' => __( 'Answers.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit' ),
				),
			),
		);
	}

	/**
	 * Get items.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function get_items( $request ) {
		$course_id = $request->get_param( 'id' );
		$args      = array();

		if ( get_post_type( $course_id ) !== $this->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		$registered = $this->get_collection_params();

		/**
		 * Filter the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$request = apply_filters( 'bbapp_tutor_get_course_question_items_request', $request );

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'page'     => 'paged',
			'per_page' => 'per_page',
		);

		/**
		 * For each known parameter which is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		$args      = $this->prepare_items_query( $args, $request );
		$questions = $this->get_questions( $course_id, $args );

		$retval = array();
		foreach ( $questions as $question ) {
			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $question, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		/**
		 * Fires after a list of Course Question and Answer items is fetched via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_course_question_items_response', $response, $request );

		return $response;
	}

	/**
	 * Save item.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function save_item( $request ) {
		$course_id = $request->get_param( 'id' );
		$question  = $request->get_param( 'question' );
		$user_id   = get_current_user_id();

		if ( ! is_user_logged_in() ) {
			return CoursesError::instance()->course_login_required();
		}

		if ( get_post_type( $course_id ) !== $this->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		if ( empty( $question ) ) {
			return QuestionsError::instance()->empty_question_not_allowed();
		}

		if ( ! Q_and_A::has_qna_access( $user_id, $course_id ) ) {
			return QuestionsError::instance()->access_denied();
		}

		$user = get_user_by( 'id', $user_id );
		$date = gmdate( 'Y-m-d H:i:s', tutor_time() );

		// Insert data prepare.
		$data = apply_filters(
			'tutor_qna_insert_data',
			array(
				'comment_post_ID'  => $course_id,
				'comment_author'   => $user->user_login,
				'comment_date'     => $date,
				'comment_date_gmt' => get_gmt_from_date( $date ),
				'comment_content'  => $question,
				'comment_approved' => 'approved',
				'comment_agent'    => 'TutorLMSPlugin',
				'comment_type'     => 'tutor_q_and_a',
				'user_id'          => $user_id,
			)
		);

		$question_id = $this->save_question( $data );

		if ( is_wp_error( $question_id ) ) {
			return $question_id;
		}

		$response = $this->prepare_item_for_response( tutor_utils()->get_qa_question( $question_id ), $request );

		return rest_ensure_response( $response );
	}

	/**
	 * Save question.
	 *
	 * @param array $data Data.
	 *
	 * @since 2.2.80
	 *
	 * @return int|WP_Error
	 */
	public function save_question( $data ) {
		global $wpdb;

		// Insert new question/answer.
		$wpdb->insert( $wpdb->comments, $data ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery

		$question_id = $wpdb->insert_id;

		if ( ! $question_id ) {
			return QuestionsError::instance()->error_saving_item();
		}

		// Get the user id who asked the question.
		$asker_id = $this->get_asker_id( $question_id );
		update_comment_meta( $question_id, 'tutor_qna_read' . ( get_current_user_id() === $asker_id ? '' : '_' . $asker_id ), 0 );
		do_action( 'tutor_after_asked_question', $data );

		return $question_id;
	}

	/**
	 * Get item.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function get_item( $request ) {
		$course_id   = $request->get_param( 'id' );
		$question_id = $request->get_param( 'question_id' );

		if ( get_post_type( $course_id ) !== $this->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		$question = tutor_utils()->get_qa_question( $question_id );

		if ( ! $question ) {
			return QuestionsError::instance()->invalid_question_id();
		}

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $question, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a single Course Question and Answer item is fetched via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_course_question_item_response', $response, $request );

		return $response;
	}

	/**
	 * Prepare item for response.
	 *
	 * @param object|WP_Comment $question Question object.
	 * @param WP_REST_Request   $request  Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response
	 */
	public function prepare_item_for_response( $question, $request ) {
		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = array(
			'id'               => $question->comment_ID,
			'course_id'        => $question->course_id,
			'question_content' => $question->comment_content,
			'user_id'          => $question->user_id,
			'display_name'     => $question->display_name,
			'user_email'       => $question->user_email,
			'user_avatar'      => get_avatar_url( $question->user_id ),
			'answer_count'     => $question->answer_count,
			'created_at'       => $question->comment_date,
			'created_at_gmt'   => $question->comment_date_gmt,
			'meta'             => $question->meta,
		);

		if ( ! empty( $request->get_param( 'with_answers' ) ) ) {
			$request->set_param( 'question_id', $question->comment_ID );
			$answers_response = AnswersRest::instance()->get_items( $request );
			$data['answers']  = $answers_response->get_data();
		}

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		/**
		 * Filter the question and answer item returned from the API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param object          $question  The original question and answer object.
		 * @param WP_REST_Request  $request   Request used to generate the response.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_prepare_course_question_item_for_response', $response, $question, $request );
	}

	/**
	 * Get questions.
	 *
	 * @param int   $course_id Course ID.
	 * @param array $args      Args.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function get_questions( $course_id, $args ) {
		$offset    = (int) ( $args['per_page'] * $args['paged'] ) - $args['per_page'];
		$per_page  = $args['per_page'];
		$questions = tutor_utils()->get_qa_questions( $offset, $per_page, '', null, null, null, null, false, array( 'course_id' => $course_id ) );

		return $questions;
	}

	/**
	 * Check if a given request has access to get items.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return bool
	 */
	public function get_items_permissions_check( $request ) {
		$course_id = $request->get_param( 'id' );
		$retval    = $this->get_permissions_check( $course_id );

		/**
		 * Filter the permissions check for getting Course Question and Answer items via the REST API.
		 *
		 * @param bool            $retval  Whether the current user can get Course Question and Answer items.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_question_get_items_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to save item.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return bool
	 */
	public function save_item_permissions_check( $request ) {
		$course_id = $request->get_param( 'id' );
		$retval    = $this->get_permissions_check( $course_id );

		/**
		 * Filter the permissions check for saving Course Question and Answer items via the REST API.
		 *
		 * @param bool            $retval  Whether the current user can save Course Question and Answer items.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_question_save_item_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to get item.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return bool
	 */
	public function get_item_permissions_check( $request ) {
		$course_id = $request->get_param( 'id' );
		$retval    = $this->get_permissions_check( $course_id );

		/**
		 * Filter the permissions check for getting Course Question and Answer item via the REST API.
		 *
		 * @param bool            $retval  Whether the current user can get Course Question and Answer item.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_question_get_item_permissions_check', $retval, $request );
	}

	/**
	 * Get permissions check.
	 *
	 * @param int $course_id Course ID.
	 *
	 * @since 2.2.80
	 *
	 * @return bool
	 */
	public function get_permissions_check( $course_id ) {
		return true;
	}

	/**
	 * Get user id who asked
	 *
	 * @param int $question_id question id.
	 *
	 * @since 2.2.80
	 * @return string author id
	 */
	protected function get_asker_id( $question_id ) {
		global $wpdb;

		$author_id = $wpdb->get_var( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT user_id
					FROM {$wpdb->comments}
					WHERE comment_ID = %d
				",
				$question_id
			)
		);

		return $author_id;
	}
}
