<?php
/**
 * Holds AnswersRest rest functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Course\QuestionAnswer
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Course\QuestionAnswer;

use BuddyBossApp\Api\TutorLMS\V1\Course\CoursesError;
use TUTOR\Q_and_A;
use WP_Comment;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * AnswersRest class.
 */
class AnswersRest extends QuestionsRest {
	/**
	 * Class isntance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * AnswersRest instance.
	 *
	 * @since 2.2.80
	 *
	 * @return AnswersRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/' . $this->rest_questions_base . '/(?P<question_id>[\d]+)/answers',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'save_item' ),
					'permission_callback' => array( $this, 'save_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Get the collection parameters.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = array(
			'page'     => array(
				'description'       => __( 'Current page of the collection.', 'buddyboss-app' ),
				'type'              => 'integer',
				'default'           => 1,
				'sanitize_callback' => 'absint',
			),
			'per_page' => array(
				'description'       => __( 'Maximum number of items to be returned in result set.', 'buddyboss-app' ),
				'type'              => 'integer',
				'default'           => 10,
				'sanitize_callback' => 'absint',
			),
		);

		/**
		 * Filter the collection parameters for the questions and answers.
		 *
		 * @param array $params Collection parameters.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_answers_get_collection_params', $params );
	}

	/**
	 * Get the item schema.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function get_public_item_schema() {
		return array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'answers',
			'type'       => 'object',
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'course_id'      => array(
					'description' => __( 'Course ID.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'content'        => array(
					'description' => __( 'Content of the question.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'user_id'        => array(
					'description' => __( 'User ID.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'display_name'   => array(
					'description' => __( 'Display name.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'user_email'     => array(
					'description' => __( 'User email.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'user_avatar'    => array(
					'description' => __( 'User avatar.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'answer_count'   => array(
					'description' => __( 'Answer count.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'created_at'     => array(
					'description' => __( 'Created at.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'created_at_gmt' => array(
					'description' => __( 'Created at GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'meta'           => array(
					'description' => __( 'Meta.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
			),
		);
	}

	/**
	 * Get items.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function get_items( $request ) {
		$question_id = $request->get_param( 'question_id' );
		$question    = tutor_utils()->get_qa_question( $question_id );

		if ( ! $question ) {
			return QuestionsError::instance()->invalid_question_id();
		}

		$registered = $this->get_collection_params();

		/**
		 * Filter the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$request = apply_filters( 'bbapp_tutor_get_course_answers_items_request', $request );

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'page'     => 'paged',
			'per_page' => 'per_page',
		);

		/**
		 * For each known parameter that is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		$answers = tutor_utils()->get_qa_answer_by_question( $question_id );
		$retval  = array();

		foreach ( $answers as $answer ) {
			if ( 0 === (int) $answer->comment_parent ) {
				continue;
			}

			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $answer, $request )
			);
		}

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a collection of Course Question and Answer items is fetched via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_course_answers_items_response', $response, $request );

		return $response;
	}

	/**
	 * Save item.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function save_item( $request ) {
		$course_id   = $request->get_param( 'id' );
		$question_id = $request->get_param( 'question_id' );
		$answer      = $request->get_param( 'answer' );
		$user_id     = get_current_user_id();

		if ( ! is_user_logged_in() ) {
			return CoursesError::instance()->course_login_required();
		}

		if ( get_post_type( $course_id ) !== $this->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		if ( ! Q_and_A::has_qna_access( $user_id, $course_id ) ) {
			return QuestionsError::instance()->access_denied();
		}

		if ( empty( $answer ) ) {
			return QuestionsError::instance()->empty_answer_not_allowed();
		}

		$user = get_user_by( 'id', $user_id );
		$date = gmdate( 'Y-m-d H:i:s', tutor_time() );

		// Insert data prepare.
		$data = apply_filters(
			'tutor_qna_insert_data',
			array(
				'comment_post_ID'  => $course_id,
				'comment_author'   => $user->user_login,
				'comment_date'     => $date,
				'comment_date_gmt' => get_gmt_from_date( $date ),
				'comment_content'  => $answer,
				'comment_approved' => 'approved',
				'comment_agent'    => 'TutorLMSPlugin',
				'comment_type'     => 'tutor_q_and_a',
				'comment_parent'   => $question_id,
				'user_id'          => $user_id,
			)
		);

		$answer_id = $this->save_answer( $data, $request );

		if ( is_wp_error( $answer_id ) ) {
			return $answer_id;
		}

		$answer   = tutor_utils()->get_qa_answer_by_question( $answer_id );
		$answer   = $answer[0];
		$response = $this->prepare_item_for_response( $answer, $request );

		return rest_ensure_response( $response );
	}

	/**
	 * Save question.
	 *
	 * @param array           $data    Data.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return int|WP_Error
	 */
	public function save_answer( $data, $request ) {
		global $wpdb;

		// Insert new question/answer.
		$wpdb->insert( $wpdb->comments, $data ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery

		$answer_id = $wpdb->insert_id;

		if ( ! $answer_id ) {
			return QuestionsError::instance()->error_saving_item();
		}

		// Get the user id who asked the question.
		// Get the user id who asked the question.
		$asker_id = $this->get_asker_id( $answer_id );
		update_comment_meta( $answer_id, 'tutor_qna_read' . ( get_current_user_id() === $asker_id ? '' : '_' . $asker_id ), 0 );

		do_action( 'tutor_after_asked_question', $data );

		if ( ! empty( $request['question_id'] ) ) {
			if ( 0 !== $request['question_id'] && ( current_user_can( 'administrator' ) || tutor_utils()->is_instructor_of_this_course(
				get_current_user_id(),
				$request['id']
			) ) ) {
				do_action( 'tutor_after_answer_to_question', $answer_id );
			}
		}

		return $answer_id;
	}


	/**
	 * Prepare item for response.
	 *
	 * @param object|WP_Comment $answer  Answer object.
	 * @param WP_REST_Request   $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response
	 */
	public function prepare_item_for_response( $answer, $request ) {
		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = array(
			'id'             => $answer->comment_ID,
			'question_id'    => $answer->comment_parent,
			'course_id'      => $answer->comment_post_ID,
			'answer_content' => $answer->comment_content,
			'user_id'        => $answer->user_id,
			'display_name'   => $answer->display_name,
			'user_email'     => get_the_author_meta( 'user_email', $answer->user_id ),
			'user_avatar'    => get_avatar_url( $answer->user_id ),
			'created_at'     => $answer->comment_date,
			'created_at_gmt' => $answer->comment_date_gmt,
		);

		$comment_meta = array();
		$metas        = get_comment_meta( $answer->comment_ID );

		if ( ! empty( $metas ) ) {
			foreach ( $metas as $key => $value ) {
				$comment_meta[ $key ] = maybe_unserialize( $value[0] );
			}
		}

		$data['meta'] = $comment_meta;
		$data         = $this->add_additional_fields_to_object( $data, $request );
		$data         = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		/**
		 * Filter the Course Question and Answer item returned from the API.
		 *
		 * @param WP_REST_Response $response The response object.
		 * @param object           $answer   Answer object.
		 * @param WP_REST_Request  $request  Request object.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_course_answer_item_response', $response, $answer, $request );
	}

	/**
	 * Check if a given request has access to get items.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return bool
	 */
	public function get_items_permissions_check( $request ) {
		$course_id = $request->get_param( 'id' );

		$retval = $this->get_permissions_check( $course_id );

		/**
		 * Filter the permissions check.
		 *
		 * @param bool            $retval  Whether the request has permission to get items.
		 * @param WP_REST_Request $request Request object.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_answers_get_items_permissions_check', $retval, $request );
	}

	/**
	 * Get permissions check.
	 *
	 * @param int $course_id Course ID.
	 *
	 * @since 2.2.80
	 *
	 * @return bool
	 */
	public function get_permissions_check( $course_id ) {
		return true;
	}
}
