<?php
/**
 * Holds TutorLMS course rest functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Course
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Course;

use BuddyBossApp\Api\TutorLMS\V1\Core\TutorRestController;
use TUTOR\Course_List;
use Tutor\Models\QuizModel;
use TUTOR_ASSIGNMENTS\Assignments;
use TUTOR_CERT\Certificate;
use TUTOR_ZOOM\Zoom;
use WP_Error;
use WP_Post;
use WP_Query;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use TUTOR_ENROLLMENTS\Enrollment_Expiry as Enroll_Expiry_A;
use TUTOR_PRO\Enrollment_Expiry as Enroll_Expiry_B;

/**
 * TutorLMS course rest class.
 */
class CoursesRest extends TutorRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Course post-type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'courses';

	/**
	 * Allow batch.
	 *
	 * @var true[] $allow_batch
	 */
	protected $allow_batch = array( 'v1' => true );

	/**
	 * CoursesRest instance.
	 *
	 * @since 2.2.80
	 * @return CoursesRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 2.2.80
	 * @return void
	 */
	public function __construct() {
		$this->rest_base = 'courses';

		parent::__construct();
	}

	/**
	 * Check if a given request has access to course items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the course `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_courses_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to course item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_item_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the course `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_course_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to course item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_item_contents_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the course `get_item_contents` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_item_contents_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'allow_batch' => $this->allow_batch,
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'single' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/content',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item_contents' ),
					'permission_callback' => array( $this, 'get_item_contents_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve Courses.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since         2.2.80
	 *
	 * @return WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/courses Get TutorLMS Courses
	 * @apiName        GetTutorCourses
	 * @apiGroup       Tutor Courses
	 * @apiDescription Retrieve Courses
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in a result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure a result set includes specific IDs.
	 * @apiParam {String} [after] Limit results to those published after a given ISO8601 compliant date.
	 * @apiParam {String} [before] Limit results to those published before a given ISO8601 compliant date.
	 * @apiParam {Array} [author] Limit results to those assigned to specific authors.
	 * @apiParam {Array} [author_exclude] Ensure results to exclude those assigned to specific authors.
	 * @apiParam {String=asc,desc} [order=desc] Sort result set by given order.
	 * @apiParam {String=date,id,title,menu_order} [orderby=date] Sort result set by given field.
	 * @apiParam {Array} [categories] Limit results to those assigned to specific categories.
	 * @apiParam {Number=0,1} [mycourses] Limit results to current user courses.
	 */
	public function get_items( $request ) {
		$registered = $this->get_collection_params();

		/**
		 * Filter the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$request = apply_filters( 'bbapp_tutor_get_courses_request', $request );

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'author'         => 'author__in',
			'author_exclude' => 'author__not_in',
			'exclude'        => 'post__not_in',
			'include'        => 'post__in',
			'offset'         => 'offset',
			'order'          => 'order',
			'orderby'        => 'orderby',
			'page'           => 'paged',
			'parent'         => 'post_parent__in',
			'parent_exclude' => 'post_parent__not_in',
			'search'         => 's',
			'slug'           => 'post_name__in',
			'status'         => 'post_status',
			'mycourses'      => 'mycourses',
			'group_id'       => 'group_id',
			'per_page'       => 'posts_per_page',
		);

		/**
		 * For each known parameter that is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		$args     = $this->prepare_items_query( $args, $request );
		$relation = ( ! empty( $request['tax_relation'] ) ) ? $request['tax_relation'] : 'OR';

		if ( ! empty( $relation ) ) {
			$args['tax_query'] = array( 'relation' => $relation ); //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
		}

		// Tutor Course Categories.
		$allowed_categories = array( 'tutor-course-filter-category', 'categories' );

		foreach ( $allowed_categories as $category ) {
			if ( ! empty( $request[ $category ] ) ) {
				if ( ! is_array( $request[ $category ] ) ) {
					$request[ $category ] = (array) $request[ $category ];
				}

				if ( in_array( 0, $request[ $category ], true ) ) {
					$args['tax_query'][] = array(
						'taxonomy' => 'course-category',
						'operator' => 'NOT EXISTS',
					);
				}

				$request['categories'] = array_filter( $request[ $category ] );

				if ( ! empty( $request[ $category ] ) ) {
					$args['tax_query'][] = array(
						'taxonomy'         => 'course-category',
						'field'            => 'term_id',
						'terms'            => $request['categories'],
						'include_children' => false,
					);
				}

				break;
			}
		}

		// Tutor Course Tags.
		$allowed_tags = array( 'tutor-course-filter-tag', 'tags' );

		foreach ( $allowed_tags as $tag ) {
			if ( ! empty( $request[ $tag ] ) ) {
				$args['tax_query'][] = array(
					'taxonomy'         => 'course-tag',
					'field'            => 'term_id',
					'terms'            => $request[ $tag ],
					'include_children' => false,
				);

				break;
			}
		}

		// Filter by my courses only.
		if ( isset( $args['mycourses'] ) && $args['mycourses'] ) {
			$mycourse_ids = tutor_utils()->get_enrolled_courses_ids_by_user( get_current_user_id() );

			if ( is_array( $mycourse_ids ) && ! empty( $mycourse_ids ) && ! is_wp_error( $mycourse_ids ) ) {
				$args['post__in'] = ! empty( $args['post__in'] ) ? array_intersect( $mycourse_ids, $args['post__in'] ) : $mycourse_ids;
			}

			/*
			 * If we intersected, but there are no post ids in common,
			 * WP_Query won't return "no posts" for post__in = array()
			 * so we have to fake it a bit.
			*/
			if ( ! $args['post__in'] ) {
				$args['post__in'] = array( 0 );
			}

			unset( $args['mycourses'] );
		}

		// Prepare course order.
		if ( empty( $args['orderby'] ) || empty( $args['order'] ) || ! empty( $request['course_order'] ) ) {
			$course_order = ! empty( $request['course_order'] ) ? $request['course_order'] : 'newest_first';

			switch ( $course_order ) {
				case 'newest_first':
					$args['orderby'] = 'post_date';
					$args['order']   = 'DESC';
					break;
				case 'oldest_first':
					$args['orderby'] = 'post_date';
					$args['order']   = 'ASC';
					break;
				case 'course_title_az':
					$args['orderby'] = 'post_title';
					$args['order']   = 'ASC';
					break;
				case 'course_title_za':
					$args['orderby'] = 'post_title';
					$args['order']   = 'DESC';
					break;
			}
		}

		// Prepare level and price type.
		$is_membership = 'pmpro' === get_tutor_option( 'monetize_by' ) && tutils()->has_pmpro();
		$level_price   = array();

		foreach ( array( 'level', 'price' ) as $type ) {
			if ( $is_membership && 'price' === $type ) {
				continue;
			}

			$type_array = tutils()->array_get( 'tutor-course-filter-' . $type, $request->get_params(), array() );
			$type_array = array_map( 'sanitize_text_field', ( is_array( $type_array ) ? $type_array : array( $type_array ) ) );

			if ( 'level' === $type && in_array( 'all_levels', $type_array, true ) ) {
				continue;
			}

			if ( count( $type_array ) > 0 ) {
				$level_price[] = array(
					'key'     => 'level' === $type ? '_tutor_course_level' : '_tutor_course_price_type',
					'value'   => $type_array,
					'compare' => 'IN',
				);
			}
		}

		count( $level_price ) ? $args['meta_query'] = $level_price : 0; //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$args = apply_filters( 'bbapp_tutor_get_courses_args', $args, $request );

		$args['post_type']      = $this->post_type;
		$posts_query            = new WP_Query();
		$courses['posts']       = $posts_query->query( $args );
		$courses['total_posts'] = $posts_query->found_posts;

		// Todo::dips Check again this code needed or not.
		if ( $courses['total_posts'] < 1 ) {
			// Out-of-bounds, run the query again without LIMIT for total count.
			unset( $args['paged'] );
			$count_query = new WP_Query();
			$count_query->query( $args );
			$courses['total_posts'] = $count_query->found_posts;
		}

		/**
		 * Fires list of Courses is fetched via Query.
		 *
		 * @param array            $courses Fetched courses.
		 * @param WP_REST_Response $args    Query arguments.
		 * @param WP_REST_Request  $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$courses = apply_filters( 'bbapp_tutor_get_courses', $courses, $args, $request );

		$retval = array();

		foreach ( $courses['posts'] as $couese ) {
			if ( ! $this->check_read_permission( $couese ) ) {
				continue;
			}

			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $couese, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		$response = bbapp_tutor_response_add_total_headers( $response, $courses['total_posts'], $args['posts_per_page'] );

		/**
		 * Fires after a list of Courses response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_course_items_response', $response, $request );

		return $response;
	}

	/**
	 * Retrieve Course.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/courses/:id Get TutorLMS Course
	 * @apiName        GetTutorCourse
	 * @apiGroup       Tutor Courses
	 * @apiDescription Retrieve single Course
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function get_item( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course    = get_post( $course_id );

		if ( empty( $course ) || $this->post_type !== $course->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		/**
		 * Fire after Course is fetched via Query.
		 *
		 * @param array   $course    Fetched course.
		 * @param integer $course_id Course id.
		 *
		 * @since 2.2.80
		 */
		$course = apply_filters( 'bbapp_tutor_get_course', $course, $course_id );

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $course, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a course response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_course_item_response', $response, $request );

		return $response;
	}

	/**
	 * Retrieve Course details like lesson/topics and quiz.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_Error | WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/courses/:id/contents Get TutorLMS Course Contents
	 * @apiName        GetTutorCourse
	 * @apiGroup       Tutor Courses
	 * @apiDescription Retrieve a single Course
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function get_item_contents( $request ) {
		$course_id       = is_numeric( $request ) ? $request : (int) $request['id'];
		$course          = get_post( $course_id );
		$GLOBALS['post'] = $course; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited --Set global post data.

		setup_postdata( $course ); // Set post data.
		if ( empty( $course ) || $this->post_type !== $course->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		$retval = array();
		// Get Course Topics.
		$retval_topics = array();
		$topics        = tutor_utils()->get_topics();

		if ( $topics->have_posts() ) {
			while ( $topics->have_posts() ) {
				$topics->the_post();
				$topic_id     = get_the_ID();
				$retval_topic = array(
					'id'      => $topic_id,
					'title'   => get_the_title( $topic_id ),
					'summary' => get_the_content(),
				);

				$topic_contents = tutor_utils()->get_course_contents_by_topic( $topic_id, - 1 );

				if ( $topic_contents->have_posts() ) {
					while ( $topic_contents->have_posts() ) {
						$topic_contents->the_post();
						$topic_content_id = get_the_ID(); // it could be lesson or quiz.

						$is_preview       = get_post_meta( $topic_content_id, '_is_preview', true );
						$is_public_course = Course_List::is_public( $course_id );
						$is_locked        = ! ( $this->is_enrolled( $course ) || $is_preview || $is_public_course );

						$topic_children            = new \stdClass();
						$topic_children->id        = $topic_content_id;
						$topic_children->title     = get_the_title( $topic_content_id );
						$topic_children->type      = get_post_type( $topic_content_id );
						$topic_children->is_locked = $is_locked;

						if ( 'tutor_quiz' === get_post_type( $topic_content_id ) ) {
							$topic_children = $this->get_tutor_quiz_data( $topic_children );
						} elseif ( 'tutor_assignments' === get_post_type( $topic_content_id ) ) {
							$topic_children = $this->get_tutor_assignment_data( $topic_children );
						} elseif ( 'tutor_zoom_meeting' === get_post_type( $topic_content_id ) ) {
							$topic_children = $this->get_tutor_zoom_meeting_data( $topic_children );
						} elseif ( 'lesson' === get_post_type( $topic_content_id ) ) {
							$topic_children = $this->get_tutor_lesson_data( $topic_children );
						}

						$retval_topic['children'][] = $topic_children;
					}
				}

				$retval_topics[] = $retval_topic;
			}

			/**
			 * Filters Course Topic list.
			 *
			 * @param array           $ret_val_lesson Return list.
			 * @param WP_REST_Request $request        Full details about the request.
			 *
			 * @since 2.2.80
			 */
			$retval['topics'] = apply_filters( 'bbapp_tutor_rest_course_content_list', $retval_topics, $request );
		}

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a course response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_course_item_contents_response', $response, $request );

		return $response;
	}

	/**
	 * Prepare a single post-output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$GLOBALS['post'] = $post; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited

		setup_postdata( $post );

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$schema  = $this->get_public_item_schema();

		/**
		 * Create Short Content from Content without more link.
		 */
		add_filter( 'excerpt_more', array( $this, 'remove_excerpt_more_link_and_add_dots' ) );
		$short_content = wp_trim_excerpt( '', $post );
		remove_filter( 'excerpt_more', array( $this, 'remove_excerpt_more_link_and_add_dots' ) );

		// Base fields for every post.
		$data = array(
			'id'           => $post->ID,
			'title'        => array(
				'raw'      => $post->post_title,
				'rendered' => get_the_title( $post->ID ),
			),
			'content'      => array(
				'raw'      => bbapp_tutor_fix_relative_urls_protocol( $post->post_content ),
				'rendered' => bbapp_tutor_fix_relative_urls_protocol( apply_filters( 'the_content', $post->post_content ) ),
				'short'    => bbapp_tutor_fix_relative_urls_protocol( $short_content ),
			),
			'date'         => mysql_to_rfc3339( $post->post_date ),
			'date_gmt'     => mysql_to_rfc3339( $post->post_date_gmt ),
			'modified'     => mysql_to_rfc3339( $post->post_modified ),
			'modified_gmt' => mysql_to_rfc3339( $post->post_modified_gmt ),
			'link'         => get_permalink( $post->ID ),
			'slug'         => $post->post_name,
			'author'       => (int) $post->post_author,
			'excerpt'      => array(
				'raw'      => bbapp_tutor_fix_relative_urls_protocol( $post->post_excerpt ),
				'rendered' => bbapp_tutor_fix_relative_urls_protocol( apply_filters( 'the_excerpt', $post->post_excerpt ) ),
			),
			'menu_order'   => (int) $post->menu_order,
		);

		/**
		 * Feature Media
		 */
		$post->featured_media            = $this->get_feature_media( $post );
		$data['featured_media']          = array();
		$data['featured_media']['small'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['small'] ) ) ? $post->featured_media['small'] : null;
		$data['featured_media']['large'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['large'] ) ) ? $post->featured_media['large'] : null;

		if ( ! empty( $schema['properties']['categories'] ) && in_array( $context, $schema['properties']['categories']['context'], true ) ) {
			$post->categories   = $this->get_categories( $post );
			$data['categories'] = $post->categories;
		}

		if ( ! empty( $schema['properties']['tags'] ) && in_array( $context, $schema['properties']['tags']['context'], true ) ) {
			$post->tags   = $this->get_tags( $post );
			$data['tags'] = $post->tags;
		}

		if ( ! empty( $schema['properties']['is_public'] ) && in_array( $context, $schema['properties']['is_public']['context'], true ) ) {
			$post->is_public   = $this->is_public( $post );
			$data['is_public'] = (bool) $post->is_public;
		}

		if ( ! empty( $schema['properties']['purchasable'] ) && in_array( $context, $schema['properties']['purchasable']['context'], true ) ) {
			$post->purchasable   = $this->is_purchasable( $post );
			$data['purchasable'] = (bool) $post->purchasable;
		}

		if ( ! empty( $schema['properties']['price'] ) && in_array( $context, $schema['properties']['price']['context'], true ) ) {
			$post->price = $this->get_price( $post );

			if ( ! empty( $post->price ) ) {
				$data['price'] = $post->price;
			}
		}

		if ( ! empty( $schema['properties']['course_full'] ) && in_array( $context, $schema['properties']['course_full']['context'], true ) ) {
			$post->course_full   = $this->get_course_full( $post );
			$data['course_full'] = $post->course_full;
		}

		if ( ! empty( $schema['properties']['buttons'] ) && in_array( $context, $schema['properties']['buttons']['context'], true ) ) {
			$post->buttons   = function_exists( 'tutor_entry_box_buttons' ) ? tutor_entry_box_buttons( $post->ID, get_current_user_id() ) : array();
			$data['buttons'] = $post->buttons;
		}

		if ( ! empty( $schema['properties']['is_wishlisted'] ) && in_array( $context, $schema['properties']['is_wishlisted']['context'], true ) ) {
			$post->is_wishlisted   = $this->is_wishlisted( $post );
			$data['is_wishlisted'] = $post->is_wishlisted;
		}

		$post->has_course_access = $this->get_has_course_access( $post );
		if ( ! empty( $schema['properties']['has_course_access'] ) && in_array( $context, $schema['properties']['has_course_access']['context'], true ) ) {
			$data['has_course_access'] = (bool) $post->has_course_access;
		}

		if ( ! empty( $schema['properties']['has_content_access'] ) && in_array( $context, $schema['properties']['has_content_access']['context'], true ) ) {
			$post->has_content_access   = $this->get_has_content_access( $post );
			$data['has_content_access'] = (bool) $post->has_content_access;
		}

		if ( ! empty( $schema['properties']['progression'] ) && in_array( $context, $schema['properties']['progression']['context'], true ) ) {
			$post->progression   = $this->get_progression( $post );
			$data['progression'] = $post->progression;
		}

		if ( ! empty( $schema['properties']['duration'] ) && in_array( $context, $schema['properties']['duration']['context'], true ) ) {
			$post->duration   = $this->get_duration( $post );
			$data['duration'] = $post->duration;
		}

		if ( ! empty( $schema['properties']['ratings'] ) && in_array( $context, $schema['properties']['ratings']['context'], true ) ) {
			$post->ratings   = $this->get_ratings( $post );
			$data['ratings'] = $post->ratings;
		}

		if ( ! empty( $schema['properties']['ratings_settings'] ) && in_array( $context, $schema['properties']['ratings_settings']['context'], true ) ) {
			$post->ratings_settings   = $this->get_ratings_settings( $post );
			$data['ratings_settings'] = $post->ratings_settings;
		}

		if ( ! empty( $schema['properties']['author_data'] ) && in_array( $context, $schema['properties']['author_data']['context'], true ) ) {
			$post->author_data   = $this->get_author_data( $post );
			$data['author_data'] = $post->author_data;
		}

		if ( ! empty( $schema['properties']['completed'] ) && in_array( $context, $schema['properties']['completed']['context'], true ) ) {
			$post->completed   = $this->is_completed( $post );
			$data['completed'] = (bool) $post->completed;
		}

		if ( ! empty( $schema['properties']['certificate_url'] ) && in_array( $context, $schema['properties']['certificate_url']['context'], true ) ) {
			$post->certificate_url   = $this->get_certificates( $post );
			$data['certificate_url'] = $post->certificate_url;
		}

		if ( ! empty( $schema['properties']['instructors'] ) && in_array( $context, $schema['properties']['instructors']['context'], true ) ) {
			$post->instructors   = $this->get_instructors( $post );
			$data['instructors'] = $post->instructors;
		}

		if ( ! empty( $schema['properties']['level'] ) && in_array( $context, $schema['properties']['level']['context'], true ) ) {
			$post->level   = $this->get_level( $post );
			$data['level'] = $post->level;
		}

		if ( ! empty( $schema['properties']['total_enrolled'] ) && in_array( $context, $schema['properties']['total_enrolled']['context'], true ) ) {
			$post->total_enrolled   = $this->get_total_enrolled( $post );
			$data['total_enrolled'] = $post->total_enrolled;
		}

		if ( ! empty( $schema['properties']['is_enrolled'] ) && in_array( $context, $schema['properties']['is_enrolled']['context'], true ) ) {
			$post->is_enrolled   = $this->is_enrolled( $post );
			$data['is_enrolled'] = (bool) $post->is_enrolled;
		}

		if ( ! empty( $schema['properties']['has_enrolled'] ) && in_array( $context, $schema['properties']['has_enrolled']['context'], true ) ) {
			$post->has_enrolled   = $this->has_enrolled( $post );
			$data['has_enrolled'] = (bool) $post->has_enrolled;
		}

		if ( ! empty( $schema['properties']['enrolled'] ) && in_array( $context, $schema['properties']['enrolled']['context'], true ) ) {
			$post->enrolled   = $this->get_enrolled( $post );
			$data['enrolled'] = $post->enrolled;
		}

		if ( ! empty( $schema['properties']['tabs'] ) && in_array( $context, $schema['properties']['tabs']['context'], true ) ) {
			$post->tabs   = $this->get_tabs( $post );
			$data['tabs'] = $post->tabs;
		}

		if ( ! empty( $schema['properties']['additional_info'] ) && in_array( $context, $schema['properties']['additional_info']['context'], true ) ) {
			$post->additional_info   = $this->get_additional_info( $post );
			$data['additional_info'] = $post->additional_info;
		}

		if ( ! empty( $schema['properties']['error_message'] ) && in_array( $context, $schema['properties']['error_message']['context'], true ) ) {
			$post->error_message = $this->get_error_message( $post );

			if ( ! empty( $post->error_message ) ) {
				$error_code            = $post->error_message->get_error_code();
				$data['error_message'] = array(
					'code'    => $error_code,
					'message' => $post->error_message->get_error_message(),
					'data'    => $post->error_message->get_error_data( $error_code ),
				);
			} else {
				$data['error_message'] = array();
			}
		}

		if ( ! empty( $schema['properties']['course_validity'] ) && in_array( $context, $schema['properties']['course_validity']['context'], true ) ) {
			$course_id       = $post->ID;
			$enrolment       = tutor_utils()->is_enrolled( $course_id, get_current_user_id() );
			$validity_string = '';
			$expiry_class    = null;

			if ( class_exists( Enroll_Expiry_B::class ) ) {
				$expiry_class = Enroll_Expiry_B::class;
			} elseif ( class_exists( Enroll_Expiry_A::class ) ) {
				$expiry_class = Enroll_Expiry_A::class;
			}

			ob_start();

			if ( $expiry_class ) {
				$expiry_instance = new $expiry_class();

				if ( $enrolment ) {
					$expiry_instance->show_expires_info( $course_id );
				} else {
					$expiry_instance->show_expires_info_not_enrolled( $course_id );
				}
			}

			$validity_string         .= ob_get_clean();
			$data['course_validity'] = $validity_string;
		}

		if ( function_exists( 'TUTOR_CERT' ) && tutils()->is_addon_enabled( TUTOR_CERT()->basename ) && ! empty( $schema['properties']['has_course_certificate'] ) && in_array( $context, $schema['properties']['has_course_certificate']['context'], true ) ) {
			$cert                           = new Certificate();
			$has_cert                       = $cert->show_course_has_certificate( array(), $post->ID );
			$data['has_course_certificate'] = ! empty( $has_cert );
		}

		if ( ! empty( $schema['properties']['course_prerequisite'] ) && in_array( $context, $schema['properties']['course_prerequisite']['context'], true ) ) {
			$data['course_prerequisite'] = $this->get_course_prerequisite( $post );
		}

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filters course response.
		 *
		 * @param WP_REST_Response $response Rest response.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_rest_prepare_course', $response, $post, $request );
	}

	/**
	 * Get the course medias.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	private function get_feature_media( $post ) {
		$return = array(
			'large' => '',
			'small' => '',
		);
		$large  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'large' );
		$small  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'medium' );

		if ( isset( $large[0] ) ) {
			$return['large'] = $large[0];
		}
		if ( isset( $small[0] ) ) {
			$return['small'] = $small[0];
		}

		return $return;
	}

	/**
	 * Get the course categories.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	private function get_categories( $post ) {
		$terms = get_the_terms( $post, 'course-category' );

		return $terms && ! is_wp_error( $terms ) ? wp_list_pluck( $terms, 'term_id' ) : array();
	}

	/**
	 * Get the course tags.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	private function get_tags( $post ) {
		$terms = get_the_terms( $post, 'course-tag' );

		return $terms && ! is_wp_error( $terms ) ? wp_list_pluck( $terms, 'term_id' ) : array();
	}

	/**
	 * Get the course access.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	private function is_public( $post ) {
		return Course_List::is_public( $post->ID );
	}

	/**
	 * If the course purchasable.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	private function is_purchasable( $post ) {
		return tutor_utils()->is_course_purchasable( $post->ID );
	}

	/**
	 * Get the course price.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	private function get_price( $post ) {
		$price      = array();
		$product_id = tutor_utils()->get_course_product_id( $post->ID );

		if ( tutor_utils()->is_course_purchasable( $post->ID ) ) {
			$monetize_by = tutor_utils()->get_option( 'monetize_by' );

			if ( tutor_utils()->has_wc() && 'wc' === $monetize_by ) {
				$product = wc_get_product( $product_id );

				if ( $product ) {
					$price['regular'] = get_woocommerce_currency_symbol() . $product->get_regular_price();
					$price['sale']    = get_woocommerce_currency_symbol() . $product->get_price();
				}
			} elseif ( 'edd' === $monetize_by && function_exists( 'edd_price' ) ) {
				$download      = new \EDD_Download( $product_id );
				$price['sale'] = \edd_price( $download->ID, false );
			}
		}

		return $price;
	}

	/**
	 * Get the course full.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	private function get_course_full( $post ) {
		return tutor_utils()->is_course_fully_booked( $post->ID );
	}

	/**
	 * Get the course wishlisted.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	private function is_wishlisted( $post ) {
		return tutor_utils()->is_wishlisted( $post->ID, get_current_user_id() );
	}

	/**
	 * Get the course access.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return string
	 */
	private function get_has_course_access( $post ) {
		return $this->is_enrolled( $post );
	}

	/**
	 * Get course content access.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	private function get_has_content_access( $post ) {
		return $post->has_course_access || tutor_utils()->has_user_course_content_access( get_current_user_id(), $post->ID );
	}

	/**
	 * Get the course progress.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	private function get_progression( $post ) {
		$progress_percentage = 0;

		if ( is_user_logged_in() ) {
			$user_id             = get_current_user_id();
			$progress            = tutor_utils()->get_course_completed_percent( $post->ID, $user_id, true );
			$progress_percentage = $progress;
		}

		return $progress_percentage;
	}

	/**
	 * Get the course duration.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	private function get_duration( $post ) {
		$duration = tutor_utils()->get_course_duration( $post->ID, true );

		return $duration['duration'] ?? $duration;
	}

	/**
	 * Get the course ratings.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return object
	 */
	private function get_ratings( $post ) {
		return tutor_utils()->get_course_rating( $post->ID );
	}

	/**
	 * Get the course ratings settings.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	private function get_ratings_settings( $post ) {
		$is_enrolled      = $this->is_enrolled( $post );
		$can_review       = (bool) $is_enrolled;
		$my_rating        = tutor_utils()->get_reviews_by_user(
			get_current_user_id(),
			0,
			150,
			false,
			$post->ID,
			array(
				'approved',
				'hold',
			)
		);
		$is_new           = ! $my_rating || empty( $my_rating->rating ) || empty( $my_rating->comment_content );
		$ratings_settings = array(
			'can_review'  => $can_review,
			'review_mode' => $is_new ? 'add' : 'edit',
		);

		return $ratings_settings;
	}

	/**
	 * Get the course author data.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	private function get_author_data( $post ) {
		$author_id = $post->post_author;
		$author    = get_userdata( $author_id );

		if ( function_exists( 'bb_theme_enable_tutorlms_override' ) && bb_theme_enable_tutorlms_override() ) {
			$avatar_url = get_avatar_url( $author_id, array( 'size' => 50 ) );
		} else {
			$profile_photo = get_user_meta( $author_id, '_tutor_profile_photo', true );
			$avatar_url    = wp_get_attachment_image_url( $profile_photo, 'thumbnail' );
		}

		if ( empty( $avatar_url ) && function_exists( 'bp_core_fetch_avatar' ) ) {
			$avatar_url = bp_core_fetch_avatar(
				array(
					'item_id' => $author_id,
					'html'    => false,
				)
			);
		}

		return array(
			'id'          => $author_id,
			'name'        => $author->display_name,
			'avatar'      => $avatar_url,
			'profile_url' => get_author_posts_url( $author_id ),
		);
	}

	/**
	 * If the course is completed.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array|bool|object|null
	 */
	private function is_completed( $post ) {
		return tutor_utils()->is_completed_course( $post->ID, get_current_user_id() );
	}

	/**
	 * Get the course certificates.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return string
	 */
	public function get_certificates( $post ) {
		return ( function_exists( 'TUTOR_CERT' ) && tutils()->is_addon_enabled( TUTOR_CERT()->basename ) ) ? ( new Certificate( true ) )->get_certificate( $post->ID ) : '';
	}

	/**
	 * Get the course instructors.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	private function get_instructors( $post ) {
		$instructors      = tutor_utils()->get_instructors_by_course( $post->ID );
		$instructors_data = array();

		foreach ( $instructors as $instructor ) {
			if ( function_exists( 'bb_theme_enable_tutorlms_override' ) && bb_theme_enable_tutorlms_override() ) {
				$profile_url      = trailingslashit( bp_core_get_user_domain( $instructor->ID ) . bp_get_profile_slug() );
				$tutor_avatar_url = get_avatar_url( $instructor->ID, array( 'size' => 50 ) );
			} else {
				$profile_url      = tutor_utils()->profile_url( $instructor->ID, true );
				$profile_photo    = get_user_meta( $instructor->ID, '_tutor_profile_photo', true );
				$tutor_avatar_url = wp_get_attachment_image_url( $profile_photo, 'thumbnail' );
			}

			if ( empty( $tutor_avatar_url ) && function_exists( 'bp_core_fetch_avatar' ) ) {
				$tutor_avatar_url = bp_core_fetch_avatar(
					array(
						'item_id' => $instructor->ID,
						'html'    => false,
					)
				);
			}

			$instructors_data[] = array(
				'id'          => $instructor->ID,
				'name'        => $instructor->display_name,
				'avatar'      => $tutor_avatar_url,
				'profile_url' => $profile_url,
			);
		}

		return $instructors_data;
	}

	/**
	 * Get the course level.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return string
	 */
	private function get_level( $post ) {
		return get_tutor_course_level( $post->ID );
	}

	/**
	 * Get the course total enrolled.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return string
	 */
	private function get_total_enrolled( $post ) {
		$total_enrolled = 0;

		if ( tutor_utils()->get_option( 'enable_course_total_enrolled' ) ) {
			$total_enrolled = tutor_utils()->count_enrolled_users_by_course( $post->ID );
			$total_enrolled = apply_filters( 'tutor_course_students', $total_enrolled, $post->ID );
		}

		return $total_enrolled;
	}

	/**
	 * Get the course is enrolled.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array|bool|object|null
	 */
	private function is_enrolled( $post ) {
		return tutor_utils()->is_enrolled( $post->ID, get_current_user_id() );
	}

	/**
	 * Get the course has enrolled.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	private function has_enrolled( $post ) {
		return tutor_utils()->has_enrolled_content_access( 'course', $post->ID, get_current_user_id() );
	}

	/**
	 * Get the course enrolled.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	private function get_enrolled( $post ) {
		$enrolled_data                  = array();
		$is_enrolled                    = $this->is_enrolled( $post );
		$post_date                      = is_object( $is_enrolled ) && isset( $is_enrolled->post_date ) ? $is_enrolled->post_date : '';
		$enrolled_data['enrolled_date'] = tutor_i18n_get_formated_date( $post_date, get_option( 'date_format' ) );
		$enrolled_data['enrolled_data'] = $is_enrolled;

		return $enrolled_data;
	}

	/**
	 * Get the course tabs.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	private function get_tabs( $post ) {
		return bbapp_tutor_course_nav_items( $post->ID, get_current_user_id() );
	}

	/**
	 * Get the course additional info.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	private function get_additional_info( $post ) {
		$additional_info                    = array();
		$course_id                          = $post->ID;
		$settings_meta                      = get_post_meta( get_the_ID(), '_tutor_course_settings', true );
		$settings_meta                      = (array) maybe_unserialize( $settings_meta );
		$additional_info['settings']        = $settings_meta;
		$additional_info['benefits']        = tutor_course_benefits( $course_id );
		$additional_info['requirements']    = tutor_course_requirements( $course_id );
		$additional_info['target_audience'] = tutor_course_target_audience( $course_id );
		$additional_info['material']        = tutor_course_material_includes( $course_id );
		$additional_info['video']           = $this->get_video_info( $course_id );

		return $additional_info;
	}

	/**
	 * Error message.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	private function get_error_message( $post ) {
		return array();
	}

	/**
	 * Remove excerpt More link as it not need and added dots.
	 *
	 * @since 2.2.80
	 * @return string
	 */
	public function remove_excerpt_more_link_and_add_dots() {
		return '...';
	}

	/**
	 * Prepare items.
	 *
	 * @param array $prepared_args Prepare item parameters.
	 * @param null  $request       Request parameters.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function prepare_items_query( $prepared_args = array(), $request = null ) {
		$query_args = array();

		foreach ( $prepared_args as $key => $value ) {
			/**
			 * Filters the query_vars used in get_items() for the constructed query.
			 *
			 * The dynamic portion of the hook name, `$key`, refers to the query_var key.
			 *
			 * @param string $value The query_var value.
			 *
			 * @since 2.2.80
			 */
			$query_args[ $key ] = apply_filters( "rest_query_var-{$key}", $value ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		}

		$query_args['ignore_sticky_posts'] = true;

		// Map to proper WP_Query orderby param.
		if ( isset( $query_args['orderby'] ) && isset( $request['orderby'] ) ) {
			$orderby_mappings = array(
				'id'            => 'ID',
				'include'       => 'post__in',
				'slug'          => 'post_name',
				'include_slugs' => 'post_name__in',
			);

			if ( isset( $orderby_mappings[ $request['orderby'] ] ) ) {
				$query_args['orderby'] = $orderby_mappings[ $request['orderby'] ];
			}
		}

		return $query_args;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array $data item object.
	 *
	 * @since 2.2.80
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {
		$links = parent::prepare_links( $data );

		/**
		 * Quiz links
		 */
		if ( isset( $data['quizzes'] ) ) {
			$links['quizzes'] = array();

			foreach ( $data['quizzes'] as $quiz ) {
				$links['quizzes'][] = array(
					'href'       => rest_url( $this->namespace . '/quiz/' . $quiz ),
					'embeddable' => true,
				);
			}
		}

		/**
		 * Lesson link
		 */
		if ( isset( $data['lessons'] ) ) {
			$links['lessons'] = array(
				'href'       => rest_url( $this->namespace . '/lessons?course_id=' . $data['id'] ),
				'embeddable' => false,
			);
		}

		/**
		 * Categories Link
		 */
		if ( ! empty( $data['categories'] ) ) {
			$params = array();

			foreach ( $data['categories'] as $catid ) {
				$params[] = 'include[]=' . $catid;
			}

			$links['categories'] = array(
				'href'       => rest_url( '/wp/v2/course-category?' . implode( '&', $params ) ),
				'embeddable' => true,
			);
		}

		/**
		 * Tags Link
		 */
		if ( ! empty( $data['tags'] ) ) {
			$params = array();

			foreach ( $data['tags'] as $catid ) {
				$params[] = 'include[]=' . $catid;
			}

			$links['tags'] = array(
				'href'       => rest_url( '/wp/v2/course-tag?' . implode( '&', $params ) ),
				'embeddable' => true,
			);
		}

		/**
		 * Author links.
		 */
		if ( defined( 'BP_PLATFORM_VERSION' ) && function_exists( 'bp_is_active' ) && bp_is_active( 'members' ) ) {
			if ( isset( $data['author'] ) ) {
				$links['members'] = array(
					'href'       => rest_url( '/buddyboss/v1/members/' . $data['author'] ),
					'embeddable' => true,
				);
			}
		}

		return $links;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['keyword'] = array(
			'description'       => __( 'Keyword to search.', 'buddyboss-app' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author'] = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['exclude']        = array(
			'description'       => __( 'Ensure result set excludes specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include']        = array(
			'description'       => __( 'Limit result set to specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['offset']         = array(
			'description'       => __( 'Offset the result set by a specific number of items.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
				'my_progress',
				'menu_order',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent IDs.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent ID.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);
		$params['slug']           = array(
			'description'       => __( 'Limit result set to posts with a specific slug.', 'buddyboss-app' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['filter']         = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.', 'buddyboss-app' ),
		);

		$params = $this->prepare_taxonomy_limit_schema( $params );

		$params['mycourses'] = array(
			'description'       => __( 'Limit response to resources which is taken by current user.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['group_id'] = array(
			'description'       => __( 'Limit response to resources that are connected with a group.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'course',
			'type'       => 'object',
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'          => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'single', 'edit', 'embed' ),
						),
					),
				),
				'content'        => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'single', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'single', 'edit' ),
						),
						'short'    => array(
							'description' => __( 'Short content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'single', 'edit' ),
						),
					),
				),
				'date'           => array(
					'description' => __( 'The date the object was published, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
				),
				'date_gmt'       => array(
					'description' => __( 'The date the object was published, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
				),
				'modified'       => array(
					'description' => __( 'The date the object was last modified, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'single', 'edit' ),
					'readonly'    => true,
				),
				'modified_gmt'   => array(
					'description' => __( 'The date the object was last modified, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'single', 'edit' ),
					'readonly'    => true,
				),
				'link'           => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'           => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'single', 'edit' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'         => array(
					'description' => __( 'The ID for the author of the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'single', 'edit' ),
				),
				'excerpt'        => array(
					'description' => __( 'The excerpt for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'single', 'edit', 'embed' ),
						),
					),
				),
				'featured_media' => array(
					'description' => __( 'Feature media object containing thumb and full URL of the image.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
				),
				'menu_order'     => array(
					'description' => __( 'The order of the object in relation to other object of its type.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'single', 'edit' ),
				),
			),
		);

		$schema['properties']['categories'] = array(
			'description' => __( 'The terms assigned to the object in the categories taxonomy.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['tags'] = array(
			'description' => __( 'The terms assigned to the object in the tags taxonomy.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['is_public'] = array(
			'description' => __( 'Whether object is public or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['purchasable'] = array(
			'description' => __( 'Whether or not object is purchasable', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['price'] = array(
			'description' => __( 'The price of the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['course_full'] = array(
			'description' => __( 'Whether or not object is fully booked.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['buttons'] = array(
			'description' => __( 'The price of the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['is_wishlisted'] = array(
			'description' => __( 'Whether or not object is wishlisted.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['has_course_access'] = array(
			'description' => __( 'Whether or not user have the object access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['has_content_access'] = array(
			'description' => __( 'Whether or not user have the object content access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['progression'] = array(
			'description' => __( 'The progression % of user for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['duration'] = array(
			'description' => __( 'The time duration for the object', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['ratings'] = array(
			'description' => __( 'The ratings for the object', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['ratings_settings'] = array(
			'description' => __( 'The ratings settings for the object', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'single', 'edit', 'embed' ),
		);

		$schema['properties']['author_data'] = array(
			'description' => __( 'The author data for the object', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['completed'] = array(
			'description' => __( 'The object is completed by current user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['certificate_url'] = array(
			'description' => __( 'The certificate URL for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'single', 'edit', 'embed' ),
		);

		$schema['properties']['instructors'] = array(
			'description' => __( 'The instructors for the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit', 'embed' ),
		);

		$schema['properties']['level'] = array(
			'description' => __( 'The level for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'single', 'edit', 'embed' ),
		);

		$schema['properties']['total_enrolled'] = array(
			'description' => __( 'The total enrolled students for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'single', 'edit', 'embed' ),
		);

		$schema['properties']['is_enrolled'] = array(
			'description' => __( 'Whether or not user is enrolled in the object.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['has_enrolled'] = array(
			'description' => __( 'Whether or not user has enrolled in the object.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['enrolled'] = array(
			'description' => __( 'The enrolled students for the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit', 'embed' ),
		);

		$schema['properties']['tabs'] = array(
			'description' => __( 'The tabs for the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit', 'embed' ),
		);

		$schema['properties']['additional_info'] = array(
			'description' => __( 'The additional info for the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit', 'embed' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['course_validity'] = array(
			'description' => __( 'Course expiry', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'single', 'edit', 'embed' ),
		);

		$schema['properties']['has_course_certificate'] = array(
			'description' => __( 'Whether or not course has certificate.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['course_prerequisite'] = array(
			'description' => __( 'Course Prerequisite(s).', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get the course quiz.
	 *
	 * @param object $data Course content data.
	 *
	 * @since 2.2.80
	 * @return object
	 */
	private function get_tutor_quiz_data( $data ) {
		$last_attempt  = ( new QuizModel() )->get_first_or_last_attempt( $data->id );
		$attempt_ended = is_object( $last_attempt ) && QuizModel::ATTEMPT_STARTED !== $last_attempt->attempt_status;
		$quiz_result   = QuizModel::get_quiz_result( $data->id );
		$data->result  = 'pass';

		if ( $attempt_ended && QuizModel::ATTEMPT_STARTED !== $last_attempt->attempt_status ) {
			if ( 'fail' === $quiz_result ) {
				$data->result = 'fail';
			}

			if ( 'pending' === $quiz_result ) {
				$data->result = 'pending';
			}
		}

		$data->attempt_ended = $attempt_ended;

		return $data;
	}

	/**
	 * Get the course lesson.
	 *
	 * @param object $data Course content data.
	 *
	 * @since 2.2.80
	 * @return object
	 */
	private function get_tutor_lesson_data( $data ) {
		$is_completed_lesson = tutor_utils()->is_completed_lesson( $data->id, get_current_user_id() );
		$data->completed     = (bool) $is_completed_lesson;

		return $data;
	}

	/**
	 * Get the course assignment
	 *
	 * @param object $data Course content data.
	 *
	 * @since 2.2.80
	 * @return object
	 */
	private function get_tutor_assignment_data( $data ) {
		$data->completed = false;
		$is_submitted    = tutor_utils()->is_assignment_submitted( $data->id, get_current_user_id() );

		if ( $is_submitted && 'submitted' === $is_submitted->comment_approved ) {
			$result          = Assignments::get_assignment_result( $data->id, get_current_user_id() );
			$data->result    = $result;
			$data->completed = true;
		}

		return $data;
	}

	/**
	 * Get the course Zoom meeting.
	 *
	 * @param object $data Course content data.
	 *
	 * @since 2.2.80
	 * @return object
	 */
	private function get_tutor_zoom_meeting_data( $data ) {
		$is_completed    = Zoom::is_zoom_lesson_done( '', $data->id, get_current_user_id() );
		$data->completed = $is_completed;

		return $data;
	}

	/**
	 * Prepares the collection schema for including and excluding items by Tutor terms.
	 *
	 * @param array $query_params Collection schema.
	 *
	 * @since 2.2.80
	 *
	 * @return array Updated schema.
	 */
	private function prepare_taxonomy_limit_schema( $query_params ) {
		$taxonomies = wp_list_filter( get_object_taxonomies( $this->post_type, 'objects' ), array( 'show_in_rest' => true ) );

		if ( ! $taxonomies ) {
			return $query_params;
		}

		$query_params['tax_relation'] = array(
			'description' => __( 'Limit result set based on relationship between multiple taxonomies.', 'buddyboss-app' ),
			'type'        => 'string',
			'enum'        => array( 'AND', 'OR' ),
		);

		$limit_schema = array(
			'type'  => array( 'object', 'array' ),
			'oneOf' => array(
				array(
					'title'       => __( 'Term ID List', 'buddyboss-app' ),
					'description' => __( 'Match terms with the listed IDs.', 'buddyboss-app' ),
					'type'        => 'array',
					'items'       => array(
						'type' => 'integer',
					),
				),
				array(
					'title'                => __( 'Term ID Taxonomy Query', 'buddyboss-app' ),
					'description'          => __( 'Perform an advanced term query.', 'buddyboss-app' ),
					'type'                 => 'object',
					'properties'           => array(
						'terms' => array(
							'description' => __( 'Term IDs.', 'buddyboss-app' ),
							'type'        => 'array',
							'items'       => array(
								'type' => 'integer',
							),
							'default'     => array(),
						),
					),
					'additionalProperties' => false,
				),
			),
		);

		$include_schema = array_merge(
			array(
				/* translators: %s: Taxonomy name. */
				'description' => __( 'Limit result set to items with specific terms assigned in the Tutor %s taxonomy.', 'buddyboss-app' ),
			),
			$limit_schema
		);
		// 'operator' is supported only for 'include' queries.
		$include_schema['oneOf'][1]['properties']['operator'] = array(
			'description' => __( 'Whether items must be assigned all or any of the specified terms.', 'buddyboss-app' ),
			'type'        => 'string',
			'enum'        => array( 'AND', 'OR' ),
			'default'     => 'OR',
		);

		$exclude_schema = array_merge(
			array(
				/* translators: %s: Taxonomy name. */
				'description' => __( 'Limit result set to items except those with specific terms assigned in the Tutor %s taxonomy.', 'buddyboss-app' ),
			),
			$limit_schema
		);

		$supported_taxonomies = array(
			'categories', // Tutor course category.
			'tags', // Tutor course tags.
		);

		foreach ( $supported_taxonomies as $supported_taxonomy ) {
			$base         = $supported_taxonomy;
			$base_exclude = $base . '_exclude';

			$query_params[ $base ]                = $include_schema;
			$query_params[ $base ]['description'] = sprintf( $query_params[ $base ]['description'], $base );

			$query_params[ $base_exclude ]                = $exclude_schema;
			$query_params[ $base_exclude ]['description'] = sprintf( $query_params[ $base_exclude ]['description'], $base );
		}

		return $query_params;
	}

	/**
	 * Get course prerequisites.
	 *
	 * @param WP_Post $course course.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function get_course_prerequisite( $course ) {
		$saved_prerequisites_ids = maybe_unserialize( get_post_meta( $course->ID, '_tutor_course_prerequisites_ids', true ) );
		$prerequisites           = array();

		if ( is_array( $saved_prerequisites_ids ) && count( $saved_prerequisites_ids ) ) {
			foreach ( $saved_prerequisites_ids as $key => $course_id ) {
				$prerequisites[ $key ]['id']        = $course_id;
				$prerequisites[ $key ]['title']     = get_the_title( $course_id );
				$prerequisites[ $key ]['thumb']     = get_the_post_thumbnail_url( $course_id );
				$prerequisites[ $key ]['completed'] = (bool) tutor_utils()->is_completed_course( $course_id );
			}
		}

		return $prerequisites;
	}

	/**
	 * Get the course video info.
	 *
	 * @param int $course_id Course id.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function get_video_info( $course_id ) {
		$video_data = array();

		if ( empty( $course_id ) ) {
			return $video_data;
		}

		$video_data                         = tutor_utils()->get_video( $course_id );
		$video_data['video_attachment_url'] = ( ! empty( $video_data['source_video_id'] ) ) ? wp_get_attachment_url( $video_data['source_video_id'] ) : '';
		$video_data['video_poster_url']     = ( ! empty( $video_data['poster'] ) ) ? wp_get_attachment_url( $video_data['poster'] ) : '';
		$video_data['video_shortcode_html'] = '';

		if ( ! empty( $video_data['source_shortcode'] ) ) {
			ob_start();
			echo do_shortcode( $video_data['source_shortcode'] );
			$video_data['video_shortcode_html'] .= ob_get_clean();
		}

		return $video_data;
	}
}
