<?php
/**
 * Holds course Resources rest functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Course
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Course;

use BuddyBossApp\Api\TutorLMS\V1\Core\TutorRestController;
use WP_Error;
use WP_Post;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Course Resources rest class.
 */
class CoursesResourcesRest extends TutorRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Course post type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'courses';

	/**
	 * CoursesResourcesRest instance.
	 *
	 * @since 2.2.80
	 * @return CoursesResourcesRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 2.2.80
	 */
	public function __construct() {
		parent::__construct();

		$this->rest_base = 'courses';
	}

	/**
	 * Check if a given request has access to course resources.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 2.2.80
	 */
	public function get_course_resources_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the course `get_course_resources` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_course_resources_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<id>\d+)/resources',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_course_resources' ),
					'permission_callback' => array( $this, 'get_course_resources_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve Course Resources.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/courses/:id/resources Get TutorLMS Course Resources
	 * @apiName        GetTutorCourseResources
	 * @apiGroup       TutorLMS Courses
	 * @apiDescription Retrieve single Courses Resources
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function get_course_resources( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course    = get_post( $course_id );

		if ( empty( $course_id ) || $this->post_type !== $course->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		$resources_list = $this->get_resources_list( $course );
		$retval         = array();

		if ( ! empty( $resources_list ) ) {
			foreach ( $resources_list as $resources ) {
				$retval[] = $resources;
			}
		}

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a course response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_course_course_resources_response', $response, $request );

		return $response;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function get_collection_params() {
		return array(
			'context' => $this->get_context_param( array( 'default' => 'view' ) ),
		);
	}

	/**
	 * Course Resources list.
	 *
	 * @param WP_Post $post Course object.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function get_resources_list( $post ) {
		return tutor_utils()->get_attachments( $post->ID );
	}
}
