<?php
/**
 * Holds course Gradebook rest functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Course
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Course;

use BuddyBossApp\Api\TutorLMS\V1\Core\TutorRestController;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Course Grade book rest class.
 */
class CoursesGradebookRest extends TutorRestController {
	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Course post-type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'courses';

	/**
	 * CoursesResourcesRest instance.
	 *
	 * @since 2.2.80
	 * @return CoursesGradebookRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 2.2.80
	 */
	public function __construct() {
		parent::__construct();

		$this->rest_base = 'courses';
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<id>\d+)/gradebook',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_course_gradebook' ),
					'permission_callback' => array( $this, 'get_course_resources_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to course resources.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_course_resources_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the course `get_course_resources` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_course_gradebook_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve Course Resources.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/courses/:id/resources Get TutorLMS Course Resources
	 * @apiName        GetTutorCourseResources
	 * @apiGroup       TutorLMS Courses
	 * @apiDescription Retrieve single Courses Resources
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function get_course_gradebook( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course    = get_post( $course_id );

		if ( empty( $course_id ) || $this->post_type !== $course->post_type ) {
			return rest_ensure_response( CoursesError::instance()->invalid_course_id() );
		}

		$registered         = $this->get_collection_params();
		$parameter_mappings = array();

		/**
		 * For each known parameter which is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		$gradebook_data = $this->get_gradebook_data( $course_id );
		$response       = rest_ensure_response( $gradebook_data );

		/**
		 * Fires after a course response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_course_course_gradebook_response', $response, $request );

		return $response;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function get_collection_params() {
		return array(
			'context' => $this->get_context_param( array( 'default' => 'view' ) ),
		);
	}

	/**
	 * Get the course grade book data.
	 *
	 * @param int $course_id Course ID.
	 *
	 * @since 2.2.80
	 * @return WP_Error | array
	 */
	private function get_gradebook_data( $course_id ) {
		$gradebooks = tutor_utils()->get_gradebooks();

		if ( ! tutor_utils()->count( $gradebooks ) ) {
			return CoursesError::instance()->no_grading_system();
		}

		$grades     = get_generated_gradebook( 'all', $course_id );
		$quiz_grade = get_quiz_gradebook_by_course( $course_id );

		if ( ! $quiz_grade || ! tutor_utils()->count( $grades ) ) {
			return CoursesError::instance()->no_gradebook_data();
		}

		$final_grade                         = get_generated_gradebook( 'final', $course_id );
		$final_stat                          = tutor_generate_grade_html( $final_grade, null );
		$return_data                         = array(
			'final_grade' => $final_stat,
		);
		$return_data['final_grade']['color'] = isset( $final_grade->grade_config ) ? tutor_utils()->array_get( 'grade_color', maybe_unserialize( $final_grade->grade_config ) ) : '';
		$return_data['grades']               = array();

		if ( ! empty( $grades ) ) {
			foreach ( $grades as $key => $grade ) {
				$return_data['grades'][ $key ]        = $grade;
				$for                                  = strtolower( $grade->result_for );
				$content_id                           = ( 'quiz' === $for ) ? $grade->quiz_id : $grade->assignment_id;
				$return_data['grades'][ $key ]->title = get_the_title( $content_id );
			}
		}

		return $return_data;
	}
}
