<?php
/**
 * Holds Courses action rest functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Course
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Course;

use Tutor\Models\CourseModel;
use WP_Error;
use WP_REST_Response;
use WP_REST_Server;
use WP_REST_Request;

/**
 * Courses action rest.
 */
class CoursesActionRest extends CoursesRest {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * CoursesActionRest instance.
	 *
	 * @since 2.2.80
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>\d+)/enroll',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'enroll' ),
					'permission_callback' => array( $this, 'get_enroll_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/complete',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'complete' ),
					'permission_callback' => array( $this, 'get_complete_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/reset',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'reset_course_progress' ),
					'permission_callback' => array( $this, 'get_complete_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to enroll course items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_enroll_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = CoursesError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the course `enroll` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_courses_enroll_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to complete course items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 2.2.80
	 */
	public function get_complete_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = CoursesError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the course `complete`  permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_courses_complete_permissions_check', $retval, $request );
	}

	/**
	 * Process Course enrollment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 * @since          2.2.80
	 *
	 * @api            {POST} /wp-json/buddyboss-app/tutor/v1/courses/:id/enroll Process TutorLMS Course Enrollment
	 * @apiName        PostTutorCourseEnroll
	 * @apiGroup       Tutor Courses
	 * @apiDescription Process Course enrollment
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function enroll( $request ) {
		$retval    = false;
		$user_id   = get_current_user_id();
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course    = get_post( $course_id );

		if ( empty( $course_id ) || $this->post_type !== $course->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		if ( ! is_user_logged_in() ) {
			return CoursesError::instance()->course_login_required();
		}

		if ( tutor_utils()->has_user_course_content_access( $user_id, $course_id ) ) {
			$retval = true; // if course already purchased.
		} else {
			$is_purchasable = tutor_utils()->is_course_purchasable( $course_id );

			if ( ! $is_purchasable ) {
				$retval = tutor_utils()->do_enroll( $course_id, 0, $user_id );
			}
		}

		if ( $retval ) {
			return $this->get_item( $request );
		}

		return CoursesError::instance()->course_purchase_required();
	}

	/**
	 * Process Course complete.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response|WP_Error|bool
	 * @api            {POST} /wp-json/buddyboss-app/tutor/v1/courses/:id/complete Process TutorLMS Course Complete
	 * @apiName        PostTutorCourseComplete
	 * @apiGroup       Tutor Courses
	 * @apiDescription Process Course Complete
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Course.
	 */
	public function complete( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course    = get_post( $course_id );
		$user_id   = get_current_user_id();

		if ( empty( $course_id ) || $this->post_type !== $course->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		if ( ! is_user_logged_in() ) {
			return CoursesError::instance()->user_not_logged_in();
		}

		$is_enrolled = tutor_utils()->is_enrolled( $course_id, $user_id );

		if ( ! $is_enrolled ) {
			return CoursesError::instance()->course_enroll_required();
		}

		$can_complete_course = CourseModel::can_complete_course( $course_id, $user_id );

		if ( $can_complete_course ) {
			$complete = CourseModel::mark_course_as_completed( $course_id, $user_id );

			if ( $complete ) {
				$completed = true;
			} else {
				$completed = false;
			}

			if ( is_wp_error( $completed ) ) {
				return $completed;
			} elseif ( true === $completed ) {
				return $this->get_item( $request );
			}
		}

		return CoursesError::instance()->course_complete_error();
	}

	/**
	 * Reset course Progress.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/tutor/v1/courses/:id/reset Process TutorLMS Course Reset
	 * @apiName        PostTutorCourseReset
	 * @apiGroup       Tutor Courses
	 * @apiDescription Process Course Reset
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Course.
	 */
	public function reset_course_progress( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course    = get_post( $course_id );

		if ( empty( $course_id ) || $this->post_type !== $course->post_type ) {
			return rest_ensure_response( CoursesError::instance()->invalid_course_id() );
		}

		if ( ! is_user_logged_in() ) {
			return rest_ensure_response( CoursesError::instance()->user_not_logged_in() );
		}

		if ( ! is_numeric( $course_id ) || ! tutor_utils()->is_enrolled( $course_id ) ) {
			return rest_ensure_response( CoursesError::instance()->course_enroll_required() );
		}

		tutor_utils()->delete_course_progress( $course_id );

		return $this->get_item( $request );
	}
}
