<?php
/**
 * Holds assignments rest functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Assignments
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Assignments;

use BuddyBossApp\Api\TutorLMS\V1\Core\TutorRestController;
use TUTOR_ASSIGNMENTS\Assignments;
use WP_Error;
use WP_Post;
use WP_Query;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Assignment rest.
 */
class AssignmentsRest extends TutorRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Assignment post type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'tutor_assignments';

	/**
	 * AssignmentsRest instance.
	 *
	 * @since 2.2.80
	 * @return AssignmentsRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 2.2.80
	 */
	public function __construct() {
		$this->rest_base = 'assignments';
		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve Assignments.
	 *
	 * @since          2.2.80
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/assignments Get TutorLMS Assignments
	 * @apiName        GetTutorAssignments
	 * @apiGroup       TutorLMS Assignments
	 * @apiDescription Retrieve Assignments
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure a result set includes specific IDs.
	 * @apiParam {String} [after] Limit results to those published after a given ISO8601 compliant date.
	 * @apiParam {String} [before] Limit results to those published before a given ISO8601 compliant date.
	 * @apiParam {Array} [author] Limit results to those assigned to specific authors.
	 * @apiParam {Array} [author_exclude] Ensure results to exclude those assigned to specific authors.
	 * @apiParam {String=asc,desc} [order=desc] Sort result set by given order.
	 * @apiParam {String=date,id,title,menu_order} [orderby=date] Sort result set by given field.
	 * @apiParam {Array} [parent] Limit results to those assigned to specific parent.
	 */
	public function get_items( $request ) {
		$user_id    = get_current_user_id();
		$registered = $this->get_collection_params();

		/**
		 * Filter the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$request = apply_filters( 'bbapp_tutor_get_assignments_request', $request );

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'author'         => 'author__in',
			'author_exclude' => 'author__not_in',
			'exclude'        => 'post__not_in',
			'include'        => 'post__in',
			'offset'         => 'offset',
			'order'          => 'order',
			'orderby'        => 'orderby',
			'page'           => 'paged',
			'parent'         => 'post_parent__in',
			'parent_exclude' => 'post_parent__not_in',
			'search'         => 's',
			'slug'           => 'post_name__in',
			'status'         => 'post_status',
			'per_page'       => 'posts_per_page',
		);

		/**
		 * For each known parameter that is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		// Check for & assign any parameters which require special handling or setting.
		$args['date_query'] = array();

		// Set before into a date query. Date query must be specified as an array of an array.
		if ( isset( $registered['before'], $request['before'] ) ) {
			$args['date_query'][0]['before'] = $request['before'];
		}

		// Set after into a date query. Date query must be specified as an array of an array.
		if ( isset( $registered['after'], $request['after'] ) ) {
			$args['date_query'][0]['after'] = $request['after'];
		}

		if ( ! empty( $user_id ) ) {
			$args['author'] = $user_id;
		}
		$args = $this->prepare_items_query( $args, $request );

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$args = apply_filters( 'bbapp_tutor_get_assignments_args', $args, $request );

		$args['post_type'] = $this->post_type;

		$posts_query                = new WP_Query();
		$assignments['posts']       = $posts_query->query( $args );
		$assignments['total_posts'] = $posts_query->found_posts;

		/**
		 * Fires list of Assignment is fetched via Query.
		 *
		 * @param array            $assignments Fetched assignments.
		 * @param WP_REST_Response $args        Query arguments.
		 * @param WP_REST_Request  $request     The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$assignments = apply_filters( 'bbapp_tutor_get_assignments', $assignments, $args, $request );

		$retval = array();

		foreach ( $assignments['posts'] as $assignment ) {
			if ( ! $this->check_read_permission( $assignment ) ) {
				continue;
			}

			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $assignment, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		$response = bbapp_tutor_response_add_total_headers( $response, $assignments['total_posts'], $args['posts_per_page'] );

		/**
		 * Fires after a list of assignment response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_assignments_items_response', $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to assignment items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the assignment `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_assignments_items_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve Assignment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/assignments/{{assignment_id}} Get TutorLMS Assignment
	 * @apiName        GetTutorAssignment
	 * @apiGroup       TutorLMS Assignments
	 * @apiDescription Retrieve single Assignments
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Assignment.
	 */
	public function get_item( $request ) {
		$assignment_id = $request->get_param( 'id' );
		$assignment    = get_post( $assignment_id );

		if ( empty( $assignment ) || $this->post_type !== $assignment->post_type ) {
			return AssignmentsError::instance()->invalid_assignment_id();
		}

		/**
		 * Fire after Assignment is fetched via Query.
		 *
		 * @param array           $assignment    Fetched assignment.
		 * @param WP_REST_Request $assignment_id assignment id.
		 *
		 * @since 2.2.80
		 */
		$assignment = apply_filters( 'bbapp_tutor_get_assignment', $assignment, $assignment_id );

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $assignment, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after an assignment response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_assignment_item_response', $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to assignment item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_item_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the assignment `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_assignments_item_permissions_check', $retval, $request );
	}

	/**
	 * Prepare a single post-output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$GLOBALS['post'] = $post; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited

		setup_postdata( $post );

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$schema  = $this->get_public_item_schema();

		// Base fields for every post.
		$data = array(
			'id'           => $post->ID,
			'title'        => array(
				'raw'      => $post->post_title,
				'rendered' => get_the_title( $post->ID ),
			),
			'content'      => array(
				'raw'      => bbapp_tutor_fix_relative_urls_protocol( $post->post_content ),
				'rendered' => bbapp_tutor_fix_relative_urls_protocol( apply_filters( 'the_content', $post->post_content ) ),
			),
			'date'         => mysql_to_rfc3339( $post->post_date ),
			'date_gmt'     => mysql_to_rfc3339( $post->post_date_gmt ),
			'modified'     => mysql_to_rfc3339( $post->post_modified ),
			'modified_gmt' => mysql_to_rfc3339( $post->post_modified_gmt ),
			'link'         => get_permalink( $post->ID ),
			'slug'         => $post->post_name,
			'author'       => (int) $post->post_author,
			'excerpt'      => array(
				'raw'      => bbapp_tutor_fix_relative_urls_protocol( $post->post_excerpt ),
				'rendered' => bbapp_tutor_fix_relative_urls_protocol( apply_filters( 'the_excerpt', $post->post_excerpt ) ),
			),
			'menu_order'   => (int) $post->menu_order,
		);

		if ( ! empty( $schema['properties']['course'] ) && in_array( $context, $schema['properties']['course']['context'], true ) ) {
			$data['course'] = $this->get_course_id( $post );
		}

		if ( ! empty( $schema['properties']['topic'] ) && in_array( $context, $schema['properties']['topic']['context'], true ) ) {
			$data['topic'] = $this->get_topic_id( $post );
		}

		if ( ! empty( $schema['properties']['attachments'] ) && in_array( $context, $schema['properties']['attachments']['context'], true ) ) {
			$data['attachments'] = $this->get_attachments( $post );
		}

		if ( ! empty( $schema['properties']['can_submit'] ) && in_array( $context, $schema['properties']['can_submit']['context'], true ) ) {
			$data['can_submit'] = $this->can_submit( $post );
		}

		if ( ! empty( $schema['properties']['is_submitting'] ) && in_array( $context, $schema['properties']['is_submitting']['context'], true ) ) {
			$data['is_submitting'] = $this->is_submitting( $post );
		}

		if ( ! empty( $schema['properties']['submit_info'] ) && in_array( $context, $schema['properties']['submit_info']['context'], true ) ) {
			$data['submit_info'] = $this->get_submit_info( $post );
		}

		if ( ! empty( $schema['properties']['results'] ) && in_array( $context, $schema['properties']['results']['context'], true ) ) {
			$data['results'] = $this->get_results( $post );
		}

		if ( ! empty( $schema['properties']['instructor_note'] ) && in_array( $context, $schema['properties']['instructor_note']['context'], true ) ) {
			$data['instructor_note'] = $this->get_instructor_note( $post );
		}

		if ( ! empty( $schema['properties']['settings'] ) && in_array( $context, $schema['properties']['settings']['context'], true ) ) {
			$data['settings'] = $this->get_settings( $post );
		}

		if ( ! empty( $schema['properties']['my_assignment'] ) && in_array( $context, $schema['properties']['my_assignment']['context'], true ) ) {
			$data['my_assignment'] = $this->get_my_assignment( $post );
		}

		if ( ! empty( $schema['properties']['error_message'] ) && in_array( $context, $schema['properties']['error_message']['context'], true ) ) {
			$post->error_message = $this->get_error_message( $post );

			if ( ! empty( $post->error_message ) ) {
				$error_code            = $post->error_message->get_error_code();
				$data['error_message'] = array(
					'code'    => $error_code,
					'message' => $post->error_message->get_error_message(),
					'data'    => $post->error_message->get_error_data( $error_code ),

				);
			} else {
				$data['error_message'] = array();
			}
		}

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filter the assignment data for a response.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_rest_prepare_assignment', $response, $post, $request );
	}

	/**
	 * Prepare items.
	 *
	 * @param array $prepared_args query parameters.
	 * @param null  $request       Request parameters.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	protected function prepare_items_query( $prepared_args = array(), $request = null ) {
		$query_args = array();

		foreach ( $prepared_args as $key => $value ) {
			/**
			 * Filters the query_vars used in get_items() for the constructed query.
			 *
			 * The dynamic portion of the hook name, `$key`, refers to the query_var key.
			 *
			 * @param string $value The query_var value.
			 *
			 * @since 4.7.0
			 */
			$query_args[ $key ] = apply_filters( "rest_query_var-{$key}", $value ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		}

		$query_args['ignore_sticky_posts'] = true;

		// Map to proper WP_Query order by param.
		if ( isset( $query_args['orderby'] ) && isset( $request['orderby'] ) ) {
			$orderby_mappings = array(
				'id'            => 'ID',
				'include'       => 'post__in',
				'slug'          => 'post_name',
				'include_slugs' => 'post_name__in',
			);

			if ( isset( $orderby_mappings[ $request['orderby'] ] ) ) {
				$query_args['orderby'] = $orderby_mappings[ $request['orderby'] ];
			}
		}

		return $query_args;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array $data item object.
	 *
	 * @since 2.2.80
	 *
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {
		$links = parent::prepare_links( $data );

		return $links;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author'] = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset']            = array(
			'description'       => __( 'Offset the result set by a specific number of items.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']['enum'][] = 'menu_order';

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent IDs.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent ID.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);

		$params['slug']   = array(
			'description'       => __( 'Limit result set to posts with a specific slug.', 'buddyboss-app' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['filter'] = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.', 'buddyboss-app' ),
		);

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'assignment',
			'type'       => 'object',
			'properties' => array(
				'id'           => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'        => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content'      => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'date'         => array(
					'description' => __( 'The date the object was published, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date_gmt'     => array(
					'description' => __( 'The date the object was published, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'modified'     => array(
					'description' => __( 'The date the object was last modified, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'modified_gmt' => array(
					'description' => __( 'The date the object was last modified, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'link'         => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'         => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'       => array(
					'description' => __( 'The ID for the author of the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'excerpt'      => array(
					'description' => __( 'The excerpt for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
			),
		);

		$schema['properties']['attachments'] = array(
			'description' => __( 'Assignment attachments.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['can_submit'] = array(
			'description' => __( 'Assignment is submitting.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['is_submitting'] = array(
			'description' => __( 'Assignment is submitting.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['submit_info'] = array(
			'description' => __( 'Assignment submit info.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['results'] = array(
			'description' => __( 'Assignment results.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['instructor_note'] = array(
			'description' => __( 'Instructor note for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['settings'] = array(
			'description' => __( 'Assignment settings.', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'view', 'edit' ),
			'properties'  => array(
				'properties' => array(),
			),
		);

		$schema['properties']['my_assignment'] = array(
			'description' => __( 'My assignment.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['course'] = array(
			'description' => __( 'Assignment associated course id.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['topic'] = array(
			'description' => __( 'Assignment associated topic id.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get course id.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @since  2.2.80
	 *
	 * @return int
	 */
	protected function get_course_id( $post ) {
		$course_id = tutor_utils()->get_course_id_by_subcontent( $post->ID );

		if ( ! isset( $course_id ) ) {
			$course_id = 0;
		}

		return $course_id;
	}

	/**
	 * Get topic id.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @since 2.2.80
	 *
	 * @return int
	 */
	protected function get_topic_id( $post ) {
		$topic_id = $post->post_parent;

		if ( ! isset( $topic_id ) ) {
			$topic_id = 0;
		}

		return $topic_id;
	}

	/**
	 * Get attachments.
	 *
	 * @param object $post Post object.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_attachments( $post ) {
		$assignment_attachments = maybe_unserialize( get_post_meta( $post->ID, '_tutor_assignment_attachments', true ) );
		$attachments            = array();

		if ( is_array( $assignment_attachments ) && count( $assignment_attachments ) ) {
			foreach ( $assignment_attachments as $attachment_id ) {
				$attachment_name = get_post_meta( $attachment_id, '_wp_attached_file', true );
				$attachment_name = substr( $attachment_name, strrpos( $attachment_name, '/' ) + 1 );
				$file_size       = tutor_utils()->get_readable_filesize( get_attached_file( $attachment_id ) );
				$attachment_url  = wp_get_attachment_url( $attachment_id );

				$attachments[] = array(
					'id'             => $attachment_id,
					'name'           => $attachment_name,
					'file_size'      => $file_size,
					'attachment_url' => $attachment_url,
				);
			}
		}

		return $attachments;
	}

	/**
	 * Can submit.
	 *
	 * @param object $post Post object.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	protected function can_submit( $post ) {
		$can_submit = tutor_utils()->is_assignment_submitted( $post->ID, get_current_user_id() );

		return ! $can_submit;
	}

	/**
	 * Is submitting.
	 *
	 * @param object $post Post object.
	 *
	 * @since 2.2.80
	 * @return int
	 */
	protected function is_submitting( $post ) {
		return tutor_utils()->is_assignment_submitting( $post->ID, get_current_user_id() );
	}

	/**
	 * Get submitted.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @since 2.2.80
	 * @return array|object|null
	 */
	protected function get_submitted( $post ) {
		$submitted_assignment = tutor_utils()->is_assignment_submitted( $post->ID, get_current_user_id() );

		if ( ! $submitted_assignment ) {
			return null;
		}

		return $submitted_assignment;
	}

	/**
	 * Submit assignment info.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	protected function get_submit_info( $post ) {
		$submitted_assignment = $this->get_submitted( $post );

		if ( ! $submitted_assignment ) {
			return null;
		}

		// Get assignment info.
		$submitted_assignment = tutor_utils()->get_assignment_submit_info( $submitted_assignment->comment_ID );

		if ( ! empty( $submitted_assignment->comment_date ) ) {
			$submitted_assignment->comment_date_readable = tutor_utils()->convert_date_into_wp_timezone( $submitted_assignment->comment_date );
		}

		return $submitted_assignment;
	}

	/**
	 * Get results.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_results( $post ) {
		$submitted_assignment = $this->get_submitted( $post );

		if ( $submitted_assignment ) {
			$assignment_results = $this->get_assignment_results( $submitted_assignment );
		} else {
			$assignment_results = array();
		}
		return $assignment_results;
	}

	/**
	 * Get assignment results.
	 *
	 * @param object $submitted_assignment Submitted assignment.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_assignment_results( $submitted_assignment ) {
		$assignment_results                  = array();
		$assignment_results['assignment_id'] = $submitted_assignment->comment_post_ID;
		$assignment_results['submit_id']     = $submitted_assignment->comment_ID;
		$assignment_results['max_mark']      = tutor_utils()->get_assignment_option( $submitted_assignment->comment_post_ID, 'total_mark' );
		$assignment_results['pass_mark']     = tutor_utils()->get_assignment_option( $submitted_assignment->comment_post_ID, 'pass_mark' );
		$assignment_results['given_mark']    = get_comment_meta( $submitted_assignment->comment_ID, 'assignment_mark', true );
		$is_reviewed_by_instructor           = get_comment_meta( $submitted_assignment->comment_ID, 'evaluate_time', true );
		$result                              = '';

		if ( $is_reviewed_by_instructor ) {
			if ( $assignment_results['given_mark'] >= $assignment_results['pass_mark'] ) {
				$result = 'passed';
			} else {
				$result = 'failed';
			}
		}

		if ( ! $is_reviewed_by_instructor ) {
			$result = 'pending';
		}

		$assignment_results['is_reviewed_by_instructor'] = $is_reviewed_by_instructor;
		$assignment_results['result']                    = $result;

		return $assignment_results;
	}

	/**
	 * Get instructor note.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_instructor_note( $post ) {
		$submitted_assignment = $this->get_submitted( $post );
		$note                 = array();

		if ( $submitted_assignment ) {
			$is_reviewed_by_instructor         = get_comment_meta( $submitted_assignment->comment_ID, 'evaluate_time', true );
			$instructor_note                   = get_comment_meta( $submitted_assignment->comment_ID, 'instructor_note', true );
			$note['is_reviewed_by_instructor'] = $is_reviewed_by_instructor;
			$note['instructor_note']           = $instructor_note;
		}

		return $note;
	}

	/**
	 * Get time duration.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_time_duration( $post ) {
		$time_duration           = tutor_utils()->get_assignment_option(
			$post->ID,
			'time_duration',
			array(
				'time'  => '',
				'value' => 0,
			)
		);
		$assignment_created_time = strtotime( $post->post_date_gmt );
		$time_duration_in_sec    = 0;

		if ( isset( $time_duration['value'] ) && isset( $time_duration['time'] ) ) {
			switch ( $time_duration['time'] ) {
				case 'hours':
					$time_duration_in_sec = 3600;
					break;
				case 'days':
					$time_duration_in_sec = 86400;
					break;
				case 'weeks':
					$time_duration_in_sec = 7 * 86400;
					break;
				default:
					$time_duration_in_sec = 0;
					break;
			}
		}

		$time_duration_in_sec = $time_duration_in_sec * (int) $time_duration['value'];
		$remaining_time       = $assignment_created_time + $time_duration_in_sec;
		$now                  = time();
		$remaining            = $now - $remaining_time;

		if ( 0 !== $time_duration['value'] ) {
			if ( $now > $remaining_time && ! $this->is_submitted( $post ) ) {
				$deadline = 'expired';
			} else {
				$deadline = $remaining;
			}
		} else {
			$deadline = 'n\a';
		}

		$time_duration['remaining']      = $remaining;
		$time_duration['remaining_time'] = $remaining_time;
		$time_duration['duration']       = $time_duration['value'] ? $time_duration['value'] . ' ' . $time_duration['time'] : __( 'No limit', 'buddyboss-app' );
		$time_duration['deadline']       = $deadline;

		return $time_duration;
	}

	/**
	 * Check if assignment is submitted.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	protected function is_submitted( $post ) {
		$assignment_comment = tutor_utils()->get_single_comment_user_post_id( $post->ID, get_current_user_id() );

		return ( $assignment_comment ) ? $assignment_comment->comment_approved : false;
	}

	/**
	 * Get my assignment.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function get_my_assignment( $post ) {
		$result               = Assignments::get_assignment_result( $post->ID, get_current_user_id() );
		$time_duration        = $this->get_time_duration( $post );
		$now                  = time();
		$submitted_assignment = $this->get_submitted( $post );

		if ( ( $time_duration['remaining_time'] > $now || 0 === $time_duration['value'] ) ) {
			$my_assignment = array(
				'result' => $result,
			);

			if ( $submitted_assignment ) {
				$my_assignment['attached_files'] = array();
				$attached_files                  = get_comment_meta( $submitted_assignment->comment_ID, 'uploaded_attachments', true );
				$attached_files_data             = ! empty( json_decode( $attached_files, true ) ) ? json_decode( $attached_files, true ) : array();

				if ( ! empty( $attached_files_data ) ) {
					$upload_dir     = wp_get_upload_dir();
					$upload_basedir = trailingslashit( $upload_dir['basedir'] ?? '' );

					foreach ( $attached_files_data as $file ) {
						$my_assignment['attached_files'][] = array(
							'url'           => $file['url'],
							'type'          => $file['type'],
							'uploaded_path' => $file['uploaded_path'],
							'name'          => $file['name'],
							'size'          => tutor_utils()->get_readable_filesize( $upload_basedir . $file['uploaded_path'] ),
						);
					}
				}

				$my_assignment['submitted_content'] = get_comment_text( $submitted_assignment->comment_ID );
			}
		} else {
			$my_assignment = array();
		}

		return $my_assignment;
	}

	/**
	 * Get settings.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_settings( $post ) {
		$allowed_upload_files = (int) tutor_utils()->get_assignment_option( $post->ID, 'upload_files_limit' );
		$total_mark           = tutor_utils()->get_assignment_option( $post->ID, 'total_mark' );
		$pass_mark            = tutor_utils()->get_assignment_option( $post->ID, 'pass_mark' );
		$file_upload_limit    = tutor_utils()->get_assignment_option( $post->ID, 'upload_file_size_limit' );
		$time_duration        = $this->get_time_duration( $post );
		$settings             = array(
			'allowed_upload_files' => $allowed_upload_files,
			'total_mark'           => $total_mark,
			'pass_mark'            => $pass_mark,
			'file_upload_limit'    => $file_upload_limit,
			'time_duration'        => $time_duration,
		);

		return $settings;
	}

	/**
	 * Get an error message.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @since 2.2.80
	 * @return array|WP_Error
	 */
	public function get_error_message( $post ) {
		$time_duration = $this->get_time_duration( $post );
		$now           = time();

		if ( ! $this->is_submitted( $post ) && 0 !== $time_duration['value'] && ( $now > $time_duration['remaining_time'] ) ) {
			return AssignmentsError::instance()->missed_submission_deadline();
		}

		return array();
	}
}
