<?php
/**
 * Holds Assignment action rest functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Assignments
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Assignments;

use Exception;
use TUTOR_ASSIGNMENTS\Assignments;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Assignment action rest.
 */
class AssignmentsActionRest extends AssignmentsRest {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * AssignmentsActionRest instance.
	 *
	 * @since 2.2.80
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/start',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'start' ),
					'permission_callback' => array( $this, 'get_start_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/submit/(?P<submission_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'submit' ),
					'permission_callback' => array( $this, 'get_submit_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/delete-attachment/(?P<submission_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'delete_attachment' ),
					'permission_callback' => array( $this, 'get_submit_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to start assignment item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_start_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = AssignmentsError::instance()->user_not_logged_in();
		}

		if ( is_user_logged_in() ) {
			$user_id = get_current_user_id();

			if ( $this->is_running_assignment( $request->get_param( 'id' ), $user_id ) ) {
				$retval = AssignmentsError::instance()->assignment_already_started();
			}
		}

		/**
		 * Filter the Assignment `start` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_assignment_start_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to submit assignment item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_submit_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = AssignmentsError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the Assignment `submit` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_assignment_submit_permissions_check', $retval, $request );
	}

	/**
	 * Start Assignment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response|WP_Error|boolean
	 * @throws Exception Exception.
	 * @api            {POST} /wp-json/buddyboss-app/tutor/v1/assignments/:id/start Start TutorLMS Assignment.
	 * @apiName        TutorAssignmentStart
	 * @apiGroup       Tutor Assignments
	 * @apiDescription Start Assignments. This endpoint requires request to be sent in "multipart/form-data" format.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Assignment.
	 */
	public function start( $request ) {
		$assignment_id = $request->get_param( 'id' );
		$user_id       = get_current_user_id();
		$deadline_date = tutor_utils()->get_assignment_deadline_date( (int) $assignment_id, 'Y-m-d H:i:s' );

		if ( ! empty( $deadline_date ) ) {
			// Convert deadline string to a DateTime object.
			$deadline_time = new \DateTime( $deadline_date );
			// Get the current time.
			$current_time = new \DateTime();

			// Compare the deadline time with the current time.
			if ( $deadline_time < $current_time ) {
				// The Deadline has expired.
				return AssignmentsError::instance()->assignment_deadline_expired();
			}
		}

		// Get if student has any submitted assignment.
		$submitted_assignment = tutor_utils()->get_single_comment_user_post_id( (int) $assignment_id, $user_id );

		if ( $submitted_assignment ) {
			return AssignmentsError::instance()->assignment_already_submitted();
		}

		// Total file count.
		$files              = $request->get_file_params();
		$allow_to_upload    = (int) tutor_utils()->get_assignment_option( (int) $assignment_id, 'upload_files_limit' );
		$current_file_count = ! empty( $files ) && isset( $files['attached_assignment_files'] ) ? count( $files['attached_assignment_files'] ) : 0;
		$total_file_count   = $current_file_count;

		// Check if total files exceed the limit.
		if ( $total_file_count > $allow_to_upload ) {
			return AssignmentsError::instance()->invalid_upload_limit();
		}

		$course_id = tutor_utils()->get_course_id_by( 'assignment', (int) $assignment_id );

		if ( ! tutor_utils()->is_enrolled( $course_id, (int) $user_id ) ) {
			return AssignmentsError::instance()->course_not_enrolled();
		}

		$assignment_id = $this->start_assignment( $assignment_id );

		if ( is_wp_error( $assignment_id ) ) {
			return $assignment_id;
		} elseif ( is_numeric( $assignment_id ) ) {
			return $this->get_item( $request );
		}

		return false;
	}

	/**
	 * Submit Assignment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response|WP_Error|boolean
	 * @throws Exception Exception.
	 * @api            {POST} /wp-json/buddyboss-app/tutor/v1/assignments/:id/submit/:submission_id Submit TutorLMS
	 *                 Assignment.
	 * @apiName        TutorAssignmentSubmit
	 * @apiGroup       Tutor Assignments
	 * @apiDescription Submit Assignments. This endpoint requires request to be sent in "multipart/form-data" format.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Assignment.
	 */
	public function submit( $request ) {
		$assignment_id    = $request->get_param( 'id' );
		$submission_id    = $request->get_param( 'submission_id' );
		$user_id          = get_current_user_id();
		$is_update_exists = get_comment( $submission_id );

		if ( ! $is_update_exists ) {
			return AssignmentsError::instance()->invalid_submission_id();
		}

		$deadline_date = tutor_utils()->get_assignment_deadline_date( (int) $assignment_id, 'Y-m-d H:i:s' );

		if ( ! empty( $deadline_date ) ) {
			// Convert deadline string to a DateTime object.
			$deadline_time = new \DateTime( $deadline_date );
			// Get the current time.
			$current_time = new \DateTime();

			// Compare the deadline time with the current time.
			if ( $deadline_time < $current_time ) {
				// Deadline has expired.
				return AssignmentsError::instance()->assignment_deadline_expired();
			}
		}

		// Check if already evaluated.
		$is_evaluated = get_comment_meta( $submission_id, 'evaluate_time' );

		if ( ! empty( $is_evaluated ) ) {
			return AssignmentsError::instance()->assignment_already_evaluated();
		}

		$previous_upload_count = 0;
		// Get previous upload count if any.
		$submitted_attachments = get_comment_meta( $submission_id, 'uploaded_attachments' );
		if ( ! empty( $submitted_attachments ) ) {
			$submitted_attachments = json_decode( $submitted_attachments[0] );
			$previous_upload_count = count( $submitted_attachments );
		}

		// Total file count.
		$files              = $request->get_file_params();
		$allow_to_upload    = (int) tutor_utils()->get_assignment_option( (int) $assignment_id, 'upload_files_limit' );
		$current_file_count = ! empty( $files ) && isset( $files['attached_assignment_files']['name'] ) ? count( $files['attached_assignment_files']['name'] ) : 0;
		$total_file_count   = $current_file_count + $previous_upload_count;

		// Check if total files exceed the limit.
		if ( $total_file_count > $allow_to_upload ) {
			return AssignmentsError::instance()->invalid_upload_limit();
		}

		$course_id = tutor_utils()->get_course_id_by( 'assignment', (int) $assignment_id );

		if ( ! tutor_utils()->is_enrolled( $course_id, (int) $user_id ) ) {
			return AssignmentsError::instance()->course_not_enrolled();
		}

		$assignment_content = $request->get_param( 'content' );

		if ( empty( $assignment_content ) ) {
			return AssignmentsError::instance()->assignment_empty_content();
		}

		$assignment_id = $this->submit_assignment( $assignment_id, $submission_id, $assignment_content );

		if ( is_wp_error( $assignment_id ) ) {
			return $assignment_id;
		} elseif ( is_numeric( $assignment_id ) ) {
			return $this->get_item( $request );
		}

		return false;
	}


	/**
	 * Start Assignment.
	 *
	 * @param int $assignment_id Assignment ID.
	 *
	 * @since 2.2.80
	 * @return int|WP_Error
	 */
	public function start_assignment( $assignment_id ) {
		global $wpdb;

		$user_id       = get_current_user_id();
		$assignment_id = absint( $assignment_id );
		$user          = get_userdata( $user_id );
		$date          = gmdate( 'Y-m-d H:i:s' );
		$assignment    = get_post( $assignment_id );

		if ( ! $assignment ) {
			return AssignmentsError::instance()->invalid_assignment_id();
		}

		$course_id = tutor_utils()->get_course_id_by( 'assignment', $assignment_id );

		do_action( 'tutor_before_assignment_submit_start' );

		$data = apply_filters(
			'tutor_assignment_start_submitting_data',
			array(
				'comment_post_ID'  => $assignment_id,
				'comment_author'   => $user->user_login,
				'comment_date'     => $date, // Submit Finished.
				'comment_date_gmt' => wp_date( 'Y-m-d H:i:s' ), // Submit Started.
				'comment_approved' => 'submitting', // submitting, submitted.
				'comment_agent'    => 'TutorLMSPlugin',
				'comment_type'     => 'tutor_assignment',
				'comment_parent'   => $course_id,
				'user_id'          => $user_id,
			)
		);

		$wpdb->insert( $wpdb->comments, $data ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$comment_id = $wpdb->insert_id;

		do_action( 'tutor_after_assignment_submit_start', $comment_id );

		return $assignment_id;
	}

	/**
	 * Submit Assignment.
	 *
	 * @param int    $assignment_id Assignment ID.
	 * @param int    $submission_id Submission ID.
	 * @param string $content       Assignment Content.
	 *
	 * @since 2.2.80
	 * @return int|WP_Error
	 */
	public function submit_assignment( $assignment_id, $submission_id, $content = '' ) {
		$user_id                          = get_current_user_id();
		$store_data                       = new \stdClass();
		$store_data->update_id            = $submission_id;
		$store_data->assignment_id        = (int) $assignment_id;
		$store_data->assignment_answer    = $content;
		$store_data->allowed_upload_files = (int) tutor_utils()->get_assignment_option( $store_data->assignment_id, 'upload_files_limit' );
		$store_data->assignment_submit_id = tutor_utils()->is_assignment_submitting( $store_data->assignment_id );
		$store_data->course_id            = tutor_utils()->get_course_id_by( 'assignment', $store_data->assignment_id );
		$store_data->student_id           = $user_id;
		$assignment                       = new Assignments( false );

		return $assignment->update_assignment_submit( $store_data );
	}

	/**
	 * Delete Assignment attachment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response
	 * @throws Exception Exception.
	 * @api            {POST} /wp-json/buddyboss-app/tutor/v1/assignments/:id/delete-attachment/:submission_id Submit
	 *                 TutorLMS Assignment.
	 * @apiName        TutorAssignmentSubmit
	 * @apiGroup       Tutor Assignments
	 * @apiDescription Submit Assignments. This endpoint requires request to be sent in "multipart/form-data" format.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Assignment.
	 */
	public function delete_attachment( $request ) {
		$attachment_id     = $request->get_param( 'submission_id' );
		$file_name         = $request->get_param( 'file_name' );
		$delete_attachment = Assignments::delete_attachment( $attachment_id, $file_name );

		if ( $delete_attachment ) {
			return rest_ensure_response( $attachment_id );
		} elseif ( is_numeric( $attachment_id ) ) {
			return rest_ensure_response( $attachment_id );
		}

		return rest_ensure_response( AssignmentsError::instance()->attachment_delete_failed() );
	}

	/**
	 * Is Assignment Started.
	 * Check if the assignment is already started by the user.
	 *
	 * @param int $assignment_id Assignment ID.
	 * @param int $user_id       User ID.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	private function is_running_assignment( $assignment_id, $user_id = null ) {
		if ( null === $user_id ) {
			$user_id = get_current_user_id();
		}

		$is_running_submit = tutor_utils()->is_assignment_submitting( $assignment_id, $user_id );

		if ( $is_running_submit > 0 ) {
			return true;
		}

		return false;
	}
}
