<?php
/**
 * Holds TutorLMS main class functionality.
 *
 * @package BuddyBossApp\Api\TutorLMS
 */

namespace BuddyBossApp\Api\TutorLMS;

use BuddyBossApp\AccessControls\AccessRule;
use BuddyBossApp\Api\TutorLMS\V1\Course\CoursesRest;
use BuddyBossApp\Api\TutorLMS\V1\RestAPI as RestAPIv1;
use BuddyBossApp\Auth\Common;
use BuddyBossApp\Auth\Jwt;
use TUTOR\RestAuth;
use TUTOR_GC\Classroom;
use TUTOR_PMPRO\PaidMembershipsPro;
use WP_Post;
use WP_REST_Request;
use WP_REST_Response;
use function rcp_user_can_access;

/**
 * TutorLMS API main class.
 */
class Main {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return Main
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Register all hooks
	 *
	 * @since 2.2.80
	 * @return void
	 */
	public function hooks() {
		/**
		 * `is_tutor_lms_active` initialise on plugin loaded.
		 */
		add_action( 'plugins_loaded', array( $this, 'load' ) );
	}

	/**
	 * Load method.
	 *
	 * @since 2.2.80
	 */
	public function load() {
		if ( function_exists( 'bbapp_is_tutor_lms_plugins_active' ) && ! bbapp_is_tutor_lms_plugins_active() ) {
			// don't load if tutorLMS/tutorLMS pro is not enabled.
			return;
		}

		require_once __DIR__ . '/functions.php';

		RestAPIv1::instance();

		add_action( 'rest_api_init', array( $this, 'tutor_lms_init' ) );
		add_filter( 'rest_request_after_callbacks', array( $this, 'override_tutor_rest_permission_check' ), 99, 3 );
	}

	/**
	 * Override tutor rest permission check.
	 *
	 * @param WP_REST_Response $response Response.
	 * @param array            $handler  Handler.
	 * @param WP_REST_Request  $request  Request.
	 *
	 * @since 2.2.80
	 * @return WP_REST_Response
	 */
	public function override_tutor_rest_permission_check( $response, $handler, $request ) {
		if (
			RestAuth::is_tutor_api_request() &&
			! empty( $request->get_header( 'Bypass-Tutor' ) ) &&
			! empty( $handler['permission_callback'] ) &&
			end( $handler['permission_callback'] ) === 'process_api_request'
		) {
			$token     = $request->get_header( 'accesstoken' );
			$user_data = Jwt::instance()->verify_token( $token );

			if ( ! is_wp_error( $user_data ) && ! empty( $user_data['user_id'] ) ) {
				wp_set_current_user( $user_data['user_id'] );
			}

			// Continue with the default permission check.
			$response = call_user_func( $handler['callback'], $request );
		}

		return $response;
	}

	/**
	 * Load the integrations.
	 *
	 * @since 2.2.80
	 */
	public function tutor_lms_init() {
		/**
		 * TutorLMS wishlist course.
		 */
		register_rest_field(
			'user',
			'wish_listed_courses',
			array(
				'get_callback'    => array( $this, 'get_wish_listed_courses' ),
				'update_callback' => array( $this, 'update_wish_listed_courses' ),
				'schema'          => array(
					'description' => __( 'Wish listed courses', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit' ),
				),
			)
		);

		add_filter( 'tutor_rest_course_single_post', array( $this, 'bbapp_tutor_rest_course_single_post' ) );
		add_filter( 'tutor_course_additional_info', array( $this, 'bbapp_tutor_course_additional_info' ) );
		add_filter( 'tutor_rest_api_response', array( $this, 'tutor_rest_response_extra_data' ) );
		add_filter( 'edd_download_price_after_html', array( $this, 'override_edd_product_price' ), 10, 3 );
		add_filter( 'tutor_enrollment_buttons', array( $this, 'pmpro_tutor_enrollment_buttons' ), 10, 4 );

		if ( RestAuth::is_tutor_api_request() ) {
			$all_headers = Common::instance()->getallheaders();

			if ( ! empty( $all_headers['accesstoken'] ) ) {
				global $tutor;

				// Use reflection to access the private property.
				$reflection = new \ReflectionClass( $tutor );
				$property   = $reflection->getProperty( 'rest_auth' );

				// Set the property to be accessible.
				$property->setAccessible( true );

				// Get the value of the private property.
				$rest_auth = $property->getValue( $tutor );
				remove_filter( 'determine_current_user', array( $rest_auth, 'api_auth' ) );
			}
		}
	}

	/**
	 * Get the instance of the class.
	 *
	 * @param WP_Post $post Post object.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function bbapp_tutor_rest_course_single_post( $post ) {
		$course_id      = $post->ID;
		$total_enrolled = 0;

		if ( tutor_utils()->get_option( 'enable_course_total_enrolled' ) ) {
			$total_enrolled = apply_filters( 'tutor_course_students', tutor_utils()->count_enrolled_users_by_course( $course_id ), $course_id );
		}

		$extra_data                    = array();
		$extra_data['total_enrolled']  = $total_enrolled;
		$extra_data['is_wish_listed']  = tutor_utils()->is_wishlisted( $course_id );
		$extra_data['is_enrolled']     = tutor_utils()->is_enrolled( $course_id, get_current_user_id() );
		$extra_data['bb_access']       = $this->bb_access_controls_data( $post );
		$extra_data['instructors']     = $this->bbapp_get_tutor_instructors_by_course( $course_id );
		$extra_data['course_progress'] = tutor_utils()->get_course_completed_percent( $course_id, 0, true );

		$post->bbapp_data = $extra_data;

		return $post;
	}

	/**
	 * Course additional info.
	 *
	 * @param array $detail Detail.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function bbapp_tutor_course_additional_info( $detail ) {
		// Get the requested URI from $_SERVER variable.
		// Todo: It's not safe to use.
		$request_uri = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$url_parts   = wp_parse_url( $request_uri );

		if ( isset( $url_parts['path'] ) ) {
			$path          = $url_parts['path'];
			$path_segments = explode( '/', $path );

			if ( in_array( 'course-detail', $path_segments, true ) ) {
				$course_id       = end( $path_segments );
				$post            = get_post( $course_id );
				$GLOBALS['post'] = $post; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited

				setup_postdata( $post );

				$total_enrolled = 0;
				$last_update    = null;
				$is_enrolled    = tutor_utils()->is_enrolled( $course_id, get_current_user_id() );
				$can_review     = (bool) $is_enrolled;
				$my_rating      = tutor_utils()->get_reviews_by_user( get_current_user_id(), 0, 150, false, $course_id, array( 'approved', 'hold' ) );
				$is_new         = ! $my_rating || empty( $my_rating->rating ) || empty( $my_rating->comment_content );
				$post_date      = is_object( $is_enrolled ) && isset( $is_enrolled->post_date ) ? $is_enrolled->post_date : '';

				if ( tutor_utils()->get_option( 'enable_course_total_enrolled' ) ) {
					$total_enrolled = apply_filters( 'tutor_course_students', tutor_utils()->count_enrolled_users_by_course( $course_id ), $course_id );
				}

				if ( get_tutor_option( 'enable_course_update_date' ) ) {
					$last_update = get_the_modified_date( get_option( 'date_format' ), $course_id );
				}

				$detail['ratings']           = tutor_utils()->get_course_rating( $course_id );
				$detail['ratings_settings']  = array(
					'can_review'  => $can_review,
					'review_mode' => $is_new ? 'add' : 'edit',
				);
				$detail['instructors']       = $this->bbapp_get_tutor_instructors_by_course( $course_id );
				$detail['total_enrolled']    = $total_enrolled;
				$detail['last_update']       = array(
					'date'         => $last_update,
					'modified'     => get_post_datetime( $course_id, 'modified' ),
					'modified_gmt' => get_post_datetime( $course_id, 'modified', 'gmt' ),
				);
				$detail['you_enrolled_date'] = tutor_i18n_get_formated_date( $post_date, get_option( 'date_format' ) );
				$detail['is_enrolled']       = $is_enrolled;
				$detail['course_nav_item']   = bbapp_tutor_course_nav_items( $course_id, get_current_user_id() );
				$detail['certificate_url']   = CoursesRest::instance()->get_certificates( $post );
			}
		}

		return $detail;
	}

	/**
	 * Add extra data to the tutor response.
	 *
	 * @param WP_REST_Response $response Response.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function tutor_rest_response_extra_data( $response ) {
		$request_uri_server = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$response_data      = $response->get_data();

		if ( str_contains( $request_uri_server, '/tutor/v1/course-rating' ) ) {
			if ( ! empty( $response_data['data'] ) && ! empty( $response_data['data']->reviews ) ) {
				foreach ( $response_data['data']->reviews as $review ) {
					$review->user_avatar = get_avatar_url( $review->user_id );
				}
			}

			$course_id                         = ( ! empty( $_REQUEST['id'] ) ) ? absint( $_REQUEST['id'] ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$response_data['data']->my_reviews = array();

			if ( ! empty( $course_id ) ) {
				$current_user_review = array();
				$my_reviews          = tutor_utils()->get_reviews_by_user(
					get_current_user_id(),
					0,
					0,
					true,
					$course_id,
					array(
						'hold',
					)
				);

				if ( ! empty( $my_reviews->results ) ) {
					foreach ( $my_reviews->results as $key => $my_review ) {
						$current_user_review[ $key ]              = $my_review;
						$current_user_review[ $key ]->user_avatar = get_avatar_url( $my_review->user_id );
					}

					$response_data['data']->my_reviews = $current_user_review;
				}
			}
		}

		if ( str_contains( $request_uri_server, '/course-announcement' ) ) {
			if ( ! empty( $response_data['data'] ) ) {
				foreach ( $response_data['data'] as $announcement ) {
					$author                      = get_post_field( 'post_author', $announcement->ID );
					$announcement->author_avatar = get_avatar_url( $author );
					$announcement->author_name   = bbaap_get_user_display_name( $author );
					$announcement->date          = esc_html( sprintf( '%1$s %2$s', human_time_diff( strtotime( get_post_field( 'post_date_gmt', $announcement->ID ) ) ), esc_html__( 'ago', 'buddyboss-app' ) ) );
				}
			}
		}

		return $response;
	}

	/**
	 * Override EDD product price.
	 *
	 * @param string $formatted_price Formatted price.
	 * @param int    $download_id     Download id.
	 * @param string $price           Price.
	 *
	 * @since 2.2.80
	 * @return string
	 */
	public function override_edd_product_price( $formatted_price, $download_id, $price ) {
		return $price;
	}

	/**
	 * PMPro tutor enrollment buttons.
	 *
	 * @param object $buttons Buttons.
	 *
	 * @since 2.2.80
	 * @return object
	 */
	public function pmpro_tutor_enrollment_buttons( $buttons ) {
		// Get the requested URI from $_SERVER variable.
		// Todo: It's not safe to use.
		$request_uri = bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.InputNotValidated
		$url_parts   = wp_parse_url( $request_uri );

		if ( ! isset( $url_parts['path'] ) ) {
			return $buttons;
		}

		$path             = $url_parts['path'];
		$path_segments    = explode( '/', $path );
		$course_id        = end( $path_segments );
		$course_post_type = get_post_type( $course_id );

		if ( tutor()->course_post_type !== $course_post_type ) {
			return $buttons;
		}

		if ( 'pmpro' === tutor_utils()->get_option( 'monetize_by' ) ) {
			$access_require = $this->has_course_access( $course_id );

			if ( true === $access_require ) {
				return $buttons;
			}

			$buttons->show_enroll_btn      = false;
			$buttons->show_add_to_cart_btn = true;
		} elseif ( 'restrict-content-pro' === tutor_utils()->get_option( 'monetize_by' ) && function_exists( 'rcp_user_can_access' ) && ! rcp_user_can_access( get_current_user_id(), $course_id ) ) {
			$buttons->show_enroll_btn      = false;
			$buttons->show_add_to_cart_btn = true;
		}

		return $buttons;
	}

	/**
	 * Get tutor instructors by course.
	 *
	 * @param int $course_id Course id.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function bbapp_get_tutor_instructors_by_course( $course_id ) {
		$instructors = tutor_utils()->get_instructors_by_course( $course_id );

		foreach ( $instructors as $key => $instructor ) {
			if ( function_exists( 'bb_theme_enable_tutorlms_override' ) && bb_theme_enable_tutorlms_override() ) {
				$profile_url      = trailingslashit( bp_core_get_user_domain( $instructor->ID ) . bp_get_profile_slug() );
				$tutor_avatar_url = get_avatar_url( $instructor->ID, array( 'size' => 50 ) );
			} else {
				$profile_url      = tutor_utils()->profile_url( $instructor->ID, true );
				$profile_photo    = get_user_meta( $instructor->ID, '_tutor_profile_photo', true );
				$tutor_avatar_url = wp_get_attachment_image_url( $profile_photo, 'thumbnail' );
			}

			if ( empty( $tutor_avatar_url ) && function_exists( 'bp_core_fetch_avatar' ) ) {
				$tutor_avatar_url = bp_core_fetch_avatar(
					array(
						'item_id' => $instructor->ID,
						'html'    => false,
					)
				);
			}

			$instructor->profile_url = $profile_url;
			$instructor->avatar_url  = $tutor_avatar_url;
		}

		return $instructors;
	}

	/**
	 * Get bb access controls data.
	 *
	 * @param WP_Post $post Post.
	 *
	 * @since 2.2.80
	 * @return mixed|null
	 */
	public function bb_access_controls_data( $post ) {
		$rule_data  = AccessRule::instance()->get_access_rule( $post->ID, $post->post_type );
		$can_access = true;

		// Check if stored access group conditions are still exist in code.
		if ( ! empty( $rule_data['rule_data'] ) ) {
			$can_access = bb_access_controls_user_can_access_rule( $rule_data['rule_data'] );
		}

		$bb_access['can_access']       = $can_access;
		$bb_access['restrict_message'] = ( true !== $can_access && isset( $rule_data['rule_data']['restricted_message'] ) ) ? bb_access_controls_get_restrict_message( $rule_data['rule_data']['restricted_message'] ) : null;

		/**
		 * Filter the bb access controls data.
		 *
		 * @param array   $bb_access Access controls data.
		 * @param WP_Post $post      Post.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'tutor_bb_access_controls_data', $bb_access, $post );
	}

	/**
	 * Get wish listed courses.
	 *
	 * @param array $user User.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function get_wish_listed_courses( $user ) {
		return get_user_meta( $user['id'], '_tutor_course_wishlist' );
	}

	/**
	 * Update wish listed courses.
	 *
	 * @param string|array $course_ids Course ids.
	 * @param object       $user       User.
	 *
	 * @since 2.2.80
	 * @return void
	 */
	public function update_wish_listed_courses( $course_ids, $user ) {
		$course_wishlist = get_user_meta( $user->ID, '_tutor_course_wishlist' );
		foreach ( $course_ids as $course_id ) {
			if ( in_array( $course_id, $course_wishlist, true ) ) {
				delete_user_meta( $user->ID, '_tutor_course_wishlist', $course_id );
			} else {
				add_user_meta( $user->ID, '_tutor_course_wishlist', $course_id );
			}
		}
	}

	/**
	 * Course nav items
	 *
	 * @param int $course_id Course id.
	 * @param int $user_id   User id.
	 *
	 * @since 2.2.80
	 *
	 * @return mixed
	 */
	public function course_nav_items( $course_id, $user_id = 0 ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		/**
		 * If current user has course content then enrollment is not
		 * required
		 *
		 * @since 2.2.80
		 */
		$is_require_enrollment = ! tutor_utils()->has_user_course_content_access( $user_id, $course_id );
		$array                 = array(
			'info'          => array(
				'title' => __( 'Course Info', 'tutor' ),
			),
			'reviews'       => array(
				'title' => __( 'Reviews', 'tutor' ),
			),
			'questions'     => array(
				'title'             => __( 'Q&A', 'tutor' ),
				'require_enrolment' => $is_require_enrollment,
			),
			'announcements' => array(
				'title'             => __( 'Announcements', 'tutor' ),
				'require_enrolment' => $is_require_enrollment,
			),
		);

		return $array;
	}

	/**
	 * Enable/Disable course nav items
	 *
	 * @param array $items Course nav items.
	 * @param int   $course_id Course id.
	 * @param int   $user_id User id.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function enable_disable_course_nav_items( $items, $course_id, $user_id = 0 ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$enable_q_and_a_on_course     = (bool) get_tutor_option( 'enable_q_and_a_on_course' );
		$disable_course_announcements = ! (bool) tutor_utils()->get_option( 'enable_course_announcements', true, true );
		$disable_qa_for_this_course   = get_post_meta( $course_id, '_tutor_enable_qa', true ) !== 'yes';

		// Whether Q&A enabled.
		if ( ! $enable_q_and_a_on_course || $disable_qa_for_this_course ) {
			if ( tutor_utils()->array_get( 'questions', $items ) ) {
				unset( $items['questions'] );
			}
		}

		// Whether announcment enabled.
		if ( $disable_course_announcements ) {
			if ( tutor_utils()->array_get( 'announcements', $items ) ) {
				unset( $items['announcements'] );
			}
		}

		// Hide review section if disabled.
		if ( ! get_tutor_option( 'enable_course_review' ) ) {
			unset( $items['reviews'] );
		}

		// Whether enrollment require.
		$is_enrolled = tutor_utils()->is_enrolled( $course_id, $user_id );

		return array_filter(
			$items,
			function ( $item ) use ( $is_enrolled ) {
				if ( isset( $item['require_enrolment'] ) && $item['require_enrolment'] ) {
					return $is_enrolled;
				}

				return true;
			}
		);
	}

	/**
	 * Course attachments course nav item
	 *
	 * @param array $items Course nav items.
	 * @param int   $course_id Course id.
	 * @param int   $user_id User id.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function course_attachments_course_nav_item( $items, $course_id, $user_id = 0 ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		/**
		 * Check settings if admin & instructor as course access and
		 * current user has permission to edit course then user should
		 * access course attachments without enrollment.
		 *
		 * @since v2.0.5
		 */
		$is_enabled           = tutor_utils()->get_option( 'course_content_access_for_ia' );
		$can_user_edit_course = tutor_utils()->can_user_edit_course( $user_id, $course_id );
		$require_enrolment    = ! ( $is_enabled && $can_user_edit_course ); // Admin and instructor of the course can see resource tab.

		if ( $course_id ) {
			$items['resources'] = array(
				'title'             => __( 'Resources', 'tutor-pro' ),
				'require_enrolment' => $require_enrolment,
			);
		}

		return $items;
	}

	/**
	 * Stream tab.
	 *
	 * @param array $nav_menus nav menu items.
	 * @param int   $course_id course id.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function google_classroom_stream_tab( $nav_menus, $course_id ) {
		if ( ( new Classroom( null, null, true ) )->is_google_class( $course_id ) && $this->is_stream_enabled( $course_id ) ) {
			$nav_menus['google-classroom-stream'] = array(
				'title'             => __( 'Stream', 'tutor-pro' ),
				'require_enrolment' => true,
			);
		}

		return $nav_menus;
	}

	/**
	 * Course nav item
	 *
	 * @param array $items     items.
	 * @param int   $course_id course id.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function grade_book_add_course_nav_item( $items, $course_id ) {
		if ( $course_id ) {
			$gading_content = get_grading_contents_by_course_id( $course_id );

			if ( tutor_utils()->count( $gading_content ) ) {
				$items['gradebook'] = array(
					'title'             => __( 'Gradebook', 'tutor-pro' ),
					'require_enrolment' => true,
				);
			}
		}

		return $items;
	}

	/**
	 * Check stream enabled
	 *
	 * @param int $course_id course id.
	 *
	 * @since 2.2.80
	 * @return boolean
	 */
	public function is_stream_enabled( $course_id = null ) {
		if ( ! $course_id ) {
			$course_id = get_the_ID();
		}

		$value = get_post_meta( $course_id, 'tutor_gc_enable_classroom_stream', true );

		return ( empty( $value ) || 'yes' === $value );
	}

	/**
	 * Just check if has membership access
	 *
	 * @param int $course_id course id.
	 * @param int $user_id user id.
	 *
	 * @since 2.2.80
	 * @return array|object|\stdClass[]|true|null
	 */
	private function has_course_access( $course_id, $user_id = null ) {
		if ( ! tutor_utils()->has_pmpro( true ) ) {
			// Check if monetization is pmpro and the plugin exists.
			return true;
		}

		// Prepare data.
		$user_id           = null === $user_id ? get_current_user_id() : $user_id;
		$has_course_access = false;

		// Get all membership levels of this user.
		$levels                         = $user_id ? pmpro_getMembershipLevelsForUser( $user_id ) : array();
		! is_array( $levels ) ? $levels = array() : 0;

		// Get course categories by id.
		$terms    = get_the_terms( $course_id, 'course-category' );
		$term_ids = array_map(
			function ( $term ) {
				return $term->term_id;
			},
			( is_array( $terms ) ? $terms : array() )
		);

		$required_cats = $this->required_levels( $term_ids );
		if ( is_array( $required_cats ) && ! count( $required_cats ) && ! $this->has_any_full_site_level() ) {
			// Has access if no full site level and the course has no category.
			return true;
		}

		// Check if any level has access to the course.
		foreach ( $levels as $level ) {
			// Remove enrolment of expired levels.
			$endtime = (int) $level->enddate;
			if ( 0 < $endtime && $endtime < tutor_time() ) {
				// Remove here.
				continue;
			}

			if ( $has_course_access ) {
				// No need further check if any level has access to the course.
				continue;
			}

			$model = get_pmpro_membership_level_meta( $level->id, 'tutor_pmpro_membership_model', true );

			if ( PaidMembershipsPro::FULL_WEBSITE_MEMBERSHIP === $model ) {
				// If any model of the user is full site then the user has membership access.
				$has_course_access = true;

			} elseif ( PaidMembershipsPro::CATEGORY_WISE_MEMBERSHIP === $model ) {
				// Check this course if attached to any category that is linked with this membership.
				$member_cats = pmpro_getMembershipCategories( $level->id );
				$member_cats = array_map(
					function ( $member ) {
						return (int) $member;
					},
					( is_array( $member_cats ) ? $member_cats : array() )
				);

				// Check if the course id in the level category.
				foreach ( $term_ids as $term_id ) {
					if ( in_array( (int) $term_id, array_map( 'intval', $member_cats ), true ) ) {
						$has_course_access = true;
						break;
					}
				}
			}
		}

		return $has_course_access ? true : $this->required_levels( $term_ids, true );
	}

	/**
	 * Required levels
	 *
	 * @param mixed   $term_ids   term ids.
	 * @param boolean $check_full check full.
	 *
	 * @since 2.2.80
	 * @return array|object|\stdClass[]|null
	 */
	private function required_levels( $term_ids, $check_full = false ) {
		global $wpdb;

		$cat_clause = count( $term_ids ) ? ( $check_full ? ' OR ' : '' ) . " (meta.meta_value='category_wise_membership' AND cat_table.category_id IN (" . implode( ',', $term_ids ) . '))' : '';
		$query_last = ( $check_full ? " meta.meta_value='full_website_membership' " : '' ) . $cat_clause;
		$query_last = ( ! $query_last || ctype_space( $query_last ) ) ? '' : ' AND (' . $query_last . ')';

		return $wpdb->get_results( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			"SELECT DISTINCT level_table.*
            FROM {$wpdb->pmpro_membership_levels} level_table
                LEFT JOIN {$wpdb->pmpro_memberships_categories} cat_table ON level_table.id=cat_table.membership_id
                LEFT JOIN {$wpdb->pmpro_membership_levelmeta} meta ON level_table.id=meta.pmpro_membership_level_id
            WHERE
                meta.meta_key='tutor_pmpro_membership_model' " . $query_last //phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		);
	}

	/**
	 * Check has any full site level.
	 *
	 * @since 2.2.80
	 * @return boolean
	 */
	private function has_any_full_site_level() {
		global $wpdb;

		$count = $wpdb->get_var( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			"SELECT level_table.id
            FROM {$wpdb->pmpro_membership_levels} level_table
                INNER JOIN {$wpdb->pmpro_membership_levelmeta} meta ON level_table.id=meta.pmpro_membership_level_id
            WHERE
                meta.meta_key='tutor_pmpro_membership_model' AND
                meta.meta_value='full_website_membership'"
		);

		return (int) $count;
	}
}
