<?php
/**
 * Notification Rest API v2.
 *
 * @package BuddyBossApp\Api\Notification\V2
 */

namespace BuddyBossApp\Api\Notification\V2;

use BuddyBossApp\Notification\Notification;
use BuddyBossApp\Notification\Push;
use WP_REST_Request;
use WP_REST_Response;

/**
 * BuddyBossApp Notification.
 * This Notification uses the same response as the latest platform API.
 *
 * Class RestAPIv2
 *
 * @package BuddyBossApp\Api\Notification\V2
 */
class RestAPI extends \WP_REST_Controller {

	/**
	 * The single instance of the class.
	 *
	 * @var null $instance
	 */
	public static $instance;

	/**
	 * Get the instance of class
	 *
	 * @return RestAPI
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->namespace = 'buddyboss-app/core/v2';
		$this->rest_base = 'notifications';
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
		add_filter( 'bp_rest_notifications_prepare_value', array( $this, 'add_title_property' ), 10, 3 );
	}

	/**
	 * Register the component routes.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				'args'   => array(
					'id' => array(
						'description' => __( 'A unique numeric ID for the notification.', 'buddyboss-app' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param(
							array(
								'default' => 'view',
							)
						),
					),
				),
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::EDITABLE ),
				),
				array(
					'methods'             => \WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve notifications.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 */
	public function get_items( $request ) {
		$args = array(
			'user_id'           => $request['user_id'],
			'item_id'           => $request['item_id'],
			'secondary_item_id' => $request['secondary_item_id'],
			'component_name'    => $request['component_name'],
			'component_action'  => $request['component_action'],
			'order_by'          => $request['order_by'],
			'sort_order'        => strtoupper( $request['sort_order'] ),
			'is_new'            => $request['is_new'],
			'page'              => $request['page'],
			'per_page'          => $request['per_page'],
		);

		if ( empty( $request['component_action'] ) ) {
			$args['component_action'] = false;
		}

		if ( empty( $request['component_name'] ) ) {
			$args['component_name'] = function_exists( 'bp_notifications_get_registered_components' ) ? bp_notifications_get_registered_components() : false;
		}

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 */
		$args = apply_filters( 'bbapp_rest_notifications_get_items_query_args', $args, $request );

		/**
		 * When BuddyPress or Platform is Present Then Use Them
		 */
		if ( class_exists( 'BP_Notifications_Notification' ) ) {

			// Actually, query it.
			$notifications = \BP_Notifications_Notification::get( $args );

		} else {
			/**
			 * When BuddyBoss Platform is not present then use the fall back to BuddyBoss App.
			 */
			// Actually, query it.
			$notifications = Notification::instance()->get_inbuilt_notifications( $args, true );

			/**
			 * Add is_new property. for making it compatible.
			 */
			foreach ( $notifications as $k => $v ) {
				$notifications[ $k ]->is_new = $v->unread;
			}
		}

		$retval = array();
		foreach ( $notifications as $notification ) {
			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $notification, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		/**
		 * Fires after notifications are fetched via the REST API.
		 *
		 * @param array             $notifications Fetched notifications.
		 * @param \WP_REST_Response $response      The response data.
		 * @param \WP_REST_Request  $request       The request sent to the API.
		 */
		do_action( 'bbapp_rest_notifications_get_items', $notifications, $response, $request );
		return $response;
	}


	/**
	 * Check if a given request has access to notification items.
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 *
	 * @return \WP_Error|bool
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() || ( get_current_user_id() !== $request['user_id'] && ! $this->can_see() ) ) {
			$retval = new \WP_Error(
				'rest_not_logged_in',
				__( 'Sorry, you are not allowed to see notifications.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		/**
		 * Filter the notifications `get_items` permissions check.
		 *
		 * @param bool|\WP_Error   $retval  Returned value.
		 * @param \WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_notifications_get_items_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve a notification.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_REST_Response
	 */
	public function get_item( $request ) {
		$notification = $this->get_notification_object( $request );

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $notification, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a notification is fetched via the REST API.
		 *
		 * @param \BP_Notifications_Notification $notification Fetched notification.
		 * @param \WP_REST_Response              $response     The response data.
		 * @param \WP_REST_Request               $request      The request sent to the API.
		 */
		do_action( 'bbapp_rest_notifications_get_item', $notification, $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to get information about a specific notification.
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 *
	 * @return \WP_Error|bool
	 */
	public function get_item_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = new \WP_Error(
				'rest_not_logged_in',
				__( 'Sorry, you are not allowed to see the notification.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		$notification = $this->get_notification_object( $request );

		if ( true === $retval && ( empty( $notification ) || is_null( $notification->item_id ) ) ) {
			$retval = new \WP_Error(
				'bbapp_rest_notification_invalid_id',
				__( 'Invalid notification ID.', 'buddyboss-app' ),
				array(
					'status' => 404,
				)
			);
		}

		if ( true === $retval && ! $this->can_see( $notification->id ) ) {
			$retval = new \WP_Error(
				'rest_not_logged_in',
				__( 'Sorry, you cannot view this notification.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		/**
		 * Filter the notifications `get_item` permissions check.
		 *
		 * @param bool|\WP_Error   $retval  Returned value.
		 * @param \WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_notifications_get_item_permissions_check', $retval, $request );
	}

	/**
	 * Update a notification.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function update_item( $request ) {
		// Setting context.
		$request->set_param( 'context', 'edit' );

		$notification = $this->get_notification_object( $request );

		if ( $request['is_new'] === $notification->is_new ) {
			return new \WP_Error(
				'bbapp_rest_user_cannot_update_notification_status',
				__( 'Notification is already with the status you are trying to update into.', 'buddyboss-app' ),
				array(
					'status' => 500,
				)
			);
		}

		if ( class_exists( 'BP_Notifications_Notification' ) ) {

			$updated = \BP_Notifications_Notification::update(
				array( 'is_new' => $request['is_new'] ),
				array( 'id' => $notification->id )
			);

		} else {

			global $wpdb;

			$table_name = bbapp_get_network_table( 'bbapp_notifications' );
			$updated    = $wpdb->update( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$table_name,
				array(
					'unread' => $request['is_new'],
				),
				array(
					'id'      => $notification->id,
					'blog_id' => get_current_blog_id(),
				)
			);

		}

		if ( ! (bool) $updated ) {
			return new \WP_Error(
				'bbapp_rest_user_cannot_update_notification',
				__( 'Cannot update the status of this notification.', 'buddyboss-app' ),
				array(
					'status' => 500,
				)
			);
		}

		// Make sure to update the status of the notification.
		$notification = $this->prepare_item_for_database( $request );

		// Update additional fields.
		$fields_update = $this->update_additional_fields_for_object( $notification, $request );

		if ( is_wp_error( $fields_update ) ) {
			return $fields_update;
		}

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $notification, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a notification is updated via the REST API.
		 *
		 * @param \BP_Notifications_Notification $notification The updated notification.
		 * @param \WP_REST_Response              $response     The response data.
		 * @param \WP_REST_Request               $request      The request sent to the API.
		 */
		do_action( 'bbapp_rest_notifications_update_item', $notification, $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to update a notification.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_Error|bool
	 */
	public function update_item_permissions_check( $request ) {
		$retval = $this->get_item_permissions_check( $request );

		/**
		 * Filter the notifications `update_item` permissions check.
		 *
		 * @param bool|\WP_Error   $retval  Returned value.
		 * @param \WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_notifications_update_item_permissions_check', $retval, $request );
	}

	/**
	 * Delete a notification.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function delete_item( $request ) {
		// Setting context.
		$request->set_param( 'context', 'edit' );

		// Get the notification before it's deleted.
		$notification = $this->get_notification_object( $request );
		$previous     = $this->prepare_item_for_response( $notification, $request );

		if ( class_exists( 'BP_Notifications_Notification' ) ) {
			if ( ! \BP_Notifications_Notification::delete( array( 'id' => $notification->id ) ) ) {
				return new \WP_Error(
					'bbapp_rest_notification_invalid_id',
					__( 'Invalid notification ID.', 'buddyboss-app' ),
					array(
						'status' => 500,
					)
				);
			}
		} else {

			$delete = Notification::instance()->delete_notification(
				array(
					'id' => $notification->id,
				)
			);

		}

		// Build the response.
		$response = new \WP_REST_Response();
		$response->set_data(
			array(
				'deleted'  => true,
				'previous' => $previous->get_data(),
			)
		);

		/**
		 * Fires after a notification is deleted via the REST API.
		 *
		 * @param \BP_Notifications_Notification $notification The deleted notification.
		 * @param \WP_REST_Response              $response     The response data.
		 * @param \WP_REST_Request               $request      The request sent to the API.
		 */
		do_action( 'bbapp_rest_notifications_delete_item', $notification, $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to delete a notification.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return bool|\WP_Error
	 */
	public function delete_item_permissions_check( $request ) {
		$retval = $this->get_item_permissions_check( $request );

		/**
		 * Filter the notifications `delete_item` permissions check.
		 *
		 * @param bool|\WP_Error   $retval  Returned value.
		 * @param \WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_notifications_delete_item_permissions_check', $retval, $request );
	}

	/**
	 * Prepares notification data for return as an object.
	 *
	 * @param \BP_Notifications_Notification $notification Notification data.
	 * @param \WP_REST_Request               $request      Full details about the request.
	 *
	 * @return \WP_REST_Response
	 */
	public function prepare_item_for_response( $notification, $request ) {
		$data = array(
			'id'                => $notification->id,
			'user_id'           => $notification->user_id,
			'item_id'           => $notification->item_id,
			'secondary_item_id' => $notification->secondary_item_id,
			'component'         => $notification->component_name,
			'action'            => $notification->component_action,
			'date'              => $this->prepare_date_response( $notification->date_notified ),
			'is_new'            => $notification->is_new,
			'description'       => array(
				'rendered' => '',
			),
			'link_url'          => '',
			'rest_actions'      => '',
		);

		global $bp;

		/**
		 * If BP or Platform
		 */
		if ( class_exists( 'BP_Notifications_Notification' ) ) {

			if ( ! isset( $bp->notifications ) ) {
				$bp->notifications = new \stdClass();
			}
			if ( ! isset( $bp->notifications->query_loop ) ) {
				$bp->notifications->query_loop = new \stdClass();
			}

			$bp->notifications->query_loop->notification = $notification;
			$data['description']['rendered']             = bp_get_the_notification_description();

			/**
			 * If Builtin BuddyBoss App.
			 */
		} else {
			/**
			 * Filters notification output.
			 *
			 * @param int    $user_id           User id.
			 * @param string $component_name    Notification component.
			 * @param string $component_action  Notification component action.
			 * @param int    $item_id           Notification item id.
			 * @param int    $secondary_item_id Notification secondary item id.
			 * @param int    $notification_id   Notification ID.
			 * @param string $format            Format of return. Either 'string' or 'object'.
			 */
			$content = apply_filters(
				'bbapp_get_notification_output',
				$notification->user_id,
				$notification->component_name,
				$notification->component_action,
				$notification->item_id,
				$notification->secondary_item_id,
				false,
				'string'
			);

			if ( is_array( $content ) ) {
				$content = $content[0];
			}

			$data['description']['rendered'] = $content;

		}

		// Link URL.
		if ( ! empty( $data['description']['rendered'] ) ) {
			// Extract the first URL from Description.
			preg_match( '/\bhttps?:\/\/[^,\s()<>]+(?:\([\w\d]+\)|([^,[:punct:]\s]|\/))/', $data['description']['rendered'], $matches_url );

			if ( isset( $matches_url[0] ) && wp_http_validate_url( $matches_url[0] ) ) {
				$data['link_url'] = $matches_url[0];
			}
		}

		// Object Type.
		$component = $notification->component_name;
		$object    = $notification->component_name;
		$item_id   = $notification->item_id;
		$object_id = $notification->item_id;

		switch ( $component ) {
			case 'groups':
				if ( ! empty( $notification->item_id ) ) {
					$object = 'group';
				}
				break;
			case 'follow':
			case 'friends':
				if ( ! empty( $notification->item_id ) ) {
					$object = 'user';
				}
				break;
			default:
				if ( ! empty( $notification->secondary_item_id ) ) {
					$object    = 'user';
					$object_id = $notification->secondary_item_id;
					$item_id   = $notification->secondary_item_id;
				} else {
					$object = 'user';
				}
				break;
		}

		/**
		 * Avatars
		 * If the buddpress or buddboss platfrom is not active then get user default avatar url.
		 */
		if ( ! function_exists( 'bp_core_fetch_avatar' ) ) {
			$args                         = array( 'size' => 96 );
			$data['avatar_urls']['full']  = get_avatar_url( $notification->secondary_item_id, apply_filters( 'bbapp_avatar_full_url_args', $args ) );
			$data['avatar_urls']['thumb'] = get_avatar_url( $notification->secondary_item_id, apply_filters( 'bbapp_avatar_thumb_url_args', $args ) );
		} else {
			$data['avatar_urls'] = array(
				'full'  => bp_core_fetch_avatar(
					array(
						'item_id' => $item_id,
						'html'    => false,
						'type'    => 'full',
						'object'  => $object,
					)
				),
				'thumb' => bp_core_fetch_avatar(
					array(
						'item_id' => $item_id,
						'html'    => false,
						'object'  => $object,
					)
				),
			);
		}

		// Notification object.
		$data['object']    = $object;
		$data['object_id'] = $object_id;

		// Works with BuddyBoss Platform API and BP API.
		$data['link_url']     = ( function_exists( 'bp_rest_namespace' ) ? $this->bp_rest_link_url_update( $data['link_url'], $notification ) : $data['link_url'] );
		$data['rest_actions'] = ( function_exists( 'bp_rest_namespace' ) ? $this->bp_rest_get_notification_actions( $notification ) : '' );

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $notification, $object ) );

		/**
		 * Filter a notification value returned from the API.
		 *
		 * @param \WP_REST_Response              $response     The response data.
		 * @param \WP_REST_Request               $request      Request used to generate the response.
		 * @param \BP_Notifications_Notification $notification Notification object.
		 */
		return apply_filters( 'bp_rest_notifications_prepare_value', $response, $request, $notification );
	}

	/**
	 * Prepare a notification for create or update.
	 *
	 * @param \WP_REST_Request $request Request object.
	 *
	 * @return \stdClass
	 */
	protected function prepare_item_for_database( $request ) {
		$prepared_notification = new \stdClass();
		$schema                = $this->get_item_schema();
		$notification          = $this->get_notification_object( $request );

		if ( ! empty( $schema['properties']['id'] ) && ! empty( $notification->id ) ) {
			$prepared_notification->id = $notification->id;
		}

		if ( ! empty( $schema['properties']['user_id'] ) && isset( $request['user_id'] ) ) {
			$prepared_notification->user_id = (int) $request['user_id'];
		} elseif ( isset( $notification->user_id ) ) {
			$prepared_notification->user_id = $notification->user_id;
		} else {
			$prepared_notification->user_id = get_current_user_id();
		}

		if ( ! empty( $schema['properties']['item_id'] ) && isset( $request['item_id'] ) ) {
			$prepared_notification->item_id = $request['item_id'];
		} elseif ( isset( $notification->item_id ) ) {
			$prepared_notification->item_id = $notification->item_id;
		}

		if ( ! empty( $schema['properties']['secondary_item_id'] ) && isset( $request['secondary_item_id'] ) ) {
			$prepared_notification->secondary_item_id = $request['secondary_item_id'];
		} elseif ( isset( $notification->secondary_item_id ) ) {
			$prepared_notification->secondary_item_id = $notification->secondary_item_id;
		}

		if ( ! empty( $schema['properties']['component'] ) && isset( $request['component'] ) ) {
			$prepared_notification->component_name = $request['component'];
		} elseif ( isset( $notification->component_name ) ) {
			$prepared_notification->component_name = $notification->component_name;
		}

		if ( ! empty( $schema['properties']['action'] ) && isset( $request['action'] ) ) {
			$prepared_notification->component_action = $request['action'];
		} elseif ( isset( $notification->component_action ) ) {
			$prepared_notification->component_action = $notification->component_action;
		}

		if ( ! empty( $schema['properties']['is_new'] ) && isset( $request['is_new'] ) ) {
			$prepared_notification->is_new = $request['is_new'];
		} elseif ( isset( $notification->is_new ) ) {
			$prepared_notification->is_new = $notification->is_new;
		}

		if ( ! empty( $schema['properties']['date'] ) && isset( $request['date'] ) ) {
			$prepared_notification->date_notified = $request['date'];
		} elseif ( isset( $notification->date_notified ) ) {
			$prepared_notification->date_notified = $notification->date_notified;
		}

		/**
		 * Filters a notification before it is inserted or updated via the REST API.
		 *
		 * @param \stdClass         $prepared_notification An object prepared for inserting or updating the database.
		 * @param \WP_REST_Request $request               Request object.
		 */
		return apply_filters( 'bp_rest_notifications_pre_insert_value', $prepared_notification, $request );
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param \BP_Notifications_Notification $notification Notification item.
	 * @param string                         $object Notification object.
	 *
	 * @return array Links for the given plugin.
	 */
	protected function prepare_links( $notification, $object ) {
		$base = sprintf( '/%s/%s/', $this->namespace, $this->rest_base );

		$users_ids = $notification->user_id;
		if ( 'user' === $object && (int) $users_ids !== (int) $notification->secondary_item_id ) {
			$users_ids = array(
				$users_ids,
				$notification->secondary_item_id,
			);
		}

		// Entity meta.
		$links = array(
			'self'       => array(
				'href' => rest_url( $base . $notification->id ),
			),
			'collection' => array(
				'href' => rest_url( $base ),
			),
			'user'       => array(
				'href'       => rest_url( $this->bbapp_bp_rest_get_user_url( $users_ids ) ),
				'embeddable' => true,
			),
		);

		/**
		 * Filter links prepared for the REST response.
		 *
		 * @param array                          $links        The prepared links of the REST response.
		 * @param \BP_Notifications_Notification $notification Notification object.
		 */
		return apply_filters( 'bp_rest_notifications_prepare_links', $links, $notification );
	}

	/**
	 * Can this user see the notification?
	 *
	 * @param int $notification_id Notification ID.
	 *
	 * @return bool
	 */
	protected function can_see( $notification_id = 0 ) {

		$notification = $this->get_notification_object( $notification_id );

		// Check notification access.
		if ( ! empty( $notification ) && (int) get_current_user_id() === (int) $notification->user_id ) {
			return true;
		}

		// Moderators as well.
		if ( current_user_can( 'bp_moderate' ) || current_user_can( 'manage_options' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Get notification object.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return Notification|string A notification object.
	 */
	public function get_notification_object( $request ) {
		$notification_id = is_numeric( $request ) ? $request : (int) $request['id'];

		/**
		 * From BP OR Platform
		 */
		if ( function_exists( 'bp_notifications_get_notification' ) ) {

			$notification = bp_notifications_get_notification( $notification_id );

			if ( empty( $notification->id ) ) {
				return '';
			}
		} else {
			/**
			 * From Builtin BuddyBoss App.
			 */

			// Actually, query it.
			$notifications = Notification::instance()->get_inbuilt_notifications( array( 'id' => $notification_id ) );

			/**
			 * Add is_new property. for making it compatible.
			 */
			foreach ( $notifications as $k => $v ) {
				$notifications[ $k ]->is_new = $v->unread;
			}

			if ( ! isset( $notifications[0] ) ) {
				return '';
			}

			$notification = $notifications[0];

		}

		return $notification;
	}

	/**
	 * Select the item schema arguments needed for the EDITABLE method.
	 *
	 * @param string $method Optional. HTTP method of the request.
	 *
	 * @return array Endpoint arguments.
	 */
	public function get_endpoint_args_for_item_schema( $method = \WP_REST_Server::CREATABLE ) {
		$args = \WP_REST_Controller::get_endpoint_args_for_item_schema( $method );
		$key  = 'get_item';

		if ( \WP_REST_Server::EDITABLE === $method ) {
			$key = 'update_item';

			// Only switching the is_new property can be achieved.
			$args                      = array_intersect_key( $args, array( 'is_new' => true ) );
			$args['is_new']['default'] = 0;
		} elseif ( \WP_REST_Server::DELETABLE === $method ) {
			$key = 'delete_item';
		}

		/**
		 * Filters the method query arguments.
		 *
		 * @param array  $args   Query arguments.
		 * @param string $method HTTP method of the request.
		 */
		return apply_filters( "bp_rest_notifications_{$key}_query_arguments", $args, $method );
	}

	/**
	 * Get the notification schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'bp_notifications',
			'type'       => 'object',
			'properties' => array(
				'id'                => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'A unique numeric ID for the notification.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'integer',
				),
				'user_id'           => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'The ID of the user the notification is addressed to.', 'buddyboss-app' ),
					'type'        => 'integer',
					'default'     => get_current_user_id(),
				),
				'item_id'           => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'The ID of the item associated with the notification.', 'buddyboss-app' ),
					'type'        => 'integer',
				),
				'secondary_item_id' => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'The ID of the secondary item associated with the notification.', 'buddyboss-app' ),
					'type'        => 'integer',
				),
				'component'         => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'The name of the BuddyPress component the notification relates to.', 'buddyboss-app' ),
					'type'        => 'string',
				),
				'action'            => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'The name of the component\'s action the notification is about.', 'buddyboss-app' ),
					'type'        => 'string',
				),
				'date'              => array(
					'description' => __( 'The date the notification was created, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit' ),
				),
				'is_new'            => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'Whether it\'s a new notification or not.', 'buddyboss-app' ),
					'type'        => 'integer',
					'default'     => 1,
				),
				'description'       => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'Whether it\'s a new notification or not.', 'buddyboss-app' ),
					'type'        => 'integer',
					'default'     => 1,
					'properties'  => array(
						'rendered' => array(
							'description' => __( 'HTML description for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
							'readonly'    => true,
						),
					),
				),
				'link_url'          => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'Link URL for the notification.', 'buddyboss-app' ),
					'type'        => 'string',
				),
			),
		);

		/**
		 * Filters the notifications schema.
		 *
		 * @param array $schema The endpoint schema.
		 */
		return apply_filters( 'bp_rest_notification_schema', $this->add_additional_fields_schema( $schema ) );
	}

	/**
	 * Get the query params for the notifications collections.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params                       = parent::get_collection_params();
		$params['context']['default'] = 'view';

		// Remove the search argument.
		unset( $params['search'] );

		$params['order_by'] = array(
			'description'       => __( 'Name of the field to order according to.', 'buddyboss-app' ),
			'default'           => 'id',
			'type'              => 'string',
			'enum'              => array(
				'id',
				'date_notified',
				'item_id',
				'secondary_item_id',
				'component_name',
				'component_action',
			),
			'sanitize_callback' => 'sanitize_key',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['sort_order'] = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'default'           => 'ASC',
			'type'              => 'string',
			'enum'              => array( 'ASC', 'DESC' ),
			'sanitize_callback' => 'sanitize_key',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['component_name'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific component', 'buddyboss-app' ),
			'default'           => '',
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['component_action'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific component\'s action name.', 'buddyboss-app' ),
			'default'           => '',
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['user_id'] = array(
			'description'       => __( 'Limit result set to notifications addressed to a specific user.', 'buddyboss-app' ),
			'default'           => get_current_user_id(),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['item_id'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific item ID.', 'buddyboss-app' ),
			'default'           => 0,
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['secondary_item_id'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific secondary item ID.', 'buddyboss-app' ),
			'default'           => 0,
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['is_new'] = array(
			'description'       => __( 'Limit result set to items from specific states.', 'buddyboss-app' ),
			'default'           => true,
			'type'              => 'boolean',
			'sanitize_callback' => 'rest_sanitize_boolean',
			'validate_callback' => 'rest_validate_request_arg',
		);

		/**
		 * Filters the collection query params.
		 *
		 * @param array $params Query params.
		 */
		return apply_filters( 'bbapp_rest_notifications_collection_params', $params );
	}

	/**
	 * Check the post_date_gmt or modified_gmt and prepare any post or
	 * modified date for single post output.
	 *
	 * @param string      $date_gmt Post GMT date.
	 * @param string|null $date     Post date.
	 *
	 * @return string|null ISO8601/RFC3339 formatted datetime.
	 */
	protected function prepare_date_response( $date_gmt, $date = null ) {
		// Use the date if passed.
		if ( isset( $date ) ) {
			return mysql_to_rfc3339( $date ); // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
		}

		// Return null if $date_gmt is empty/zeros.
		if ( '0000-00-00 00:00:00' === $date_gmt ) {
			return null;
		}

		// Return the formatted datetime.
		return mysql_to_rfc3339( $date_gmt ); // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
	}

	/**
	 * Action to perform.
	 *
	 * @param \BP_Notifications_Notification $notification A notification object.
	 *
	 * @return array|string
	 */
	public function bp_rest_get_notification_actions( $notification ) {
		$component_action = $notification->component_action;

		$data = array(
			'status' => '',
			'accept' => array(),
			'reject' => array(),
		);

		switch ( $component_action ) {
			case 'friendship_accepted':
			case 'membership_request_accepted':
			case 'membership_request_rejected':
			case 'member_promoted_to_admin':
			case 'member_promoted_to_mod':
				break;

			case 'friendship_request':
				if (
					! empty( $notification->secondary_item_id )
					&& bp_is_active( 'friends' )
					&& class_exists( 'BP_Friends_Friendship' )
				) {
					$friendship = new \BP_Friends_Friendship( $notification->secondary_item_id );
					if ( $friendship->id === $notification->secondary_item_id ) {

						if ( ! empty( $friendship->is_confirmed ) ) {
							$data['status'] = __( 'Accepted', 'buddyboss-app' );
						} else {
							$data['status']             = __( 'Pending', 'buddyboss-app' );
							$data['accept']['endpoint'] = rest_url( bp_rest_namespace() . '/' . bp_rest_version() . '/' . buddypress()->friends->id . '/' . $friendship->id );
							$data['accept']['method']   = 'PATCH';
							$data['accept']['link_url'] = bp_loggedin_user_domain() . bp_get_friends_slug();

							$data['reject']['endpoint'] = rest_url( bp_rest_namespace() . '/' . bp_rest_version() . '/' . buddypress()->friends->id . '/' . $friendship->id );
							$data['reject']['method']   = 'DELETE';
							$data['reject']['link_url'] = bp_loggedin_user_domain() . bp_get_friends_slug();
						}
					} else {
						$data['status'] = __( 'Rejected', 'buddyboss-app' );
					}
				}
				break;

			case 'new_membership_request':
				if (
					! empty( $notification->secondary_item_id )
					&& bp_is_active( 'groups' )
					&& function_exists( 'groups_get_requests' )
				) {
					$group     = groups_get_group( $notification->item_id );
					$is_member = groups_is_user_member( $notification->secondary_item_id, $notification->item_id );
					if ( ! empty( $is_member ) ) {
						$data['status'] = __( 'Accepted', 'buddyboss-app' );
					} else {
						$requests = groups_get_requests(
							array(
								'user_id' => $notification->secondary_item_id,
								'item_id' => $notification->item_id,
							)
						);

						if ( ! empty( $requests ) ) {
							$current_request = current( $requests );
							if ( ! empty( $current_request->accepted ) ) {
								$data['status'] = __( 'Accepted', 'buddyboss-app' );
							} else {
								$data['status']             = __( 'Pending', 'buddyboss-app' );
								$data['accept']['endpoint'] = rest_url( bp_rest_namespace() . '/' . bp_rest_version() . '/' . buddypress()->groups->id . '/membership-requests/' . $current_request->id );
								$data['accept']['method']   = 'PATCH';
								$data['accept']['link_url'] = trailingslashit( bp_get_group_permalink( $group ) . 'members' );

								$data['reject']['endpoint'] = rest_url( bp_rest_namespace() . '/' . bp_rest_version() . '/' . buddypress()->groups->id . '/membership-requests/' . $current_request->id );
								$data['reject']['method']   = 'DELETE';
								$data['reject']['link_url'] = trailingslashit( bp_get_group_permalink( $group ) . 'members' );
							}
						} else {
							$data['status'] = __( 'Rejected', 'buddyboss-app' );
						}
					}
				}
				break;

			case 'group_invite':
				if ( bp_is_active( 'groups' ) && function_exists( 'groups_get_invites' ) ) {
					$group     = groups_get_group( $notification->item_id );
					$is_member = groups_is_user_member( $notification->user_id, $notification->item_id );
					if ( ! empty( $is_member ) ) {
						$data['status'] = __( 'Accepted', 'buddyboss-app' );
					} else {
						$invites = groups_get_invites(
							array(
								'user_id' => $notification->user_id,
								'item_id' => $notification->item_id,
							)
						);

						if ( ! empty( $invites ) ) {
							$current_invites = current( $invites );
							if ( ! empty( $current_invites->accepted ) ) {
								$data['status'] = __( 'Accepted', 'buddyboss-app' );
							} else {
								$data['status']             = __( 'Pending', 'buddyboss-app' );
								$data['accept']['endpoint'] = rest_url( bp_rest_namespace() . '/' . bp_rest_version() . '/' . buddypress()->groups->id . '/invites/' . $current_invites->id );
								$data['accept']['method']   = 'PATCH';
								$data['accept']['link_url'] = bp_get_group_permalink( $group );

								$data['reject']['endpoint'] = rest_url( bp_rest_namespace() . '/' . bp_rest_version() . '/' . buddypress()->groups->id . '/invites/' . $current_invites->id );
								$data['reject']['method']   = 'DELETE';
								$data['reject']['link_url'] = bp_get_group_permalink( $group );
							}
						} else {
							$data['status'] = __( 'Rejected', 'buddyboss-app' );
						}
					}
				}
				break;
		}

		if ( array(
			'status' => '',
			'accept' => array(),
			'reject' => array(),
		) === $data ) {
			return '';
		}

		return $data;
	}

	/**
	 * Update Link URL after request accept/reject.
	 *
	 * @param string                         $url          Link URL for the notification.
	 * @param \BP_Notifications_Notification $notification A notification object.
	 *
	 * @return string
	 */
	public function bp_rest_link_url_update( $url, $notification ) {
		$component_action = $notification->component_action;

		switch ( $component_action ) {
			case 'friendship_accepted':
			case 'membership_request_accepted':
			case 'membership_request_rejected':
			case 'member_promoted_to_admin':
			case 'member_promoted_to_mod':
				break;

			case 'friendship_request':
				if (
					! empty( $notification->secondary_item_id )
					&& bp_is_active( 'friends' )
					&& class_exists( 'BP_Friends_Friendship' )
				) {
					$friendship = new \BP_Friends_Friendship( $notification->secondary_item_id );
					if ( $friendship->id === $notification->secondary_item_id ) {
						if ( ! empty( $friendship->is_confirmed ) ) {
							$url = bp_loggedin_user_domain() . bp_get_friends_slug();
						}
					} else {
						$url = bp_loggedin_user_domain() . bp_get_friends_slug();
					}
				}
				break;

			case 'new_membership_request':
				if (
					! empty( $notification->secondary_item_id )
					&& bp_is_active( 'groups' )
					&& function_exists( 'groups_get_requests' )
				) {
					$group     = groups_get_group( $notification->item_id );
					$is_member = groups_is_user_member( $notification->secondary_item_id, $notification->item_id );
					if ( ! empty( $is_member ) ) {
						$url = trailingslashit( bp_get_group_permalink( $group ) . 'members' );
					} else {
						$requests = groups_get_requests(
							array(
								'user_id' => $notification->secondary_item_id,
								'item_id' => $notification->item_id,
							)
						);

						if ( empty( $requests ) ) {
							$url = trailingslashit( bp_get_group_permalink( $group ) . 'members' );
						}
					}
				}
				break;

			case 'group_invite':
				if ( bp_is_active( 'groups' ) && function_exists( 'groups_get_invites' ) ) {
					$group     = groups_get_group( $notification->item_id );
					$is_member = groups_is_user_member( $notification->user_id, $notification->item_id );
					if ( ! empty( $is_member ) ) {
						$url = bp_get_group_permalink( $group );
					} else {
						$invites = groups_get_invites(
							array(
								'user_id' => $notification->user_id,
								'item_id' => $notification->item_id,
							)
						);

						if ( empty( $invites ) ) {
							$url = bp_get_group_permalink( $group );
						}
					}
				}
				break;
		}

		return $url;
	}


	/**
	 * Update Members endpoint url.
	 *
	 * @param integer $user_id User's ID.
	 *
	 * @return string
	 */
	public function bbapp_bp_rest_get_user_url( $user_id ) {
		if ( function_exists( 'bp_rest_get_user_url' ) ) {
			return bp_rest_get_user_url( $user_id );
		} elseif ( function_exists( 'boss_bp_rest_get_user_url' ) ) {
			return boss_bp_rest_get_user_url( $user_id );
		}
		return '/wp/v2/users/' . $user_id;
	}

	/**
	 * BuddyBoss App Related Function Only.
	 *
	 * @param \WP_REST_Response              $response     The response data.
	 * @param \WP_REST_Request               $request      Full details about the request.
	 * @param \BP_Notifications_Notification $notification Notification object.
	 *
	 * @return WP_REST_Response
	 */
	public function add_title_property( $response, $request, $notification ) {

		$response->data['title'] = '';

		if ( 'manual_push' === $response->data['action'] ) {
			$notification = Push::instance()->get_notification( $response->data['item_id'] );
			if ( ! empty( $notification ) ) {
				$response->data['title'] = $notification->primary_text;
			}
		}

		return $response;
	}
}
