<?php
/**
 * Notification Rest API v1.
 *
 * @package BuddyBossApp\Api\Notification\V1
 */

namespace BuddyBossApp\Api\Notification\V1;

use BuddyBossApp\Notification\Notification;
use BuddyBossApp\RestErrors;

/**
 * Remove the rest API and convert it into dummy once all old codes are migrated.
 *
 * @deprecated
 * @todo :- Remove the rest API and convert it into dummy once all old codes are migrated.
 */

/**
 * Contain functionality for required additional rest api endpoints.
 * v1 Standard
 * Class RestAPI
 *
 * @package BuddyBossApp\Api\Notification\V1
 */
class RestAPI extends \WP_REST_Controller {

	/**
	 * The single instance of the class.
	 *
	 * @var null $instance
	 */
	public static $instance;

	/**
	 * Namespace slug.
	 *
	 * @var string $instance
	 */
	protected $namespace_slug = '';

	/**
	 * API namespace.
	 *
	 * @var string $instance
	 */
	protected $namespace = 'buddyboss-app/core/v1';

	/**
	 * Allow batch.
	 *
	 * @var true[] $allow_batch
	 */
	protected $allow_batch = array( 'v1' => true );

	/**
	 * Get the instance of class
	 *
	 * @return RestAPI
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Load initial hooks.
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
		add_filter( 'rest_pre_serve_request', array( $this, 'register_header' ), 20 );
		add_filter( 'bbapp_notification_api_response_entry', array( $this, 'bbapp_notification_api_response_entry' ), 99, 2 );
	}

	/**
	 * Register routes.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/notification/settings',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_notification_settings' ),
					'permission_callback' => '__return_true',
					'args'                => array(),
				),
				'allow_batch' => $this->allow_batch,
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'update_notification_settings' ),
					'permission_callback' => '__return_true',
					'args'                => array(),
				),
			)
		);
	}

	/**
	 * Notification settings.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
	 * @apiPrivate
	 * @api {GET} /wp-json/buddyboss-app/core/v1/notification/settings Core notification settings
	 * @apiName GetCoreNotificationSettings
	 * @apiGroup CoreNotifications
	 * @apiVersion 1.0.0
	 * @apiPermission Public
	 * @apiDescription Get core notification settings
	 */
	public function get_notification_settings( $request ) {
		global $bbapp_var;

		/**
		 * Check user logged in permissions.
		 */
		if ( ! is_user_logged_in() ) {
			return RestErrors::instance()->user_not_logged_in();
		}

		$settings_list = Notification::instance()->get_push_notification_subs_for_user( get_current_user_id() );

		$response = array();

		foreach ( $settings_list as $sub_name => $sub ) {
			$response[] = array_merge(
				array(
					'name' => $sub_name,
				),
				$sub
			);
		}

		return rest_ensure_response( $response );

	}

	/**
	 * Update notification settings.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
	 * @apiPrivate
	 * @api {POST} /wp-json/buddyboss-app/core/v1/notification/settings Update core notification settings
	 * @apiName UpdateCoreNotificationSettings
	 * @apiGroup CoreNotifications
	 * @apiVersion 1.0.0
	 * @apiPermission Public
	 * @apiDescription Update core notification settings
	 */
	public function update_notification_settings( $request ) {
		global $bbapp_var;

		/**
		 * Check user logged in permissions.
		 */
		if ( ! is_user_logged_in() ) {
			return RestErrors::instance()->user_not_logged_in();
		}

		$notification_settings_list = Notification::instance()->get_push_notification_subs_for_user( get_current_user_id() );

		$sub_data = Notification::instance()->get_push_notification_subs( get_current_user_id() );

		foreach ( $notification_settings_list as $notification_setting ) {
			if ( isset( $notification_setting['settings'] ) ) {
				foreach ( $notification_setting['settings'] as $sub_name => $sub ) {
					if ( isset( $request[ $sub_name ] ) && $sub['available'] ) {
						if ( 1 === (int) $request[ $sub_name ] ) {
							$sub_data[ $sub_name ] = true;
						} else {
							$sub_data[ $sub_name ] = false;
						}
					}
				}
			}
		}

		Notification::instance()->update_push_notification_subs( get_current_user_id(), $sub_data );

		return $this->get_notification_settings( $request );

	}

	/**
	 * Add the headers for notification rest API.
	 *
	 * @return mixed
	 */
	public function register_header( $response ) {
		$this->unread_notifications_count();
		return $response;
	}

	/**
	 * Get unread notifications count.
	 *
	 * @since 2.3.20
	 * @return void
	 */
	public function unread_notifications_count() {
		if ( is_user_logged_in() && function_exists( 'bbapp_notifications' ) ) {
			$count = bbapp_notifications()->get_notification_count( get_current_user_id() );
			header( 'bbapp-unread-notifications:' . $count );
			// Deprecated Support.
			header( 'bbp-unread-notifications:' . $count );
		}
	}

	/**
	 * Prepares notification data for return as an object.
	 *
	 * @param array $val Notification items.
	 *
	 * @return mixed
	 */
	public function prepare_notifications_response( $val ) {

		$response = rest_ensure_response( $val );

		$uris = array();

		$uris[] = array( 'author', rest_url( '/wp/v2/users/' . $val['author_id'] ) );

		if ( ! empty( $val['user_id'] ) && (int) $val['user_id'] !== (int) $val['author_id'] ) {
			$uris[] = array( 'author', rest_url( '/wp/v2/users/' . $val['user_id'] ) );
		}

		foreach ( $uris as $uri ) {
			$response->add_link( $uri[0], $uri[1], array( 'embeddable' => true ) );
		}

		return $response;

	}

	/**
	 * Adds the common embeddable on Notification Rest API.
	 *
	 * @param \WP_REST_Response $response The response data.
	 * @param array             $args     Get notification parameters.
	 *
	 * @return mixed
	 */
	public function bbapp_notification_api_response_entry( $response, $args ) {

		$data = $response->get_data();

		if ( ! is_array( $data ) || 'push_notification' !== $data['namespace'] ) {
			return $response;
		}

		// Due to Old API Release We have to Modify Some Properties for Manual Push Notification Type.

		$uris = array();

		if ( 'manual_push' === $data['action'] ) {

			$data['item_id']           = $data['primary_text'];
			$data['secondary_item_id'] = $data['secondary_text'];
			$data['action']            = 'manual_push_notification';

			if ( ! empty( $data['user_id'] ) ) {
				$uris[] = array( 'author', rest_url( '/wp/v2/users/' . $data['user_id'] ) );
			}
			if ( ! empty( $data['author_id'] ) ) {
				$uris[] = array( 'author', rest_url( '/wp/v2/users/' . $data['author_id'] ) );
			}

			$response->set_data( $data ); // update the modified properties.

		}

		foreach ( $uris as $uri ) {
			$response->add_link( $uri[0], $uri[1], array( 'embeddable' => true ) );
		}

		return $response;
	}
}
