<?php
/**
 * Holds Learndash API common function.
 *
 * @package BuddyBossApp\Api\LearnDash
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

if ( ! function_exists( 'bbapp_learners_response_add_total_headers' ) ) {
	/**
	 * Set headers to let the Client Script be aware of the pagination.
	 *
	 * @param WP_REST_Response $response The response data.
	 * @param integer          $total    The total number of found items.
	 * @param integer          $per_page The number of items per page of results.
	 *
	 * @return WP_REST_Response $response The response data.
	 * @since 0.1.0
	 */
	function bbapp_learners_response_add_total_headers( WP_REST_Response $response, $total = 0, $per_page = 0 ) {
		if ( ! $total || ! $per_page ) {
			return $response;
		}

		$total_items = (int) $total;
		$max_pages   = ceil( $total_items / (int) $per_page );

		$response->header( 'X-WP-Total', $total_items );
		$response->header( 'X-WP-TotalPages', (int) $max_pages );

		return $response;
	}
}

if ( ! function_exists( 'bbapp_learners_download_html' ) ) {
	/**
	 * Helper Function fro Download endpoint.
	 *
	 * @param string $folder_path The path to download.
	 * @param string $content The content to be downloaded.
	 */
	function bbapp_learners_download_html( $folder_path, $content ) {
		$content = str_replace( '&#8221;', '"', $content );
		$content = html_entity_decode( $content );

		/**
		 * Help URL https://regex101.com/r/F42TTK/1
		 */
		$pattern = "/src=['\"]((.+)?(?:.+))['\"]/iU";
		preg_match_all( $pattern, $content, $images );
		preg_replace_callback(
			$pattern,
			function ( $match ) use ( $folder_path, &$content ) {
				$assets_name = uniqid() . basename( $match[1] );

				$data = \BuddyBossApp\Helpers\BBAPP_File::read_file( $match[1] );
				bbapp_learners_write_file( $folder_path . $assets_name, $data );

				$content = str_replace( $match[1], './' . urlencode( $assets_name ), $content ); //phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.urlencode_urlencode

			},
			$content
		);

		return $content;
	}
}

if ( ! function_exists( 'bbapp_learners_setpath' ) ) {
	/**
	 * Creates Directory.
	 *
	 * @param string $dir Directory to be created.
	 */
	function bbapp_learners_setpath( $dir ) {
		$upload_dir = wp_upload_dir();
		if ( ! empty( $upload_dir['basedir'] ) ) {
			$bbapp_learners_dirname = trailingslashit( $upload_dir['basedir'] ) . trailingslashit( 'bbapp-learners' ) . $dir;
			if ( ! file_exists( $bbapp_learners_dirname ) ) {
				wp_mkdir_p( $bbapp_learners_dirname );
			}
		}

		return $bbapp_learners_dirname;
	}
}

if ( ! function_exists( 'bbapp_learners_getpath' ) ) {
	/**
	 * Get's the Directory path.
	 *
	 * @param string $dir Directory to be get.
	 */
	function bbapp_learners_getpath( $dir ) {
		$upload_dir = wp_upload_dir();
		if ( ! empty( $upload_dir['basedir'] ) ) {
			$bbapp_learners_dirname = trailingslashit( $upload_dir['basedir'] ) . trailingslashit( 'bbapp-learners' ) . $dir;
		}

		return $bbapp_learners_dirname;
	}
}

if ( ! function_exists( 'bbapp_learners_delete_directory' ) ) {
	/**
	 * To delete directory.
	 *
	 * @param string $dirname Directory to be deleted.
	 */
	function bbapp_learners_delete_directory( $dirname ) {
		if ( is_dir( $dirname ) && file_exists( $dirname ) ) {
			$dir_handle = opendir( $dirname );
		}

		if ( ! isset( $dir_handle ) || empty( $dir_handle ) ) {
			return false;
		}

		$file = readdir( $dir_handle );

		while ( $file ) {
			if ( '.' !== $file && '..' !== $file ) {
				if ( ! is_dir( $dirname . '/' . $file ) ) {
					wp_delete_file( $dirname . '/' . $file );
				} else {
					bbapp_learners_delete_directory( $dirname . '/' . $file );
				}
			}
		}

		closedir( $dir_handle );
		rmdir( $dirname );

		return true;
	}
}

if ( ! function_exists( 'bbapp_learners_geturl' ) ) {
	/**
	 * Get's the Directory url.
	 *
	 * @param string $dir Directory to be get.
	 */
	function bbapp_learners_geturl( $dir ) {
		$upload_dir = wp_upload_dir();
		if ( ! empty( $upload_dir['baseurl'] ) ) {
			$bbapp_learners_dirname = trailingslashit( $upload_dir['baseurl'] ) . trailingslashit( 'bbapp-learners' ) . $dir;
		}

		return $bbapp_learners_dirname;
	}
}

if ( ! function_exists( 'bbapp_learners_write_file' ) ) {
	/**
	 * Write file to given path.
	 *
	 * @param string $file File path to be written.
	 * @param string $content Content to be written.
	 */
	function bbapp_learners_write_file( $file, $content ) {
		\BuddyBossApp\Helpers\BBAPP_File::write_file( $file, $content );
	}
}

if ( ! function_exists( 'bbapp_learners_zip_create' ) ) {
	/**
	 * Create zip of particular folder.
	 *
	 * @param string $folder_path File path to be zipped.
	 * @param string $zip_name Zip name.
	 */
	function bbapp_learners_zip_create( $folder_path, $zip_name ) {

		$root_path = realpath( $folder_path );

		$zip_name = $zip_name . '.zip';
		$zip_path = bbapp_learners_setpath( '' );
		$zip_path = $zip_path . $zip_name;

		$zip = new \ZipArchive();
		$zip->open( $zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE );

		// Create recursive directory iterator.
		// @var SplFileInfo[] $files.
		$files = new \RecursiveIteratorIterator(
			new RecursiveDirectoryIterator( $root_path ),
			RecursiveIteratorIterator::LEAVES_ONLY
		);

		foreach ( $files as $name => $file ) {
			// Skip directories (they would be added automatically).
			if ( ! $file->isDir() ) {
				// Get real and relative path for current file.
				$file_path     = $file->getRealPath();
				$relative_path = substr( $file_path, strlen( $root_path ) + 1 );

				// Add current file to archive.
				$zip->addFile( $file_path, $relative_path );
			}
		}
		// Zip archive will be created only after closing object.
		$zip->close();

		if ( file_exists( $zip_path ) ) {
			return bbapp_learners_geturl( $zip_name );
		}

		return false;
	}
}


/**
 * Helper function
 */
if ( ! function_exists( 'bbapp_learners_fix_relative_urls_protocol' ) ) {

	/**
	 * Fix the relative content urls protocol.
	 * eg. changes //example.com to https://example.com
	 *
	 * @param string $content Post content.
	 *
	 * @return mixed
	 */
	function bbapp_learners_fix_relative_urls_protocol( $content ) {
		$content = preg_replace( '/(src|href)=["|\']\/\/(.*?)["|\']/', '$1="https://$2"', $content );

		return trim( $content );
	}
}

/**
 * Helper function
 */
if ( ! function_exists( 'build_video_cookie_key' ) ) {
	/**
	 * Build unique video progress cookie key. This is used to track the video state
	 * in the user's browser.
	 *
	 * @param WP_Post $post    Post object.
	 *
	 * @return string $cookie_key.
	 */
	function build_video_cookie_key( $post ) {
		$cookie_key    = ld_get_nonce_slug( $post );
		$step_settings = learndash_get_setting( $post );
		if ( ( isset( $step_settings['lesson_video_url'] ) ) && ( ! empty( $step_settings['lesson_video_url'] ) ) ) {
			$lesson_video_url = trim( $step_settings['lesson_video_url'] );
			$lesson_video_url = html_entity_decode( $lesson_video_url );

			$cookie_key .= '_' . $lesson_video_url;
		}
		$cookie_key = 'learndash-video-progress-' . md5( $cookie_key );

		return $cookie_key;
	}
}

/**
 * Helper function
 */
if ( ! function_exists( 'get_nonce_slug' ) ) {
	/**
	 * Utility function to get the nonce slug.
	 *
	 * @param WP_Post $post    Post object.
	 *
	 * @return string
	 * @since 3.2.3
	 */
	function ld_get_nonce_slug( $post ) {
		$step_id   = (int) $post->ID;
		$course_id = (int) learndash_get_course_id( $post->ID );
		if ( is_user_logged_in() ) {
			$user_id = (int) get_current_user_id();
		} else {
			$user_id = 0;
		}

		return 'learndash_video_' . $user_id . '_' . $course_id . '_' . $step_id;
	}
}
