<?php
/**
 * Holds topics error rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Topics
 */

namespace BuddyBossApp\Api\LearnDash\V1\Topic;

use BuddyBossApp\RestErrors;
use WP_Error;

/**
 * Topics error.
 */
class TopicsError {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return TopicsError
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * When user is not logged in this error message should be used.
	 *
	 * @return WP_Error
	 */
	public function user_not_logged_in() {
		return RestErrors::instance()->user_not_logged_in();
	}

	/**
	 * When topic id not correct.
	 *
	 * @return WP_Error
	 */
	public function invalid_topic_id() {
		return new WP_Error( 'learndash_json_topic_invalid_id', __( 'Topic not found.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * When Lesson's lesson id not correct on topic actions.
	 *
	 * @return WP_Error
	 */
	public function invalid_lesson_id() {
		return new WP_Error( 'learndash_json_topic_invalid_lesson', __( 'Topic\'s Lesson not found! For which Lesson are you performing the action?', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When Lesson's course id not correct on topic actions.
	 *
	 * @return WP_Error
	 */
	public function invalid_course_id() {
		return new WP_Error( 'learndash_json_topic_invalid_course', __( 'Topic\'s Course not found! For which Course are you performing the action?', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user don't have topic access.
	 *
	 * @return WP_Error
	 */
	public function invalid_topic_access() {
		return new WP_Error( 'learndash_json_topic_invalid_access', __( 'You do not have access to this topic.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user not completed previous Lesson/topic.
	 *
	 * @return WP_Error
	 */
	public function topic_previous_not_completed() {
		return new WP_Error( 'learndash_json_topic_previous_not_completed', __( 'You must complete each lesson/topic in sequence.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user not completed lesson's quiz and trying to complete Lesson.
	 *
	 * @return WP_Error
	 */
	public function topic_quiz_not_completed() {
		return new WP_Error( 'learndash_json_lesson_quiz_not_completed', __( 'Topic\'s quiz is not completed.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user not completed lesson's assignment and trying to complete Lesson.
	 *
	 * @return WP_Error
	 */
	public function topic_assignment_not_completed() {
		return new WP_Error( 'learndash_json_topic_assignment_not_completed', __( 'Please upload assignment to complete this topic.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user not completed topic's assignment and trying to complete topic.
	 *
	 * @param int $count Count of assignments.
	 *
	 * @since 2.1.00
	 *
	 * @return WP_Error
	 */
	public function topic_assignment_not_approved( $count ) {
		$message = sprintf( _n( 'You have an assignment awaiting approval', 'You have an assignments awaiting approval', $count, 'buddyboss-app' ), $count );

		return new WP_Error( 'learndash_json_topic_assignment_not_approved', $message, array( 'status' => 400 ) );
	}

	/**
	 * When user doesn't match required course prerequities.
	 *
	 * @param string $topic_access_from Time string to show when topic available.
	 *
	 * @since 1.7.3
	 *
	 * @return WP_Error
	 */
	public function topic_not_available( $topic_access_from ) {
		$message = sprintf( '%1$s %2$s', __( 'Available on:', 'buddyboss-app' ), esc_html( $topic_access_from ) );

		return new WP_Error( 'learndash_json_topic_not_available', $message, array( 'status' => 400 ) );
	}

	/**
	 * When Lesson complete fails.
	 *
	 * @return WP_Error
	 */
	public function topic_complete_error() {
		return new WP_Error( 'learndash_json_lesson_complete_error', __( 'Something is wrong with complete process.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When topic incomplete fails.
	 *
	 * @return WP_Error
	 * @since          1.6.4
	 */
	public function topic_incomplete_error() {
		return new WP_Error( 'learndash_json_topic_incomplete_error', __( 'Something is wrong with complete process.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When assignment dir not writeable.
	 *
	 * @return WP_Error
	 */
	public function assignment_dir_not_writeable() {
		return new WP_Error( 'learndash_json_assignment_dir_not_writeable', __( 'Unable to write to directory. Is this directory writable by the server?', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When lesson zip is not created.
	 *
	 * @return WP_Error
	 */
	public function invalid_topic_zip() {
		return new WP_Error( 'learndash_json_topic_invalid_zip', __( 'Topic Zip file not created.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}
}
