<?php
/**
 * Topics action rest.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Topic
 */

namespace BuddyBossApp\Api\LearnDash\V1\Topic;

use BuddyBossApp\Api\LearnDash\V1\Assignments\AssignmentsActionRest;
use BuddyBossApp\Api\LearnDash\V1\Course\CoursesError;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Topics action rest.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Topic
 */
class TopicsActionRest extends TopicsRest {

	/**
	 * The single instance of the class.
	 *
	 * @var null $instance
	 */
	protected static $instance;

	/**
	 * TopicsActionRest instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/complete',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'complete' ),
					'permission_callback' => array( $this, 'get_complete_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/download',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'download_item' ),
					'permission_callback' => array( $this, 'get_download_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/incomplete',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'incomplete' ),
					'permission_callback' => array( $this, 'get_incomplete_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to complete topic item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_complete_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = TopicsError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the topic `complete` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_topic_complete_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to incomplete topic item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 1.6.4
	 */
	public function get_incomplete_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = TopicsError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the topic `incomplete` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_topic_incomplete_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to download topic item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_download_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = TopicsError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the topic `get_download_permissions_check` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_topic_download_permissions_check', $retval, $request );
	}

	/**
	 * Process Topic complete.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          0.1.0
	 *
	 * @return WP_Error|WP_REST_Response|bool
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/topics/:id/complete Process LearnDash Topic Complete
	 * @apiName        PostLDTopicComplete
	 * @apiGroup       LD Topics
	 * @apiDescription Process Topic enrollment
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the topic.
	 */
	public function complete( $request ) {
		$topic_id  = is_numeric( $request ) ? $request : (int) $request['id'];
		$lesson_id = isset( $request['lesson_id'] ) ? (int) $request['lesson_id'] : 0;
		$course_id = isset( $request['course_id'] ) ? (int) $request['course_id'] : 0;
		$user_id   = get_current_user_id();

		$topic = get_post( $topic_id );

		if ( empty( $topic ) || $this->post_type !== $topic->post_type ) {
			return TopicsError::instance()->invalid_topic_id();
		}

		if ( empty( $lesson_id ) ) {
			return TopicsError::instance()->invalid_lesson_id();
		}

		if ( empty( $course_id ) ) {
			return TopicsError::instance()->invalid_course_id();
		}

		if ( ! sfwd_lms_has_access( $lesson_id, $user_id ) ) {
			return TopicsError::instance()->invalid_topic_access();
		}

		if ( function_exists( 'learndash_is_topic_complete' ) && ! learndash_is_topic_complete( $user_id, $topic_id, $course_id ) ) {
			if ( ! bbapp_lms_is_content_access( $topic, 'points_access' ) ) {
				return CoursesError::instance()->invalid_course_access_point();
			}

			if ( ! bbapp_lms_is_content_access( $topic, 'prerequities_completed' ) ) {
				return CoursesError::instance()->invalid_course_prerequities();
			}

			$topic_access_from = bbapp_lesson_visible_after( $topic );
			if ( ! empty( $topic_access_from ) && true !== $topic_access_from ) {
				$topic_access_from = learndash_adjust_date_time_display( $topic_access_from );

				return TopicsError::instance()->topic_not_available( $topic_access_from );
			}

			if ( ! bbapp_lms_is_content_access( $topic, 'previous_lesson_completed' ) || ! bbapp_lms_is_content_access( $topic, 'previous_topic_completed' ) ) {
				return TopicsError::instance()->topic_previous_not_completed();
			}

			if ( ! bbapp_lms_quiz_completed( $topic ) ) {
				return TopicsError::instance()->topic_quiz_not_completed();
			}

			if ( $this->get_lesson_assignment_upload( $topic ) ) {

				/**
				 * Process topic assignment to complete topic
				 */
				$created_assignment_id = AssignmentsActionRest::instance()->upload_attachment_file( $lesson_id, $topic_id );
				if ( is_wp_error( $created_assignment_id ) ) {
					return $created_assignment_id;
				} else {
					$completed = ( 0 === $created_assignment_id ) ? false : true;
				}
			} else {
				// To stop "Uncanny LearnDash Toolkit" Redirection after complete event for rest endpoint.
				$_REQUEST['doing_rest'] = 1;

				$completed = learndash_process_mark_complete( $user_id, $topic_id );
			}
		} else {
			/**
			 * Topic already completed
			 */
			$completed = true;
		}

		if ( is_wp_error( $completed ) ) {
			return $completed;
		} elseif ( true === $completed ) {
			return $this->get_item( $request );
		}

		return TopicsError::instance()->topic_complete_error();
	}

	/**
	 * Process Topic incomplete.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          1.6.4
	 *
	 * @return WP_Error|WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/topics/:id/incomplete Process LearnDash Topic Incomplete
	 * @apiName        PostLDTopicIncomplete
	 * @apiGroup       LD Topics
	 * @apiDescription Process Topic un-enrollment.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the topic.
	 */
	public function incomplete( $request ) {
		$topic_id  = is_numeric( $request ) ? $request : (int) $request['id'];
		$lesson_id = isset( $request['lesson_id'] ) ? (int) $request['lesson_id'] : 0;
		$course_id = isset( $request['course_id'] ) ? (int) $request['course_id'] : 0;
		$user_id   = get_current_user_id();

		$topic = get_post( $topic_id );

		if ( empty( $topic ) || $this->post_type !== $topic->post_type ) {
			return TopicsError::instance()->invalid_topic_id();
		}

		if ( empty( $lesson_id ) ) {
			return TopicsError::instance()->invalid_lesson_id();
		}

		if ( empty( $course_id ) ) {
			return TopicsError::instance()->invalid_course_id();
		}

		if ( ! sfwd_lms_has_access( $lesson_id, $user_id ) ) {
			return TopicsError::instance()->invalid_topic_access();
		}

		$incomplete = false;

		if ( function_exists( 'learndash_is_topic_complete' ) && learndash_is_topic_complete( $user_id, $topic_id ) ) {
			if ( ! bbapp_lms_is_content_access( $topic, 'points_access' ) ) {
				return CoursesError::instance()->invalid_course_access_point();
			}

			if ( ! bbapp_lms_is_content_access( $topic, 'prerequities_completed' ) ) {
				return CoursesError::instance()->invalid_course_prerequities();
			}

			$topic_access_from = bbapp_lesson_visible_after( $topic );
			if ( ! empty( $topic_access_from ) && true !== $topic_access_from ) {
				$topic_access_from = learndash_adjust_date_time_display( $topic_access_from );

				return TopicsError::instance()->topic_not_available( $topic_access_from );
			}

			if ( ! bbapp_lms_is_content_access( $topic, 'previous_lesson_completed' ) || ! bbapp_lms_is_content_access( $topic, 'previous_topic_completed' ) ) {
				return TopicsError::instance()->topic_previous_not_completed();
			}

			if ( ! bbapp_lms_quiz_completed( $topic ) ) {
				return TopicsError::instance()->topic_quiz_not_completed();
			}

			/**
			 * Process topic incomplete is assignment is not required
			 */
			$course_id = bbapp_learndash_get_course_id( $topic_id );

			$incomplete = learndash_process_mark_incomplete( $user_id, $course_id, $topic_id );

		} else {
			/**
			 * Topic already incomplete
			 */
			$incomplete = true;
		}

		if ( is_wp_error( $incomplete ) ) {
			return $incomplete;
		} elseif ( true === $incomplete ) {
			return $this->get_item( $request );
		}

		return TopicsError::instance()->topic_incomplete_error();
	}

	/**
	 * Process Topic Download.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @param string          $parent_path Parent path.
	 *
	 * @return WP_REST_Response|WP_Error
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/topics/:id/download Process LearnDash Topic Download
	 * @apiName        PostLDTopicDownload
	 * @apiGroup       LD Topics
	 * @apiDescription Process Topic Download
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the topic.
	 */
	public function download_item( $request, $parent_path = '' ) {

		$topic_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$topic    = get_post( $topic_id );

		if ( empty( $topic ) || $this->post_type !== $topic->post_type ) {
			return TopicsError::instance()->invalid_topic_id();
		}

		// Set lesson folder path.
		$folder_name = 'topic_' . $topic_id . '/';

		// Set parent path if given.
		if ( ! empty( $parent_path ) ) {
			$folder_name = trailingslashit( $parent_path ) . $folder_name;
		}

		// Delete lesson directory if exist.
		$folder_path = bbapp_learners_getpath( $folder_name );
		bbapp_learners_delete_directory( $folder_path );

		// Create topic directory.
		$folder_path = bbapp_learners_setpath( $folder_name );

		// Create assests directory for lesson.
		$assests_path = bbapp_learners_setpath( $folder_name ) . 'topics/';
		if ( ! file_exists( $assests_path ) ) {
			wp_mkdir_p( $assests_path );
		}

		$data = $this->prepare_item_for_response( $topic, $request );

		// Download topic html and assests.
		$html_attrs = apply_filters( 'bbapp_ld_topic_html_attrs', array( 'content' ) );
		foreach ( $html_attrs as $attr ) {
			if ( isset( $data->data[ $attr ]['rendered'] ) ) {
				$data->data[ $attr ]['rendered'] = bbapp_learners_download_html( $assests_path, $data->data[ $attr ]['rendered'] );
				$value                           = $data->data[ $attr ]['rendered'];
			} else {
				$data->data[ $attr ] = bbapp_learners_download_html( $assests_path, $data->data[ $attr ] );
				$value               = $data->data[ $attr ];
			}
			$file_name = 'lesson_' . $attr . '.html';
			bbapp_learners_write_file( $assests_path . $file_name, $value );
		}

		// Create topic.json.
		$file_data = wp_json_encode( $data );
		$file_name = 'topic_' . $topic_id . '.json';
		bbapp_learners_write_file( $folder_path . $file_name, $file_data );

		// Create info.json.
		$info_data = wp_json_encode(
			array(
				'id'           => $topic_id,
				'hash_key'     => sha1( $topic->post_modified ),
				'modified'     => mysql_to_rfc3339( $topic->post_modified ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
				'modified_gmt' => mysql_to_rfc3339( $topic->post_modified_gmt ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			)
		);
		$file_name = 'info.json';
		bbapp_learners_write_file( $folder_path . $file_name, $info_data );

		/**
		 * If `$parent_path` is not empty means topics data added in lesson zip so no need to create zip for tipicss.
		 */
		$zip_url = '';

		if ( empty( $parent_path ) ) {
			// Create zip file.
			$zip_url = bbapp_learners_zip_create( $folder_path, 'topic_' . $topic_id );
			if ( empty( $zip_url ) ) {
				return TopicsError::instance()->invalid_topic_zip();
			}
		}

		return rest_ensure_response( array( 'zip_url' => $zip_url ) );
	}
}
