<?php
/**
 * Holds statement different errors.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Statement
 */

namespace BuddyBossApp\Api\LearnDash\V1\Statement;

use BuddyBossApp\Api\LearnDash\V1\Core\LDRestController;
use BuddyBossApp\Api\LearnDash\V1\Course\CoursesError;
use BuddyBossApp\Api\LearnDash\V1\Lesson\LessonsError;
use BuddyBossApp\Api\LearnDash\V1\Topic\TopicsError;
use WP_Error;
use \WP_REST_Server;
use \WP_REST_Request;
use \WP_REST_Response;

/**
 * Class StatmentRest
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Statement
 */
class StatementRest extends LDRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * StatmentRest instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'statement';
		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'upload_progress' ),
					'permission_callback' => array( $this, 'get_statement_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
			)
		);
	}

	/**
	 * Check if a given request has access to start quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_statement_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = CoursesError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the quiz `statement` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_statement_permissions_check', $retval, $request );
	}

	/**
	 * Process Course Statements.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error|mixed
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/statement Process LearnDash Course Statements
	 * @apiName        PostLDCourseStatements
	 * @apiGroup       LD Statements
	 * @apiDescription Process Course Statements.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 * @apiParam {Array} [statements] statements data to perform action. action,object,item_id are required.  .
	 */
	public function upload_progress( $request ) {
		$retval       = array();
		$temp_request = $_REQUEST; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$statements   = isset( $request['statements'] ) ? $request['statements'] : array();

		if ( ! empty( $statements ) ) {
			foreach ( $statements as $statement ) {
				$action      = isset( $statement['action'] ) ? $statement['action'] : '';
				$object      = isset( $statement['object'] ) ? $statement['object'] : '';
				$object_data = isset( $statement['data'] ) ? $statement['data'] : array();

				if ( empty( $action ) || empty( $object ) ) {
					$retval[] = CoursesError::instance()->invalid_course_statement();
				}

				if ( 'completed' === $action ) {
					switch ( $object ) {
						case 'lesson':
							$course_id = isset( $statement['primary_item_id'] ) ? (int) $statement['primary_item_id'] : 0;
							$lesson_id = isset( $statement['item_id'] ) ? (int) $statement['item_id'] : 0;

							if ( empty( $course_id ) ) {
								// error empty action.
								$retval[] = CoursesError::instance()->invalid_course_id();
								break;
							}

							if ( empty( $lesson_id ) ) {
								// error empty action.
								$retval[] = LessonsError::instance()->invalid_lesson_id();
								break;
							}

							// Topdo: Change endpoint.
							$lesson_request = new WP_REST_Request( 'POST', '/buddyboss-app/learndash/v1/lessons/' . $lesson_id . '/complete' );

							$lesson_request->set_url_params( array( 'id' => $lesson_id ) );
							$lesson_request->set_param( 'course_id', $course_id );
							$lesson_request->set_body( $object_data );
							if ( ! empty( $object_data ) ) {
								foreach ( $object_data as $key => $data ) {
									$lesson_request->set_param( $key, $data );
								}
							}

							$_GET            = $lesson_request->get_params();
							$_POST           = $lesson_request->get_params();
							$_REQUEST        = $lesson_request->get_params();
							$topic_temp_file = $_FILES;
							if ( isset( $_FILES[ $lesson_id ] ) ) {
								$_FILES = $_FILES[ $lesson_id ]; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
							} else {
								unset( $_FILES );
							}
							$_FILES = $topic_temp_file;

							$server   = rest_get_server();
							$retval[] = $server->dispatch( $lesson_request );

							break;
						case 'topic':
							$course_id = isset( $statement['primary_item_id'] ) ? (int) $statement['primary_item_id'] : 0;
							$lesson_id = isset( $statement['secondary_item_id'] ) ? (int) $statement['secondary_item_id'] : 0;
							$topic_id  = isset( $statement['item_id'] ) ? (int) $statement['item_id'] : 0;

							if ( empty( $course_id ) ) {
								// error empty action.
								$retval[] = CoursesError::instance()->invalid_course_id();
								break;
							}

							if ( empty( $lesson_id ) ) {
								// error empty action.
								$retval[] = LessonsError::instance()->invalid_lesson_id();
								break;
							}

							if ( empty( $topic_id ) ) {
								// error empty action.
								$retval[] = TopicsError::instance()->invalid_topic_id();
								break;
							}

							// Topdo: Change endpoint.
							$topic_request = new WP_REST_Request( 'POST', '/buddyboss-app/learndash/v1/topics/' . $topic_id . '/complete' );

							$topic_request->set_url_params( array( 'id' => $topic_id ) );
							$topic_request->set_param( 'course_id', $course_id );
							$topic_request->set_param( 'lesson_id', $lesson_id );
							$topic_request->set_body( $object_data );
							if ( ! empty( $object_data ) ) {
								foreach ( $object_data as $key => $data ) {
									$topic_request->set_param( $key, $data );
								}
							}

							$_GET            = $topic_request->get_params();
							$_POST           = $topic_request->get_params();
							$_REQUEST        = $topic_request->get_params();
							$topic_temp_file = $_FILES;
							if ( isset( $_FILES[ $topic_id ] ) ) {
								$_FILES = $_FILES[ $topic_id ]; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized,WordPress.Security.ValidatedSanitizedInput.MissingUnslash
							} else {
								unset( $_FILES );
							}
							$_FILES = $topic_temp_file;

							$server   = rest_get_server();
							$retval[] = $server->dispatch( $topic_request );

							break;
					}
				}
			}
		}

		$_GET     = $temp_request;
		$_POST    = $temp_request;
		$_REQUEST = $temp_request;

		return $retval;
	}
}
