<?php
/**
 * Holds rest functionality for rest.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Quiz
 */

namespace BuddyBossApp\Api\LearnDash\V1\Quiz;

use BuddyBossApp\Api\LearnDash\V1\Core\LDRestController;
use BuddyBossApp\Api\LearnDash\V1\Course\CoursesError;
use LDLMS_Factory_Post;
use LearnDash_Settings_Section;
use WP_Error;
use WP_Post;
use WP_Query;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use WpProQuiz_Controller_Quiz;
use WpProQuiz_Model_Form;
use WpProQuiz_Model_FormMapper;
use WpProQuiz_Model_Quiz;
use WpProQuiz_Model_QuizMapper;

/**
 * Quiz rest class.
 */
class QuizRest extends LDRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Quiz post type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'sfwd-quiz';

	/**
	 * QuizRest instance.
	 *
	 * @since 0.1.0
	 *
	 * @return QuizRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'quiz';
		parent::__construct();
	}

	/**
	 * Check if a given request has access to Quiz items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the Quiz `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quizzes_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the quiz `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}


	/**
	 * Retrieve Quizzes.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/quiz Get LearnDash Quizzes
	 * @apiName        GetLDQuizzes
	 * @apiGroup       LD Quizzes
	 * @apiDescription Retrieve Quizzes
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure result set includes specific IDs.
	 * @apiParam {String} [after]  Limit results to those published after a given ISO8601 compliant date.
	 * @apiParam {String} [before] Limit results to those published before a given ISO8601 compliant date.
	 * @apiParam {Array} [author] Limit results to those assigned to specific authors.
	 * @apiParam {Array} [author_exclude] Ensure results to excludes those assigned to specific authors.
	 * @apiParam {String=asc,desc} [order=desc] Sort result set by given order.
	 * @apiParam {String=date,id,title,menu_order} [orderby=date] Sort result set by given field.
	 * @apiParam {Array} [parent] Limit results to those assigned to specific parent.
	 */
	public function get_items( $request ) {
		$user_id    = get_current_user_id();
		$registered = $this->get_collection_params();

		/**
		 * Filter the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$request = apply_filters( 'bbapp_ld_get_quizzes_request', $request );

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'author'         => 'author__in',
			'author_exclude' => 'author__not_in',
			'exclude'        => 'post__not_in',
			'include'        => 'post__in',
			'offset'         => 'offset',
			'order'          => 'order',
			'orderby'        => 'orderby',
			'page'           => 'paged',
			'parent'         => 'post_parent__in',
			'parent_exclude' => 'post_parent__not_in',
			'search'         => 's',
			'slug'           => 'post_name__in',
			'status'         => 'post_status',
			'per_page'       => 'posts_per_page',
		);

		/**
		 * For each known parameter which is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		// Check for & assign any parameters which require special handling or setting.
		$args['date_query'] = array();

		// Set before into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['before'], $request['before'] ) ) {
			$args['date_query'][0]['before'] = $request['before'];
		}

		// Set after into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['after'], $request['after'] ) ) {
			$args['date_query'][0]['after'] = $request['after'];
		}

		$args = $this->prepare_items_query( $args, $request );

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$args = apply_filters( 'bbapp_ld_get_quizzes_args', $args, $request );

		$args['post_type'] = $this->post_type;

		/**
		 * If User pass course id & lesson id then return topic list.
		 *  - We want list to list out topic on lesson page
		 * else return topic if user has access of the topic course
		 */
		global $wpdb;

		if ( isset( $request['course_id'] ) ) {
			$course_id = isset( $request['course_id'] ) ? (int) $request['course_id'] : false;

			/**
			 * Support lesson/topics order setting of course
			 */
			$order           = bbapp_learndash_get_course_meta_setting( $course_id, 'course_lesson_order' );
			$orderby         = bbapp_learndash_get_course_meta_setting( $course_id, 'course_lesson_orderby' );
			$default_args    = sfwd_lms_get_post_options( $this->post_type );
			$args['order']   = ! empty( $order ) ? $order : $default_args['order'];
			$args['orderby'] = ! empty( $orderby ) ? $orderby : $default_args['orderby'];

			// Get Quiz ids by course and ite's lesson and topics.
			if ( 'yes' === LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'enabled' ) ) {
				$ld_course_steps_object = LDLMS_Factory_Post::course_steps( $course_id );
				$quiz_ids               = array();

				if ( $ld_course_steps_object ) {
					$ld_course_steps_object->load_steps();
					$steps = $ld_course_steps_object->get_steps();

					if ( isset( $steps['sfwd-lessons'] ) ) {
						foreach ( $steps['sfwd-lessons'] as $lesson ) {
							if ( isset( $lesson['sfwd-topic'] ) ) {
								foreach ( $lesson['sfwd-topic'] as $topic ) {
									if ( isset( $topic['sfwd-quiz'] ) ) {
										$quiz_ids = array_merge( $quiz_ids, array_keys( $topic['sfwd-quiz'] ) );
									}
								}
							}

							if ( isset( $lesson['sfwd-quiz'] ) ) {
								$quiz_ids = array_merge( $quiz_ids, array_keys( $lesson['sfwd-quiz'] ) );
							}
						}
					}

					if ( isset( $steps['sfwd-quiz'] ) ) {
						$quiz_ids = array_merge( $quiz_ids, array_keys( $steps['sfwd-quiz'] ) );
					}
				}

				$args['post__in'] = ! empty( $quiz_ids ) ? $quiz_ids : array( - 1 );
			} else {
				$args['meta_query'][] = array(
					'key'     => 'course_id',
					'value'   => $course_id,
					'compare' => '=',
				);
			}
		} else {
			if ( 'yes' === LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'enabled' ) ) {
				$_course_ids      = ld_get_mycourses( $user_id, array() );
				$_course_str      = implode( ',', $_course_ids );
				$quiz_ids         = $wpdb->get_col( "SELECT `post_id` FROM $wpdb->postmeta WHERE meta_value IN ($_course_str) AND (`meta_key` LIKE '%ld_course_%' OR  `meta_key`LIKE 'course_id' ) ORDER BY `meta_id` ASC" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$args['post__in'] = ! empty( $quiz_ids ) ? $quiz_ids : array( - 1 );
			} else {
				$args['meta_query'][] = array(
					'key'     => 'course_id',
					'value'   => 0,
					'compare' => '!=',
				);
			}
		}

		/**
		 * Taxonomy Filter query
		 */
		$taxonomies = wp_list_filter( get_object_taxonomies( $this->post_type, 'objects' ), array( 'show_in_rest' => true ) );

		foreach ( $taxonomies as $taxonomy ) {
			$base = ! empty( $taxonomy->rest_base ) ? $taxonomy->rest_base : $taxonomy->name;

			if ( ! empty( $request[ $base ] ) ) {
				$args['tax_query'][] = array(
					'taxonomy'         => $taxonomy->name,
					'field'            => 'term_id',
					'terms'            => $request[ $base ],
					'include_children' => false,
				);
			}
		}

		$posts_query            = new WP_Query();
		$quizzes['posts']       = $posts_query->query( $args );
		$quizzes['total_posts'] = $posts_query->found_posts;

		// Todo::dips Check again this code needed or not.
		if ( $quizzes['total_posts'] < 1 ) {
			// Out-of-bounds, run the query again without LIMIT for total count.
			unset( $args['paged'] );
			$count_query = new WP_Query();
			$count_query->query( $args );
			$quizzes['total_posts'] = $count_query->found_posts;
		}

		/**
		 * Fires list of Quizzes is fetched via Query.
		 *
		 * @param array            $quizzes Fetched Quizzes.
		 * @param WP_REST_Response $args    Query arguments.
		 * @param WP_REST_Request  $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$quizzes = apply_filters( 'bbapp_ld_get_quizzes', $quizzes, $args, $request );

		$retval = array();

		foreach ( $quizzes['posts'] as $quize ) {
			if ( ! $this->check_read_permission( $quize ) ) {
				continue;
			}

			$retval[] = $this->prepare_response_for_collection( $this->prepare_item_for_response( $quize, $request ) );
		}

		$response = rest_ensure_response( $retval );
		$response = bbapp_learners_response_add_total_headers( $response, $quizzes['total_posts'], $quizzes['posts_per_page'] );

		/**
		 * Fires after a list of quizzes response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_quiz_items_response', $response, $request );

		return $response;
	}


	/**
	 * Retrieve Quiz.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/quiz/:id Get LearnDash Topic
	 * @apiName        GetLDQuiz
	 * @apiGroup       LD Quizzes
	 * @apiDescription Retrieve single Quiz
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 */
	public function get_item( $request ) {
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$quiz    = get_post( $quiz_id );

		if ( empty( $quiz ) || $this->post_type !== $quiz->post_type ) {
			return QuizError::instance()->invalid_quiz_id();
		}

		/**
		 * Fire after quiz is fetched via Query.
		 *
		 * @param array           $quiz    Fetched Quiz.
		 * @param WP_REST_Request $quiz_id Quiz id.
		 *
		 * @since 0.1.0
		 */
		$quiz = apply_filters( 'bbapp_ld_get_quiz', $quiz, $quiz_id );

		$retval   = $this->prepare_response_for_collection( $this->prepare_item_for_response( $quiz, $request ) );
		$response = rest_ensure_response( $retval );

		/**
		 * Fires after an quiz response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_quiz_item_response', $response, $request );

		return $response;
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$GLOBALS['post'] = $post; //phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited

		setup_postdata( $post );

		$context                  = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$schema                   = $this->get_public_item_schema();
		$post->has_content_access = $this->get_has_content_access( $post );

		// Base fields for every post.
		$data = array(
			'id'           => $post->ID,
			'title'        => array(
				'raw'      => $post->post_title,
				'rendered' => get_the_title( $post->ID ),
			),
			'content'      => array(
				'raw'      => ( $post->has_content_access ) ? bbapp_learners_fix_relative_urls_protocol( $post->post_content ) : '',
				'rendered' => ( $post->has_content_access ) ? bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_content', $post->post_content ) ) : '',
			),
			'date'         => mysql_to_rfc3339( $post->post_date ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'date_gmt'     => mysql_to_rfc3339( $post->post_date_gmt ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'modified'     => mysql_to_rfc3339( $post->post_modified ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'modified_gmt' => mysql_to_rfc3339( $post->post_modified_gmt ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'link'         => get_permalink( $post->ID ),
			'slug'         => $post->post_name,
			'author'       => (int) $post->post_author,
			'excerpt'      => array(
				'raw'      => bbapp_learners_fix_relative_urls_protocol( $post->post_excerpt ),
				'rendered' => bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_excerpt', $post->post_excerpt ) ),
			),
			'menu_order'   => (int) $post->menu_order,
		);

		if ( ! empty( $schema['properties']['has_course_access'] ) && in_array( $context, $schema['properties']['has_course_access']['context'], true ) ) {
			$post->has_course_access   = $this->get_has_course_access( $post );
			$data['has_course_access'] = (bool) $post->has_course_access;
		}

		if ( ! empty( $schema['properties']['has_content_access'] ) && in_array( $context, $schema['properties']['has_content_access']['context'], true ) ) {
			$data['has_content_access'] = (bool) $post->has_content_access;
		}

		/**
		 * Feature Media.
		 */
		$post->featured_media            = $this->get_feature_media( $post );
		$data['featured_media']          = array();
		$data['featured_media']['small'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['small'] ) ) ? $post->featured_media['small'] : null;
		$data['featured_media']['large'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['large'] ) ) ? $post->featured_media['large'] : null;

		if ( ! empty( $schema['properties']['mode'] ) && in_array( $context, $schema['properties']['mode']['context'], true ) ) {
			$post->mode   = $this->get_mode( $post );
			$data['mode'] = $post->mode;
		}

		if ( ! empty( $schema['properties']['course'] ) && in_array( $context, $schema['properties']['course']['context'], true ) ) {
			$post->course   = $this->get_course_id( $post );
			$data['course'] = (int) $post->course;
		}

		if ( ! empty( $schema['properties']['lesson'] ) && in_array( $context, $schema['properties']['lesson']['context'], true ) ) {
			$post->lesson   = $this->get_lesson_id( $post );
			$data['lesson'] = (int) $post->lesson;
		}

		if ( ! empty( $schema['properties']['topic'] ) && in_array( $context, $schema['properties']['topic']['context'], true ) ) {
			$post->topic   = $this->get_topic_id( $post );
			$data['topic'] = (int) $post->topic;
		}

		if ( ! empty( $schema['properties']['completed'] ) && in_array( $context, $schema['properties']['completed']['context'], true ) ) {
			$post->completed   = $this->is_completed( $post, learndash_get_course_id( $post->ID ) );
			$data['completed'] = (bool) $post->completed;
		}

		if ( ! empty( $schema['properties']['can_take_again'] ) && in_array( $context, $schema['properties']['can_take_again']['context'], true ) ) {
			$post->can_take_again   = $this->get_can_take_again( $post );
			$data['can_take_again'] = (bool) $post->can_take_again;
		}

		if ( ! empty( $schema['properties']['form'] ) && in_array( $context, $schema['properties']['form']['context'], true ) ) {
			$post->form   = $this->get_from( $post );
			$data['form'] = $post->form;
		}

		if ( ! empty( $schema['properties']['settings'] ) && in_array( $context, $schema['properties']['settings']['context'], true ) ) {
			$post->settings   = $this->get_settings( $post );
			$data['settings'] = $post->settings;
		}

		if ( ! empty( $schema['properties']['quiz_resume_activity'] ) && in_array( $context, $schema['properties']['quiz_resume_activity']['context'], true ) && isset( $data['settings']['_quiz_saving'] ) && true === $data['settings']['_quiz_saving']['quiz_resume'] ) {
			$post->quiz_resume_activity   = $this->get_quiz_resume_activity( $post );
			$data['quiz_resume_activity'] = $post->quiz_resume_activity;
		}

		if ( ! empty( $schema['properties']['error_message'] ) && in_array( $context, $schema['properties']['error_message']['context'], true ) ) {
			$post->error_message = $this->get_error_message( $post );

			if ( ! empty( $post->error_message ) ) {
				$error_code            = $post->error_message->get_error_code();
				$data['error_message'] = array(
					'code'    => $error_code,
					'message' => $post->error_message->get_error_message(),
					'data'    => $post->error_message->get_error_data( $error_code ),

				);
			} else {
				$data['error_message'] = array();
			}
		}

		if ( ! empty( $schema['properties']['materials'] ) && in_array( $context, $schema['properties']['materials']['context'], true ) ) {
			$post->materials   = $this->get_materials( $post );
			$data['materials'] = ( $post->has_content_access ) ? $post->materials : '';
		}

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filters quiz response.
		 *
		 * @type WP_REST_Response $response Quiz response.
		 * @type WP_Post          $post     Post object.
		 * @type WP_REST_Request  $request  Request object.
		 */
		return apply_filters( 'bbapp_ld_rest_prepare_quiz', $response, $post, $request );
	}

	/**
	 * Prepare items.
	 *
	 * @param array $prepared_args query parameters.
	 * @param null  $request Request parameters.
	 *
	 * @return array
	 */
	protected function prepare_items_query( $prepared_args = array(), $request = null ) {
		$query_args = array();

		foreach ( $prepared_args as $key => $value ) {
			/**
			 * Filters the query_vars used in get_items() for the constructed query.
			 *
			 * The dynamic portion of the hook name, `$key`, refers to the query_var key.
			 *
			 * @param string $value The query_var value.
			 *
			 * @since 4.7.0
			 */
			$query_args[ $key ] = apply_filters( "rest_query_var-{$key}", $value ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		}

		$query_args['ignore_sticky_posts'] = true;

		// Map to proper WP_Query orderby param.
		if ( isset( $query_args['orderby'] ) && isset( $request['orderby'] ) ) {
			$orderby_mappings = array(
				'id'            => 'ID',
				'include'       => 'post__in',
				'slug'          => 'post_name',
				'include_slugs' => 'post_name__in',
			);

			if ( isset( $orderby_mappings[ $request['orderby'] ] ) ) {
				$query_args['orderby'] = $orderby_mappings[ $request['orderby'] ];
			}
		}

		return $query_args;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array $data item object.
	 *
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {

		$links = parent::prepare_links( $data );

		// Todo::dips add filter here for course/lesson/topic.

		return $links;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author'] = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset']            = array(
			'description'       => __( 'Offset the result set by a specific number of items.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']['enum'][] = 'menu_order';

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent IDs.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent ID.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);

		$params['slug']   = array(
			'description'       => __( 'Limit result set to posts with a specific slug.', 'buddyboss-app' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['filter'] = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.', 'buddyboss-app' ),
		);

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'quiz',
			'type'       => 'object',
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'          => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content'        => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'date'           => array(
					'description' => __( 'The date the object was published, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date_gmt'       => array(
					'description' => __( 'The date the object was published, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'modified'       => array(
					'description' => __( 'The date the object was last modified, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'modified_gmt'   => array(
					'description' => __( 'The date the object was last modified, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'link'           => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'           => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'         => array(
					'description' => __( 'The ID for the author of the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'excerpt'        => array(
					'description' => __( 'The excerpt for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'featured_media' => array(
					'description' => __( 'Feature media object containing thumb and full URL of image.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'menu_order'     => array(
					'description' => __( 'The order of the object in relation to other object of its type.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
			),
		);

		$schema['properties']['mode'] = array(
			'description' => __( 'Mode of the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['course'] = array(
			'description' => __( 'The Course id for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['lesson'] = array(
			'description' => __( 'The Lesson id for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['topic'] = array(
			'description' => __( 'The Topic id for the object.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['has_course_access'] = array(
			'description' => __( 'Whether or not user have the object access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['has_content_access'] = array(
			'description' => __( 'Whether or not user have the object content access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['completed'] = array(
			'description' => __( 'The object is completed by current user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['can_take_again'] = array(
			'description' => __( 'Whether or not user can take the object again.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['form']     = array(
			'description' => __( 'Pre request form for object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view' ),
		);
		$schema['properties']['settings'] = array(
			'description' => __( 'Settings for object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view' ),
		);

		$schema['properties']['quiz_resume_activity'] = array(
			'description' => __( 'Quiz activity for object shows when quiz saving feature enable.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['materials'] = array(
			'description' => __( 'Materials for the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get feature media.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return array
	 */
	protected function get_feature_media( $post ) {
		$return = array(
			'large' => null,
			'small' => null,
		);
		$large  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'large' );
		$small  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'medium' );

		if ( isset( $large[0] ) ) {
			$return['large'] = $large[0];
		}

		if ( isset( $small[0] ) ) {
			$return['small'] = $small[0];
		}

		return $return;
	}

	/**
	 * Get course id.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return int
	 */
	protected function get_course_id( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post->ID );

		if ( ! isset( $course_id ) ) {
			$course_id = 0;
		}

		return $course_id;
	}

	/**
	 * Get lesson id.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return int
	 */
	protected function get_lesson_id( $post ) {
		$lesson_id = bbapp_learndash_get_lesson_id( $post->ID );

		if ( 'sfwd-topic' === get_post_type( $lesson_id ) ) {
			$lesson_id = bbapp_learndash_get_lesson_id( $lesson_id );
		}

		if ( ! isset( $lesson_id ) ) {
			$lesson_id = 0;
		}

		return $lesson_id;
	}

	/**
	 * Get topic id.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return int
	 */
	protected function get_topic_id( $post ) {
		$topic_id = bbapp_learndash_get_lesson_id( $post->ID );

		if ( 'sfwd-lessons' === get_post_type( $topic_id ) ) {
			unset( $topic_id );
		}

		if ( ! isset( $topic_id ) ) {
			$topic_id = 0;
		}

		return $topic_id;
	}

	/**
	 * Get quiz mode.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return string
	 */
	protected function get_mode( $post ) {
		$quiz_settings = learndash_get_setting( $post->ID );
		$quiz_pro      = $quiz_settings['quiz_pro'];
		$quiz_mapper   = new WpProQuiz_Model_QuizMapper();
		$quiz          = $quiz_mapper->fetch( $quiz_pro );
		$quiz_mode_id  = (int) $quiz->getQuizModus();
		$quiz_mode     = '';

		switch ( $quiz_mode_id ) {
			case WpProQuiz_Model_Quiz::QUIZ_MODUS_NORMAL:
				$quiz_mode = 'Normal';
				break;
			case WpProQuiz_Model_Quiz::QUIZ_MODUS_BACK_BUTTON:
				$quiz_mode = 'Back Button';
				break;
			case WpProQuiz_Model_Quiz::QUIZ_MODUS_CHECK:
				$quiz_mode = 'Check';
				break;
			case WpProQuiz_Model_Quiz::QUIZ_MODUS_SINGLE:
				$quiz_mode = 'Single';
				break;
		}

		return $quiz_mode;
	}

	/**
	 * If user has course accesss.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return mixed
	 */
	public function get_has_course_access( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post->ID );

		return sfwd_lms_has_access( $course_id );
	}

	/**
	 * Check if quiz is accessible.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return bool
	 */
	private function get_quiz_accessable( $post ) {
		if ( is_user_logged_in() ) {
			$course_id                  = bbapp_learndash_get_course_id( $post->ID );
			$lesson_progression_enabled = learndash_lesson_progression_enabled( $course_id );

			if ( ! empty( $lesson_progression_enabled ) ) {
				return $lesson_progression_enabled && function_exists( 'learndash_is_quiz_accessable' ) ? learndash_is_quiz_accessable( null, $post, false, $course_id ) : is_quiz_accessable( null, $post, false, $course_id );
			}

			return true;
		}

		return false;
	}

	/**
	 * Check if quiz is lock.
	 *
	 * @param WP_Post $post quiz post object.
	 * @param string  $type Quick is lock.
	 *
	 * @return bool
	 */
	private function check_quiz_lock( $post, $type = 'lock' ) {
		$quiz_settings = learndash_get_setting( $post->ID );
		$quiz_pro      = $quiz_settings['quiz_pro'];
		$quiz_mapper   = new WpProQuiz_Model_QuizMapper();
		$quiz          = $quiz_mapper->fetch( $quiz_pro );

		if ( $quiz->isQuizRunOnce() || $quiz->isPrerequisite() || $quiz->isStartOnlyRegisteredUser() ) {
			$quiz_id = ( ! empty( $_POST['quizId'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['quizId'] ) ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( ! empty( $quiz_id ) ) {
				$_POST['quizId'] = $quiz_pro;
			}

			$quiz_controller = new WpProQuiz_Controller_Quiz();
			$lock            = $quiz_controller->isLockQuiz( $quiz_id );

			if ( isset( $lock[ $type ] ) ) {
				if ( 'lock' === $type ) {
					return ! ( $lock['lock']['is'] && $lock['lock']['pre'] );
				} else {
					return ! ( $lock[ $type ] );
				}
			}
		}

		return true;
	}

	/**
	 * If user has content access.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return bool
	 */
	public function get_has_content_access( $post ) {
		return bbapp_lms_is_content_access( $post, 'prerequities_completed' ) &&
		       bbapp_lms_is_content_access( $post, 'points_access' ) &&
		       (
			       $this->get_quiz_accessable( $post ) ||
			       sfwd_lms_has_access( $post->ID )
		       ) &&
		       $this->check_quiz_lock( $post, 'prerequisite' ) &&
		       empty( bbapp_lesson_visible_after( $post ) );
	}

	/**
	 * Check if user can take quiz again.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return bool
	 */
	public function get_can_take_again( $post ) {
		if ( ! is_user_logged_in() ) {
			return false;
		}

		$user_id        = get_current_user_id();
		$quiz_settings  = learndash_get_setting( $post->ID );
		$repeats        = ( ! empty( $quiz_settings['repeats'] ) ) ? trim( $quiz_settings['repeats'] ) : '';
		$attempts_count = 0;

		if ( '' !== $repeats ) {
			$usermeta = get_user_meta( $user_id, '_sfwd-quizzes', true );
			$usermeta = maybe_unserialize( $usermeta );

			if ( ! is_array( $usermeta ) ) {
				$usermeta = array();
			}

			if ( ! empty( $usermeta ) ) {
				foreach ( $usermeta as $v ) {
					if ( ( (int) $v['quiz'] === $post->ID ) ) {
						$attempts_count ++;
					}
				}
			}
		}

		$bypass_course_limits_admin_users = learndash_can_user_bypass( $user_id, 'learndash_course_lesson_access_from', $post );
		// For logged-in users to allow an override filter.
		/** This filter is documented in themes/ld30/includes/helpers.php */
		$bypass_course_limits_admin_users = apply_filters( 'learndash_prerequities_bypass', $bypass_course_limits_admin_users, $user_id, $post->ID, $post );

		return ( true === $bypass_course_limits_admin_users ) || ( '' === $repeats || $repeats >= $attempts_count ) && $this->check_quiz_lock( $post );
	}

	/**
	 * This function we use in Course/Lesson/Topic endpoint so it should be public
	 *
	 * @param WP_Post $post      quiz post object.
	 * @param int     $course_id Course id.
	 *
	 * @return mixed
	 */
	public function is_completed( $post, $course_id = 0 ) {
		$user_id = get_current_user_id();

		return learndash_is_quiz_complete( $user_id, $post->ID, $course_id );
	}

	/**
	 * Get form.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return array
	 */
	protected function get_from( $post ) {
		$quiz_settings = learndash_get_setting( $post->ID );
		$quiz_pro      = $quiz_settings['quiz_pro'];
		$form_mapper   = new WpProQuiz_Model_FormMapper();
		$forms         = $form_mapper->fetch( $quiz_pro );
		$arr_form      = array();

		foreach ( $forms as $key => $form_field ) {
			$arr_form[ $key ]['id']        = $form_field->getFormId();
			$arr_form[ $key ]['fieldname'] = $form_field->getFieldname();
			$arr_form[ $key ]['type']      = $this->get_form_type( $form_field->getType() );
			$arr_form[ $key ]['required']  = $form_field->isRequired();
			$arr_form[ $key ]['data']      = $form_field->getData();
		}

		return $arr_form;
	}

	/**
	 * Get form field type.
	 *
	 * @param string $type Field type.
	 *
	 * @return string
	 */
	private function get_form_type( $type ) {
		switch ( $type ) {
			case WpProQuiz_Model_Form::FORM_TYPE_TEXT:
				$type = 'text';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_EMAIL:
				$type = 'email';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_NUMBER:
				$type = 'number';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_TEXTAREA:
				$type = 'textarea';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_CHECKBOX:
				$type = 'checkbox';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_DATE:
				$type = 'date';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_RADIO:
				$type = 'radio';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_SELECT:
				$type = 'select';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_YES_NO:
				$type = 'yesno';
				break;
		}

		return $type;
	}

	/**
	 * Get settings.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return mixed
	 */
	protected function get_settings( $post ) {
		$quiz_settings           = learndash_get_setting( $post->ID );
		$quiz_pro                = $quiz_settings['quiz_pro'];
		$quiz_mapper             = new WpProQuiz_Model_QuizMapper();
		$quiz                    = $quiz_mapper->fetch( $quiz_pro );
		$setting                 = $quiz->get_object_as_array();
		$setting['_toplistData'] = maybe_unserialize( $setting['_toplistData'] );

		if ( $quiz_settings['retry_restrictions'] ) {
			$setting['_restrict_quiz_retakes'] = array(
				'retry_restrictions' => $quiz_settings['retry_restrictions'],
				'repeats'            => $quiz_settings['repeats'],
			);
		}

		$quiz_saving             = $this->get_quiz_saving( $post );
		$setting['_quiz_saving'] = $quiz_saving;

		return $setting;
	}

	/**
	 * Function to save quiz setting.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	protected function get_quiz_saving( $post ) {
		$quiz_settings                                = learndash_get_setting( $post->ID );
		$quiz_saving                                  = array();
		$quiz_saving['quiz_resume']                   = $quiz_settings['quiz_resume'];
		$quiz_saving['quiz_resume_cookie_send_timer'] = $quiz_settings['quiz_resume_cookie_send_timer'];

		return $quiz_saving;
	}

	/**
	 * Function to get quiz resume activity.
	 *
	 * @param object $post quiz post object.
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	protected function get_quiz_resume_activity( $post ) {
		$user_id                  = get_current_user_id();
		$course_id                = bbapp_learndash_get_course_id( $post->ID );
		$quiz_resume_id           = 0;
		$quiz_resume_data         = array();
		$quiz_resume_quiz_started = 0;
		$quiz_resume_activity     = \LDLMS_User_Quiz_Resume::get_user_quiz_resume_activity( $user_id, $post->ID, $course_id );

		if ( ( is_a( $quiz_resume_activity, 'LDLMS_Model_Activity' ) ) && ( property_exists( $quiz_resume_activity, 'activity_id' ) ) && ( ! empty( $quiz_resume_activity->activity_id ) ) ) {
			$quiz_resume_id = $quiz_resume_activity->activity_id;

			if ( ( property_exists( $quiz_resume_activity, 'activity_meta' ) ) && ( ! empty( $quiz_resume_activity->activity_meta ) ) ) {
				$quiz_resume_data = $quiz_resume_activity->activity_meta;
				$quiz_resume_data = $this->prepare_quiz_resume_data( $quiz_resume_data );
			}

			if ( ( property_exists( $quiz_resume_activity, 'activity_started' ) ) && ( ! empty( $quiz_resume_activity->activity_started ) ) ) {
				$quiz_resume_quiz_started = $quiz_resume_activity->activity_started;
			}
		}

		return array(
			'id'           => $quiz_resume_id,
			'data'         => $quiz_resume_data,
			'quiz_started' => $quiz_resume_quiz_started,
		);
	}

	/**
	 * Function to prepare quiz resume data.
	 *
	 * @param array $data Quiz resume data.
	 *
	 * @since 1.5.0
	 *
	 * @return mixed
	 */
	protected function prepare_quiz_resume_data( $data ) {
		$resume_dat    = array();
		$question_data = array();
		$review_box    = ! empty( $data['reviewBox'] ) ? $data['reviewBox'] : array();
		$last_question = ! empty( $data['lastQuestion'] ) ? $data['lastQuestion'] : 0;
		$next_question = ! empty( $data['nextQuestion'] ) ? $data['nextQuestion'] : 0;

		unset( $data['reviewBox'], $data['lastQuestion'], $data['nextQuestion'] );

		if ( ! empty( $data ) ) {
			foreach ( $data as $key => $data_item ) {
				if ( ! in_array( $key, array( 'randomQuestions', 'randomOrder' ), true ) ) {
					$question_data[ $key ] = $data_item;
				}
			}
		}

		foreach ( $review_box as $key => $review_box_item ) {
			if ( ! empty( $review_box_item ) ) {
				foreach ( $data as $id => $quiz_item ) {
					if ( isset( $quiz_item['index'] ) && $quiz_item['index'] === $key ) {
						$review_box[ $key ]['item'] = $quiz_item;
						unset( $data[ $id ] );
					}
				}
			} else {
				$review_box[ $key ] = (object) $review_box_item;
			}
		}

		$resume_dat['main']['reviewBox']    = $review_box;
		$resume_dat['main']['lastQuestion'] = $last_question;
		$resume_dat['main']['nextQuestion'] = $next_question;
		$resume_dat['items']                = $data;
		$resume_dat['question_data']        = $question_data;

		return $resume_dat;
	}

	/**
	 * Get error message.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @return WP_Error|array
	 */
	public function get_error_message( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post->ID );
		$user_id   = get_current_user_id();

		if ( ! bbapp_lms_is_content_access( $post, 'prerequities_completed' ) ) {
			$prerequisite_posts_all = learndash_get_course_prerequisites( $course_id );
			$course_str             = '';

			if ( ! empty( $prerequisite_posts_all ) ) {
				foreach ( $prerequisite_posts_all as $pre_post_id => $pre_status ) {
					if ( false === $pre_status ) {
						if ( ! empty( $course_str ) ) {
							$course_str .= ', ';
						}

						$course_str .= get_the_title( $pre_post_id );
					}
				}
			}

			return CoursesError::instance()->invalid_course_prerequities( $course_str );
		}

		if ( ! bbapp_lms_is_content_access( $post, 'points_access' ) ) {
			$course_access_points = learndash_get_course_points_access( $course_id );
			$user_course_points   = learndash_get_user_course_points( $user_id );

			return CoursesError::instance()->invalid_course_access_point( $course_access_points, $user_course_points );
		}

		if ( ! $this->get_quiz_accessable( $post ) && ! learndash_is_sample( $post ) ) {
			$course_id            = bbapp_learndash_get_course_id( $post->ID );
			$last_incomplete_step = function_exists( 'learndash_is_quiz_accessable' ) ? learndash_is_quiz_accessable( null, $post, true, $course_id ) : is_quiz_accessable( null, $post, true, $course_id );

			if ( 1 !== $last_incomplete_step ) {
				return QuizError::instance()->quiz_previous_module_not_completed( $last_incomplete_step );
			}
		}

		$step_access_from = bbapp_lesson_visible_after( $post );
		if ( ! empty( $step_access_from ) && true !== $step_access_from ) {
			$step_access_from = learndash_adjust_date_time_display( $step_access_from );

			return QuizError::instance()->quiz_not_available( $step_access_from );
		}

		if ( ! sfwd_lms_has_access( $post->ID, $user_id ) ) {
			return QuizError::instance()->invalid_quiz_access();
		}

		if ( ! $this->check_quiz_lock( $post, 'prerequisite' ) ) {
			return QuizError::instance()->quiz_previous_not_completed();
		}

		$attempts_count = $this->check_quiz_attempt_alert( $post );

		if ( $attempts_count ) {
			return QuizError::instance()->quiz_attempts_alert( $attempts_count );
		}
	}

	/**
	 * Get quiz data.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @return mixed
	 */
	public function get_additional_data( $post ) {
		$post->mode               = $this->get_mode( $post );
		$post->has_course_access  = $this->get_has_course_access( $post );
		$post->has_content_access = $this->get_has_content_access( $post );
		$post->can_take_again     = $this->get_can_take_again( $post );
		$post->completed          = $this->is_completed( $post, learndash_get_course_id( $post->ID ) );
		$post->featured_media     = $this->get_feature_media( $post );
		$post->course             = $this->get_course_id( $post );
		$post->lesson             = $this->get_lesson_id( $post );
		$post->topic              = $this->get_topic_id( $post );
		$post->link               = get_permalink( $post->ID );
		$post->form               = $this->get_from( $post );
		$post->settings           = $this->get_settings( $post );
		$error_message            = $this->get_error_message( $post );

		if ( ! empty( $error_message ) ) {
			$error_code          = $error_message->get_error_code();
			$post->error_message = array(
				'code'    => $error_code,
				'message' => $error_message->get_error_message(),
				'data'    => $error_message->get_error_data( $error_code ),

			);
		} else {
			$post->error_message = array();
		}

		return $post;
	}

	/**
	 * Check Quiz attempt alert.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @return false|int
	 */
	public function check_quiz_attempt_alert( $post ) {
		$quiz_id        = absint( $post->ID );
		$course_id      = bbapp_learndash_get_course_id( $post->ID );
		$user_id        = get_current_user_id();
		$attempts_count = 0;

		if ( $user_id ) {
			$usermeta = get_user_meta( $user_id, '_sfwd-quizzes', true );
			$usermeta = maybe_unserialize( $usermeta );

			if ( ! is_array( $usermeta ) ) {
				$usermeta = array();
			}

			if ( ! empty( $usermeta ) ) {
				foreach ( $usermeta as $k => $v ) {
					if ( ( intval( $v['quiz'] ) === $quiz_id ) ) {
						if ( ! empty( $course_id ) ) {
							if ( ( isset( $v['course'] ) ) && ( ! empty( $v['course'] ) ) && ( absint( $v['course'] ) === absint( $course_id ) ) ) {
								// Count the number of time the student has taken the quiz where the course_id matches.
								$attempts_count ++;
							}
						} elseif ( empty( $course_id ) ) {
							if ( ( isset( $v['course'] ) ) && ( empty( $v['course'] ) ) && ( absint( $v['course'] ) === absint( $course_id ) ) ) {
								// Count the number of time the student has taken the quiz where the course_id is zero.
								$attempts_count ++;
							}
						}
					}
				}
			}
		} else {
			$quiz_settings = learndash_get_setting( $post->ID );
			$quiz_pro      = $quiz_settings['quiz_pro'];
			$quiz_mapper   = new WpProQuiz_Model_QuizMapper();
			$quiz          = $quiz_mapper->fetch( $quiz_pro );
			$cookie_time   = $quiz->getQuizRunOnceTime();
			$_cookie       = stripslashes_deep( $_COOKIE );

			if ( isset( $_cookie['wpProQuiz_lock'] ) ) {
				$cookie_json = json_decode( $_cookie['wpProQuiz_lock'], true );

				if ( ( false !== $cookie_json ) && ( isset( $cookie_json[ $quiz_pro ] ) ) ) {
					$cookie_quiz = $cookie_json[ $quiz_pro ];
					$cookie_quiz = learndash_quiz_convert_lock_cookie( $cookie_quiz );

					if ( $cookie_quiz['time'] === $cookie_time ) {
						$attempts_count = absint( $cookie_quiz['count'] );
					}
				}
			}
		}

		return false === $this->get_can_take_again( $post ) ? $attempts_count : false;
	}

	/**
	 * Quizze materials content.
	 *
	 * @since BuddyBoss-APP 1.3.5
	 *
	 * @param object $post Quizze post data.
	 *
	 * @return string
	 */
	protected function get_materials( $post ) {

		$quiz_materials_enabled = learndash_get_setting( $post->ID, 'quiz_materials_enabled' );

		if ( 'on' === $quiz_materials_enabled ) {
			$quiz_materials = learndash_get_setting( $post->ID, 'quiz_materials' );
			$quiz_materials = wp_kses_post( wp_specialchars_decode( $quiz_materials, ENT_QUOTES ) );

			if ( ! empty( $quiz_materials ) ) {
				$quiz_materials = do_shortcode( $quiz_materials );
			};

			return $quiz_materials;
		}

		return '';
	}
}
