<?php
/**
 * Hold quiz question rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Quiz
 */

namespace BuddyBossApp\Api\LearnDash\V1\Quiz;

use WP_Error;
use WP_Post;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Quiz question rest.
 */
class QuizQuestionsRest extends QuizRest {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Quiz helper.
	 *
	 * @var mixed $helper
	 */
	protected $helper;

	/**
	 * QuizQuestionsRest instance..
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'quiz';
		$this->helper    = QuizQuestionsHelper::instance();
		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/start',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'start' ),
					'permission_callback' => array( $this, 'get_start_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/next',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'next' ),
					'permission_callback' => array( $this, 'get_next_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/prev',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'prev' ),
					'permission_callback' => array( $this, 'get_prev_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/check',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'check' ),
					'permission_callback' => array( $this, 'get_check_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/upload',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'upload' ),
					'permission_callback' => array( $this, 'get_upload_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/save',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'save' ),
					'permission_callback' => array( $this, 'get_save_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/cookies',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'cookies' ),
					'permission_callback' => array( $this, 'get_cookies_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to start quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_start_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() && ! $this->is_open_course_quiz( $request['id'] ) ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the quiz `start` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_start_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to next quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_next_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() && ! $this->is_open_course_quiz( $request['id'] ) ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the quiz `next` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_next_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to prev quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_prev_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() && ! $this->is_open_course_quiz( $request['id'] ) ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the quiz `prev` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_prev_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to check quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_check_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() && ! $this->is_open_course_quiz( $request['id'] ) ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the quiz `check` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_check_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to upload quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_upload_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the quiz `upload` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_upload_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to save quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_save_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() && ! $this->is_open_course_quiz( $request['id'] ) ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the quiz `save` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_save_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to cookie quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 1.5.0
	 * @return bool|WP_Error
	 */
	public function get_cookies_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() && ! $this->is_open_course_quiz( $request['id'] ) ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the quiz `cookie` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 1.5.0
		 */
		return apply_filters( 'bbapp_ld_quiz_cookie_permissions_check', $retval, $request );
	}


	/**
	 * Start Quiz.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return array|WP_Error|WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/quiz/:id/start Start LearnDash Quiz
	 * @apiName        StartLDQuiz
	 * @apiGroup       LD Quizzes
	 * @apiDescription Start Quiz
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 */
	public function start( $request ) {
		global $wp_rest_server;

		$user_id = is_user_logged_in() ? get_current_user_id() : $request->get_param( 'guest_user_id' );
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$quiz    = get_post( $quiz_id );

		if ( ! empty( $request->get_param( 'course' ) ) && is_numeric( $request->get_param( 'course' ) ) ) {
			$_GET['course_id'] = (int) $request->get_param( 'course' );
		}

		if ( empty( $quiz ) || $this->post_type !== $quiz->post_type ) {
			return QuizError::instance()->invalid_quiz_id();
		}

		if ( empty( $user_id ) || ! $this->is_open_course_quiz( $request['id'] ) && ! $this->get_has_content_access( $quiz ) ) {
			return QuizError::instance()->invalid_quiz_access();
		}

		/**
		 * Check Quiz attempts is available
		 */
		$quiz_settings  = learndash_get_setting( $quiz_id );
		$attempts_count = 0;
		$repeats        = (int) trim( $quiz_settings['repeats'] );
		$usermeta       = get_user_meta( $user_id, '_sfwd-quizzes', true );
		$usermeta       = maybe_unserialize( $usermeta );

		if ( is_array( $usermeta ) && ! empty( $usermeta ) ) {
			foreach ( $usermeta as $k => $v ) {
				if ( (int) $v['quiz'] === $quiz_id ) {
					$attempts_count ++;
				}
			}
		}

		$attempts_left = ( empty( $repeats ) || $repeats > $attempts_count );

		if ( false === $attempts_left && isset( $quiz_settings['retry_restrictions'] ) && 'on' === $quiz_settings['retry_restrictions'] ) {
			return QuizError::instance()->quiz_attempts_left_empty();
		}

		$response        = array();
		$fetch_questions = $this->helper->get_questions( $quiz_settings['quiz_pro'], $request );

		/**
		 * Prepare session data for Quiz Question Which is use to keep quiz Question progression.
		 */
		$index     = 1;
		$questions = array();

		if ( ! empty( $fetch_questions ) ) {
			foreach ( $fetch_questions as $question ) {
				$questions[ $index ] = array(
					'id'         => $question->ID,
					'start_time' => 0,
					'end_time'   => 0,
					'answer'     => array(),
					'responce'   => array(),
					'point'      => ( isset( $post->points ) ) ? $question->points : 0,
				);
				$question->index     = $index;
				$question->answer    = array();
				$temp_res            = $this->prepare_item_for_response( $question, $request );
				$response[]          = $wp_rest_server->response_to_data( rest_ensure_response( $temp_res ), $request['_embed'] );
				$index ++;
			}
		}

		/**
		 * Store session data in meta so we can use this while Quiz complete as On web this stuff managing with cookies & session.
		 * Prepare session data for Quiz Question Which is use to keep quiz progression
		 */
		$quiz_meta = array(
			'start_time' => current_time( 'timestamp' ), // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp
			'end_time'   => 0,
			'questions'  => $questions,
			'form'       => array(),
			'next'       => ( ! empty( $questions ) ) ? 1 : - 1,
			'prev'       => - 1,
		);
		$this->set_quiz_lock( $user_id, $quiz_id, $quiz_meta );

		/**
		 * Store session data in meta so we can use this while Quiz complete
		 * Store form Data which is required to start quiz [ LD setting ]
		 */
		if ( ! empty( $request['form'] ) ) {
			$this->save_form_meta( $user_id, $quiz_id, $request['form'] );
		}

		/**
		 * Fires after an quiz is start and response prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_quiz_start_response', $response, $request );

		return $response;
	}

	/**
	 * Next Quiz Question.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/quiz/:id/next Next LearnDash Quiz Question
	 * @apiName        LDQuizQuestionNext
	 * @apiGroup       LD Quizzes
	 * @apiDescription Next Quiz Question
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 */
	public function next( $request ) {
		$user_id = is_user_logged_in() ? get_current_user_id() : $request->get_param( 'guest_user_id' );
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$quiz    = get_post( $quiz_id );

		if ( empty( $quiz ) || $this->post_type !== $quiz->post_type ) {
			return QuizError::instance()->invalid_quiz_id();
		}

		if ( empty( $user_id ) || ! $this->is_open_course_quiz( $request['id'] ) && ! $this->get_has_content_access( $quiz ) ) {
			return QuizError::instance()->invalid_quiz_access();
		}

		// Store answer Question in Meta.
		if ( ! empty( $request['answer'] ) ) {
			$this->save_answer_meta( $user_id, $quiz_id, $request['answer'] );
		}

		/**
		 * Check if Quiz session is exist or not.
		 */
		$quiz_meta = $this->get_quiz_lock( $user_id, $quiz_id );

		if ( empty( $quiz_meta ) ) {
			return QuizError::instance()->quiz_not_started();
		}

		$question_index = (int) $quiz_meta['next'];

		if ( - 1 === $question_index ) {
			return QuizError::instance()->invalid_quiz_next_question_id();
		}

		/**
		 * If Quiz start and it's first question then lock quiz session.
		 */
		if ( empty( $quiz_meta['questions'][ $question_index ]['start_time'] ) ) {
			$quiz_meta['questions'][ $question_index ]['start_time'] = current_time( 'timestamp' ); // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp

			$this->set_quiz_lock( $user_id, $quiz_id, $quiz_meta );
		}

		$quiz_settings         = learndash_get_setting( $quiz_id );
		$next_question         = $this->helper->get_guestion( $quiz_meta['questions'][ $question_index ]['id'], $quiz_settings['quiz_pro'] );
		$next_question->index  = $question_index;
		$next_question->answer = $quiz_meta['questions'][ $question_index ]['answer'];

		if ( ! isset( $quiz_meta['questions'][ $question_index + 1 ] ) ) {
			$next_question->is_last = true;
		}

		$data     = $this->prepare_item_for_response( $next_question, $request );
		$response = rest_ensure_response( $data );

		return $response;
	}

	/**
	 * Previous Quiz Question.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/quiz/:id/prev Previous LearnDash Quiz Question
	 * @apiName        LDQuizQuestionPrevious
	 * @apiGroup       LD Quizzes
	 * @apiDescription Previous Quiz Question
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 */
	public function prev( $request ) {
		$user_id = is_user_logged_in() ? get_current_user_id() : $request->get_param( 'guest_user_id' );
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$quiz    = get_post( $quiz_id );

		if ( empty( $quiz ) || $this->post_type !== $quiz->post_type ) {
			return QuizError::instance()->invalid_quiz_id();
		}

		if ( empty( $user_id ) || ! $this->is_open_course_quiz( $request['id'] ) && ! $this->get_has_content_access( $quiz ) ) {
			return QuizError::instance()->invalid_quiz_access();
		}

		/**
		 * Check if Quiz session is exist or not.
		 */
		$quiz_meta = $this->get_quiz_lock( $user_id, $quiz_id );

		if ( empty( $quiz_meta ) ) {
			return QuizError::instance()->quiz_not_started();
		}

		$question_index = (int) $quiz_meta['prev'];

		if ( - 1 === $question_index ) {
			return QuizError::instance()->invalid_quiz_question_id();
		}

		/**
		 * If Quiz start and it's first question then lock quiz session.
		 */
		if ( empty( $quiz_meta['questions'][ $question_index ]['start_time'] ) ) {
			$quiz_meta['questions'][ $question_index ]['start_time'] = current_time( 'timestamp' ); // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp
		}

		/**
		 * Update quiz session next & previous question id.
		 * if previous and next question not exist if current question is first or last in that case it's set '-1'.
		 */
		$next_question_id  = $question_index;
		$prev_question_id  = $question_index - 1;
		$quiz_meta['next'] = ( isset( $quiz_meta['questions'][ $next_question_id ] ) ) ? $next_question_id : - 1;
		$quiz_meta['prev'] = ( isset( $quiz_meta['questions'][ $prev_question_id ] ) ) ? $prev_question_id : - 1;

		$this->set_quiz_lock( $user_id, $quiz_id, $quiz_meta );

		$quiz_settings         = learndash_get_setting( $quiz_id );
		$next_question         = $this->helper->get_guestion( $quiz_meta['questions'][ $question_index ]['id'], $quiz_settings['quiz_pro'] );
		$next_question->index  = $question_index;
		$next_question->answer = $quiz_meta['questions'][ $question_index ]['answer'];

		if ( ! isset( $quiz_meta['questions'][ $question_index + 1 ] ) ) {
			$next_question->is_last = true;
		}

		$data     = $this->prepare_item_for_response( $next_question, $request );
		$response = rest_ensure_response( $data );

		return $response;
	}

	/**
	 * Check Quiz Question is correct or not.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/quiz/:id/check Check LearnDash Quiz Question is correct or not.
	 * @apiName        LDQuizQuestionCheck
	 * @apiGroup       LD Quizzes
	 * @apiDescription Check Quiz Question is correct or not.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 * @apiParam {array} [answer] Answer array for quiz questions.
	 */
	public function check( $request ) {
		$user_id = is_user_logged_in() ? get_current_user_id() : $request->get_param( 'guest_user_id' );
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$quiz    = get_post( $quiz_id );

		if ( empty( $quiz ) || $this->post_type !== $quiz->post_type ) {
			return QuizError::instance()->invalid_quiz_id();
		}

		if ( empty( $user_id ) || ! $this->is_open_course_quiz( $request['id'] ) && ! $this->get_has_content_access( $quiz ) ) {
			return QuizError::instance()->invalid_quiz_access();
		}

		/**
		 * Check if Quiz session is exist or not.
		 */
		$quiz_meta = $this->get_quiz_lock( $user_id, $quiz_id );

		if ( empty( $quiz_meta ) ) {
			return QuizError::instance()->quiz_not_started();
		}

		// $question_index = (int) $quiz_meta['next']; this is for session current Question
		$answers = $request['answer'];

		if ( empty( $answers ) ) {
			return QuizError::instance()->quiz_question_empty_answer();
		}

		/**
		 * Get Question post id using index and map it with answer
		 */
		$answer_data = array();

		foreach ( $answers as $index => $answer ) {
			$question_index = (int) $index;
			$answer_data[ $quiz_meta['questions'][ $question_index ]['id'] ] = $answer;
		}

		/**
		 * Check if same answers is already check and response store in quiz session data.
		 */
		if ( $answer_data[ $quiz_meta['questions'][ $question_index ]['id'] ] === $quiz_meta['questions'][ $question_index ]['answer'] && ! empty( $quiz_meta['questions'][ $question_index ]['responce'] ) ) {
			$data = $quiz_meta['questions'][ $question_index ]['responce'];
		} else {

			$quiz_settings = learndash_get_setting( $quiz_id );
			$data          = $this->helper->check_answers( $answer_data, $quiz_settings['quiz_pro'], $quiz_id );

			if ( ! empty( $data['data'] ) ) {
				$quiz_meta['questions'][ $question_index ]['responce'] = $data;
				$quiz_meta['questions'][ $question_index ]['answer']   = $answer_data[ $quiz_meta['questions'][ $question_index ]['id'] ];
				$this->set_quiz_lock( $user_id, $quiz_id, $quiz_meta );
			}
		}

		if ( ! empty( $data['data'] ) ) {
			/**
			 * Update quiz session next & previous question id.
			 * if previous and next question not exist if current question is first or last in that case it's set '-1'.
			 */
			$next_question_id  = $question_index + 1;
			$prev_question_id  = $question_index;
			$quiz_meta['next'] = ( isset( $quiz_meta['questions'][ $next_question_id ] ) ) ? $next_question_id : - 1;
			$quiz_meta['prev'] = ( isset( $quiz_meta['questions'][ $prev_question_id ] ) ) ? $prev_question_id : - 1;

			$this->set_quiz_lock( $user_id, $quiz_id, $quiz_meta );

			return rest_ensure_response( $data['data'] );
		}

		return QuizError::instance()->quiz_question_check_error();
	}

	/**
	 * Upload Quiz Question answers file.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/quiz/:id/upload Upload LearnDash Quiz Question Answers file.
	 * @apiName        LDQuizQuestionUpload
	 * @apiGroup       LD Quizzes
	 * @apiDescription Upload Quiz Question answers file. This endpoint requires request to be sent in "multipart/form-data" format.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 * @apiParam {Number} [ question_id ] A unique numeric ID for the Quiz Question.
	 * @apiParam {File} [ attachment ] A file to upload for question answers.
	 */
	public function upload( $request ) {
		$quiz_id     = is_numeric( $request ) ? $request : (int) $request['id'];
		$question_id = (int) ! empty( $request['question_id'] ) ? $request['question_id'] : 0;
		$quiz        = get_post( $quiz_id );

		if ( empty( $quiz ) || $this->post_type !== $quiz->post_type ) {
			return QuizError::instance()->invalid_quiz_id();
		}

		if ( ! $this->get_has_content_access( $quiz ) ) {
			return QuizError::instance()->invalid_quiz_access();
		}

		if ( empty( $question_id ) ) {
			return QuizError::instance()->invalid_quiz_question_id();
		}

		if ( empty( $_FILES ) ) {
			return QuizError::instance()->quiz_question_empty_answers_file();
		}

		if ( function_exists( 'learndash_essay_fileupload_process' ) && ! empty( $_FILES['attachment'] ) ) {
			$data = learndash_essay_fileupload_process( $_FILES['attachment'], $question_id ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		}

		if ( ! empty( $data ) ) {
			return rest_ensure_response( $data );
		}

		return QuizError::instance()->quiz_question_upload_error();
	}

	/**
	 * Save quiz question answers.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/quiz/:id/save Save LearnDash Quiz Question Answers.
	 * @apiName        LDQuizQuestionSave
	 * @apiGroup       LD Quizzes
	 * @apiDescription Save quiz question answers.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 * @apiParam {array} [answer] Answer for quiz questions.
	 * @apiParam {array} [Form] Form details for quiz.
	 */
	public function save( $request ) {
		$user_id = is_user_logged_in() ? get_current_user_id() : $request->get_param( 'guest_user_id' );
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$quiz    = get_post( $quiz_id );

		if ( empty( $quiz ) || $this->post_type !== $quiz->post_type ) {
			return QuizError::instance()->invalid_quiz_id();
		}

		/**
		 * Store Quiz answer & From data in session if it's not added or changed.
		 */
		$answers = $request['answer'];

		if ( ! empty( $answers ) ) {
			$this->save_answer_meta( $user_id, $quiz_id, $answers, true );
		}

		$form = $request['form'];

		if ( ! empty( $form ) ) {
			$this->save_form_meta( $user_id, $quiz_id, $form );
		}

		$quiz_meta = $this->get_quiz_lock( $user_id, $quiz_id );

		if ( empty( $quiz_meta ) ) {
			return QuizError::instance()->quiz_not_started();
		}

		/**
		 * Set Quiz endtime in quiz session.
		 */
		$quiz_meta['end_time'] = current_time( 'timestamp' ); // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp.Requested

		$this->set_quiz_lock( $user_id, $quiz_id, $quiz_meta );

		$data = $this->helper->save_quiz( $quiz_meta, $quiz_id );

		if ( ! empty( $quiz_id ) && ! is_wp_error( $data ) ) {
			$this->bbapp_ld_process_to_mark_complete( $user_id, $data, $quiz_id );
		}

		if ( is_wp_error( $data ) ) {
			return $data;
		}

		if ( true === $data['finished'] ) {
			$this->delete_quiz_lock( $user_id, $quiz_id );
		}

		return rest_ensure_response( $data );
	}

	/**
	 * Save cookies quiz question answers.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 * @since          1.5.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/quiz/:id/cookies Save cookies LearnDash Quiz Question Answers.
	 * @apiName        LDQuizQuestionSave
	 * @apiGroup       LD Quizzes
	 * @apiDescription Save cookies quiz question answers.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 * @apiParam {array} [answer] Answer for quiz questions.
	 * @apiParam {array} [Form] Form details for quiz.
	 */
	public function cookies( $request ) {
		$data      = array(
			'saved' => false,
		);
		$user_id   = is_user_logged_in() ? get_current_user_id() : $request->get_param( 'guest_user_id' );
		$quiz_id   = is_numeric( $request ) ? $request : (int) $request['id'];
		$course_id = ! empty( $request['course_id'] ) ? $request['course_id'] : 0;
		$quiz      = get_post( $quiz_id );

		if ( empty( $quiz ) || $this->post_type !== $quiz->post_type ) {
			return QuizError::instance()->invalid_quiz_id();
		}

		/**
		 * Store Quiz answer & From data in session if it's not added or changed.
		 */
		if ( ( isset( $request['results'] ) ) && ( ! empty( $request['results'] ) ) ) {
			$results      = (array) $request['results'];
			$quiz_started = 0;

			if ( isset( $request['quiz_started'] ) ) {
				$quiz_started = absint( $request['quiz_started'] / 1000 );
			}

			if ( ( ! empty( $quiz_id ) ) && ( ! empty( $quiz_id ) ) ) {
				$success = \LDLMS_User_Quiz_Resume::update_user_quiz_resume_metadata( $user_id, $quiz_id, $course_id, $quiz_started, $results );
				if ( $success ) {
					$data['saved'] = true;
				} else {
					/* translators: $s: Quiz label. */
					$data = new WP_Error( 'learndash_json_quiz_not_saved', sprintf( esc_html_x( '%s data could not be saved to the server. Please reload the app and try again. If this error persists, please contact support.', 'placeholder: Quiz', 'buddyboss-app' ), esc_html( \LearnDash_Custom_Label::get_label( 'quiz' ) ) ), array( 'status' => 404 ) );
				}
			}
		}

		return rest_ensure_response( $data );
	}


	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$schema  = $this->get_public_item_schema();

		// Get logged in user ID.
		$user_id     = get_current_user_id();
		$quiz_id     = $request->get_param( 'id' ); // Get the quiz ID from request.
		$course_id   = 0;

		// Get the course ID from quiz ID.
		if ( ! empty( $quiz_id ) ) {
			$course_id = bbapp_learndash_get_course_id( $quiz_id );

			if ( ! $course_id ) {
				$course_id = ( ! empty( $request->get_param( 'course' ) ) ) ? $request->get_param( 'course' ) : 0;
			}
		}

		$quiz_resume_data     = array();
		$quiz_resume_activity = \LDLMS_User_Quiz_Resume::get_user_quiz_resume_activity( $user_id, $quiz_id, $course_id );

		if ( ( is_a( $quiz_resume_activity, 'LDLMS_Model_Activity' ) ) && ( property_exists( $quiz_resume_activity, 'activity_id' ) ) && ( ! empty( $quiz_resume_activity->activity_id ) ) ) {
			if ( ( property_exists( $quiz_resume_activity, 'activity_meta' ) ) && ( ! empty( $quiz_resume_activity->activity_meta ) ) && ( ! empty( $quiz_resume_activity->activity_meta[ $post->ID ] ) ) ) {
				$quiz_resume_data = $quiz_resume_activity->activity_meta[ $post->ID ];
			}
		}

		// Base fields for every post.
		$data = array(
			'id'    => $post->ID,
			'title' => array(
				'raw'      => $post->title,
				'rendered' => apply_filters( 'the_content', $post->title ), // Question content is a title.
			),
		);

		// Set resume data if not empty.
		if ( ! empty( $quiz_resume_data ) ) {
			$data['resume_data'] = $quiz_resume_data;
		}

		if ( ! empty( $schema['properties']['index'] ) && in_array( $context, $schema['properties']['index']['context'], true ) ) {
			$data['index'] = (int) $post->index;
		}

		if ( ! empty( $schema['properties']['points'] ) && in_array( $context, $schema['properties']['points']['context'], true ) ) {
			if ( ! empty( $post->points ) ) {
				$data['points'] = (int) $post->points;
			}
		}

		if ( ! empty( $schema['properties']['category'] ) && in_array( $context, $schema['properties']['category']['context'], true ) ) {
			$data['category'] = $post->category;
		}

		if ( ! empty( $schema['properties']['question_type'] ) && in_array( $context, $schema['properties']['question_type']['context'], true ) ) {
			$data['question_type'] = $post->question_type;
		}

		if ( ! empty( $schema['properties']['hint'] ) && in_array( $context, $schema['properties']['hint']['context'], true ) ) {
			$data['hint'] = $post->hint;
		}

		if ( ! empty( $schema['properties']['options'] ) && in_array( $context, $schema['properties']['options']['context'], true ) ) {
			$data['options'] = $post->options;
		}

		if ( ! empty( $schema['properties']['options_1'] ) && in_array( $context, $schema['properties']['options_1']['context'], true ) ) {
			if ( ! empty( $post->options_1 ) ) {
				$data['options_1'] = $post->options_1;
			}
		}

		if ( ! empty( $schema['properties']['answer'] ) && in_array( $context, $schema['properties']['answer']['context'], true ) ) {
			$data['answer'] = $post->answer;
		}

		if ( ! empty( $schema['properties']['gradedType'] ) && in_array( $context, $schema['properties']['gradedType']['context'], true ) ) {
			if ( isset( $post->graded_type ) ) { //phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				$data['gradedType'] = $post->graded_type; //phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			}
		}

		if ( ! empty( $schema['properties']['is_last'] ) && in_array( $context, $schema['properties']['is_last']['context'], true ) ) {
			if ( isset( $post->is_last ) ) {
				$data['is_last'] = $post->is_last;
			}
		}

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filters quiz question.
		 *
		 * @param WP_REST_Response $response Quiz response.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 */
		return apply_filters( 'bbapp_ld_rest_prepare_quiz_question', $response, $post, $request );
	}


	/**
	 * Prepare links for the request.
	 *
	 * @param array $data Item object.
	 *
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {
		$links = array();

		return $links;
	}


	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'ld_quiz_question',
			'type'       => 'object',
			'properties' => array(
				'id'    => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title' => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
			),
		);

		$schema['properties']['index'] = array(
			'description' => __( 'Index for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['points'] = array(
			'description' => __( 'Point for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['category'] = array(
			'description' => __( 'Category for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['question_type'] = array(
			'description' => __( 'Question type for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['hint'] = array(
			'description' => __( 'Hint for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['options'] = array(
			'description' => __( 'Answers Option for the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['options_1'] = array(
			'description' => __( 'Answers additional Option for the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['answer'] = array(
			'description' => __( 'Answers for the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['gradedType'] = array(
			'description' => __( 'Type of essay Answers for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['is_last'] = array(
			'description' => __( 'Whether it is a last object or not', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get quiz lock.
	 *
	 * @param int $user_id User id.
	 * @param int $quiz_id Quiz id.
	 *
	 * @return array
	 */
	protected function get_quiz_lock( $user_id, $quiz_id ) {
		$quiz_locks = $this->get_bbapp_ld_quiz_data( $user_id );

		if ( isset( $quiz_locks[ $quiz_id ] ) ) {
			return $quiz_locks[ $quiz_id ];
		}

		return array();
	}

	/**
	 * Set quiz lock.
	 *
	 * @param int   $user_id   User id.
	 * @param int   $quiz_id   Quiz id.
	 * @param array $quiz_meta Quiz meta.
	 *
	 * @return mixed
	 */
	protected function set_quiz_lock( $user_id, $quiz_id, $quiz_meta ) {
		$quiz_locks = $this->get_bbapp_ld_quiz_data( $user_id );

		if ( empty( $quiz_locks ) ) {
			$quiz_locks = array();
		}

		/**
		 * Filters quiz lock data.
		 *
		 * @type int|string $quiz_meta Quiz meta.
		 * @type int        $user_id   User id.
		 * @type int        $quiz_id   Quiz id.
		 */
		$quiz_meta              = apply_filters( 'bbapp_ld_quiz_lock', $quiz_meta, $user_id, $quiz_id );
		$quiz_locks[ $quiz_id ] = $quiz_meta;

		return $this->update_bbapp_ld_quiz_data( $user_id, $quiz_locks );
	}

	/**
	 * Remove quzi lock.
	 *
	 * @param int $user_id User id.
	 * @param int $quiz_id Quiz id.
	 *
	 * @return mixed
	 */
	protected function delete_quiz_lock( $user_id, $quiz_id ) {
		$quiz_locks = $this->get_bbapp_ld_quiz_data( $user_id );

		unset( $quiz_locks[ $quiz_id ] );

		if ( empty( $quiz_locks ) ) {
			return $this->delete_bbapp_ld_quiz_data( $user_id );
		} else {
			return $this->update_bbapp_ld_quiz_data( $user_id, $quiz_locks );
		}
	}

	/**
	 * Save quiz form.
	 *
	 * @param int   $user_id User id.
	 * @param int   $quiz_id Quiz id.
	 * @param array $form    Quiz form.
	 *
	 * @return bool|mixed
	 */
	protected function save_form_meta( $user_id, $quiz_id, $form ) {
		$quiz_meta = $this->get_quiz_lock( $user_id, $quiz_id );

		if ( empty( $quiz_meta ) ) {
			return false;
		}

		$quiz_meta['form'] = $form;

		return $this->set_quiz_lock( $user_id, $quiz_id, $quiz_meta );
	}

	/**
	 * Save answer.
	 *
	 * @param int   $user_id User id.
	 * @param int   $quiz_id Quiz id.
	 * @param array $answers Answers.
	 * @param bool  $is_all  All answers.
	 *
	 * @return bool|mixed
	 */
	protected function save_answer_meta( $user_id, $quiz_id, $answers, $is_all = false ) {
		$quiz_meta = $this->get_quiz_lock( $user_id, $quiz_id );

		if ( empty( $quiz_meta ) ) {
			return false;
		}

		foreach ( $answers as $index => $answer ) {
			$question_index = (int) $quiz_meta['next'];

			if ( (int) $index === $question_index || $is_all ) {
				if ( $answer !== $quiz_meta['questions'][ $index ]['answer'] ) {
					$quiz_meta['questions'][ $index ]['answer']   = $answer;
					$quiz_meta['questions'][ $index ]['responce'] = array();
				}

				if ( empty( $quiz_meta['questions'][ $index ]['end_time'] ) ) {
					$quiz_meta['questions'][ $index ]['end_time'] = current_time( 'timestamp' ); // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp
				}

				$quiz_meta['prev'] = $index;
				$quiz_meta['next'] = $index + 1;

				if ( ! isset( $quiz_meta['questions'][ $quiz_meta['prev'] ] ) ) {
					$quiz_meta['prev'] = - 1;
				}

				if ( ! isset( $quiz_meta['questions'][ $quiz_meta['next'] ] ) ) {
					$quiz_meta['next'] = - 1;
				}
			}
		}

		return $this->set_quiz_lock( $user_id, $quiz_id, $quiz_meta );
	}

	/**
	 * Is Open Course Quiz or not.
	 *
	 * @param int $quiz_id Quiz id.
	 *
	 * @return bool
	 */
	protected function is_open_course_quiz( $quiz_id = 0 ) {
		$course_id   = bbapp_learndash_get_course_id( $quiz_id );
		$course_type = bbapp_learndash_get_course_meta_setting( $course_id, 'course_price_type' );

		return 'open' === $course_type;
	}

	/**
	 * Get Quiz Data.
	 *
	 * @param int $user_id User id.
	 *
	 * @return false|mixed|void
	 */
	protected function get_bbapp_ld_quiz_data( $user_id = 0 ) {
		if ( ! empty( $user_id ) ) {
			if ( is_user_logged_in() && is_numeric( $user_id ) ) {
				return get_user_meta( $user_id, 'bbapp_ld_quiz', true );
			}

			return get_option( '_bbapp_ld_quiz[' . $user_id . ']' );
		}

		return array();
	}

	/**
	 * Update Quiz Data.
	 *
	 * @param int    $user_id User id.
	 * @param string $quiz_locks Quiz lock.
	 *
	 * @return bool|int|array
	 */
	protected function update_bbapp_ld_quiz_data( $user_id = 0, $quiz_locks = '' ) {
		if ( ! empty( $user_id ) ) {
			if ( is_user_logged_in() && is_numeric( $user_id ) ) {
				return update_user_meta( $user_id, 'bbapp_ld_quiz', $quiz_locks );
			}

			return update_option( '_bbapp_ld_quiz[' . $user_id . ']', $quiz_locks );
		}

		return array();
	}

	/**
	 * Delete Quiz data.
	 *
	 * @param int $user_id User id.
	 *
	 * @return bool|array
	 */
	protected function delete_bbapp_ld_quiz_data( $user_id = 0 ) {
		if ( ! empty( $user_id ) ) {
			if ( is_user_logged_in() && is_numeric( $user_id ) ) {
				return delete_user_meta( $user_id, 'bbapp_ld_quiz' );
			}

			return delete_option( '_bbapp_ld_quiz[' . $user_id . ']' );
		}

		return array();
	}

	/**
	 * Function will mark complete for topic/lesson after last quiz will complete.
	 *
	 * @param int   $user_id User id.
	 * @param array $data    Quiz data.
	 * @param int   $quiz_id Quiz id.
	 */
	public function bbapp_ld_process_to_mark_complete( $user_id, $data, $quiz_id ) {
		if ( isset( $data['quiz'] ) ) {
			$course_id = isset( $data['quiz']->course ) && ! empty( $data['quiz']->course ) ? $data['quiz']->course : learndash_get_course_id( $quiz_id );

			if ( ! empty( $course_id ) ) {
				$quiz_parent_post_id = 0;
				if ( isset( $data['quiz']->topic ) && ! empty( $data['quiz']->topic ) ) {
					$quiz_parent_post_id = $data['quiz']->topic;
				} elseif ( isset( $data['quiz']->lesson ) && ! empty( $data['quiz']->lesson ) ) {
					$quiz_parent_post_id = $data['quiz']->lesson;
				}

				if ( ! empty( $quiz_parent_post_id ) ) {
					$all_quizzes_complete = true;
					$quizzes              = learndash_get_lesson_quiz_list( $quiz_parent_post_id, $user_id );

					if ( ( ! empty( $quizzes ) ) && ( is_array( $quizzes ) ) ) {
						foreach ( $quizzes as $quiz ) {
							if ( ( isset( $quiz['status'] ) ) && ( 'completed' !== $quiz['status'] ) ) {
								$all_quizzes_complete = false;
								break;
							}
						}
					}

					if ( true === $all_quizzes_complete ) {
						learndash_process_mark_complete( $user_id, $quiz_parent_post_id, false, $course_id );
					}
				}
			}
		}
	}
}
