<?php
/**
 * Holds quiz question helper functions.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Quiz
 */

namespace BuddyBossApp\Api\LearnDash\V1\Quiz;

use stdClass;
use WP_Error;
use WP_REST_Response;
use WpProQuiz_Controller_Statistics;
use WpProQuiz_Model_Category;
use WpProQuiz_Model_CategoryMapper;
use WpProQuiz_Model_QuestionMapper;
use WpProQuiz_Model_QuizMapper;
use function GuzzleHttp\Psr7\str;

/**
 * Quiz question helper class.
 */
class QuizQuestionsHelper {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * QuizQuestionsHelper instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
	}

	/**
	 * Get the Questions list.
	 *
	 * @param int    $quiz_pro Quiz id.
	 * @param object $request  Request data.
	 *
	 * @since 1.6.8 Added the `$request` argument.
	 * @return array
	 */
	public function get_questions( $quiz_pro, $request ) {
		$quiz_mapper     = new WpProQuiz_Model_QuizMapper();
		$question_mapper = new WpProQuiz_Model_QuestionMapper();
		$quiz            = $quiz_mapper->fetch( $quiz_pro );
		$fetch_questions = array();

		/**
		 * Fetch the question according to setting
		 * - Show all quiz questions
		 * - Show predefine number of question [ LD Setting ]
		 * - Show predefine % of question [ LD Setting ]
		 * - Fetch All question Random order [ LD setting ]
		 */
		$user_id = get_current_user_id();

		if ( $quiz->isShowMaxQuestion() && $quiz->getShowMaxQuestionValue() > 0 ) {
			$learndash_quiz_resume_enabled = false;
			$learndash_quiz_resume_data    = array();
			$quiz_post_id                  = $quiz->getPostId();

			if ( ! empty( $quiz_post_id ) && $user_id ) {
				$learndash_quiz_resume_enabled = learndash_get_setting( $quiz_post_id, 'quiz_resume' );

				if ( true === $learndash_quiz_resume_enabled ) {
					$course_id = bbapp_learndash_get_course_id( $quiz_post_id );

					if ( ! $course_id ) {
						$course_id = ( ! empty( $request->get_param( 'course' ) ) ) ? $request->get_param( 'course' ) : 0;
					}

					$learndash_quiz_resume_activity = \LDLMS_User_Quiz_Resume::get_user_quiz_resume_activity( $user_id, $quiz_post_id, $course_id );

					if ( ( is_a( $learndash_quiz_resume_activity, 'LDLMS_Model_Activity' ) ) && ( property_exists( $learndash_quiz_resume_activity, 'activity_id' ) ) && ( ! empty( $learndash_quiz_resume_activity->activity_id ) ) ) {
						if ( ( property_exists( $learndash_quiz_resume_activity, 'activity_meta' ) ) && ( ! empty( $learndash_quiz_resume_activity->activity_meta ) ) ) {
							$learndash_quiz_resume_data = $learndash_quiz_resume_activity->activity_meta;
						}
					}
				}
			}

			$value = $quiz->getShowMaxQuestionValue();

			if ( $quiz->isShowMaxQuestionPercent() ) {
				$count = $question_mapper->count( $quiz_pro );

				$value = ceil( $count * $value / 100 );
			}

			if ( $learndash_quiz_resume_enabled ) {
				if ( ! empty( $learndash_quiz_resume_data ) && isset( $learndash_quiz_resume_data['randomQuestions'] ) ) {
					if ( isset( $learndash_quiz_resume_data['randomOrder'] ) ) {
						foreach ( $learndash_quiz_resume_data['randomOrder'] as $id => $value ) {
							$fetch_questions[] = $question_mapper->fetchById( $value );
						}
					}
				} else {
					$fetch_questions = $question_mapper->fetchAll( $quiz, true, $value );
				}
			} else {
				$fetch_questions = $question_mapper->fetchAll( $quiz, true, $value );
			}
		} else {
			$is_rendom_fetch = $quiz->isQuestionRandom();
			$fetch_questions = $question_mapper->fetchAll( $quiz, $is_rendom_fetch );
		}

		$questions = array();

		$question_order           = ( ! empty( $request->get_param( 'random_order' ) ) && is_array( $request->get_param( 'random_order' ) ) ) ? array_map( 'absint', array_unique( $request->get_param( 'random_order' ) ) ) : array();
		$is_question_random_order = ! empty( $question_order );

		foreach ( $fetch_questions as $que_post ) {
			$que_data = $this->prepare_question_data( $que_post );

			/**
			 * Shuffle option if quiz setting is enabled.
			 */
			if ( $quiz->isAnswerRandom() ) {
				if ( ! empty( $que_data->options_1 ) ) {
					shuffle( $que_data->options_1 );
				} else {
					shuffle( $que_data->options );
				}
			}

			if ( $is_question_random_order ) {
				$position               = array_search( $que_data->ID, $question_order, true );
				$questions[ $position ] = $que_data;
			} else {
				$questions[] = $que_data;
			}
		}

		if ( $is_question_random_order ) {
			ksort( $questions );
		}

		if ( $quiz->isSortCategories() ) {
			usort( $questions, array( $this, 'sort_categories' ) );
		}

		return $questions;
	}

	/**
	 *
	 * Get the single Question
	 *
	 * @param int $question_id Question id.
	 * @param int $quiz_pro    Quiz id.
	 *
	 * @return mixed
	 */
	public function get_guestion( $question_id, $quiz_pro ) {
		$question_mapper = new WpProQuiz_Model_QuestionMapper();
		$que_post        = $question_mapper->fetchById( $question_id );

		return $this->prepare_question_data( $que_post );
	}

	/**
	 * Check Question Answer
	 *
	 * @param array $answers  Answers.
	 * @param int   $quiz_pro Quiz id.
	 * @param int   $quiz_id  Quiz id.
	 *
	 * @return array|bool
	 */
	public function check_answers( $answers, $quiz_pro, $quiz_id ) {
		/**
		 * Ref_Files: sfwd-lms/includes/quiz/ld-quiz-pro.php
		 * Ref_Funcation: checkAnswers -> Ajax Request
		 */
		$user_id = get_current_user_id();

		/**
		 * On check endpoint we checking one question answer. we use loop because of we sending answer Array using question id so we need to use loop.
		 * End of loop we adding break statement so loop only work one time
		 */
		foreach ( $answers as $ques_id => $answer ) {
			$answer_data = $this->prepare_question_answer( $answer, $ques_id );
			$data        = array(
				'quizId'     => $quiz_pro,
				'quiz'       => $quiz_id,
				'quiz_nonce' => wp_create_nonce( 'sfwd-quiz-nonce-' . $quiz_id . '-' . $quiz_pro . '-' . $user_id ),
				'responses'  => array(
					$ques_id => array( 'response' => $answer_data ),
				),
			);

			if ( ! isset( $this->ld_quiz_pro ) ) {
				$this->ld_quiz_pro = new LdQuizpro();
			}

			$responses = $this->ld_quiz_pro->checkAnswers( $data );
			break;
		}

		if ( ! empty( $responses ) ) {
			$question_mapper = new WpProQuiz_Model_QuestionMapper();
			$result_data     = array();
			$responses       = (array) json_decode( $responses );

			foreach ( $responses as $q_id => $response ) {
				$answer['id'] = $q_id;
				$que_post     = $question_mapper->fetchById( $q_id );

				/**
				 * Filters quiz question and answer.
				 *
				 * @param array  $question_data Question data.
				 * @param object $que_post      Question post.
				 */
				$result_data[ $q_id ]['data'] = apply_filters(
					'bbapp_ld_prepare_quiz_question_answer',
					array(
						'id'               => $q_id,
						'question_type'    => $response->e->type,
						'isCorrect'        => $response->c,
						'message'          => $response->e->AnswerMessage,
						'point'            => $response->p,
						'possiblePoints'   => $response->e->possiblePoints,
						'sentItems'        => $answers[ $ques_id ],
						'sentItemsCorrect' => $this->get_item_correct( $response ),
						'possibleItems'    => isset( $response->e->c ) ? $response->e->c : '',
					),
					$que_post
				);

				if ( 'essay' === $response->e->type ) {
					$result_data[ $q_id ]['data']['graded_id']     = $response->e->graded_id;
					$result_data[ $q_id ]['data']['graded_status'] = $response->e->graded_status;
				}

				$result_data[ $q_id ]['responce'] = $response;
			}

			return $result_data[ $q_id ];
		}

		return false;
	}

	/**
	 * Save quiz question answers
	 *
	 * @param array $quiz_meta Quiz meta.
	 * @param int   $quiz_id Quiz id.
	 *
	 * @return array|WP_Error
	 */
	public function save_quiz( $quiz_meta, $quiz_id ) {
		/**
		 * Ref_Files: sfwd-lms/includes/quiz/ld-quiz-pro.php
		 * Ref_Function: checkAnswers -> Ajax Request
		 */
		$user_id              = get_current_user_id();
		$quiz_settings        = learndash_get_setting( $quiz_id );
		$quiz_pro             = $quiz_settings['quiz_pro'];
		$quiz_mapper          = new WpProQuiz_Model_QuizMapper();
		$quiz                 = $quiz_mapper->fetch( $quiz_pro );
		$question_mapper      = new WpProQuiz_Model_QuestionMapper();
		$statistic_controller = new WpProQuiz_Controller_Statistics();

		/**
		 * Prepare quiz result for learndash.
		 */
		$quiz_result           = array();
		$quiz_answers          = array();
		$total_awarded_points  = 0;
		$total_possible_points = 0;
		$total_correct         = 0;
		$cat_points            = array();
		$responses             = $this->prepare_question_answer_response( $quiz_meta, $quiz_pro, $quiz_id );

		/**
		 * Either user answer is empty or CheckAnswer process returning some error
		 */
		if ( empty( $responses ) && is_wp_error( $responses ) ) {
			return QuizError::instance()->quiz_check_question_answers_error();
		}

		$categories = $this->get_question_categories_list( $quiz_pro );

		/**
		 * Prepare Quiz answers & results data for response
		 */
		foreach ( $quiz_meta['questions'] as $index => $question ) {
			$ques_id  = $question['id'];
			$response = $responses[ $ques_id ];

			/**
			 * Prepare quiz question category total possible points count
			 */
			$que_post = $question_mapper->fetchById( $ques_id );

			if ( ! isset( $cat_points[ $que_post->getCategoryId() ] ) ) {
				$cat_points[ $que_post->getCategoryId() ] = 0;
			}

			$cat_points[ $que_post->getCategoryId() ] += intval( $que_post->getPoints() );

			/**
			 * Prepare quiz question category total awarded points count
			 */
			$categories[ $que_post->getCategoryId() ]['result'] += intval( $response->p );

			/**
			 * Prepare quiz total correct answer count
			 */
			if ( $response->c ) {
				++ $total_correct;
			}

			/**
			 * Prepare quiz total awarded points counts
			 */
			$total_awarded_points += intval( $response->p );

			/**
			 * Prepare quiz total possible points counts
			 */
			$total_possible_points += intval( $response->e->possiblePoints );

			/**
			 * Prepare User's answers
			 */
			$user_response = array();

			if ( isset( $response->e->r ) ) {
				$user_response = is_array( $response->e->r ) ? $response->e->r : array( $response->e->r );

				if ( ! empty( $user_response ) ) {
					foreach ( $user_response as $key => $value ) {
						if ( true === $value ) {
							$user_response[ $key ] = 1;
						} elseif ( false === $value ) {
							$user_response[ $key ] = 0;
						}
					}
				}
			}

			/**
			 * Prepare Quiz Question Result data
			 */
			$quiz_result[ $question['id'] ] = array(
				'time'           => $question['end_time'] - $question['start_time'],
				'points'         => $response->p,
				'correct'        => $response->c,
				'data'           => $user_response,
				'possiblePoints' => $response->e->possiblePoints,
				'a_nonce'        => $response->a_nonce,
				'p_nonce'        => $response->p_nonce,
			);

			/**
			 * Prepare & Update Quiz Question answers data.
			 */
			$quiz_answers[ $index ] = apply_filters(
				'bbapp_ld_prepare_quiz_question_answer',
				array(
					'id'               => $ques_id,
					'question_type'    => $response->e->type,
					'isCorrect'        => $response->c,
					'message'          => $response->e->AnswerMessage,
					'point'            => $response->p,
					'possiblePoints'   => $response->e->possiblePoints,
					'sentItems'        => $question['answer'],
					'sentItemsCorrect' => $this->get_item_correct( $response ),
					'possibleItems'    => isset( $response->e->c ) ? $response->e->c : '',
				),
				$que_post
			);

			if ( 'essay' === $response->e->type ) {
				$quiz_result[ $question['id'] ]['graded_id']     = $response->e->graded_id;
				$quiz_result[ $question['id'] ]['graded_status'] = $response->e->graded_status;
				$quiz_result[ $question['id'] ]['data']          = array( $response->e->graded_id );
				$quiz_answers[ $index ]['graded_id']             = $response->e->graded_id;
				$quiz_answers[ $index ]['graded_status']         = $response->e->graded_status;
			}
		}

		/**
		 * Calculate Quiz question categories percentage result.
		 */
		foreach ( $categories as $id => $value ) {
			if ( ! empty( $cat_points[ $id ] ) ) {
				$categories[ $id ]['result'] = round( $categories[ $id ]['result'] / $cat_points[ $id ] * 100 * 100 ) / 100;
			} else {
				$categories[ $id ]['result'] = 0;
			}
		}

		/**
		 * $total_possible_points is empty if quiz have only question which support answer point
		 */
		$result = 0;

		if ( ! empty( $total_possible_points ) ) {
			$result = ( $total_awarded_points / $total_possible_points * 100 * 100 ) / 100;
		}

		/**
		 * To do quiz from mobile we added trick to time limit fixed.
		 */
		$timelimit = $quiz->getTimeLimit();

		if ( ! empty( $timelimit ) ) {
			$diff = $quiz_meta['end_time'] - $quiz_meta['start_time'];

			if ( $diff > $timelimit ) {
				$quiz_meta['end_time'] = $quiz_meta['start_time'] + $timelimit;
			}
		}

		/**
		 * Prepare quiz result `comp` data.
		 */
		$quiz_result['comp'] = array(
			'points'             => $total_awarded_points,
			'correctQuestions'   => $total_correct,
			'quizTime'           => $quiz_meta['end_time'] - $quiz_meta['start_time'],
			'quizEndTimestamp'   => $quiz_meta['end_time'] * 1000,
			'quizStartTimestamp' => $quiz_meta['start_time'] * 1000,
			'result'             => $result,
			'cats'               => $categories,
		);

		/*******************************************
		 * Complete quiz using `completedQuiz`
		 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/controller/WpProQuiz_Controller_Admin.php
		 * Ref_Funcation: completedQuiz -> Ajax Request
		 */

		/**
		 * To complete quiz we need to update GET, POST or REQUEST data so take copy before update
		 */
		$_temp_request = $_REQUEST; // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended
		$_temp_post    = $_POST; // phpcs:ignore WordPress.Security.NonceVerification.Missing
		$_temp_get     = $_GET; // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended

		if ( empty( $_POST['results'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
			$_REQUEST           = array();
			$_POST              = array();
			$_GET               = array();
			$_POST['quizId']    = $quiz_pro;
			$_POST['quiz']      = $quiz_id;
			$_POST['results']   = $quiz_result;
			$_POST['timespent'] = $quiz_result['comp']['quizTime'];
			$_POST['forms']     = isset( $quiz_meta['form'] ) ? $quiz_meta['form'] : array();
			$_POST['course_id'] = isset( $_temp_request['course_id'] ) ? intval( $_temp_request['course_id'] ) : bbapp_learndash_get_course_id( $quiz_id );
			$_POST['lesson_id'] = isset( $_temp_request['lesson_id'] ) ? intval( $_temp_request['lesson_id'] ) : 0;
			$_POST['topic_id']  = isset( $_temp_request['topic_id'] ) ? intval( $_temp_request['topic_id'] ) : 0;
			$_REQUEST['quiz']   = $quiz_id;
		}

		if ( empty( $_POST['quiz_nonce'] ) && ( isset( $_POST['quiz'] ) && isset( $_POST['quizId'] ) ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.NonceVerification.Missing
			$_POST['quiz_nonce'] = wp_create_nonce( 'sfwd-quiz-nonce-' . intval( $_POST['quiz'] ) . '-' . intval( $_POST['quizId'] ) . '-' . $user_id ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.NonceVerification.Missing
		}

		if ( ! isset( $_POST['results'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.NonceVerification.Missing
			return QuizError::instance()->invalid_quiz_result_data();
		}

		/**
		 * To stop "Uncanny LearnDash Toolkit" Redirection after complete event for rest endpoint
		 */
		$_REQUEST['doing_rest'] = 1;

		$is_submitted = $this->completed_quiz();

		/**
		 * If completedQuiz fails with some error.
		 */
		if ( is_wp_error( $is_submitted ) ) {
			return $is_submitted;
		}

		/*******************************************
		 *        Store leaderboard data
		 */

		if ( $quiz->isToplistActivated() && $quiz->isToplistDataAddAutomatic() ) {
			LeaderboardRest::instance()->createitem(
				$quiz,
				array(
					'point'  => $total_awarded_points,
					'result' => $quiz_result['comp']['result'],
					'ip'     => '',
				)
			);
		}

		/**
		 * Make GET, POST or REQUEST reverted as quiz complete process is done.
		 */
		$_REQUEST = $_temp_request;
		$_POST    = $_temp_post;
		$_GET     = $_temp_get;

		/*******************************************
		 * Prepare rest response for quiz complete
		 */

		$quiz_post         = get_post( $quiz_id );
		$passingpercentage = intVal( $quiz_settings['passingpercentage'] );
		$pass              = ( $quiz_result['comp']['result'] >= $passingpercentage ) ? true : false;
		$reponse           = array(
			'quiz'               => QuizRest::instance()->get_additional_data( $quiz_post ),
			'finished'           => $is_submitted,
			'is_pass'            => $pass,
			'result'             => $quiz_result['comp']['result'],
			'answers'            => $quiz_answers,
			'passing_percentage' => $passingpercentage,
		);

		/**
		 * Calculate average quiz result for users.
		 */
		if ( $quiz->isShowAverageResult() ) {
			$reponse['average_result'] = $statistic_controller->getAverageResult( $quiz_pro );
		}

		/**
		 * Prepare certificate if it's attached with quiz
		 */
		$certificate_post_id = intval( $quiz_settings['certificate'] );

		if ( ! empty( $certificate_post_id ) ) {
			$certificate_post = get_post( $certificate_post_id );

			if ( ! empty( $certificate_post ) ) {
				$certificate = learndash_certificate_details( $quiz_id );

				if ( ( $reponse['result'] / 100 >= $certificate['certificate_threshold'] ) ) {
					$reponse['certificate']['title']    = get_the_title( $quiz_id );
					$reponse['certificate']['filename'] = sanitize_file_name( $quiz_post->post_title ) . '-' . sanitize_file_name( $certificate_post->post_title ) . '.pdf';
					$reponse['certificate']['link']     = $certificate['certificateLink'];
				}
			}
		}

		/**
		 * Prepare categories point response
		 */
		if ( $quiz->isShowCategoryScore() ) {
			$reponse['cats'] = array_values( $quiz_result['comp']['cats'] );
		}

		/**
		 * Added quiz complete time
		 */
		if ( ! $quiz->isHideResultQuizTime() ) {
			$reponse['time'] = $quiz_result['comp']['quizTime'];
		}

		/**
		 * Added quiz points data
		 */
		if ( ! $quiz->isHideResultPoints() ) {
			$reponse['awarded_points']  = $total_awarded_points;
			$reponse['possible_points'] = $total_possible_points;
		}

		/**
		 * Manage Quiz grade result manage
		 */
		if ( $quiz->isResultGradeEnabled() ) {
			$_result      = $reponse['result'];
			$_result_text = $quiz->getResultText();
			$diff         = 99999;

			foreach ( $_result_text['prozent'] as $i => $value ) {
				if ( ( $_result >= $value ) && ( ( $_result - $value ) < $diff ) ) {
					$diff               = $_result - $value;
					$reponse['message'] = $_result_text['text'][ $i ];
				}
			}
			$reponse['message'] = apply_filters( 'comment_text', $reponse['message'], null, null );
		} else {
			$reponse['message'] = apply_filters( 'comment_text', $quiz->getResultText(), null, null );
		}

		return $reponse;
	}

	/**
	 * Prepare question data.
	 *
	 * @param object $que_post Question post.
	 *
	 * @return mixed
	 */
	private function prepare_question_data( $que_post ) {
		/**
		 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/view/WpProQuiz_View_FrontQuiz.php
		 * Ref_Funcation: showQuizBox
		 */
		$question           = new stdClass();
		$question->ID       = $que_post->getId();
		$question->title    = $que_post->getQuestion();
		$question->category = $que_post->getCategoryName();

		if ( ! $que_post->isAnswerPointsActivated() ) {
			$question->points = $que_post->getPoints();
		}

		$question->question_type = $que_post->getAnswerType();
		$question->hint          = ( $que_post->isTipEnabled() ) ? do_shortcode( apply_filters( 'comment_text', $que_post->getTipMsg(), null, null ) ) : '';
		$question->options       = array();

		if ( 'matrix_sort_answer' === $question->question_type ) {
			$question->options_1 = array();
		}

		if ( ! in_array( $question->question_type, array( 'free_answer', 'cloze_answer', 'assessment_answer', 'essay' ), true ) ) {
			$answers      = $que_post->getAnswerData();
			$answer_index = 0;

			foreach ( $answers as $answer_post ) {
				if ( $que_post->isAnswerPointsActivated() ) {
					$question->points[ $answer_index ] = $answer_post->getPoints();
				}

				$answer        = new stdClass();
				$answer->title = $answer_post->getAnswer();
				$answer->value = $answer_post->getAnswer();

				if ( 'sort_answer' === $question->question_type ) {
					$answer->value = LdQuizpro::datapos( $question->ID, $answer_index );
				}

				if ( 'sort_answer' !== $question->question_type ) {
					$answer->pos = $answer_index;
				}

				if ( 'matrix_sort_answer' === $question->question_type ) {
					$answer_1              = new stdClass();
					$answer_1->title       = $answer_post->getSortString();
					$answer_1->value       = LdQuizpro::datapos( $question->ID, $answer_index );
					$question->options_1[] = $answer_1;
				}

				// Rendered shortcode when question options are in HTML mode.
				if ( $answer_post->isHtml() && function_exists( 'bbapp_remove_height_width' ) ) {
					$answer->title = bbapp_remove_height_width( bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_content', $answer->title ) ) );
					$answer->value = bbapp_remove_height_width( bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_content', $answer->value ) ) );
				}

				$question->options[] = $answer;

				$answer_index ++;
			}
		} elseif ( 'cloze_answer' === $question->question_type ) {
			$answers = $que_post->getAnswerData();

			foreach ( $answers as $answer_post ) {
				$cloze_data = bbapp_lms_fetch_cloze( $answer_post->getAnswer() );

				if ( $que_post->isAnswerPointsActivated() ) {
					$question->points = $cloze_data['points'];
				}

				$answer              = new stdClass();
				$answer->title       = str_replace( '@@wpProQuizCloze@@', '{___}', $cloze_data['replace'] );
				$answer->title       = nl2br( $answer->title ); // Convert nl to <br />.
				$question->options[] = $answer;
			}
		} elseif ( 'assessment_answer' === $question->question_type ) {
			$answers = $que_post->getAnswerData();

			foreach ( $answers as $answer_post ) {
				$get_answer          = $answer_post->getAnswer();
				$question->options[] = $this->format_assessment_answer( $get_answer );
			}
		} elseif ( 'essay' === $question->question_type ) {
			$answers = $que_post->getAnswerData();

			foreach ( $answers as $answer_post ) {
				$question->graded_type = $answer_post->getGradedType();
			}
		}

		/**
		 * Shuffle Question answer if Question type is sort_answer or matrix_sort_answer.
		 */
		if ( 'sort_answer' === $question->question_type || 'matrix_sort_answer' === $question->question_type ) {
			if ( ! empty( $question->options_1 ) ) {
				shuffle( $question->options_1 );
			} else {
				shuffle( $question->options );
			}
		}

		return $question;
	}

	/**
	 * Formats the assessment answer according to learner format.
	 *
	 * @param string $question Question.
	 *
	 * @return array
	 */
	private function format_assessment_answer( $question ) {
		$field_text = $question;

		// Split all html from answer.
		$get_html = preg_split( '/{.*?}/', $field_text );

		// split all fields.
		preg_match_all( '/{.*?}/', $field_text, $matched_field );

		if ( isset( $matched_field[0] ) ) {
			$matched_field = $matched_field[0];
		}

		$data = array();
		$i    = 0;

		foreach ( $get_html as $html ) {
			$data[] = array(
				'type'  => 'html',
				'value' => bbapp_learndash_fix_html( $html ),
			);

			if ( isset( $matched_field[ $i ] ) ) {
				// format the values from fields into array.
				preg_match_all( '/\[(.*?)\]/', $matched_field[ $i ], $formated );

				if ( isset( $formated[1] ) ) {
					$formated = $formated[1];
				} else {
					$formated = array();
				}

				$data[] = array(
					'type'  => 'radio',
					'value' => $formated,
				);
			}

			$i ++;
		}

		return $data;
	}

	/**
	 * Prepare Answer of question to check with learndash functions.
	 *
	 * @param string $answer  Answer.
	 * @param int    $ques_id Question id.
	 *
	 * @return array|int|mixed
	 */
	private function prepare_question_answer( $answer, $ques_id ) {
		$question_mapper = new WpProQuiz_Model_QuestionMapper();
		$que_post        = $question_mapper->fetchById( $ques_id );
		$que_post        = $this->prepare_question_data( $que_post );
		$answer_data     = array();

		if ( ! empty( $que_post->options ) && 'assessment_answer' !== $que_post->question_type ) {
			$answer_index = 0;

			foreach ( $que_post->options as $option ) {
				if ( 'single' === $que_post->question_type ) {
					$answer_data[ $answer_index ] = in_array( $option->value, $answer, true );
				} elseif ( 'multiple' === $que_post->question_type ) {
					$answer_data[ $answer_index ] = 'true' === (string) $answer[ $answer_index ];
				} elseif ( 'cloze_answer' === $que_post->question_type ) {
					$answer      = array_map( array( $this, 'cleanup_curly_quotes' ), $answer );
					$answer_data = $answer;
				} else {
					$answer_data[ $answer_index ] = $answer[ $answer_index ];
				}

				$answer_index ++;
			}
		} elseif ( 'assessment_answer' === $que_post->question_type ) {
			$answer_data = 0;

			foreach ( $answer as $ans ) {
				$answer_data += intval( $ans );
			}
		} else {
			$answer_data = $answer[0];
		}

		return $answer_data;
	}

	/**
	 * Prepare question answer check response.
	 *
	 * @param array $quiz_meta Quiz meta.
	 * @param int   $quiz_pro  Quiz id.
	 * @param int   $quiz_id   Quiz id.
	 *
	 * @return array
	 */
	private function prepare_question_answer_response( $quiz_meta, $quiz_pro, $quiz_id ) {
		$responses = array();
		$user_id   = get_current_user_id();
		$data      = array(
			'quizId'     => $quiz_pro,
			'quiz'       => $quiz_id,
			'quiz_nonce' => wp_create_nonce( 'sfwd-quiz-nonce-' . $quiz_id . '-' . $quiz_pro . '-' . $user_id ),
			'responses'  => array(),
		);

		/**
		 * Check response is stroe in session data if not then prepare data to check answers.
		 */
		foreach ( $quiz_meta['questions'] as $question ) {
			$ques_id = $question['id'];

			if ( ! empty( $question['responce'] ) ) {
				/**
				 * If Question answer already check with check endpoint then use response from quiz session data.
				 */
				$responses[ $ques_id ] = (object) $question['responce']['responce'];
			} else {
				$answer      = $question['answer'];
				$answer_data = $this->prepare_question_answer( $answer, $ques_id );

				/**
				 * Add User's answer to check it's correct or not
				 */
				$data['responses'][ $ques_id ] = array( 'response' => $answer_data );
			}
		}

		/**
		 * Check question answer if quiz is submitted with our checking answer.
		 */
		if ( ! empty( $data['responses'] ) ) {
			if ( ! isset( $this->ld_quiz_pro ) ) {
				$this->ld_quiz_pro = new LdQuizpro();
			}

			$ste_responses = $this->ld_quiz_pro->checkAnswers( $data );

			if ( empty( $ste_responses ) ) {

				/**
				 * Fails checkAnswers with some error
				 */
				return array();
			}

			$arr_responses = json_decode( $ste_responses );

			foreach ( $data['responses'] as $ques_id => $ans ) {
				$responses[ $ques_id ] = $arr_responses->$ques_id;
			}
		}

		return $responses;
	}

	/**
	 * Prepare Question categories list data.
	 *
	 * @param int $quiz_pro Quiz id.
	 *
	 * @return array
	 */
	private function get_question_categories_list( $quiz_pro ) {
		$categories       = array();
		$category_mapper  = new WpProQuiz_Model_CategoryMapper();
		$fetch_categories = $category_mapper->fetchByQuiz( $quiz_pro );

		/**
		 * Filter to modify quiz categories list data.
		 *
		 * @param array $fetchCategories Array of quiz categories
		 * @param int   $quiz_pro        Quiz ID
		 *
		 * @since 1.8.70
		 */
		$fetch_categories = apply_filters( 'bbapp_fetch_by_quiz_category', $fetch_categories, $quiz_pro );

		foreach ( $fetch_categories as $catgory ) {
			/**
			 * LD quiz model variable.
			 *
			 * @var $catgory WpProQuiz_Model_Category
			 */

			if ( ! $catgory->getCategoryId() ) {
				$categories[0] = array(
					'name'   => __( 'Not categorized', 'buddyboss-app' ),
					'result' => 0,
				);
				continue;
			}

			$categories[ $catgory->getCategoryId() ] = array(
				'name'   => $catgory->getCategoryName(),
				'result' => 0,
			);
		}

		return $categories;
	}

	/**
	 * Sort question by quiz question category.
	 *
	 * @param object $a Category data.
	 * @param object $b Category data.
	 *
	 * @return int
	 */
	private function sort_categories( $a, $b ) {
		return strcmp( $b->category, $a->category );
	}

	/**
	 * Clean Up Extra space and make user's answers in lower case so we can support lower and upper both.
	 *
	 * @param string $answer Quiz answer.
	 *
	 * @return string
	 */
	private function cleanup_curly_quotes( $answer ) {
		if ( version_compare( LEARNDASH_VERSION, '2.5', '<' ) ) {
			$answer = strtolower( $answer );
		}

		$answer = trim( $answer );

		return $answer;
	}

	/**
	 * Prepare Correct answers response
	 *
	 * @param WP_REST_Response $response Rest response.
	 *
	 * @return array
	 */
	private function get_item_correct( $response ) {
		switch ( $response->e->type ) {
			case 'cloze_answer':
				$sent_items_correct = array_values( (array) $response->s );
				break;
			case 'multiple':
				$sent_items_correct = array();
				foreach ( $response->e->c as $index => $value ) {
					$sent_items_correct[] = ( $response->e->r[ $index ] === (bool) $value ) ? true : false;
				}
				break;
			case 'sort_answer':
			case 'matrix_sort_answer':
				$sent_items_correct = array();

				foreach ( $response->e->c as $index => $value ) {
					$sent_items_correct[] = ( $response->e->r[ $index ] === $value ) ? true : false;
				}
				break;
			default:
				$sent_items_correct = $response->c;
		}

		return $sent_items_correct;
	}

	/**
	 * Complete quiz
	 */
	private function completed_quiz() {
		/**
		 * Check if $_POST data is json encode then decode it
		 */
		if ( ( ! empty( $_POST['results'] ) ) && ( is_string( $_POST['results'] ) ) ) {
			$_POST['results'] = json_decode( wp_unslash( $_POST['results'] ), true ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}

		$user_id  = ( is_user_logged_in() ) ? get_current_user_id() : 0;
		$quiz_pro = ( isset( $_POST['quizId'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['quizId'] ) ) : 0; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$quiz_id  = ( isset( $_POST['quiz'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['quiz'] ) ) : 0; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		/**
		 * If the results is not present then abort.
		 */
		if ( ! isset( $_POST['results'] ) ) {
			return QuizError::instance()->invalid_quiz_result_data();
		}

		/**
		 * LD 2.4.3 - Change in logic. Instead of accepting the values for points, correct etc from request we now result array [ question answer with none per answer ].
		 * Now we first verify that question nonce and than calculate points here
		 */
		$total_awarded_points = 0;
		$total_correct        = 0;

		/**
		 * Loop over the 'results' items. We verify and calculate the points+correct counts as well as the student response 'data'. When we get to the 'comp' results element
		 * we set the award points and correct as well as determine the total possible points.
		 */
		foreach ( $_POST['results'] as $r_idx => $result ) { //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.NonceVerification.Missing
			if ( 'comp' === $r_idx ) {
				$_POST['results'][ $r_idx ]['points']           = intval( $total_awarded_points );
				$_POST['results'][ $r_idx ]['correctQuestions'] = intval( $total_correct );
				continue;
			}

			$points_array = array(
				'points'         => function_exists( 'learndash_format_course_points' ) ? learndash_format_course_points( $result['points'] ) : intval( $result['points'] ),
				'correct'        => intval( $result['correct'] ),
				'possiblePoints' => function_exists( 'learndash_format_course_points' ) ? learndash_format_course_points( $result['possiblePoints'] ) : intval( $result['possiblePoints'] ),
			);

			$points_array['correct'] = ( true === $points_array['correct'] ) ? 1 : $points_array['correct'];
			$points_array['correct'] = ( false === $points_array['correct'] ) ? 0 : $points_array['correct'];
			$points_str              = maybe_serialize( $points_array );

			if ( ! wp_verify_nonce( $result['p_nonce'], 'ld_quiz_pnonce' . $user_id . '_' . $quiz_pro . '_' . $quiz_id . '_' . $r_idx . '_' . $points_str ) ) {
				$_POST['results'][ $r_idx ]['points']         = 0;
				$_POST['results'][ $r_idx ]['correct']        = 0;
				$_POST['results'][ $r_idx ]['possiblePoints'] = 0;
			}

			$total_awarded_points += isset( $_POST['results'][ $r_idx ]['points'] ) ? intval( $_POST['results'][ $r_idx ]['points'] ) : 0;
			$total_correct        += isset( $_POST['results'][ $r_idx ]['correct'] ) ? intval( $_POST['results'][ $r_idx ]['correct'] ) : 0;
			$response_str          = maybe_serialize( $result['data'] );

			if ( ! wp_verify_nonce( $result['a_nonce'], 'ld_quiz_anonce' . $user_id . '_' . $quiz_pro . '_' . $quiz_id . '_' . $r_idx . '_' . $response_str ) ) {
				$_POST['results'][ $r_idx ]['data'] = array();
			}
		}

		$quiz = new QuizComplete();

		return $quiz->completedQuiz();
	}
}
