<?php
/**
 * Holds quiz different errors.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Quiz
 */

namespace BuddyBossApp\Api\LearnDash\V1\Quiz;

use BuddyBossApp\RestErrors;
use WP_Error;
use WP_Post;

/**
 * Quiz error class.
 */
class QuizError {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return QuizError
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * When user is not logged in this error message should be used.
	 *
	 * @return WP_Error
	 */
	public function user_not_logged_in() {
		return RestErrors::instance()->user_not_logged_in();
	}

	/**
	 * When quiz id not correct.
	 *
	 * @return WP_Error
	 */
	public function invalid_quiz_id() {
		return new WP_Error( 'learndash_json_quiz_invalid_id', __( 'Quiz not found.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * When user don't have quiz access.
	 *
	 * @return WP_Error
	 */
	public function invalid_quiz_access() {
		return new WP_Error( 'learndash_json_quiz_invalid_access', __( 'You do not have access to this quiz.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user not completed previous module for quiz.
	 *
	 * @param int $last_incomplete_step Last incomplete step.
	 *
	 * @return WP_Error
	 */
	public function quiz_previous_module_not_completed( $last_incomplete_step ) {
		if ( ! empty( $previous_item ) && $previous_item instanceof WP_Post ) {
			switch ( $previous_item->post_type ) {
				case ( 'sfwd-quiz' ):
					/* translators: %s: Quiz label. */
					$message = sprintf( esc_html_x( 'Please go back and complete the previous %s.', 'placeholders: quiz label', 'buddyboss-app' ), learndash_get_custom_label_lower( 'quiz' ) );
					break;
				case ( 'sfwd-topic' ):
					/* translators: %s: topic label. */
					$message = sprintf( esc_html_x( 'Please go back and complete the previous %s.', 'placeholders: topic label', 'buddyboss-app' ), learndash_get_custom_label_lower( 'topic' ) );
					break;
				default:
					/* translators: %s: Lesson label. */
					$message = sprintf( esc_html_x( 'Please go back and complete the previous %s.', 'placeholders: lesson label', 'buddyboss-app' ), learndash_get_custom_label_lower( 'lesson' ) );
					break;
			}
		} else {
			/* translators: %s: Lesson label. */
			$message = sprintf( esc_html_x( 'Please go back and complete the previous %s.', 'placeholders lesson', 'buddyboss-app' ), learndash_get_custom_label_lower( 'lesson' ) );
		}

		return new WP_Error( 'learndash_json_quiz_previous_module_not_completed', $message, array( 'status' => 400 ) );
	}

	/**
	 * When user not completed previous quiz.
	 *
	 * @return WP_Error
	 */
	public function quiz_previous_not_completed() {
		return new WP_Error( 'learndash_json_quiz_previous_not_completed', __( 'You have to pass the previous Module’s quiz in order to start this quizTest Quiz Pre.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user not completed previous quiz.
	 *
	 * @return WP_Error
	 */
	public function quiz_attempts_left_empty() {
		return new WP_Error( 'learndash_json_quiz_attempts_Left_empty', __( 'Not allowed to take a quiz.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user not completed previous quiz.
	 *
	 * @return WP_Error
	 */
	public function quiz_not_started() {
		return new WP_Error( 'learndash_json_quiz_not_started', __( 'Please start the quiz.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * Quiz Result data is empty when complete a quiz.
	 *
	 * @return WP_Error
	 */
	public function invalid_quiz_result_data() {
		return new WP_Error( 'learndash_json_quiz_result_data_invalid', __( 'Quiz result data is not valid.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}


	/**
	 * When next question is not found.
	 *
	 * @return WP_Error
	 */
	public function invalid_quiz_next_question_id() {
		return new WP_Error( 'learndash_json_quiz_next_question_invalid_id', __( 'Next Quiz Question not found.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When question id is incorrect.
	 *
	 * @return WP_Error
	 */
	public function invalid_quiz_question_id() {
		return new WP_Error( 'learndash_json_quiz_question_invalid_id', __( 'Quiz Question not found.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When Quiz quiestion answer is empty.
	 *
	 * @return WP_Error
	 */
	public function quiz_question_empty_answer() {
		return new WP_Error( 'learndash_json_quiz_question_answer_empty', __( "Quiz question answer shouldn't be empty.", 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When Quiz question answer check fails.
	 *
	 * @return WP_Error
	 */
	public function quiz_question_check_error() {
		return new WP_Error( 'learndash_json_quiz_question_check_error', __( 'Something is wrong with the question check process.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When Quiz quiestion answers file is empty.
	 *
	 * @return WP_Error
	 */
	public function quiz_question_empty_answers_file() {
		return new WP_Error( 'learndash_json_quiz_question_answers_file_empty', __( "Quiz question answers file shouldn't be empty.", 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When Quiz question answer file upload fails.
	 *
	 * @return WP_Error
	 */
	public function quiz_question_upload_error() {
		return new WP_Error( 'learndash_json_quiz_question_upload_error', __( 'Something is wrong with question answer file process.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * If checkAnswers is not successfull.
	 *
	 * @return WP_Error
	 */
	public function quiz_check_question_answers_error() {
		return new WP_Error( 'learndash_json_quiz_question_upload_error', __( 'Something is wrong with check question answer process.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * If Leaderboard point is empty while create entry.
	 *
	 * @return WP_Error
	 */
	public function quiz_leaderboard_point_empty() {
		return new WP_Error( 'learndash_json_leaderboard_point_empty', __( 'Leaderboard Point data missing!', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * If Leaderboard result is empty while create entry.
	 *
	 * @return WP_Error
	 */
	public function quiz_leaderboard_result_empty() {
		return new WP_Error( 'learndash_json_leaderboard_Result_empty', __( 'Leaderboard result data missing!', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * If Leaderboard is not active.
	 *
	 * @return WP_Error
	 */
	public function quiz_leaderboard_not_active() {
		return new WP_Error( 'learndash_json_leaderboard_not_active', __( 'Leaderboard is not activated for quiz.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * If user don't have Leaderboard permission.
	 *
	 * @return WP_Error
	 */
	public function quiz_leaderboard_permission_invalid() {
		return new WP_Error( 'learndash_json_leaderboard_permission_invalid', __( 'Leaderboard permission is not allowed.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * If user entry already exist.
	 *
	 * @return WP_Error
	 */
	public function quiz_leaderboard_entry_exist() {
		return new WP_Error( 'learndash_json_leaderboard_entry_exist', __( 'Leaderboard entry already exists. You cannot enter it again.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * If user Attempts limit cross.
	 *
	 * @param int $attempts_count Number of attempts.
	 *
	 * @return WP_Error
	 */
	public function quiz_attempts_alert( $attempts_count = 0 ) {
		/* translators: %1$s: Quiz name, %2$s: Number of attempts */
		return new WP_Error( 'learndash_json_quiz_attempts_alert', sprintf( __( 'You have already taken this %1$s %2$d time(s) and are not allowed to take it again.', 'buddyboss-app' ), learndash_get_custom_label_lower( 'quiz' ), $attempts_count ), array( 'status' => 400 ) );
	}

	/**
	 * When user doesn't match required course prerequities.
	 *
	 * @param string $quiz_access_from Time string to show when quiz available.
	 *
	 * @since 1.7.3
	 *
	 * @return WP_Error
	 */
	public function quiz_not_available( $quiz_access_from ) {
		$message = sprintf( '%1$s %2$s', __( 'Available on:', 'buddyboss-app' ), esc_html( $quiz_access_from ) );

		return new WP_Error( 'learndash_json_quiz_not_available', $message, array( 'status' => 400 ) );
	}
}
