<?php
/**
 * Holds quiz complete.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Quiz
 */

namespace BuddyBossApp\Api\LearnDash\V1\Quiz;

use LearnDash_Custom_Label;
use WP_User;
use WpProQuiz_Controller_Controller;
use WpProQuiz_Controller_Statistics;
use WpProQuiz_Controller_Toplist;
use WpProQuiz_Model_Category;
use WpProQuiz_Model_CategoryMapper;
use WpProQuiz_Model_Form;
use WpProQuiz_Model_FormMapper;
use WpProQuiz_Model_GlobalSettingsMapper;
use WpProQuiz_Model_Lock;
use WpProQuiz_Model_LockMapper;
use WpProQuiz_Model_PrerequisiteMapper;
use WpProQuiz_Model_Quiz;
use WpProQuiz_Model_QuizMapper;
use WpProQuiz_Model_Template;
use WpProQuiz_Model_TemplateMapper;
use WpProQuiz_View_QuizEdit;
use WpProQuiz_View_View;

/**
 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/controller/WpProQuiz_Controller_Quiz.php
 * Remove Exit in completedQuiz method;
 */
class QuizComplete extends WpProQuiz_Controller_Controller {

	/**
	 * Quiz view.
	 *
	 * @var $view
	 */
	private $view;

	/**
	 * Quiz route.
	 *
	 * @param null $get  Get request data.
	 * @param null $post Quiz data.
	 */
	public function route( $get = null, $post = null ) {
		if ( empty( $get ) ) {
			$get = $_GET; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		}

		$action = isset( $get['action'] ) ? $get['action'] : 'show';

		switch ( $action ) {
			case 'show':
				$this->showAction();
				break;
			case 'addEdit':
				$this->addEditQuiz( $get, $post );
				break;
			case 'delete':
				if ( isset( $get['id'] ) ) {
					$this->deleteAction( intval( $get['id'] ) );
				}
				break;
			case 'reset_lock':
				if ( isset( $get['id'] ) ) {
					$this->resetLock( intval( $get['id'] ) );
				}
				break;
		}
	}

	/**
	 * Add quiz edit data.
	 *
	 * @param null $get  Get request data.
	 * @param null $post Quiz data.
	 */
	private function addEditQuiz( $get = null, $post = null ) {
		if ( empty( $get ) ) {
			$get = $_GET; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		}

		if ( ! empty( $post ) ) {
			$this->_post = $post;
		}

		$quiz_id = isset( $get['quizId'] ) ? (int) $get['quizId'] : 0;

		if ( $quiz_id ) {
			if ( ! current_user_can( 'wpProQuiz_edit_quiz' ) ) {
				wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'buddyboss-app' ) );
			}
		} else {
			if ( ! current_user_can( 'wpProQuiz_add_quiz' ) ) {
				wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'buddyboss-app' ) );
			}
		}

		$pre_requisite_mapper    = new WpProQuiz_Model_PrerequisiteMapper();
		$quiz_mapper             = new WpProQuiz_Model_QuizMapper();
		$form_mapper             = new WpProQuiz_Model_FormMapper();
		$template_mapper         = new WpProQuiz_Model_TemplateMapper();
		$quiz                    = new WpProQuiz_Model_Quiz();
		$forms                   = null;
		$pre_requisite_quiz_list = array();

		if ( ! empty( $get['post_id'] ) ) {
			$quiz_post           = get_post( $get['post_id'] );
			$this->_post['name'] = $quiz_post->post_title;
		}

		if ( 0 === $quiz_id && $quiz_mapper->exists( $quiz_id ) ) {
			/* translators: %s: Quiz label */
			WpProQuiz_View_View::admin_notices( sprintf( esc_html_x( '%s not found', 'Quiz not found', 'buddyboss-app' ), LearnDash_Custom_Label::get_label( 'quiz' ) ), 'error' );

			return;
		}

		if ( isset( $this->_post['template'] ) || ( isset( $this->_post['templateLoad'] ) && isset( $this->_post['templateLoadId'] ) ) ) {
			if ( isset( $this->_post['template'] ) ) {
				$template = $this->saveTemplate();
			} else {
				$template = $template_mapper->fetchById( $this->_post['templateLoadId'] );
			}

			$data = $template->getData();

			if ( null !== $data ) {
				$quiz = $data['quiz'];

				$quiz->setId( $quiz_id );
				$quiz->setName( $this->_post['name'] );
				$quiz->setText( 'AAZZAAZZ' );
				$quiz_mapper->save( $quiz );

				if ( empty( $quiz_id ) && ! empty( $get['post_id'] ) ) {
					learndash_update_setting( $get['post_id'], 'quiz_pro', $quiz->getId() );
				}

				$quiz_id                 = $quiz->getId();
				$forms                   = $data['forms'];
				$pre_requisite_quiz_list = $data['prerequisiteQuizList'];
			}
		} elseif ( isset( $this->_post['form'] ) ) {
			if ( isset( $this->_post['resultGradeEnabled'] ) ) {
				$this->_post['result_text'] = $this->filterResultTextGrade( $this->_post );
			}

			// Patch to only set Statistics on if post from form save.
			// LEARNDASH-1434 & LEARNDASH-1481.
			if ( ! isset( $this->_post['statisticsOn'] ) ) {
				$this->_post['statisticsOn']          = '0';
				$this->_post['viewProfileStatistics'] = '0';
			}

			$quiz = new WpProQuiz_Model_Quiz( $this->_post );

			$quiz->setId( $quiz_id );

			if ( $this->checkValidit( $this->_post ) ) {
				if ( $quiz_id ) {
					/* translators: %s: Quiz label */
					WpProQuiz_View_View::admin_notices( sprintf( esc_html_x( '%s edited', 'Quiz edited', 'buddyboss-app' ), LearnDash_Custom_Label::get_label( 'quiz' ) ), 'info' );
				} else {
					/* translators: %s: Quiz label */
					WpProQuiz_View_View::admin_notices( sprintf( esc_html_x( '%s created', 'Quiz created', 'buddyboss-app' ), LearnDash_Custom_Label::get_label( 'quiz' ) ), 'info' );
				}

				$quiz->setText( 'AAZZAAZZ' );

				$quiz_mapper->save( $quiz );

				if ( empty( $quiz_id ) && ! empty( $get['post_id'] ) ) {
					learndash_update_setting( $get['post_id'], 'quiz_pro', $quiz->getId() );
				}

				if ( ( isset( $get['post_id'] ) ) && ( ! empty( $get['post_id'] ) ) ) {
					if ( isset( $this->_post['viewProfileStatistics'] ) ) {
						$quiz->setViewProfileStatistics( true );
						update_post_meta( $get['post_id'], '_viewProfileStatistics', 1 );
					} else {
						$quiz->setViewProfileStatistics( false );
						update_post_meta( $get['post_id'], '_viewProfileStatistics', 0 );
					}
				}

				$quiz_id = $quiz->getId();

				$pre_requisite_mapper->delete( $quiz_id );

				if ( $quiz->isPrerequisite() && ! empty( $this->_post['prerequisiteList'] ) ) {
					$pre_requisite_mapper->save( $quiz_id, $this->_post['prerequisiteList'] );
					$quiz_mapper->activateStatitic( $this->_post['prerequisiteList'], 1440 );
				}

				if ( ! $this->formHandler( $quiz->getId(), $this->_post ) ) {
					$quiz->setFormActivated( false );
					$quiz->setText( 'AAZZAAZZ' );
					$quiz_mapper->save( $quiz );
				}

				$forms                   = $form_mapper->fetch( $quiz_id );
				$pre_requisite_quiz_list = $pre_requisite_mapper->fetchQuizIds( $quiz_id );
			} else {
				/* translators: %1$s: Quiz label, %2$s: Quiz label, */
				WpProQuiz_View_View::admin_notices( sprintf( esc_html_x( '%1$s title or %2$s description are not filled', 'Quiz title or quiz description are not filled', 'buddyboss-app' ), LearnDash_Custom_Label::get_label( 'quiz' ), LearnDash_Custom_Label::label_to_lower( 'quiz' ) ) );
			}
		} elseif ( $quiz_id ) {
			$quiz                    = $quiz_mapper->fetch( $quiz_id );
			$forms                   = $form_mapper->fetch( $quiz_id );
			$pre_requisite_quiz_list = $pre_requisite_mapper->fetchQuizIds( $quiz_id );
		}

		$this->view                       = new WpProQuiz_View_QuizEdit();
		$this->view->quiz                 = $quiz;
		$this->view->forms                = $forms;
		$this->view->prerequisiteQuizList = $pre_requisite_quiz_list;
		$this->view->templates            = $template_mapper->fetchAll( WpProQuiz_Model_Template::TEMPLATE_TYPE_QUIZ, false );
		$this->view->quizList             = $quiz_mapper->fetchAllAsArray(
			array(
				'id',
				'name',
			),
			$quiz_id ? array( $quiz_id ) : array()
		);
		$this->view->captchaIsInstalled   = class_exists( 'ReallySimpleCaptcha' );
		/* translators: %s: Quiz label. */
		$this->view->header = $quiz_id ? sprintf( esc_html_x( 'Edit %s', 'Edit quiz', 'buddyboss-app' ), LearnDash_Custom_Label::label_to_lower( 'quiz' ) ) : sprintf( esc_html_x( 'Create %s', 'Create quiz', 'buddyboss-app' ), LearnDash_Custom_Label::label_to_lower( 'quiz' ) );

		$this->view->show( $get );
	}

	/**
	 * Is quiz lock.
	 *
	 * @param int $quiz_id Quiz id.
	 *
	 * @return array|null
	 */
	public function isLockQuiz( $quiz_id ) {
		$quiz_id              = (int) $this->_post['quizId'];
		$user_id              = get_current_user_id();
		$data                 = array();
		$lock_mapper          = new WpProQuiz_Model_LockMapper();
		$quiz_mapper          = new WpProQuiz_Model_QuizMapper();
		$pre_requisite_mapper = new WpProQuiz_Model_PrerequisiteMapper();
		$quiz                 = $quiz_mapper->fetch( $this->_post['quizId'] );

		if ( null === $quiz || $quiz->getId() <= 0 ) {
			return null;
		}

		if ( $this->isPreLockQuiz( $quiz ) ) {
			$lock_ip     = $lock_mapper->isLock( $this->_post['quizId'], $this->getIp(), $user_id, WpProQuiz_Model_Lock::TYPE_QUIZ );
			$lock_cookie = false;
			$cookie_time = $quiz->getQuizRunOnceTime();

			if ( isset( $this->_cookie['wpProQuiz_lock'] ) && 0 === $user_id && $quiz->isQuizRunOnceCookie() ) {
				$cookie_json = json_decode( $this->_cookie['wpProQuiz_lock'], true );

				if ( false !== $cookie_json ) {
					if ( isset( $cookie_json[ $this->_post['quizId'] ] ) && (int) $cookie_json[ $this->_post['quizId'] ] === $cookie_time ) {
						$lock_cookie = true;
					}
				}
			}

			$data['lock'] = array(
				'is'  => ( $lock_ip || $lock_cookie ),
				'pre' => true,
			);
		}

		if ( $quiz->isPrerequisite() ) {
			$quiz_ids = array();

			if ( $user_id > 0 ) {
				$quiz_ids = $pre_requisite_mapper->getNoPrerequisite( $quiz_id, $user_id );
			} else {
				$check_ids = $pre_requisite_mapper->fetchQuizIds( $quiz_id );

				if ( isset( $this->_cookie['wpProQuiz_result'] ) ) {
					$r = json_decode( $this->_cookie['wpProQuiz_result'], true );

					if ( is_array( $r ) ) {
						foreach ( $check_ids as $id ) {
							if ( ! isset( $r[ $id ] ) || ! $r[ $id ] ) {
								$quiz_ids[] = $id;
							}
						}
					}
				} else {
					$quiz_ids = $check_ids;
				}
			}

			if ( ! empty( $quiz_ids ) ) {
				$post_quiz_ids = array();

				foreach ( $quiz_ids as $pro_quiz_id ) {
					$post_quiz_id = learndash_get_quiz_id_by_pro_quiz_id( $pro_quiz_id );

					if ( ! empty( $post_quiz_id ) ) {
						$post_quiz_ids[ $post_quiz_id ] = $pro_quiz_id;
					}
				}

				if ( ! empty( $post_quiz_ids ) ) {
					$post_quiz_ids = learndash_is_quiz_notcomplete( $user_id, $post_quiz_ids, true );

					if ( ! empty( $post_quiz_ids ) ) {
						$quiz_ids = array_values( $post_quiz_ids );
					} else {
						$quiz_ids = array();
					}
				}

				if ( ! empty( $quiz_ids ) ) {
					$names = $quiz_mapper->fetchCol( $quiz_ids, 'name' );

					if ( ! empty( $names ) ) {
						$data['prerequisite'] = implode( ', ', $names );
					}
				}
			}
		}

		if ( $quiz->isStartOnlyRegisteredUser() ) {
			$data['startUserLock'] = (int) ! is_user_logged_in();
		}

		return $data;
	}

	/**
	 * Load quiz data.
	 *
	 * @return array
	 */
	public function loadQuizData() {
		$quiz_id              = ( ! empty( $_POST['quizId'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['quizId'] ) ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$quiz_mapper          = new WpProQuiz_Model_QuizMapper();
		$top_list_controller  = new WpProQuiz_Controller_Toplist();
		$statistic_controller = new WpProQuiz_Controller_Statistics();
		$quiz                 = $quiz_mapper->fetch( $quiz_id );
		$data                 = array();

		if ( null === $quiz || $quiz->getId() <= 0 ) {
			return array();
		}

		$data['toplist'] = $top_list_controller->getAddToplist( $quiz );

		if ( $quiz->isShowAverageResult() ) {
			$data['averageResult'] = $statistic_controller->getAverageResult( $quiz_id );
		} else {
			$data['averageResult'] = 0;
		}

		return $data;
	}

	/**
	 * Reset quiz lock.
	 *
	 * @param int $quiz_id Quiz id.
	 */
	private function resetLock( $quiz_id ) {
		if ( ! current_user_can( 'wpProQuiz_edit_quiz' ) ) {
			exit;
		}

		$lm = new WpProQuiz_Model_LockMapper();
		$qm = new WpProQuiz_Model_QuizMapper();
		$q  = $qm->fetch( $quiz_id );

		if ( $q->getId() > 0 ) {
			$q->setQuizRunOnceTime( time() );
			$qm->save( $q );
			$lm->deleteByQuizId( $quiz_id, WpProQuiz_Model_Lock::TYPE_QUIZ );
		}
		exit;
	}

	/**
	 * Quiz show action.
	 */
	private function showAction() {
		if ( ! current_user_can( 'wpProQuiz_show' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'buddyboss-app' ) );
		}

		$this->view = new WpProQuiz_View_QuizOverall();

		$m                = new WpProQuiz_Model_QuizMapper();
		$this->view->quiz = $m->fetchAll();

		$this->view->show();
	}

	/**
	 * Quiz edit action.
	 *
	 * @param int $id Quiz id.
	 */
	private function editAction( $id ) {

		if ( ! current_user_can( 'wpProQuiz_edit_quiz' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'buddyboss-app' ) );
		}

		$pre_requisite_mapper = new WpProQuiz_Model_PrerequisiteMapper();
		$quiz_mapper          = new WpProQuiz_Model_QuizMapper();
		$form_mapper          = new WpProQuiz_Model_FormMapper();
		$template_mapper      = new WpProQuiz_Model_TemplateMapper();
		$m                    = new WpProQuiz_Model_QuizMapper();
		$this->view           = new WpProQuiz_View_QuizEdit();
		/* translators: %s: Lowercase quiz label. */
		$this->view->header      = sprintf( esc_html_x( 'Edit %s', 'Edit quiz', 'buddyboss-app' ), LearnDash_Custom_Label::label_to_lower( 'quiz' ) );
		$forms                   = $form_mapper->fetch( $id );
		$pre_requisite_quiz_list = $pre_requisite_mapper->fetchQuizIds( $id );

		if ( 0 === (int) $m->exists( $id ) ) {
			/* translators: %s: Quiz label. */
			WpProQuiz_View_View::admin_notices( sprintf( esc_html_x( '%s not found', 'Quiz not found', 'buddyboss-app' ), LearnDash_Custom_Label::get_label( 'quiz' ) ), 'error' );

			return;
		}

		if ( isset( $this->_post['submit'] ) ) {
			if ( isset( $this->_post['resultGradeEnabled'] ) ) {
				$this->_post['result_text'] = $this->filterResultTextGrade( $this->_post );
			}

			$quiz = new WpProQuiz_Model_Quiz( $this->_post );
			$quiz->setId( $id );

			if ( $this->checkValidit( $this->_post ) ) {
				/* translators: %s: Quiz label. */
				WpProQuiz_View_View::admin_notices( sprintf( esc_html_x( '%s edited', 'Quiz edited', 'buddyboss-app' ), LearnDash_Custom_Label::get_label( 'quiz' ) ), 'info' );

				$pre_requisite_mapper = new WpProQuiz_Model_PrerequisiteMapper();

				$pre_requisite_mapper->delete( $id );

				if ( $quiz->isPrerequisite() && ! empty( $this->_post['prerequisiteList'] ) ) {
					$pre_requisite_mapper->save( $id, $this->_post['prerequisiteList'] );
					$quiz_mapper->activateStatitic( $this->_post['prerequisiteList'], 1440 );
				}

				if ( ! $this->formHandler( $quiz->getId(), $this->_post ) ) {
					$quiz->setFormActivated( false );
				}

				$quiz_mapper->save( $quiz );
				$this->showAction();

				return;
			} else {
				/* translators: %1$s: Quiz label, %2$s: Lowercase quiz label. */
				WpProQuiz_View_View::admin_notices( sprintf( esc_html_x( '%1$s title or %2$s description are not filled', 'Quiz title or quiz description are not filled', 'buddyboss-app' ), LearnDash_Custom_Label::get_label( 'quiz' ), LearnDash_Custom_Label::label_to_lower( 'quiz' ) ) );
			}
		} elseif ( isset( $this->_post['template'] ) || isset( $this->_post['templateLoad'] ) ) {
			if ( isset( $this->_post['template'] ) ) {
				$template = $this->saveTemplate();
			} else {
				$template = $template_mapper->fetchById( $this->_post['templateLoadId'] );
			}

			$data = $template->getData();

			if ( null !== $data ) {
				$quiz                    = $data['quiz'];
				$forms                   = $data['forms'];
				$pre_requisite_quiz_list = $data['prerequisiteQuizList'];
			}
		} else {
			$quiz = $m->fetch( $id );
		}

		$this->view->quiz                 = $quiz;
		$this->view->prerequisiteQuizList = $pre_requisite_quiz_list;
		$this->view->quizList             = $m->fetchAllAsArray( array( 'id', 'name' ), array( $id ) );
		$this->view->captchaIsInstalled   = class_exists( 'ReallySimpleCaptcha' );
		$this->view->forms                = $forms;
		$this->view->templates            = $template_mapper->fetchAll( WpProQuiz_Model_Template::TEMPLATE_TYPE_QUIZ, false );
		$this->view->show();
	}

	/**
	 * Quiz create action.
	 */
	private function createAction() {
		if ( ! current_user_can( 'wpProQuiz_add_quiz' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'buddyboss-app' ) );
		}

		$this->view = new WpProQuiz_View_QuizEdit();
		/* translators: %s: Lowercase quiz label. */
		$this->view->header      = sprintf( esc_html_x( 'Create %s', 'Create quiz', 'buddyboss-app' ), LearnDash_Custom_Label::label_to_lower( 'quiz' ) );
		$forms                   = null;
		$pre_requisite_quiz_list = array();
		$m                       = new WpProQuiz_Model_QuizMapper();
		$template_mapper         = new WpProQuiz_Model_TemplateMapper();

		if ( isset( $this->_post['submit'] ) ) {
			if ( isset( $this->_post['resultGradeEnabled'] ) ) {
				$this->_post['result_text'] = $this->filterResultTextGrade( $this->_post );
			}

			$quiz        = new WpProQuiz_Model_Quiz( $this->_post );
			$quiz_mapper = new WpProQuiz_Model_QuizMapper();

			if ( $this->checkValidit( $this->_post ) ) {
				/* translators: %s: Lowercase quiz label. */
				WpProQuiz_View_View::admin_notices( sprintf( esc_html_x( 'Create %s', 'Create quiz', 'buddyboss-app' ), LearnDash_Custom_Label::label_to_lower( 'quiz' ) ), 'info' );

				$quiz_mapper->save( $quiz );

				$id                   = $quiz_mapper->getInsertId();
				$pre_requisite_mapper = new WpProQuiz_Model_PrerequisiteMapper();

				if ( $quiz->isPrerequisite() && ! empty( $this->_post['prerequisiteList'] ) ) {
					$pre_requisite_mapper->save( $id, $this->_post['prerequisiteList'] );
					$quiz_mapper->activateStatitic( $this->_post['prerequisiteList'], 1440 );
				}

				if ( ! $this->formHandler( $id, $this->_post ) ) {
					$quiz->setFormActivated( false );
					$quiz_mapper->save( $quiz );
				}

				$this->showAction();

				return;
			} else {
				/* translators: %1$s: Quiz label, %2$s: Lowercase quiz label.  */
				WpProQuiz_View_View::admin_notices( sprintf( esc_html_x( '%1$s title or %2$s description are not filled', 'Quiz title or quiz description are not filled', 'buddyboss-app' ), LearnDash_Custom_Label::get_label( 'quiz' ), LearnDash_Custom_Label::label_to_lower( 'quiz' ) ) );
			}
		} elseif ( isset( $this->_post['template'] ) || isset( $this->_post['templateLoad'] ) ) {
			if ( isset( $this->_post['template'] ) ) {
				$template = $this->saveTemplate();
			} else {
				$template = $template_mapper->fetchById( $this->_post['templateLoadId'] );
			}

			$data = $template->getData();

			if ( null !== $data ) {
				$quiz                    = $data['quiz'];
				$forms                   = $data['forms'];
				$pre_requisite_quiz_list = $data['prerequisiteQuizList'];
			}
		} else {
			$quiz = new WpProQuiz_Model_Quiz();
		}

		$this->view->quiz                 = $quiz;
		$this->view->prerequisiteQuizList = $pre_requisite_quiz_list;
		$this->view->quizList             = $m->fetchAllAsArray( array( 'id', 'name' ) );
		$this->view->captchaIsInstalled   = class_exists( 'ReallySimpleCaptcha' );
		$this->view->forms                = $forms;
		$this->view->templates            = $template_mapper->fetchAll( WpProQuiz_Model_Template::TEMPLATE_TYPE_QUIZ, false );
		$this->view->show();
	}

	/**
	 * Save quiz template.
	 *
	 * @return WpProQuiz_Model_Template
	 */
	private function saveTemplate() {
		$template_mapper = new WpProQuiz_Model_TemplateMapper();

		if ( isset( $this->_post['resultGradeEnabled'] ) ) {
			$this->_post['result_text'] = $this->filterResultTextGrade( $this->_post );
		}

		$quiz = new WpProQuiz_Model_Quiz( $this->_post );

		if ( $quiz->isPrerequisite() && ! empty( $this->_post['prerequisiteList'] ) && ! $quiz->isStatisticsOn() ) {
			$quiz->setStatisticsOn( true );
			$quiz->setStatisticsIpLock( 1440 );
		}

		$form = $this->_post['form'];

		unset( $form[0] );

		$forms = array();

		foreach ( $form as $f ) {
			$f['fieldname'] = trim( $f['fieldname'] );

			if ( empty( $f['fieldname'] ) ) {
				continue;
			}

			if ( (int) $f['form_id'] && (int) $f['form_delete'] ) {
				continue;
			}

			if ( WpProQuiz_Model_Form::FORM_TYPE_SELECT === (int) $f['type'] || WpProQuiz_Model_Form::FORM_TYPE_RADIO === (int) $f['type'] ) {
				if ( ! empty( $f['data'] ) ) {
					$items     = explode( "\n", $f['data'] );
					$f['data'] = array();

					foreach ( $items as $item ) {
						$item = trim( $item );

						if ( ! empty( $item ) ) {
							$f['data'][] = $item;
						}
					}
				}
			}

			if ( empty( $f['data'] ) || ! is_array( $f['data'] ) ) {
				$f['data'] = null;
			}

			$forms[] = new WpProQuiz_Model_Form( $f );
		}

		WpProQuiz_View_View::admin_notices( __( 'Template stored', 'buddyboss-app' ), 'info' );

		$data = array(
			'quiz'                 => $quiz,
			'forms'                => $forms,
			'prerequisiteQuizList' => isset( $this->_post['prerequisiteList'] ) ? $this->_post['prerequisiteList'] : array(),
		);

		$template = new WpProQuiz_Model_Template();

		if ( '0' === (string) $this->_post['templateSaveList'] ) {
			$template->setName( trim( $this->_post['templateName'] ) );
		} else {
			$template = $template_mapper->fetchById( $this->_post['templateSaveList'], false );
		}

		$template->setType( WpProQuiz_Model_Template::TEMPLATE_TYPE_QUIZ );
		$template->setData( $data );

		$template_mapper->save( $template );

		return $template;
	}

	/**
	 * Quiz form handler.
	 *
	 * @param int   $quiz_id Quiz id.
	 * @param array $post    Quiz posst data.
	 *
	 * @return bool
	 */
	private function formHandler( $quiz_id, $post ) {
		if ( ! isset( $post['form'] ) ) {
			return false;
		}

		$form = $post['form'];

		unset( $form[0] );

		if ( empty( $form ) ) {
			return false;
		}

		$form_mapper = new WpProQuiz_Model_FormMapper();
		$delete_ids  = array();
		$forms       = array();
		$sort        = 0;

		foreach ( $form as $f ) {
			if ( ( ! isset( $f['fieldname'] ) ) || ( empty( $f['fieldname'] ) ) ) {
				continue;
			}

			$f['fieldname'] = trim( $f['fieldname'] );

			if ( (int) $f['form_id'] && (int) $f['form_delete'] ) {
				$delete_ids[] = (int) $f['form_id'];
				continue;
			}

			$f['sort']   = $sort ++;
			$f['quizId'] = $quiz_id;

			if ( WpProQuiz_Model_Form::FORM_TYPE_SELECT === (int) $f['type'] || WpProQuiz_Model_Form::FORM_TYPE_RADIO === (int) $f['type'] ) {
				if ( ! empty( $f['data'] ) ) {
					$items     = explode( "\n", $f['data'] );
					$f['data'] = array();

					foreach ( $items as $item ) {
						$item = trim( $item );

						if ( ! empty( $item ) ) {
							$f['data'][] = $item;
						}
					}
				}
			}

			if ( empty( $f['data'] ) || ! is_array( $f['data'] ) ) {
				$f['data'] = null;
			}

			$forms[] = new WpProQuiz_Model_Form( $f );
		}

		if ( ! empty( $delete_ids ) ) {
			$form_mapper->deleteForm( $delete_ids, $quiz_id );
		}

		$form_mapper->update( $forms );

		return ! empty( $forms );
	}

	/**
	 * Delete.
	 *
	 * @param int $id Quiz id.
	 */
	private function deleteAction( $id ) {
		if ( ! current_user_can( 'wpProQuiz_delete_quiz' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'buddyboss-app' ) );
		}

		$m = new WpProQuiz_Model_QuizMapper();

		$m->deleteAll( $id );

		/* translators: %s: Quiz label. */
		WpProQuiz_View_View::admin_notices( sprintf( esc_html_x( '%s deleted', 'Quiz deleted', 'buddyboss-app' ), LearnDash_Custom_Label::get_label( 'quiz' ) ), 'info' );

		$this->showAction();
	}

	/**
	 * Check quiz validity.
	 *
	 * @param array $post Quiz data.
	 *
	 * @return bool
	 */
	private function checkValidit( $post ) {
		return ( isset( $post['name'] ) && ! empty( $post['name'] ) && isset( $post['text'] ) && ! empty( $post['text'] ) );
	}

	/**
	 * Filter result text grade.
	 *
	 * @param array $post Quiz data.
	 *
	 * @return array
	 */
	private function filterResultTextGrade( $post ) {
		$activ  = array_keys( $post['resultTextGrade']['activ'], '1', true );
		$result = array();

		foreach ( $activ as $k ) {
			$result['text'][]    = $post['resultTextGrade']['text'][ $k ];
			$result['prozent'][] = (float) str_replace( ',', '.', $post['resultTextGrade']['prozent'][ $k ] );
		}

		return $result;
	}

	/**
	 * Set result cookie data.
	 *
	 * @param WpProQuiz_Model_Quiz $quiz Quiz data.
	 */
	private function setResultCookie( WpProQuiz_Model_Quiz $quiz ) {
		$prerequisite = new WpProQuiz_Model_PrerequisiteMapper();

		if ( 0 === get_current_user_id() && $prerequisite->isQuizId( $quiz->getId() ) ) {
			$cookie_data = array();

			if ( isset( $this->_cookie['wpProQuiz_result'] ) ) {
				$d = json_decode( $this->_cookie['wpProQuiz_result'], true );

				if ( null !== $d && is_array( $d ) ) {
					$cookie_data = $d;
				}
			}

			$cookie_data[ $quiz->getId() ] = 1;
			$url                           = wp_parse_url( get_bloginfo( 'url' ) );

			setcookie( 'wpProQuiz_result', wp_json_encode( $cookie_data ), time() + 60 * 60 * 24 * 300, empty( $url['path'] ) ? '/' : $url['path'] );
		}
	}

	/**
	 * Quiz complete.
	 *
	 * @return bool
	 */
	public function completedQuiz() {
		$lock_mapper     = new WpProQuiz_Model_LockMapper();
		$quiz_mapper     = new WpProQuiz_Model_QuizMapper();
		$category_mapper = new WpProQuiz_Model_CategoryMapper();
		$is_100_p        = 100 === (int) $this->_post['results']['comp']['result'];
		$user_id         = get_current_user_id();
		$quiz            = $quiz_mapper->fetch( $this->_post['quizId'] );

		if ( null === $quiz || $quiz->getId() <= 0 ) {
			exit;
		}

		$categories = $category_mapper->fetchByQuiz( $quiz->getId() );

		$this->setResultCookie( $quiz );
		$this->emailNote( $quiz, $this->_post['results']['comp'], $categories );

		if ( ! $this->isPreLockQuiz( $quiz ) ) {
			$statistics              = new WpProQuiz_Controller_Statistics();
			$statistic_ref_mapper_id = $statistics->save( $quiz );

			/**
			 * Hooks on quiz completion.
			 *
			 * @param bool $statistic_ref_mapper_id Quiz statistic mapper id.
			 */
			do_action( 'wp_pro_quiz_completed_quiz', $statistic_ref_mapper_id );

			if ( $is_100_p ) {
				/**
				 * Hooks on 100% of quiz completion.
				 */
				do_action( 'wp_pro_quiz_completed_quiz_100_percent' );
			}

			return true;
		}

		$lock_mapper->deleteOldLock( 60 * 60 * 24 * 7, $this->_post['quizId'], time(), WpProQuiz_Model_Lock::TYPE_QUIZ, 0 );

		$lock_ip     = $lock_mapper->isLock( $this->_post['quizId'], $this->getIp(), get_current_user_id(), WpProQuiz_Model_Lock::TYPE_QUIZ );
		$lock_cookie = false;
		$cookie_time = $quiz->getQuizRunOnceTime();
		$cookie_json = null;

		if ( isset( $this->_cookie['wpProQuiz_lock'] ) && 0 === get_current_user_id() && $quiz->isQuizRunOnceCookie() ) {
			$cookie_json = json_decode( $this->_cookie['wpProQuiz_lock'], true );

			if ( false !== $cookie_json ) {
				if ( isset( $cookie_json[ $this->_post['quizId'] ] ) && (int) $cookie_json[ $this->_post['quizId'] ] === $cookie_time ) {
					$lock_cookie = true;
				}
			}
		}

		if ( ! $lock_ip && ! $lock_cookie ) {
			$statistics              = new WpProQuiz_Controller_Statistics();
			$statistic_ref_mapper_id = $statistics->save( $quiz );

			/**
			 * Hooks on quiz completion.
			 *
			 * @param bool $statistic_ref_mapper_id Quiz statistic mapper id.
			 */
			do_action( 'wp_pro_quiz_completed_quiz', $statistic_ref_mapper_id );

			if ( $is_100_p ) {
				/**
				 * Hooks on 100% of quiz completion.
				 */
				do_action( 'wp_pro_quiz_completed_quiz_100_percent' );
			}

			if ( 0 === get_current_user_id() && $quiz->isQuizRunOnceCookie() ) {
				$cookie_data = array();

				if ( null !== $cookie_json || false !== $cookie_json ) {
					$cookie_data = $cookie_json;
				}

				$cookie_data[ $this->_post['quizId'] ] = $quiz->getQuizRunOnceTime();
				$url                                   = wp_parse_url( get_bloginfo( 'url' ) );

				setcookie( 'wpProQuiz_lock', wp_json_encode( $cookie_data ), time() + 60 * 60 * 24 * 60, empty( $url['path'] ) ? '/' : $url['path'] );
			}

			if ( empty( $user_id ) ) {
				if ( ! $lock_mapper->isLock( $this->_post['quizId'], $this->getIp(), $user_id, WpProQuiz_Model_Lock::TYPE_QUIZ ) ) {
					$lock = new WpProQuiz_Model_Lock();

					$lock->setUserId( get_current_user_id() );
					$lock->setQuizId( $this->_post['quizId'] );
					$lock->setLockDate( time() );
					$lock->setLockIp( $this->getIp() );
					$lock->setLockType( WpProQuiz_Model_Lock::TYPE_QUIZ );
					$lock_mapper->insert( $lock );
				}
			}
		}

		/**
		 * Core File Updated: Remove Exit from original funcation and return true.
		 */
		return true;
	}

	/**
	 * Quiz is pre lock.
	 *
	 * @param WpProQuiz_Model_Quiz $quiz Quiz data.
	 *
	 * @return bool
	 */
	public function isPreLockQuiz( WpProQuiz_Model_Quiz $quiz ) {
		$user_id = get_current_user_id();

		if ( $quiz->isQuizRunOnce() ) {
			switch ( $quiz->getQuizRunOnceType() ) {
				case WpProQuiz_Model_Quiz::QUIZ_RUN_ONCE_TYPE_ALL:
					return true;
				case WpProQuiz_Model_Quiz::QUIZ_RUN_ONCE_TYPE_ONLY_USER:
					return $user_id > 0;
				case WpProQuiz_Model_Quiz::QUIZ_RUN_ONCE_TYPE_ONLY_ANONYM:
					return 0 === $user_id;
			}
		}

		return false;
	}

	/**
	 * Get IP.
	 *
	 * @return mixed|string
	 */
	private function getIp() {
		if ( get_current_user_id() > 0 ) {
			return '0';
		} else {
			return isset( $_SERVER['REMOTE_ADDR'] ) ? filter_var( wp_unslash( $_SERVER['REMOTE_ADDR'] ), FILTER_VALIDATE_IP ) : 0;
		}
	}

	/**
	 * Email note.
	 *
	 * @param WpProQuiz_Model_Quiz $quiz       QUiz data.
	 * @param array                $result     Quiz result.
	 * @param array                $categories Categories.
	 */
	private function emailNote( WpProQuiz_Model_Quiz $quiz, $result, $categories ) {
		$global_mapper = new WpProQuiz_Model_GlobalSettingsMapper();
		$admin_email   = $global_mapper->getEmailSettings();
		$user_email    = $global_mapper->getUserEmailSettings();
		$user          = wp_get_current_user();
		$r             = array(
			'$userId'     => $user->ID,
			'$username'   => $user->display_name,
			'$quizname'   => $quiz->getName(),
			'$result'     => $result['result'] . '%',
			'$points'     => $result['points'],
			'$ip'         => isset( $_SERVER['REMOTE_ADDR'] ) ? filter_var( wp_unslash( $_SERVER['REMOTE_ADDR'] ), FILTER_VALIDATE_IP ) : 0,
			'$categories' => empty( $result['cats'] ) ? '' : $this->setCategoryOverview( $result['cats'], $categories ),
		);

		if ( 0 === (int) $user->ID ) {
			$r['$username'] = $r['$ip'];
		}

		if ( $quiz->isUserEmailNotification() ) {
			$msg     = str_replace( array_keys( $r ), $r, $user_email['message'] );
			$msg     = apply_filters( 'learndash_quiz_email_note_user_message', $msg, $r, $quiz, $result, $categories );
			$headers = '';

			if ( ( isset( $user_email['from'] ) ) && ( ! empty( $user_email['from'] ) ) && ( is_email( $user_email['from'] ) ) ) {
				if ( empty( $user_email['from_name'] ) ) {
					$user_email['from_name'] = '';
					$email_user              = get_user_by( 'emal', $user_email['from'] );

					if ( $email_user instanceof WP_User ) {
						$user_email['from_name'] = $email_user->display_name;
					}
				}

				$headers .= 'From: ';

				if ( ( isset( $user_email['from_name'] ) ) && ( ! empty( $user_email['from_name'] ) ) ) {
					$headers .= $user_email['from_name'] . ' <' . $user_email['from'] . '>';
				} else {
					$headers .= $user_email['from'];
				}
			}

			if ( $user_email['html'] ) {
				add_filter( 'wp_mail_content_type', array( $this, 'htmlEmailContent' ) );
			}

			$email_params = array(
				'email'   => $user->user_email,
				'subject' => $user_email['subject'],
				'msg'     => $msg,
				'headers' => $headers,
			);
			$email_params = apply_filters( 'learndash_quiz_email', $email_params, $quiz );

			wp_mail( $email_params['email'], $email_params['subject'], $email_params['msg'], $email_params['headers'] );

			if ( $user_email['html'] ) {
				remove_filter( 'wp_mail_content_type', array( $this, 'htmlEmailContent' ) );
			}
		}

		if ( $quiz->getEmailNotification() === WpProQuiz_Model_Quiz::QUIZ_EMAIL_NOTE_ALL || ( get_current_user_id() > 0 && $quiz->getEmailNotification() === WpProQuiz_Model_Quiz::QUIZ_EMAIL_NOTE_REG_USER ) ) {
			$msg     = str_replace( array_keys( $r ), $r, $admin_email['message'] );
			$msg     = apply_filters( 'learndash_quiz_email_note_admin_message', $msg, $r, $quiz, $result, $categories );
			$headers = '';

			if ( empty( $admin_email['from'] ) || ( ! is_email( $admin_email['from'] ) ) ) {
				$admin_email['from'] = get_option( 'admin_email' );
			}

			if ( empty( $admin_email['from_name'] ) ) {
				$admin_email['from_name'] = '';

				if ( ! empty( $admin_email['from'] ) ) {
					$email_user = get_user_by( 'emal', $admin_email['from'] );

					if ( $email_user instanceof WP_User ) {
						$admin_email['from_name'] = $email_user->display_name;
					}
				}
			}

			if ( ! empty( $admin_email['from'] ) ) {
				$headers .= 'From: ';

				if ( ( isset( $admin_email['from_name'] ) ) && ( ! empty( $admin_email['from_name'] ) ) ) {
					$headers .= $admin_email['from_name'] . ' <' . $admin_email['from'] . '>';
				} else {
					$headers .= $admin_email['from'];
				}
			}

			if ( isset( $admin_email['html'] ) && $admin_email['html'] ) {
				add_filter( 'wp_mail_content_type', array( $this, 'htmlEmailContent' ) );
			}

			$email_params = array(
				'email'   => $admin_email['to'],
				'subject' => $admin_email['subject'],
				'msg'     => $msg,
				'headers' => $headers,
			);
			$email_params = apply_filters( 'learndash_quiz_email_admin', $email_params, $quiz );

			wp_mail( $email_params['email'], $email_params['subject'], $email_params['msg'], $email_params['headers'] );

			if ( isset( $admin_email['html'] ) && $admin_email['html'] ) {
				remove_filter( 'wp_mail_content_type', array( $this, 'htmlEmailContent' ) );
			}
		}
	}

	/**
	 * Html email content type.
	 *
	 * @param string $content_type Email content type.
	 *
	 * @return string
	 */
	public function htmlEmailContent( $content_type ) {
		return 'text/html';
	}

	/**
	 * Set category overview.
	 *
	 * @param array $cat_array  Categories array.
	 * @param array $categories Categories array.
	 *
	 * @return string
	 */
	private function setCategoryOverview( $cat_array, $categories ) {
		$cats = array();

		foreach ( $categories as $cat ) {
			/**
			 * WpProQuiz_Model_Category class variable.
			 *
			 * @var $cat WpProQuiz_Model_Category
			 */

			if ( ! $cat->getCategoryId() ) {
				$cat->setCategoryName( __( 'Not categorized', 'buddyboss-app' ) );
			}

			$cats[ $cat->getCategoryId() ] = $cat->getCategoryName();
		}

		$a = esc_html__( 'Categories', 'buddyboss-app' ) . ":\n";

		foreach ( $cat_array as $id => $value ) {
			if ( ! isset( $cats[ $id ] ) ) {
				continue;
			}

			$a .= '* ' . str_pad( $cats[ $id ], 35, '.' ) . ( (float) $value ) . "%\n";
		}

		return $a;
	}
}
