<?php
/**
 * Holds quiz leaderboard functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Quiz
 */

namespace BuddyBossApp\Api\LearnDash\V1\Quiz;

use WP_Post;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use WpProQuiz_Helper_Until;
use WpProQuiz_Model_Quiz;
use WpProQuiz_Model_QuizMapper;
use WpProQuiz_Model_Toplist;
use WpProQuiz_Model_ToplistMapper;

/**
 * Leaderboard class.
 */
class LeaderboardRest extends QuizRest {

	/**
	 * Class isntance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * LeaderboardRest instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'quiz';
		parent::__construct();
	}

	/**
	 * Check if a given request has access to Quiz Leaderboard items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the Quiz Leaderboard `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_leaderboard_items_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to Quiz Leaderboard item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the Quiz Leaderboard `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_leaderboard_item_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/leaderboard',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/leaderboard',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'save_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}


	/**
	 * Retrieve quiz Leaderboard items.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/quiz/:id/leaderboard Get LearnDash Quiz Leaderboard items
	 * @apiName        GetLDQuizLeaderboard
	 * @apiGroup       LD Quiz Leaderboard
	 * @apiDescription Retrieve quiz Leaderboard items
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function get_items( $request ) {
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$quiz    = get_post( $quiz_id );

		if ( empty( $quiz ) || $this->post_type !== $quiz->post_type ) {
			return QuizError::instance()->invalid_quiz_id();
		}

		$registered = $this->get_collection_params();

		/**
		 * Filter the the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$request = apply_filters( 'bbapp_ld_get_leaderboard_items_request', $request );

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'page'     => 'paged',
			'per_page' => 'per_page',
		);

		/**
		 * For each known parameter which is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		$args              = $this->prepare_items_query( $args, $request );
		$leaderboard_items = $this->get_leaderboard( $args, $quiz_id );
		$retval            = array();

		foreach ( $leaderboard_items['posts'] as $item ) {
			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $item, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		$response = bbapp_learners_response_add_total_headers( $response, $leaderboard_items['total_posts'], $args['per_page'] );

		/**
		 * Fires after a list of Leaderboard items response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_leaderboard_items_response', $response, $request );

		return $response;
	}

	/**
	 * Save leaderboard items.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return boolean
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/quiz/:id/leaderboard Save LearnDash Quiz Leaderboard item
	 * @apiName        SaveLDQuizLeaderboard
	 * @apiGroup       LD Quiz Leaderboard
	 * @apiDescription Save leaderboard item
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function save_item( $request ) {
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$quiz    = get_post( $quiz_id );

		if ( empty( $quiz ) || $this->post_type !== $quiz->post_type ) {
			return QuizError::instance()->invalid_quiz_id();
		}

		$quiz_settings = learndash_get_setting( $quiz_id );
		$quiz_pro      = $quiz_settings['quiz_pro'];
		$quiz_mapper   = new WpProQuiz_Model_QuizMapper();
		$quiz          = $quiz_mapper->fetch( $quiz_pro );
		$point         = isset( $request['point'] ) ? (int) $request['point'] : false;
		$result        = isset( $request['result'] ) ? $request['result'] : false;

		if ( false === $point ) {
			QuizError::instance()->quiz_leaderboard_point_empty();
		}

		if ( false === $result ) {
			QuizError::instance()->quiz_leaderboard_result_empty();
		}

		return $this->createitem(
			$quiz,
			array(
				'point'  => $point,
				'result' => $result,
				'ip'     => '',
			)
		);
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		// Base fields for every post.
		$data = array(
			'id'     => $post->ID,
			'name'   => $post->name,
			'date'   => mysql_to_rfc3339( $post->date ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'result' => $post->result,
			'point'  => (int) $post->point,
		);
		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );
		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filter LearnDash certificate response.
		 *
		 * @param WP_REST_Response $response API response.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 */
		return apply_filters( 'bbapp_ld_rest_prepare_certificate', $response, $post, $request );
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array $data item object.
	 *
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {
		$links = parent::prepare_links( $data );

		// Todo::dips add filter here for course/lesson/topic.

		return $links;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'ld_leaderboard',
			'type'       => 'object',
			'properties' => array(
				'id'   => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'name' => array(
					'description' => __( 'The Name of user.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date' => array(
					'description' => __( "The date the object was created, in the site's timezone.", 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
			),
		);

		$schema['properties']['point'] = array(
			'description' => __( 'Point for object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['result']        = array(
			'description' => __( 'Result for object.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);
		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get Leaderboard data
	 *
	 * @param array $args    Leaderboard get arguments.
	 * @param int   $quiz_id Quiz id.
	 *
	 * @return array
	 */
	private function get_leaderboard( $args, $quiz_id ) {
		$results         = array();
		$top_list_mapper = new WpProQuiz_Model_ToplistMapper();
		$quiz_settings   = learndash_get_setting( $quiz_id );
		$quiz_pro        = $quiz_settings['quiz_pro'];
		$quiz_mapper     = new WpProQuiz_Model_QuizMapper();
		$quiz            = $quiz_mapper->fetch( $quiz_pro );
		$total_results   = $top_list_mapper->count( $quiz_pro );

		if ( ! isset( $args['per_page'] ) ) {
			$args['per_page'] = $quiz->getToplistDataShowLimit();
		}

		$per_page = $args['per_page'];
		$offset   = $args['per_page'] * ( $args['paged'] - 1 );
		$toplist  = $top_list_mapper->fetch( $quiz_pro, $per_page, $quiz->getToplistDataSort(), $offset );

		foreach ( $toplist as $tp ) {
			$results[] = (object) array(
				'ID'     => $tp->getToplistId(),
				'name'   => $tp->getName(),
				'date'   => WpProQuiz_Helper_Until::convertTime( $tp->getDate(), get_option( 'wpProQuiz_toplistDataFormat', 'Y/m/d g:i A' ) ),
				'point'  => $tp->getPoints(),
				'result' => $tp->getResult(),
			);
		}

		return array(
			'posts'       => $results,
			'total_posts' => $total_results,
		);
	}

	/**
	 * Create leaderboard item
	 *
	 * @param object $quiz Quiz data.
	 * @param array  $data Quiz create data.
	 *
	 * @return bool
	 */
	public function createitem( $quiz, $data ) {
		$user_id         = get_current_user_id();
		$user            = wp_get_current_user();
		$email           = $user->user_email;
		$name            = $user->display_name;
		$ip              = $data['ip'];
		$point           = $data['point'];
		$result          = $data['result'];
		$top_list_mapper = new WpProQuiz_Model_ToplistMapper();

		if ( null === $quiz || 0 === (int) $quiz->getId() || ! $quiz->isToplistActivated() ) {
			QuizError::instance()->quiz_leaderboard_not_active();
		}

		if ( ! $this->preCheck( $quiz->getToplistDataAddPermissions(), $user_id ) ) {
			QuizError::instance()->quiz_leaderboard_permission_invalid();
		}

		$clear_time = null;

		if ( $quiz->isToplistDataAddMultiple() ) {
			$clear_time = $quiz->getToplistDataAddBlock() * 60;
		}

		if ( $top_list_mapper->countUser( $quiz->getId(), $user_id, $clear_time ) ) {
			QuizError::instance()->quiz_leaderboard_entry_exist();
		}

		$toplist = new WpProQuiz_Model_Toplist();
		$toplist->setQuizId( $quiz->getId() )->setUserId( $user_id )->setDate( time() )->setName( $name )->setEmail( $email )->setPoints( $point )->setResult( round( $result, 2 ) )->setIp( $ip );
		$top_list_mapper->save( $toplist );

		return true;
	}

	/**
	 * Pre check.
	 *
	 * @param string $type    Type.
	 * @param int    $user_id User id.
	 *
	 * @return bool
	 */
	private function preCheck( $type, $user_id ) {
		switch ( $type ) {
			case WpProQuiz_Model_Quiz::QUIZ_TOPLIST_TYPE_ALL:
				return true;
			case WpProQuiz_Model_Quiz::QUIZ_TOPLIST_TYPE_ONLY_ANONYM:
				return 0 === $user_id;
			case WpProQuiz_Model_Quiz::QUIZ_TOPLIST_TYPE_ONLY_USER:
				return $user_id > 0;
		}

		return false;
	}
}
