<?php
/**
 * Holds lesson rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Lesson
 */

namespace BuddyBossApp\Api\LearnDash\V1\Lesson;

use BuddyBossApp\Api\LearnDash\V1\Core\LDRestController;
use BuddyBossApp\Api\LearnDash\V1\Course\CoursesError;
use LearnDash_Settings_Section;
use WP_Error;
use WP_Post;
use WP_Query;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Lesson rest class.
 */
class LessonsRest extends LDRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Post type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'sfwd-lessons';

	/**
	 * LessonsRest instance.
	 *
	 * @since 0.1.0
	 * @return LessonsRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'lessons';

		parent::__construct();
	}

	/**
	 * Check if a given request has access to lesson items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the lesson `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_lessons_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to lesson item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the lesson `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_lesson_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}


	/**
	 * Retrieve Lessons.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/lessons Get LearnDash Lessons
	 * @apiName        GetLDLessons
	 * @apiGroup       LD Lessons
	 * @apiDescription Retrieve Lessons
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure result set includes specific IDs.
	 * @apiParam {String} [after]  Limit results to those published after a given ISO8601 compliant date.
	 * @apiParam {String} [before] Limit results to those published before a given ISO8601 compliant date.
	 * @apiParam {Array} [author] Limit results to those assigned to specific authors.
	 * @apiParam {Array} [author_exclude] Ensure results to excludes those assigned to specific authors.
	 * @apiParam {String=asc,desc} [order=desc] Sort result set by given order.
	 * @apiParam {String=date,id,title,menu_order} [orderby=date] Sort result set by given field.
	 * @apiParam {Array} [parent] Limit results to those assigned to specific parent.
	 */
	public function get_items( $request ) {
		$user_id    = get_current_user_id();
		$registered = $this->get_collection_params();

		/**
		 * Filter the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$request = apply_filters( 'bbapp_ld_get_lessons_request', $request );

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'author'         => 'author__in',
			'author_exclude' => 'author__not_in',
			'exclude'        => 'post__not_in',
			'include'        => 'post__in',
			'offset'         => 'offset',
			'order'          => 'order',
			'orderby'        => 'orderby',
			'page'           => 'paged',
			'parent'         => 'post_parent__in',
			'parent_exclude' => 'post_parent__not_in',
			'search'         => 's',
			'slug'           => 'post_name__in',
			'status'         => 'post_status',
			'per_page'       => 'posts_per_page',
		);

		/**
		 * For each known parameter which is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		// Check for & assign any parameters which require special handling or setting.
		$args['date_query'] = array();

		// Set before into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['before'], $request['before'] ) ) {
			$args['date_query'][0]['before'] = $request['before'];
		}

		// Set after into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['after'], $request['after'] ) ) {
			$args['date_query'][0]['after'] = $request['after'];
		}

		$args = $this->prepare_items_query( $args, $request );

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$args = apply_filters( 'bbapp_ld_get_lessons_args', $args, $request );

		$args['post_type'] = $this->post_type;

		/**
		 * If User pass course id then return Lesson list of that course.
		 * else return lessons if user has access of the lesson course
		 */
		if ( isset( $request['course_id'] ) ) {
			$course_id = isset( $request['course_id'] ) ? (int) $request['course_id'] : false;

			/**
			 * Support lesson/topics order setting of course & Default setting
			 */
			// We are not supporting custom pagination option.
			$default_args = sfwd_lms_get_post_options( $this->post_type );

			if ( isset( $default_args['lesson_topic_order_enabled'] ) && 'yes' === $default_args['lesson_topic_order_enabled'] ) {
				$args['order']   = ! empty( $default_args['lesson_topic_order'] ) ? $default_args['lesson_topic_order'] : $args['order'];
				$args['orderby'] = ! empty( $default_args['lesson_topic_orderby'] ) ? $default_args['lesson_topic_orderby'] : $args['orderby'];
			}

			$is_custom_order = bbapp_learndash_get_course_meta_setting( $course_id, 'course_lesson_order_enabled' );

			if ( 'on' === $is_custom_order ) {
				$order           = bbapp_learndash_get_course_meta_setting( $course_id, 'course_lesson_order' );
				$orderby         = bbapp_learndash_get_course_meta_setting( $course_id, 'course_lesson_orderby' );
				$args['order']   = ! empty( $order ) ? $order : $args['order'];
				$args['orderby'] = ! empty( $orderby ) ? $orderby : $args['orderby'];
			}

			$_course_ids = array( $course_id );
		} else {
			$_course_ids = ld_get_mycourses( $user_id, array() );
		}

		global $wpdb;

		if ( ! empty( $_course_ids ) ) {
			// If Course builder enabled then fetch lesson by that otherwise fetch by meta.
			if ( 'yes' === LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'enabled' ) ) {

				/**
				 * Filter lesson ids by course which users has access
				 * Instead of using core function to fetch lesson of course one by one we use custom query to improve performance.
				 */
				$_course_str = implode( ',', $_course_ids );
				$lesson_ids  = $wpdb->get_col( "SELECT `post_id` FROM $wpdb->postmeta WHERE meta_value IN ($_course_str) AND (`meta_key` LIKE '%ld_course_%' OR  `meta_key`LIKE 'course_id' ) ORDER BY `meta_id` ASC" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

				if ( ! empty( $lesson_ids ) ) {
					if ( ! empty( $args['post__in'] ) ) {
						if ( ! is_array( $args['post__in'] ) ) {
							$args['post__in'] = array( $args['post__in'] );
						}
						$lesson_ids = array_intersect( $args['post__in'], $lesson_ids );
						if ( empty( $lesson_ids ) ) {
							$lesson_ids = array( - 1 );
						}
					}
					$args['post__in'] = $lesson_ids;

					/**
					 * Set order if we requesting more then one course lessons
					 * For single course lesson request we use Course setting order
					 */
					if ( ! isset( $request['course_id'] ) ) {
						$args['orderby'] = ( isset( $args['s'] ) ) ? 'relevance' : 'post__in';
						unset( $args['order'] );
					}
				} else {
					$args['post__in'] = array( - 1 );
				}
			} else {
				/**
				 * Fetch lesson with old structure
				 */
				$args['meta_query'][] = array(
					'key'     => 'course_id',
					'value'   => $_course_ids,
					'compare' => 'IN',
				);
			}
		}

		/**
		 * Taxonomy Filter query
		 */
		$taxonomies = wp_list_filter( get_object_taxonomies( $this->post_type, 'objects' ), array( 'show_in_rest' => true ) );

		foreach ( $taxonomies as $taxonomy ) {
			$base = ! empty( $taxonomy->rest_base ) ? $taxonomy->rest_base : $taxonomy->name;

			if ( ! empty( $request[ $base ] ) ) {
				$args['tax_query'][] = array(
					'taxonomy'         => $taxonomy->name,
					'field'            => 'term_id',
					'terms'            => $request[ $base ],
					'include_children' => false,
				);
			}
		}

		$posts_query            = new WP_Query();
		$leesons['posts']       = $posts_query->query( $args );
		$leesons['total_posts'] = $posts_query->found_posts;

		/**
		 * Fires list of Lesson is fetched via Query.
		 *
		 * @param array            $leesons Fetched lessons.
		 * @param WP_REST_Response $args    Query arguments.
		 * @param WP_REST_Request  $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$leesons = apply_filters( 'bbapp_ld_get_lessons', $leesons, $args, $request );

		$retval = array();

		foreach ( $leesons['posts'] as $couese ) {
			if ( ! $this->check_read_permission( $couese ) ) {
				continue;
			}

			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $couese, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		$response = bbapp_learners_response_add_total_headers( $response, $leesons['total_posts'], $args['posts_per_page'] );

		/**
		 * Fires after a list of lessons response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_lessons_items_response', $response, $request );

		return $response;
	}

	/**
	 * Retrieve Lesson.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/lessons/:id Get LearnDash Lesson
	 * @apiName        GetLDLesson
	 * @apiGroup       LD Lessons
	 * @apiDescription Retrieve single Lesson
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Lesson.
	 */
	public function get_item( $request ) {
		$lesson_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$lesson    = get_post( $lesson_id );

		if ( empty( $lesson ) || $this->post_type !== $lesson->post_type ) {
			return LessonsError::instance()->invalid_lesson_id();
		}

		/**
		 * Fire after Lesson is fetched via Query.
		 *
		 * @param array           $lesson    Fetched lesson.
		 * @param WP_REST_Request $lesson_id lesson id.
		 *
		 * @since 0.1.0
		 */
		$lesson = apply_filters( 'bbapp_ld_get_lesson', $lesson, $lesson_id );

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $lesson, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after an lesson response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_lesson_item_response', $response, $request );

		return $response;
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$GLOBALS['post'] = $post; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited

		setup_postdata( $post );

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$schema  = $this->get_public_item_schema();

		$post->has_content_access = $this->get_has_content_access( $post );

		// When we start course in the app then course and lesson is not inserted in DB.
		// At that time we will not get the last activity time because both the activity start date and activity update date will be empty.
		// In the web, when we start the course, at that time both course and lesson are inserted in DB with activity start date and activity updated date.
		if ( is_user_logged_in() && $post->has_content_access ) {
			$lesson_id = isset( $request['id'] ) && is_numeric( $request['id'] ) ? (int) $request['id'] : 0;
			if ( ! empty( $lesson_id ) ) {
				$this->bbapp_ld_start_cource_and_lesson( $lesson_id );
			}
		}

		// Base fields for every post.
		$data = array(
			'id'           => $post->ID,
			'title'        => array(
				'raw'      => $post->post_title,
				'rendered' => get_the_title( $post->ID ),
			),
			'content'      => array(
				'raw'      => ( $post->has_content_access ) ? bbapp_learners_fix_relative_urls_protocol( $post->post_content ) : '',
				'rendered' => ( $post->has_content_access ) ? bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_content', $post->post_content ) ) : '',
			),
			'date'         => mysql_to_rfc3339( $post->post_date ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'date_gmt'     => mysql_to_rfc3339( $post->post_date_gmt ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'modified'     => mysql_to_rfc3339( $post->post_modified ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'modified_gmt' => mysql_to_rfc3339( $post->post_modified_gmt ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'link'         => get_permalink( $post->ID ),
			'slug'         => $post->post_name,
			'author'       => (int) $post->post_author,
			'excerpt'      => array(
				'raw'      => bbapp_learners_fix_relative_urls_protocol( $post->post_excerpt ),
				'rendered' => bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_excerpt', $post->post_excerpt ) ),
			),
			'menu_order'   => (int) $post->menu_order,
		);

		if ( ! empty( $schema['properties']['has_course_access'] ) && in_array( $context, $schema['properties']['has_course_access']['context'], true ) ) {
			$post->has_course_access   = $this->get_has_course_access( $post );
			$data['has_course_access'] = (bool) $post->has_course_access;
		}

		if ( ! empty( $schema['properties']['has_content_access'] ) && in_array( $context, $schema['properties']['has_content_access']['context'], true ) ) {
			$data['has_content_access'] = (bool) $post->has_content_access;
		}

		/**
		 * Feature Media
		 */
		$post->featured_media            = $this->get_feature_media( $post );
		$data['featured_media']          = array();
		$data['featured_media']['small'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['small'] ) ) ? $post->featured_media['small'] : null;
		$data['featured_media']['large'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['large'] ) ) ? $post->featured_media['large'] : null;

		if ( ! empty( $schema['properties']['course'] ) && in_array( $context, $schema['properties']['course']['context'], true ) ) {
			$post->course   = $this->get_course_id( $post );
			$data['course'] = (int) $post->course;
		}

		if ( ! empty( $schema['properties']['next_lesson'] ) && in_array( $context, $schema['properties']['next_lesson']['context'], true ) ) {
			$post->next_lesson   = $this->get_next_lesson_id( $post );
			$data['next_lesson'] = (int) $post->next_lesson;
		}

		if ( ! empty( $schema['properties']['duration'] ) && in_array( $context, $schema['properties']['duration']['context'], true ) ) {
			$post->duration                 = $this->get_duration( $post );
			$data['duration']               = array();
			$data['duration']['min']        = (int) ( is_array( $post->duration ) && isset( $post->duration['min'] ) ) ? $post->duration['min'] : 0;
			$data['duration']['cookie_key'] = function_exists( 'learndash_forced_lesson_time_cookie_key' ) ? 'learndash_timer_cookie_' . learndash_forced_lesson_time_cookie_key( $post->ID ) : '';
		}

		if ( ! empty( $schema['properties']['materials'] ) && in_array( $context, $schema['properties']['materials']['context'], true ) ) {
			$post->materials   = $this->get_materials( $post );
			$data['materials'] = ( $post->has_content_access ) ? $post->materials : '';
		}

		if ( ! empty( $schema['properties']['video'] ) && in_array( $context, $schema['properties']['video']['context'], true ) ) {
			$post->video   = $this->get_video( $post );
			$data['video'] = ( $post->has_content_access ) ? $post->video : '';
		}

		if ( ! empty( $schema['properties']['assignment_upload'] ) && in_array( $context, $schema['properties']['assignment_upload']['context'], true ) ) {
			$post->assignment_upload   = $this->get_lesson_assignment_upload( $post );
			$data['assignment_upload'] = $post->assignment_upload;
		}

		if ( ! empty( $schema['properties']['is_sample'] ) && in_array( $context, $schema['properties']['is_sample']['context'], true ) ) {
			$post->is_sample   = $this->get_is_sample( $post );
			$data['is_sample'] = (bool) $post->is_sample;
		}

		if ( ! empty( $schema['properties']['topics'] ) && in_array( $context, $schema['properties']['topics']['context'], true ) ) {
			$post->topics   = $this->get_topics( $post );
			$data['topics'] = $post->topics;
		}

		if ( ! empty( $schema['properties']['quizzes'] ) && in_array( $context, $schema['properties']['quizzes']['context'], true ) ) {
			$post->quizzes   = $this->get_quizzes( $post );
			$data['quizzes'] = $post->quizzes;
		}

		if ( ! empty( $schema['properties']['completed'] ) && in_array( $context, $schema['properties']['completed']['context'], true ) ) {
			$post->completed   = $this->is_completed( $post );
			$data['completed'] = (bool) $post->completed;
		}

		if ( ! empty( $schema['properties']['quiz_completed'] ) && in_array( $context, $schema['properties']['quiz_completed']['context'], true ) ) {
			$post->quiz_completed   = $this->is_quiz_completed( $post, learndash_get_course_id( $post->ID ) );
			$data['quiz_completed'] = (bool) $post->quiz_completed;
		}

		if ( ! empty( $schema['properties']['settings'] ) && in_array( $context, $schema['properties']['settings']['context'], true ) ) {
			$post->settings   = $this->get_settings( $post );
			$data['settings'] = $post->settings;
		}

		if ( ! empty( $schema['properties']['can_complete'] ) && in_array( $context, $schema['properties']['can_complete']['context'], true ) ) {
			$post->can_complete   = $this->get_can_complete( $post );
			$data['can_complete'] = $post->can_complete;
		}

		if ( ! empty( $schema['properties']['can_comment'] ) && in_array( $context, $schema['properties']['can_comment']['context'], true ) ) {
			$post->can_comment   = bbapp_learndash_comments_support_enabled( $post->post_type );
			$data['can_comment'] = $post->can_comment;
		}

		if ( ! empty( $schema['properties']['error_message'] ) && in_array( $context, $schema['properties']['error_message']['context'], true ) ) {
			$post->error_message = $this->get_error_message( $post );
			if ( ! empty( $post->error_message ) ) {
				$error_code            = $post->error_message->get_error_code();
				$data['error_message'] = array(
					'code'    => $error_code,
					'message' => $post->error_message->get_error_message(),
					'data'    => $post->error_message->get_error_data( $error_code ),

				);
			} else {
				$data['error_message'] = array();
			}
		}

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filters lesson response.
		 *
		 * @param WP_REST_Response $response Lessons response.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 */
		return apply_filters( 'bbapp_ld_rest_prepare_lesson', $response, $post, $request );
	}

	/**
	 * Prepare item query.
	 *
	 * @param array $prepared_args Item query arguments.
	 * @param null  $request Request parameters.
	 *
	 * @return array
	 */
	protected function prepare_items_query( $prepared_args = array(), $request = null ) {
		$query_args = array();

		foreach ( $prepared_args as $key => $value ) {
			/**
			 * Filters the query_vars used in get_items() for the constructed query.
			 *
			 * The dynamic portion of the hook name, `$key`, refers to the query_var key.
			 *
			 * @param string $value The query_var value.
			 *
			 * @since 4.7.0
			 */
			$query_args[ $key ] = apply_filters( "rest_query_var-{$key}", $value ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		}

		$query_args['ignore_sticky_posts'] = true;

		// Map to proper WP_Query orderby param.
		if ( isset( $query_args['orderby'] ) && isset( $request['orderby'] ) ) {
			$orderby_mappings = array(
				'id'            => 'ID',
				'include'       => 'post__in',
				'slug'          => 'post_name',
				'include_slugs' => 'post_name__in',
			);

			if ( isset( $orderby_mappings[ $request['orderby'] ] ) ) {
				$query_args['orderby'] = $orderby_mappings[ $request['orderby'] ];
			}
		}

		return $query_args;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array $data item object.
	 *
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {
		$links = parent::prepare_links( $data );

		/**
		 * Quiz links.
		 */
		if ( isset( $data['quizzes'] ) ) {
			$links['quizzes'] = array();
			foreach ( $data['quizzes'] as $quiz ) {
				$links['quizzes'][] = array(
					'href'       => rest_url( $this->namespace . '/quiz/' . $quiz ),
					'embeddable' => true,
				);
			}
		}

		/**
		 * Course.
		 */
		if ( isset( $data['course'] ) ) {
			$links['course'] = array(
				'href'       => rest_url( $this->namespace . '/courses/' . $data['course'] ),
				'embeddable' => false,
			);
		}

		/**
		 * Topic link.
		 */
		if ( isset( $data['topics'] ) ) {
			$links['topics'] = array();
			$links['topics'] = array(
				'href'       => rest_url( $this->namespace ) . '/topics?course_id=' . $data['course'] . '&lesson_id=' . $data['id'],
				'embeddable' => false,
			);
		}

		/**
		 * Next Lesson
		 */
		if ( isset( $data['next_lesson'] ) ) {
			$links['next_lesson'] = array(
				'href'       => rest_url( $this->namespace . '/lessons/' . $data['next_lesson'] ),
				'embeddable' => false,
			);
		}

		return $links;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		$params = parent::get_collection_params();

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author'] = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset']            = array(
			'description'       => __( 'Offset the result set by a specific number of items.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']['enum'][] = 'menu_order';

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent IDs.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent ID.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);

		$params['slug']   = array(
			'description'       => __( 'Limit result set to posts with a specific slug.', 'buddyboss-app' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['filter'] = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.', 'buddyboss-app' ),
		);

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'lesson',
			'type'       => 'object',
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'          => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content'        => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'date'           => array(
					'description' => __( 'The date the object was published, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date_gmt'       => array(
					'description' => __( 'The date the object was published, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'modified'       => array(
					'description' => __( 'The date the object was last modified, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'modified_gmt'   => array(
					'description' => __( 'The date the object was last modified, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'link'           => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'           => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'         => array(
					'description' => __( 'The ID for the author of the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'excerpt'        => array(
					'description' => __( 'The excerpt for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'featured_media' => array(
					'description' => __( 'Feature media object containing thumb and full URL of image.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'menu_order'     => array(
					'description' => __( 'The order of the object in relation to other object of its type.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
			),
		);

		$schema['properties']['module'] = array(
			'description' => __( 'The terms assigned to the object in the module taxonomy.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['complexity'] = array(
			'description' => __( 'Complexity of the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['course'] = array(
			'description' => __( 'The Course id for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['next_lesson'] = array(
			'description' => __( 'The next lesson id for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['materials'] = array(
			'description' => __( 'Materials for the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['video'] = array(
			'description' => __( 'video for the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['duration'] = array(
			'description' => __( 'The time duration for the object', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'view', 'edit' ),
			'properties'  => array(
				'properties' => array(
					'min'        => array(
						'description' => __( 'Minimum required time for this object.', 'buddyboss-app' ),
						'type'        => 'integer',
						'context'     => array( 'view', 'edit' ),
					),
					'cookie_key' => array(
						'description' => __( 'Timer cookie key that helps to force the timer.', 'buddyboss-app' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
				),
			),
		);

		$schema['properties']['assignment_upload'] = array(
			'description' => __( 'Assignment upload enable for the object', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['has_course_access'] = array(
			'description' => __( 'Whether or not user have the object access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['has_content_access'] = array(
			'description' => __( 'Whether or not user have the object content access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['is_sample'] = array(
			'description' => __( 'Whether or not object is sample.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['quizzes'] = array(
			'description' => __( 'The quizzes list which has access of the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['topics'] = array(
			'description' => __( 'The topics list which has access to the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['completed'] = array(
			'description' => __( 'The object is completed by current user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['quiz_completed'] = array(
			'description' => __( 'The object Quizzes is completed by current user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['settings'] = array(
			'description' => __( 'The object settings.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['can_complete'] = array(
			'description' => __( 'The object can complete by user or not.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['can_comment'] = array(
			'description' => __( 'The object can comment by user or not.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get lesson feature medias.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return array
	 */
	protected function get_feature_media( $post ) {
		$return = array(
			'large' => null,
			'small' => null,
		);
		$large  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'large' );
		$small  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'medium' );

		if ( isset( $large[0] ) ) {
			$return['large'] = $large[0];
		}
		if ( isset( $small[0] ) ) {
			$return['small'] = $small[0];
		}

		return $return;
	}

	/**
	 * Get lesson course id.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return int
	 */
	protected function get_course_id( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post->ID );

		if ( ! isset( $course_id ) ) {
			$course_id = 0;
		}

		return $course_id;
	}

	/**
	 * Get next lesson id.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return int|string
	 */
	protected function get_next_lesson_id( $post ) {
		$next = bbapp_lms_next_post_id( $post );

		if ( ! empty( $next ) ) {
			return $next;
		} else {
			return 0;
		}
	}

	/**
	 * Get lesson duration.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return array
	 */
	protected function get_duration( $post ) {
		$timeval = learndash_get_setting( $post->ID, 'forced_lesson_time' );

		if ( ! empty( $timeval ) ) {
			$time_sections = explode( ' ', $timeval );
			$h             = 0;
			$m             = 0;
			$s             = 0;

			foreach ( $time_sections as $k => $v ) {
				$value = trim( $v );

				if ( strpos( $value, 'h' ) ) {
					$h = intVal( $value );
				} elseif ( strpos( $value, 'm' ) ) {
					$m = intVal( $value );
				} elseif ( strpos( $value, 's' ) ) {
					$s = intVal( $value );
				}
			}

			$time = $h * 60 * 60 + $m * 60 + $s;

			if ( 0 === (int) $time ) {
				$time = (int) $timeval;
			}
		} else {
			$time = 0;
		}

		return array(
			'min' => $time,
		);
	}

	/**
	 * Get lesson assignments.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return bool
	 */
	protected function get_lesson_assignment_upload( $post ) {
		return function_exists( 'learndash_lesson_hasassignments' ) ? learndash_lesson_hasassignments( $post ) : lesson_hasassignments( $post );
	}

	/**
	 * Get lesson assignment upload settings.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return array
	 */
	protected function get_lesson_assignment_upload_settings( $post ) {
		$settings        = learndash_get_setting( $post->ID );
		$file_extensions = array_keys( learndash_get_allowed_upload_mime_extensions_for_post( $post->ID ) );

		$php_max_upload = ini_get( 'upload_max_filesize' );

		if ( ( isset( $settings['assignment_upload_limit_size'] ) ) && ( ! empty( $settings['assignment_upload_limit_size'] ) ) ) {
			if ( ( learndash_return_bytes_from_shorthand( $settings['assignment_upload_limit_size'] ) < learndash_return_bytes_from_shorthand( $php_max_upload ) ) ) {
				$php_max_upload = $settings['assignment_upload_limit_size'];
			}
		}

		$file_size_limit              = $php_max_upload;
		$enable_points                = ! empty( $settings['lesson_assignment_points_enabled'] ) && 'on' === $settings['lesson_assignment_points_enabled'];
		$points_amount                = ! empty( $settings['lesson_assignment_points_amount'] ) ? $settings['lesson_assignment_points_amount'] : 1;
		$grading_type['auto_approve'] = ! empty( $settings['auto_approve_assignment'] ) && 'on' === $settings['auto_approve_assignment'];

		if ( false === $grading_type['auto_approve'] ) {
			$grading_type['upload_limit_count'] = ! empty( $settings['assignment_upload_limit_count'] ) ? $settings['assignment_upload_limit_count'] : 1;
			$grading_type['can_delete']         = ! empty( $settings['lesson_assignment_deletion_enabled'] ) && 'on' === $settings['lesson_assignment_deletion_enabled'];
		}

		return array(
			'file_extensions' => $file_extensions,
			'file_size_limit' => $file_size_limit,
			'enable_points'   => $enable_points,
			'points_amount'   => $points_amount,
			'grading_type'    => $grading_type,
		);
	}

	/**
	 * Get lesson materials.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return string
	 */
	protected function get_materials( $post ) {
		$lesson_materials_enabled = learndash_get_setting( $post->ID, 'lesson_materials_enabled' );

		if ( ( isset( $lesson_materials_enabled ) ) && ( 'on' === $lesson_materials_enabled ) ) {
			$lesson_materials = learndash_get_setting( $post->ID, 'lesson_materials' );
			$lesson_materials = wp_kses_post( wp_specialchars_decode( $lesson_materials, ENT_QUOTES ) );

			if ( ! empty( $lesson_materials ) ) {
				$lesson_materials = do_shortcode( $lesson_materials );
			};

			return $lesson_materials;
		}

		return '';
	}

	/**
	 * Get lesson video.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return string
	 */
	protected function get_video( $post ) {
		$lesson_video_enabled = learndash_get_setting( $post->ID, 'lesson_video_enabled' );

		if ( ( isset( $lesson_video_enabled ) ) && ( 'on' === $lesson_video_enabled ) ) {
			return learndash_get_setting( $post->ID, 'lesson_video_url' );
		}

		return '';
	}

	/**
	 * If user has course access.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return mixed
	 */
	public function get_has_course_access( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post->ID );

		return sfwd_lms_has_access( $course_id );
	}

	/**
	 * If user has lesson content access.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return bool
	 */
	public function get_has_content_access( $post ) {
		return bbapp_lms_lesson_access_from( $post ) &&
		       bbapp_lms_is_content_access( $post, 'prerequities_completed' ) &&
		       bbapp_lms_is_content_access( $post, 'points_access' ) &&
		       (
			       bbapp_lms_is_content_access( $post, 'previous_lesson_completed' ) ||
			       sfwd_lms_has_access( $post->ID )
		       ) &&
		       empty( bbapp_lesson_visible_after( $post ) );
	}

	/**
	 * If lesson is sample lesson.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return mixed
	 */
	protected function get_is_sample( $post ) {
		return learndash_is_sample( $post );
	}

	/**
	 * Get lesson quizzes.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return array
	 */
	protected function get_quizzes( $post ) {
		$quizes = learndash_get_lesson_quiz_list( $post );
		$posts  = array();

		foreach ( $quizes as $quiz ) {
			$posts[] = $quiz['post']->ID;
		}

		return $posts;
	}

	/**
	 * Get lesson topics.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return array
	 */
	protected function get_topics( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post );
		$topics    = learndash_get_topic_list( $post->ID, $course_id );
		$posts     = array();

		if ( ! empty( $topics ) ) {
			foreach ( $topics as $topic ) {
				$posts[] = $topic->ID;
			}
		}

		return $posts;
	}

	/**
	 * This function we use in Course endpoint so it should be public
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return mixed
	 */
	public function is_completed( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post );

		return learndash_is_lesson_complete( null, $post->ID, $course_id );
	}

	/**
	 * If lesson quiz completed.
	 *
	 * @param WP_Post $post      Lesson post.
	 * @param int     $course_id Course id.
	 *
	 * @return bool
	 */
	protected function is_quiz_completed( $post, $course_id = 0 ) {
		return bbapp_lms_quiz_completed( $post, $course_id );
	}

	/**
	 * Get lesson settings.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return array
	 */
	protected function get_settings( $post ) {
		$settings             = array();
		$lesson_settings      = learndash_get_setting( $post->ID );
		$lesson_video_enabled = learndash_get_setting( $post->ID, 'lesson_video_enabled' );

		if ( isset( $lesson_video_enabled ) && 'on' === $lesson_video_enabled ) {
			$settings['video'] = array(
				'shown'         => isset( $lesson_settings['lesson_video_shown'] ) ? $lesson_settings['lesson_video_shown'] : false,
				'auto_start'    => isset( $lesson_settings['lesson_video_auto_start'] ) ? 'on' === $lesson_settings['lesson_video_auto_start'] : false,
				'show_controls' => isset( $lesson_settings['lesson_video_show_controls'] ) ? 'on' === $lesson_settings['lesson_video_show_controls'] : false,
				'cookie_key'    => build_video_cookie_key( $post ),
			);

			if ( 'AFTER' === $settings['video']['shown'] ) {
				$settings['video']['shown_after_lesson_video_auto_complete']        = isset( $lesson_settings['lesson_video_auto_complete'] ) ? 'on' === $lesson_settings['lesson_video_auto_complete'] : false;
				$settings['video']['shown_after_lesson_video_auto_complete_delay']  = $lesson_settings['lesson_video_auto_complete_delay'];
				$settings['video']['shown_after_lesson_video_show_complete_button'] = isset( $lesson_settings['lesson_video_show_complete_button'] ) ? 'on' === $lesson_settings['lesson_video_show_complete_button'] : false;
			}
		} else {
			$settings['video'] = array(
				'shown'         => '',
				'auto_start'    => false,
				'show_controls' => true,
				'cookie_key'    => null,
			);
		}

		if ( isset( $post->assignment_upload ) && ! empty( $post->assignment_upload ) ) {
			$settings['assignment_upload'] = $this->get_lesson_assignment_upload_settings( $post );
		}

		return $settings;
	}

	/**
	 * Current user complete course or not.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return bool
	 */
	private function get_can_complete( $post ) {
		$can_complete = false;

		if ( is_user_logged_in() ) {
			$can_complete = true;
		}

		return $can_complete;
	}

	/**
	 * Get error message.
	 *
	 * @param WP_Post $post Lesson post.
	 *
	 * @return WP_Error|array
	 */
	public function get_error_message( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post->ID );
		$user_id   = get_current_user_id();

		if ( ! bbapp_lms_is_content_access( $post, 'prerequities_completed' ) ) {
			$prerequisite_posts_all = learndash_get_course_prerequisites( $course_id );
			$course_str             = '';

			if ( ! empty( $prerequisite_posts_all ) ) {
				foreach ( $prerequisite_posts_all as $pre_post_id => $pre_status ) {
					if ( false === $pre_status ) {
						if ( ! empty( $course_str ) ) {
							$course_str .= ', ';
						}

						$course_str .= get_the_title( $pre_post_id );
					}
				}
			}

			return CoursesError::instance()->invalid_course_prerequities( $course_str );
		}

		if ( ! bbapp_lms_is_content_access( $post, 'points_access' ) ) {
			$course_access_points = learndash_get_course_points_access( $course_id );
			$user_course_points   = learndash_get_user_course_points( $user_id );

			return CoursesError::instance()->invalid_course_access_point( $course_access_points, $user_course_points );
		}

		$lesson_access_from = bbapp_lesson_visible_after( $post );
		if ( ! empty( $lesson_access_from ) && true !== $lesson_access_from ) {
			$lesson_access_from = learndash_adjust_date_time_display( $lesson_access_from );

			return LessonsError::instance()->lesson_not_available( $lesson_access_from );
		}

		if ( ! sfwd_lms_has_access( $post->ID, $user_id ) ) {
			return LessonsError::instance()->invalid_lesson_access();
		}

		if ( ! bbapp_lms_is_content_access( $post, 'previous_lesson_completed' ) && ! learndash_is_sample( $post->ID ) ) {
			return LessonsError::instance()->lesson_previous_not_completed();
		}

		return array();
	}

	/**
	 * Function will insert data in db for course and lesson.
	 *
	 * @param int $lesson_id Lesson id.
	 *
	 * @return mixed
	 */
	public function bbapp_ld_start_cource_and_lesson( $lesson_id ) {
		$course_id       = bbapp_learndash_get_course_id( $lesson_id );
		$current_user_id = get_current_user_id();
		// We insert the Course started record before the Lesson.
		$course_args = array(
			'course_id'        => $course_id,
			'user_id'          => $current_user_id,
			'post_id'          => $course_id,
			'activity_type'    => 'course',
			'activity_status'  => false,
			'activity_started' => time(),
			'activity_meta'    => array(
				'steps_total'     => learndash_get_course_steps_count( $course_id ),
				'steps_completed' => learndash_course_get_completed_steps( $current_user_id, $course_id ),
				'steps_last_id'   => $course_id,
			),
		);

		$course_activity = learndash_get_user_activity( $course_args );

		if ( ( ! $course_activity ) || ( ! isset( $course_activity->activity_started ) ) ) {
			learndash_update_user_activity( $course_args );
		}

		// We insert the Lesson.
		$lesson_args = array(
			'course_id'        => $course_id,
			'user_id'          => $current_user_id,
			'post_id'          => $lesson_id,
			'activity_type'    => 'lesson',
			'activity_status'  => false,
			'activity_started' => time(),
			'activity_meta'    => array(
				'steps_total'     => learndash_get_course_steps_count( $course_id ),
				'steps_completed' => learndash_course_get_completed_steps( $current_user_id, $course_id ),
			),
		);

		$lesson_activity = learndash_get_user_activity( $lesson_args );

		if ( ( ! $lesson_activity ) || ( ! isset( $lesson_activity->activity_started ) ) ) {
			learndash_update_user_activity( $lesson_args );
		}
	}
}
