<?php
/**
 * Holds Lessons related API functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Lesson
 */

namespace BuddyBossApp\Api\LearnDash\V1\Lesson;

use BuddyBossApp\RestErrors;
use WP_Error;

/**
 * Class LessonsError
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Lesson
 */
class LessonsError {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return LessonsError
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * When user is not logged in this error message should be used.
	 *
	 * @return WP_Error
	 */
	public function user_not_logged_in() {
		return RestErrors::instance()->user_not_logged_in();
	}

	/**
	 * When Lesson id not correct.
	 *
	 * @return WP_Error
	 */
	public function invalid_lesson_id() {
		return new WP_Error( 'learndash_json_lesson_invalid_id', __( 'Lesson not found.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * When Lesson's course id not correct on lesson actions.
	 *
	 * @return WP_Error
	 */
	public function invalid_course_id() {
		return new WP_Error( 'learndash_json_lesson_invalid_course', __( 'Lesson\'s course not found! On which you performing action.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user don't have lesson access.
	 *
	 * @return WP_Error
	 */
	public function invalid_lesson_access() {
		return new WP_Error( 'learndash_json_lesson_invalid_access', __( 'You do not have access to this lesson.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user doesn't match required course prerequities.
	 *
	 * @param string $lesson_access_from Time string to show when lesson available.
	 *
	 * @return WP_Error
	 */
	public function lesson_not_available( $lesson_access_from ) {
		/* translators: %s: Time string to show when lesson available. */
		$message = sprintf( __( 'Available on: %s', 'buddyboss-app' ), $lesson_access_from );

		return new WP_Error( 'learndash_json_lesson_not_available', $message, array( 'status' => 400 ) );
	}

	/**
	 * When user not completed previous Lesson.
	 *
	 * @return WP_Error
	 */
	public function lesson_previous_not_completed() {
		return new WP_Error( 'learndash_json_lesson_previous_not_completed', __( 'You must complete each lessons in sequence.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user not completed lesson's topic and trying to complete lesson.
	 *
	 * @return WP_Error
	 */
	public function lesson_topics_not_completed() {
		return new WP_Error( 'learndash_json_lesson_topics_not_completed', __( "Lesson's topic is not completed.", 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user not completed lesson's quiz and trying to complete Lesson.
	 *
	 * @return WP_Error
	 */
	public function lesson_quiz_not_completed() {
		return new WP_Error( 'learndash_json_lesson_quiz_not_completed', __( "Lesson's quiz is not completed.", 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user not completed lesson's assignment and trying to complete lesson.
	 *
	 * @param int $count Count of assignments.
	 *
	 * @since 2.1.00
	 *
	 * @return WP_Error
	 */
	public function lesson_assignment_not_approved( $count ) {
		$message = sprintf( _n( 'You have an assignment awaiting approval', 'You have an assignments awaiting approval', $count, 'buddyboss-app' ), $count );

		return new WP_Error( 'learndash_json_lesson_assignment_not_approved', $message, array( 'status' => 400 ) );
	}

	/**
	 * When user not completed lesson's assignment and trying to complete Lesson.
	 *
	 * @return WP_Error
	 */
	public function lesson_assignment_not_completed() {
		return new WP_Error( 'learndash_json_lesson_assignment_not_completed', __( 'Please upload assignment to complete this Lesson.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When Lesson complete fails.
	 *
	 * @return WP_Error
	 */
	public function lesson_complete_error() {
		return new WP_Error( 'learndash_json_lesson_complete_error', __( 'Something is wrong with complete process.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When Lesson Incomplete fails.
	 *
	 * @return WP_Error
	 * @since          1.6.4
	 */
	public function lesson_incomplete_error() {
		return new WP_Error( 'learndash_json_lesson_incomplete_error', __( 'Something wrong with incomplete process!', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When assignment dir not writeable.
	 *
	 * @return WP_Error
	 */
	public function assignment_dir_not_writeable() {
		return new WP_Error( 'learndash_json_assignment_dir_not_writeable', __( 'Unable to write to directory. Is this directory writable by the server?', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When lesson zip is not created.
	 *
	 * @return WP_Error
	 */
	public function invalid_lesson_zip() {
		return new WP_Error( 'learndash_json_lesson_invalid_zip', __( 'Lesson Zip file not created.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}
}
