<?php
/**
 * Holds lesson action rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Lesson
 */

namespace BuddyBossApp\Api\LearnDash\V1\Lesson;

use BuddyBossApp\Api\LearnDash\V1\Assignments\AssignmentsActionRest;
use BuddyBossApp\Api\LearnDash\V1\Course\CoursesError;
use BuddyBossApp\Api\LearnDash\V1\Topic\TopicsActionRest;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Lesson action rest class.
 */
class LessonsActionRest extends LessonsRest {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * LessonsActionRest instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/complete',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'complete' ),
					'permission_callback' => array( $this, 'get_complete_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/download',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'download_item' ),
					'permission_callback' => array( $this, 'get_download_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/incomplete',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'incomplete' ),
					'permission_callback' => array( $this, 'get_incomplete_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to lesson item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_complete_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = LessonsError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the lesson `complete` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_lesson_complete_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to lesson item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 1.6.4
	 */
	public function get_incomplete_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = LessonsError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the lesson `incomplete` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 1.6.4
		 */
		return apply_filters( 'bbapp_ld_lesson_incomplete_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to lesson item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_download_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = LessonsError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the lesson `download` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_lesson_download_permissions_check', $retval, $request );
	}

	/**
	 * Process Lesson complete.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          0.1.0
	 *
	 * @return WP_Error|WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/lessons/:id/complete Process LearnDash Lesson Complete
	 * @apiName        PostLDLessonComplete
	 * @apiGroup       LD Lessons
	 * @apiDescription Process Lesson Complete
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the lesson.
	 */
	public function complete( $request ) {
		$lesson_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course_id = isset( $request['course_id'] ) ? (int) $request['course_id'] : 0;
		$user_id   = get_current_user_id();
		$lesson    = get_post( $lesson_id );

		if ( empty( $lesson ) || $this->post_type !== $lesson->post_type ) {
			return LessonsError::instance()->invalid_lesson_id();
		}

		if ( empty( $course_id ) ) {
			return LessonsError::instance()->invalid_course_id();
		}

		if ( ! sfwd_lms_has_access( $lesson_id, $user_id ) ) {
			return LessonsError::instance()->invalid_lesson_access();
		}

		if ( function_exists( 'learndash_is_lesson_complete' ) && ! learndash_is_lesson_complete( $user_id, $lesson_id ) ) {
			if ( ! bbapp_lms_is_content_access( $lesson, 'points_access' ) ) {
				return CoursesError::instance()->invalid_course_access_point();
			}

			if ( ! bbapp_lms_is_content_access( $lesson, 'prerequities_completed' ) ) {
				return CoursesError::instance()->invalid_course_prerequities();
			}

			$lesson_access_from = bbapp_lesson_visible_after( $lesson );
			if ( ! empty( $lesson_access_from ) && true !== $lesson_access_from ) {
				$lesson_access_from = learndash_adjust_date_time_display( $lesson_access_from );

				return LessonsError::instance()->lesson_not_available( $lesson_access_from );
			}

			if ( ! bbapp_lms_is_content_access( $lesson, 'previous_lesson_completed' ) ) {
				return LessonsError::instance()->lesson_previous_not_completed();
			}

			if ( ! learndash_lesson_topics_completed( $lesson_id ) ) {
				if ( ! apply_filters( 'learndash_previous_step_completed', false, $lesson_id, $user_id ) ) {
					return LessonsError::instance()->lesson_topics_not_completed();
				}
			}

			if ( ! bbapp_lms_quiz_completed( $lesson ) ) {
				return LessonsError::instance()->lesson_quiz_not_completed();
			}

			if ( $this->get_lesson_assignment_upload( $lesson ) ) {
				/**
				 * Process lesson assignment to complete lesson
				 */
				$created_assignment_id = AssignmentsActionRest::instance()->upload_attachment_file( $lesson_id );

				if ( is_wp_error( $created_assignment_id ) ) {
					return $created_assignment_id;
				} else {
					$completed = ( 0 === $created_assignment_id ) ? false : true;
				}
			} else {
				// To stop "Uncanny LearnDash Toolkit" Redirection after complete event for rest endpoint.
				$_REQUEST['doing_rest'] = 1;

				$completed = learndash_process_mark_complete( $user_id, $lesson_id );
			}
		} else {
			/**
			 * Lesson already completed
			 */
			$completed = true;
		}

		if ( is_wp_error( $completed ) ) {
			return $completed;
		} elseif ( true === $completed ) {
			return $this->get_item( $request );
		}

		return LessonsError::instance()->lesson_complete_error();
	}

	/**
	 * Process Lesson incomplete.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          1.6.4
	 *
	 * @return WP_Error|WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/lessons/:id/incomplete Process LearnDash Lesson Incomplete
	 * @apiName        PostLDLessonIncomplete
	 * @apiGroup       LD Lessons
	 * @apiDescription Process Lesson Incomplete
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the lesson.
	 */
	public function incomplete( $request ) {
		$lesson_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course_id = isset( $request['course_id'] ) ? (int) $request['course_id'] : 0;
		$user_id   = get_current_user_id();
		$lesson    = get_post( $lesson_id );

		if ( empty( $lesson ) || $this->post_type !== $lesson->post_type ) {
			return LessonsError::instance()->invalid_lesson_id();
		}

		if ( empty( $course_id ) ) {
			return LessonsError::instance()->invalid_course_id();
		}

		if ( ! sfwd_lms_has_access( $lesson_id, $user_id ) ) {
			return LessonsError::instance()->invalid_lesson_access();
		}

		if ( function_exists( 'learndash_is_lesson_complete' ) && learndash_is_lesson_complete( $user_id, $lesson_id ) ) {
			if ( ! bbapp_lms_is_content_access( $lesson, 'points_access' ) ) {
				return CoursesError::instance()->invalid_course_access_point();
			}

			if ( ! bbapp_lms_is_content_access( $lesson, 'prerequities_completed' ) ) {
				return CoursesError::instance()->invalid_course_prerequities();
			}

			$lesson_access_from = bbapp_lesson_visible_after( $lesson );
			if ( ! empty( $lesson_access_from ) && true !== $lesson_access_from ) {
				$lesson_access_from = learndash_adjust_date_time_display( $lesson_access_from );

				return LessonsError::instance()->lesson_not_available( $lesson_access_from );
			}

			if ( ! bbapp_lms_is_content_access( $lesson, 'previous_lesson_completed' ) ) {
				return LessonsError::instance()->lesson_previous_not_completed();
			}

			if ( ! learndash_lesson_topics_completed( $lesson_id ) ) {
				if ( ! apply_filters( 'learndash_previous_step_completed', false, $lesson_id, $user_id ) ) {
					return LessonsError::instance()->lesson_topics_not_completed();
				}
			}

			if ( ! bbapp_lms_quiz_completed( $lesson ) ) {
				return LessonsError::instance()->lesson_quiz_not_completed();
			}

			/**
			 * Process lesson to incomplete.
			 */
			$course_id  = bbapp_learndash_get_course_id( $lesson_id );
			$incomplete = learndash_process_mark_incomplete( $user_id, $course_id, $lesson_id );
		} else {
			/**
			 * Lesson already incomplete.
			 */
			$incomplete = true;
		}

		if ( is_wp_error( $incomplete ) ) {
			return $incomplete;
		} elseif ( true === $incomplete ) {
			return $this->get_item( $request );
		}

		return LessonsError::instance()->lesson_incomplete_error();
	}

	/**
	 * Process Lesson Download.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/lessons/:id/download Process LearnDash Lesson Download
	 * @apiName        PostLDLessonDownload
	 * @apiGroup       LD Lessons
	 * @apiDescription Process Lesson Download
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the lesson.
	 */
	public function download_item( $request ) {
		$course_id = isset( $request['course_id'] ) ? (int) $request['course_id'] : 0;
		$lesson_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$lesson    = get_post( $lesson_id );

		if ( empty( $lesson ) || $this->post_type !== $lesson->post_type ) {
			return LessonsError::instance()->invalid_lesson_id();
		}

		// Set lesson folder path.
		$folder_name = 'lesson_' . $lesson_id . '/';

		// Delete lesson directory if exist.
		$folder_path = bbapp_learners_getpath( $folder_name );
		bbapp_learners_delete_directory( $folder_path );

		// Create lesson directory.
		$folder_path = bbapp_learners_setpath( $folder_name );

		// Create assets directory for lesson.
		$assests_path = bbapp_learners_setpath( $folder_name ) . 'lessons/';

		if ( ! file_exists( $assests_path ) ) {
			wp_mkdir_p( $assests_path );
		}

		$data = $this->prepare_item_for_response( $lesson, $request );

		/**
		 * Download lesson html and assets.
		 *
		 * @param array $assets Download asets.
		 */
		$html_attrs = apply_filters( 'bbapp_ld_lesson_html_attrs', array( 'content', 'materials' ) );

		foreach ( $html_attrs as $attr ) {
			if ( isset( $data->data[ $attr ]['rendered'] ) ) {
				$value                           = bbapp_learners_download_html( $assests_path, $data->data[ $attr ]['rendered'] );
				$data->data[ $attr ]['rendered'] = $value;
			} else {
				$value               = bbapp_learners_download_html( $assests_path, $data->data[ $attr ] );
				$data->data[ $attr ] = $value;
			}

			$file_name = 'lesson_' . $attr . '.html';

			bbapp_learners_write_file( $assests_path . $file_name, $value );
		}

		// Download lesson's topics.
		if ( ! empty( $data->data['topics'] ) ) {
			/**
			 * Create topics folder if not exist.
			 */
			$topics_path = bbapp_learners_setpath( $folder_name ) . 'topics/';

			if ( ! file_exists( $topics_path ) ) {
				wp_mkdir_p( $topics_path );
			}

			$topic_rest = TopicsActionRest::instance();

			foreach ( $data->data['topics'] as $topic ) {
				$topic_request = new WP_REST_Request( 'GET', $this->namespace . '/topics/' . $topic['id'] . '/download?course_id=' . $course_id . '&lesson_id=' . $lesson_id );
				$topic_request->set_url_params( array( 'id' => $topic['id'] ) );
				$topic_request->set_param( 'course_id', $course_id );
				$topic_request->set_param( 'lesson_id', $lesson_id );
				$topic_rest->download_item( $topic_request, $folder_name . 'topics/' );
			}
		}

		// Create lesson.json.
		$file_data = wp_json_encode( $data );
		$file_name = 'lesson_' . $lesson_id . '.json';

		bbapp_learners_write_file( $folder_path . $file_name, $file_data );

		// Create info.json.
		$info_data = wp_json_encode(
			array(
				'id'           => $lesson_id,
				'hash_key'     => sha1( $lesson->post_modified ),
				'modified'     => mysql_to_rfc3339( $lesson->post_modified ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
				'modified_gmt' => mysql_to_rfc3339( $lesson->post_modified_gmt ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			)
		);
		$file_name = 'info.json';

		bbapp_learners_write_file( $folder_path . $file_name, $info_data );

		// Create zip file.
		$zip_url = bbapp_learners_zip_create( $folder_path, 'lesson_' . $lesson_id );

		if ( empty( $zip_url ) ) {
			return LessonsError::instance()->invalid_lesson_zip();
		}

		return rest_ensure_response( array( 'zip_url' => $zip_url ) );
	}
}
