<?php
/**
 * Holds LearnDash course rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Course
 */

namespace BuddyBossApp\Api\LearnDash\V1\Course;

use BuddyBossApp\Admin\Settings;
use BuddyBossApp\Api\LearnDash\V1\Certificate\CertificateRest;
use BuddyBossApp\Api\LearnDash\V1\Core\LDRestController;
use BuddyBossApp\Api\LearnDash\V1\Lesson\LessonsRest;
use BuddyBossApp\Api\LearnDash\V1\Quiz\QuizRest;
use BuddyBossApp\Api\LearnDash\V1\Topic\TopicsRest;
use BuddyBossApp\Helpers\PostCover;
use LearnDash_Settings_Section;
use WP_Error;
use WP_Post;
use WP_Query;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * LearnDash course rest class.
 */
class CoursesRest extends LDRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Course post type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'sfwd-courses';

	/**
	 * Course progress.
	 *
	 * @var $_my_course_progress
	 */
	protected $my_course_progress;

	/**
	 * Allow batch.
	 *
	 * @var true[] $allow_batch
	 */
	protected $allow_batch = array( 'v1' => true );

	/**
	 * CoursesRest instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'courses';

		parent::__construct();
	}

	/**
	 * Check if a given request has access to course items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the course `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_courses_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to course item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the course `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_course_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'allow_batch' => $this->allow_batch,
				'schema'      => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'single' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/details',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item_details' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
					'allow_batch'         => $this->allow_batch,
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}


	/**
	 * Retrieve Courses.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/courses Get LearnDash Courses
	 * @apiName        GetLDCourses
	 * @apiGroup       LD Courses
	 * @apiDescription Retrieve Courses
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure result set includes specific IDs.
	 * @apiParam {String} [after]  Limit results to those published after a given ISO8601 compliant date.
	 * @apiParam {String} [before] Limit results to those published before a given ISO8601 compliant date.
	 * @apiParam {Array} [author] Limit results to those assigned to specific authors.
	 * @apiParam {Array} [author_exclude] Ensure results to excludes those assigned to specific authors.
	 * @apiParam {String=asc,desc} [order=desc] Sort result set by given order.
	 * @apiParam {String=date,id,title,menu_order} [orderby=date] Sort result set by given field.
	 * @apiParam {Array} [categories] Limit results to those assigned to specific categories.
	 * @apiParam {Number=0,1} [mycourses] Limit results to current user courses.
	 */
	public function get_items( $request ) {
		$user_id    = get_current_user_id();
		$registered = $this->get_collection_params();

		/**
		 * Filter the the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$request = apply_filters( 'bbapp_ld_get_courses_request', $request );

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'author'         => 'author__in',
			'author_exclude' => 'author__not_in',
			'exclude'        => 'post__not_in',
			'include'        => 'post__in',
			'offset'         => 'offset',
			'order'          => 'order',
			'orderby'        => 'orderby',
			'page'           => 'paged',
			'parent'         => 'post_parent__in',
			'parent_exclude' => 'post_parent__not_in',
			'search'         => 's',
			'slug'           => 'post_name__in',
			'status'         => 'post_status',
			'mycourses'      => 'mycourses',
			'group_id'       => 'group_id',
			'per_page'       => 'posts_per_page',
		);

		/**
		 * For each known parameter which is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		// Check for & assign any parameters which require special handling or setting.
		$args['date_query'] = array();

		// Set before into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['before'], $request['before'] ) ) {
			$args['date_query'][0]['before'] = $request['before'];
		}

		// Set after into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['after'], $request['after'] ) ) {
			$args['date_query'][0]['after'] = $request['after'];
		}

		$args     = $this->prepare_items_query( $args, $request );
		$relation = ( ! empty( $request['tax_relation'] ) ) ? $request['tax_relation'] : '';

		if ( ! empty( $relation ) ) {
			$args['tax_query'] = array( 'relation' => $relation ); //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
		}

		// LD Course Categories.
		if ( ! empty( $request['categories'] ) ) {
			if ( ! is_array( $request['categories'] ) ) {
				$request['categories'] = (array) $request['categories'];
			}

			if ( in_array( 0, $request['categories'], true ) ) {
				$args['tax_query'][] = array(
					'taxonomy' => 'ld_course_category',
					'operator' => 'NOT EXISTS',
				);
			}

			$request['categories'] = array_filter( $request['categories'] );

			if ( ! empty( $request['categories'] ) ) {
				$args['tax_query'][] = array(
					'taxonomy'         => 'ld_course_category',
					'field'            => 'term_id',
					'terms'            => $request['categories'],
					'include_children' => false,
				);
			}
		}

		if ( ! empty( $request['categories_exclude'] ) ) {
			$args['tax_query'][] = array(
				'taxonomy'         => 'ld_course_category',
				'field'            => 'term_id',
				'terms'            => $request['categories_exclude'],
				'include_children' => false,
				'operator'         => 'NOT IN',
			);
		}

		// LD Course Tags.
		if ( ! empty( $request['tags'] ) ) {
			$args['tax_query'][] = array(
				'taxonomy'         => 'ld_course_tag',
				'field'            => 'term_id',
				'terms'            => $request['tags'],
				'include_children' => false,
			);
		}

		if ( ! empty( $request['tags_exclude'] ) ) {
			$args['tax_query'][] = array(
				'taxonomy'         => 'ld_course_tag',
				'field'            => 'term_id',
				'terms'            => $request['tags_exclude'],
				'include_children' => false,
				'operator'         => 'NOT IN',
			);
		}

		if ( isset( $args['mycourses'] ) && $args['mycourses'] ) {
			$mycourse_ids = ld_get_mycourses( $user_id, array() );

			if ( ! empty( $mycourse_ids ) && ! is_wp_error( $mycourse_ids ) ) {
				$args['post__in'] = ! empty( $args['post__in'] ) ? array_intersect( $mycourse_ids, $args['post__in'] ) : $mycourse_ids;
			}

			/*
			 * If we intersected, but there are no post ids in common,
			 * WP_Query won't return "no posts" for post__in = array()
			 * so we have to fake it a bit.
			*/
			if ( ! $args['post__in'] ) {
				$args['post__in'] = array( 0 );
			}

			unset( $args['mycourses'] );
		}

		if ( isset( $args['group_id'] ) && ! empty( $args['group_id'] ) ) {
			$course_ids = array();

			if ( function_exists( 'bp_ld_sync' ) ) {
				$group_id   = bp_ld_sync( 'buddypress' )->helpers->getLearndashGroupId( $args['group_id'] );
				$course_ids = learndash_group_enrolled_courses( $group_id );
			}

			$args['post__in'] = ! empty( $course_ids ) ? $course_ids : array( 0 );

			unset( $args['group_id'] );
		}

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$args = apply_filters( 'bbapp_ld_get_courses_args', $args, $request );

		$args['post_type'] = $this->post_type;
		add_filter( 'posts_distinct', array( $this, 'bbapp_posts_distinct' ), 10, 2 );
		add_filter( 'posts_join', array( $this, 'bbapp_posts_join' ), 10, 2 );
		add_filter( 'posts_where', array( $this, 'bbapp_posts_where' ), 10, 1 );
		$posts_query            = new WP_Query();
		$courses['posts']       = $posts_query->query( $args );
		$courses['total_posts'] = $posts_query->found_posts;

		// Todo::dips Check again this code needed or not.
		if ( $courses['total_posts'] < 1 ) {
			// Out-of-bounds, run the query again without LIMIT for total count.
			unset( $args['paged'] );
			$count_query = new WP_Query();
			$count_query->query( $args );
			$courses['total_posts'] = $count_query->found_posts;
		}

		remove_filter( 'posts_where', array( $this, 'bbapp_posts_where' ), 10 );
		remove_filter( 'posts_join', array( $this, 'bbapp_posts_join' ), 10 );
		remove_filter( 'posts_distinct', array( $this, 'bbapp_posts_distinct' ), 10 );

		/**
		 * Fires list of Courses is fetched via Query.
		 *
		 * @param array            $courses Fetched courses.
		 * @param WP_REST_Response $args    Query arguments.
		 * @param WP_REST_Request  $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$courses = apply_filters( 'bbapp_ld_get_courses', $courses, $args, $request );

		$retval = array();

		foreach ( $courses['posts'] as $couese ) {
			if ( ! $this->check_read_permission( $couese ) ) {
				continue;
			}

			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $couese, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		$response = bbapp_learners_response_add_total_headers( $response, $courses['total_posts'], $args['posts_per_page'] );

		/**
		 * Fires after a list of Courses response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_course_items_response', $response, $request );

		return $response;
	}

	/**
	 * Get Open Free and user enrolled courses.
	 *
	 * @param string $where Where clause.
	 *
	 * @return mixed|string
	 */
	public function bbapp_posts_where( $where ) {
		global $wpdb;

		$settings        = \BuddyBossApp\ManageApp::instance()->get_app_settings();
		$where_condition = '';

		if ( is_user_logged_in() ) {
			remove_filter( 'posts_where', array( $this, 'bbapp_posts_where' ), 10 );
			remove_filter( 'posts_join', array( $this, 'bbapp_posts_join' ), 10 );
			remove_filter( 'posts_distinct', array( $this, 'bbapp_posts_distinct' ), 10 );
			$mycourse_ids = ld_get_mycourses( get_current_user_id(), array() );
			add_filter( 'posts_distinct', array( $this, 'bbapp_posts_distinct' ), 10, 2 );
			add_filter( 'posts_join', array( $this, 'bbapp_posts_join' ), 10, 2 );
			add_filter( 'posts_where', array( $this, 'bbapp_posts_where' ), 10, 1 );

			if ( ! empty( $mycourse_ids ) ) {
				$post__in        = implode( ',', array_unique( array_map( 'absint', $mycourse_ids ) ) );
				$where_condition = " AND post_id NOT IN($post__in)";
			}
		}

		$hide_in_app_query = " AND {$wpdb->posts}.ID NOT IN (SELECT post_id FROM {$wpdb->postmeta}  WHERE meta_key='_hide_in_app' AND meta_value = 'yes' $where_condition ) ";

		if ( isset( $settings['learndash_reader_app_compatibility'] ) && '1' === (string) $settings['learndash_reader_app_compatibility'] ) {
			$user_enrolled_course_where      = '';
			$free_open_not_hide_course_where = '';

			if ( isset( $post__in ) ) {
				$user_enrolled_course_where = " {$wpdb->posts}.ID IN ($post__in) OR ";
			}

			$free_open_not_hide_course_where .= "( bbpm.meta_key = '_sfwd-courses' AND ( bbpm.meta_value LIKE '%open%' OR bbpm.meta_value LIKE '%free%' ) ) {$hide_in_app_query} ";
			$where                           .= " AND ( {$user_enrolled_course_where} {$free_open_not_hide_course_where} )";
		} else {
			$where .= $hide_in_app_query;
		}

		return $where;
	}

	/**
	 * Get Open Free and user enrolled courses.
	 *
	 * @param string $join  Join clause.
	 * @param string $query Query.
	 *
	 * @return mixed|string
	 */
	public function bbapp_posts_join( $join, $query ) {
		global $wpdb;

		$join .= " INNER JOIN {$wpdb->postmeta} AS bbpm ON ( {$wpdb->posts}.ID = bbpm.post_id ) ";

		return $join;
	}

	/**
	 * Get Open Free and user enrolled courses.
	 *
	 * @param string $distinct Distuinct value.
	 * @param string $query    Query.
	 *
	 * @return mixed|string
	 */
	public function bbapp_posts_distinct( $distinct, $query ) {
		return ' DISTINCT ';
	}


	/**
	 * Retrieve Course.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/courses/:id Get LearnDash Course
	 * @apiName        GetLDCourse
	 * @apiGroup       LD Courses
	 * @apiDescription Retrieve single Course
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function get_item( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course    = get_post( $course_id );

		if ( empty( $course ) || $this->post_type !== $course->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		/**
		 * Fire after Course is fetched via Query.
		 *
		 * @param array   $course    Fetched course.
		 * @param integer $course_id Course id.
		 *
		 * @since 0.1.0
		 */
		$course = apply_filters( 'bbapp_ld_get_course', $course, $course_id );

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $course, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after an course response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_course_item_response', $response, $request );

		return $response;
	}

	/**
	 * Retrieve Course details like lesson/topics and quiz.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          0.1.0
	 *
	 * @return WP_Error | WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/courses/:id/details Get LearnDash Course Details
	 * @apiName        GetLDCourse
	 * @apiGroup       LD Courses
	 * @apiDescription Retrieve single Course
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function get_item_details( $request ) {
		$course_id       = is_numeric( $request ) ? $request : (int) $request['id'];
		$course          = get_post( $course_id );
		$GLOBALS['post'] = $course; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited --Set global post data.
		setup_postdata( $course ); // Set post data.
		if ( empty( $course ) || $this->post_type !== $course->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		$retval = array();
		// Get Course lesson.
		$pre_next          = array();
		$retval['lessons'] = array();

		/**
		 * Both functions will run based on learndash version.
		 */
		if ( function_exists( 'learndash_course_get_lessons' ) ) {
			$lessons = learndash_course_get_lessons( $course_id, array( 'per_page' => 0 ) );
		} else {
			$lessons = learndash_get_lesson_list( $course_id, array( 'num' => - 1 ) );
		}

		if ( ! empty( $lessons ) ) {
			$_GET['course_id'] = $course_id;

			foreach ( $lessons as $lesson ) {
				$ret_val_lesson = array(
					'id'                 => $lesson->ID,
					'title'              => get_the_title( $lesson->ID ),
					'completed'          => LessonsRest::instance()->is_completed( $lesson ),
					'has_course_access'  => LessonsRest::instance()->get_has_course_access( $lesson ),
					'has_content_access' => LessonsRest::instance()->get_has_content_access( $lesson ),
				);
				$pre_next[]     = array(
					'id'        => $lesson->ID,
					'type'      => 'lesson',
					'completed' => $ret_val_lesson['completed'],
				);

				/**
				 * Added error message for lesson
				 */
				$error_message = LessonsRest::instance()->get_error_message( $lesson );

				if ( ! empty( $error_message ) ) {
					$error_code                      = $error_message->get_error_code();
					$ret_val_lesson['error_message'] = array(
						'code'    => $error_code,
						'message' => $error_message->get_error_message(),
						'data'    => $error_message->get_error_data( $error_code ),
					);
				} else {
					$ret_val_lesson['error_message'] = array();
				}

				/**
				 * Get topic data for each lesson
				 */
				$topics                   = learndash_get_topic_list( $lesson->ID, $course_id );
				$ret_val_lesson['topics'] = array();

				if ( ! empty( $topics ) ) {
					$_GET['lesson_id'] = $lesson->ID;

					foreach ( $topics as $topic ) {
						$ret_val_topics = array(
							'id'                 => $topic->ID,
							'title'              => get_the_title( $topic->ID ),
							'completed'          => TopicsRest::instance()->is_completed( $topic ),
							'has_course_access'  => TopicsRest::instance()->get_has_course_access( $topic ),
							'has_content_access' => TopicsRest::instance()->get_has_content_access( $topic ),
						);
						$pre_next[]     = array(
							'id'        => $topic->ID,
							'parent'    => $lesson->ID,
							'type'      => 'topic',
							'completed' => $ret_val_topics['completed'],
						);

						/**
						 * Added error message for topics
						 */
						$error_message = TopicsRest::instance()->get_error_message( $topic );

						if ( ! empty( $error_message ) ) {
							$error_code                      = $error_message->get_error_code();
							$ret_val_topics['error_message'] = array(
								'code'    => $error_code,
								'message' => $error_message->get_error_message(),
								'data'    => $error_message->get_error_data( $error_code ),
							);
						} else {
							$ret_val_topics['error_message'] = array();
						}

						/**
						 * Get topic Quiz
						 */
						$ret_val_topics['quiz'] = array();
						$quizes                 = learndash_get_lesson_quiz_list( $topic->ID );

						if ( ! empty( $quizes ) ) {
							$_GET['topic_id'] = $topic->ID;

							foreach ( $quizes as $quiz ) {
								$quiz               = $quiz['post'];
								$ret_val_topic_quiz = array(
									'id'                 => $quiz->ID,
									'title'              => get_the_title( $quiz->ID ),
									'completed'          => QuizRest::instance()->is_completed( $quiz, $course_id ),
									'has_course_access'  => QuizRest::instance()->get_has_course_access( $quiz ),
									'has_content_access' => QuizRest::instance()->get_has_content_access( $quiz ),
									'can_take_again'     => QuizRest::instance()->get_can_take_again( $quiz ),
								);
								$pre_next[]         = array(
									'id'        => $quiz->ID,
									'parent'    => $topic->ID,
									'type'      => 'quiz',
									'completed' => $ret_val_topic_quiz['completed'],
								);

								/**
								 * Added error message for course quiz
								 */
								$error_message = QuizRest::instance()->get_error_message( $quiz );

								if ( ! empty( $error_message ) ) {
									$error_code                          = $error_message->get_error_code();
									$ret_val_topic_quiz['error_message'] = array(
										'code'    => $error_code,
										'message' => $error_message->get_error_message(),
										'data'    => $error_message->get_error_data( $error_code ),
									);
								} else {
									$ret_val_topic_quiz['error_message'] = array();
								}

								$ret_val_topics['quiz'][] = $ret_val_topic_quiz;
							}

							unset( $_GET['topic_id'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
						}

						$ret_val_lesson['topics'][] = $ret_val_topics;
					}

					unset( $_GET['lesson_id'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				}

				/**
				 * Get lesson Quiz
				 */
				$ret_val_lesson['quiz'] = array();
				$quizes                 = learndash_get_lesson_quiz_list( $lesson->ID );

				if ( ! empty( $quizes ) ) {
					$_GET['lesson_id'] = $lesson->ID;

					foreach ( $quizes as $quiz ) {
						$quiz                = $quiz['post'];
						$ret_val_lesson_quiz = array(
							'id'                 => $quiz->ID,
							'title'              => get_the_title( $quiz->ID ),
							'completed'          => QuizRest::instance()->is_completed( $quiz, $course_id ),
							'has_course_access'  => QuizRest::instance()->get_has_course_access( $quiz ),
							'has_content_access' => QuizRest::instance()->get_has_content_access( $quiz ),
							'can_take_again'     => QuizRest::instance()->get_can_take_again( $quiz ),
						);
						$pre_next[]          = array(
							'id'        => $quiz->ID,
							'parent'    => $lesson->ID,
							'type'      => 'quiz',
							'completed' => $ret_val_lesson_quiz['completed'],
						);

						/**
						 * Added error message for course quiz
						 */
						$error_message = QuizRest::instance()->get_error_message( $quiz );

						if ( ! empty( $error_message ) ) {
							$error_code                           = $error_message->get_error_code();
							$ret_val_lesson_quiz['error_message'] = array(
								'code'    => $error_code,
								'message' => $error_message->get_error_message(),
								'data'    => $error_message->get_error_data( $error_code ),
							);
						} else {
							$ret_val_lesson_quiz['error_message'] = array();
						}

						$ret_val_lesson['quiz'][] = $ret_val_lesson_quiz;
					}

					unset( $_GET['lesson_id'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				}

				/**
				 * Filters Course lesson list.
				 *
				 * @param array           $ret_val_lesson Return list.
				 * @param WP_REST_Request $request      Full details about the request.
				 */
				$retval['lessons'][] = apply_filters( 'bbapp_ld_rest_course_detail_lessons_list', $ret_val_lesson, $request );
			}

			unset( $_GET['course_id'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		}

		/**
		 * Get Course Quiz
		 */
		if ( $request['course_quiz'] ) {
			$retval['quiz'] = array();
			$quizes         = learndash_get_course_quiz_list( $course_id );

			if ( ! empty( $quizes ) ) {
				$_GET['course_id'] = $course_id;

				foreach ( $quizes as $quiz ) {
					$quiz                = $quiz['post'];
					$ret_val_course_quiz = array(
						'id'                 => $quiz->ID,
						'title'              => get_the_title( $quiz->ID ),
						'completed'          => QuizRest::instance()->is_completed( $quiz, $course_id ),
						'has_course_access'  => QuizRest::instance()->get_has_course_access( $quiz ),
						'has_content_access' => QuizRest::instance()->get_has_content_access( $quiz ),
						'can_take_again'     => QuizRest::instance()->get_can_take_again( $quiz ),
					);
					$pre_next[]          = array(
						'id'        => $quiz->ID,
						'parent'    => $course_id,
						'type'      => 'quiz',
						'completed' => $ret_val_course_quiz['completed'],
					);

					/**
					 * Added error message for course quiz
					 */
					$error_message = QuizRest::instance()->get_error_message( $quiz );

					if ( ! empty( $error_message ) ) {
						$error_code                           = $error_message->get_error_code();
						$ret_val_course_quiz['error_message'] = array(
							'code'    => $error_code,
							'message' => $error_message->get_error_message(),
							'data'    => $error_message->get_error_data( $error_code ),

						);
					} else {
						$ret_val_course_quiz['error_message'] = array();
					}

					/**
					 * Filters course details quiz list.
					 *
					 * @param array           $ret_val_course_quiz Course quiz data.
					 * @param WP_REST_Request $request             Full details about the request.
					 */
					$retval['quiz'][] = apply_filters( 'bbapp_ld_rest_course_detail_course_quiz_list', $ret_val_course_quiz, $request );
				}

				unset( $_GET['course_id'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			}
		}

		if ( ! empty( $pre_next ) ) {
			/**
			 * Filters the next question data.
			 *
			 * @param array           $pre_next Next question.
			 * @param WP_REST_Request $request  Full details about the request.
			 */
			$retval['prenext'] = apply_filters( 'bbapp_ld_rest_course_detail_pre_next', $pre_next, $request );
		}

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after an course response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_course_item_details_response', $response, $request );

		return $response;
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$GLOBALS['post'] = $post; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited

		setup_postdata( $post );

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$schema  = $this->get_public_item_schema();

		/**
		 * Create Short Content from Content without more link.
		 */
		add_filter( 'excerpt_more', array( $this, 'remove_excerpt_more_link_and_add_dots' ) );
		$short_content = wp_trim_excerpt( '', $post );
		remove_filter( 'excerpt_more', array( $this, 'remove_excerpt_more_link_and_add_dots' ) );

		// Base fields for every post.
		$data = array(
			'id'           => $post->ID,
			'title'        => array(
				'raw'      => $post->post_title,
				'rendered' => get_the_title( $post->ID ),
			),
			'content'      => array(
				'raw'      => bbapp_learners_fix_relative_urls_protocol( $post->post_content ),
				'rendered' => bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_content', $post->post_content ) ),
				'short'    => bbapp_learners_fix_relative_urls_protocol( $short_content ),
			),
			'date'         => mysql_to_rfc3339( $post->post_date ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'date_gmt'     => mysql_to_rfc3339( $post->post_date_gmt ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'modified'     => mysql_to_rfc3339( $post->post_modified ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'modified_gmt' => mysql_to_rfc3339( $post->post_modified_gmt ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'link'         => get_permalink( $post->ID ),
			'slug'         => $post->post_name,
			'author'       => (int) $post->post_author,
			'excerpt'      => array(
				'raw'      => bbapp_learners_fix_relative_urls_protocol( $post->post_excerpt ),
				'rendered' => bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_excerpt', $post->post_excerpt ) ),
			),
			'menu_order'   => (int) $post->menu_order,
		);

		/**
		 * Feature Media
		 */
		$post->featured_media            = $this->get_feature_media( $post );
		$data['featured_media']          = array();
		$data['featured_media']['small'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['small'] ) ) ? $post->featured_media['small'] : null;
		$data['featured_media']['large'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['large'] ) ) ? $post->featured_media['large'] : null;
		$post->cover_media               = $this->get_cover_media( $post );
		$data['cover_media']             = array();
		$data['cover_media']['small']    = ( is_array( $post->cover_media ) && isset( $post->cover_media['small'] ) ) ? $post->cover_media['small'] : null;
		$data['cover_media']['large']    = ( is_array( $post->cover_media ) && isset( $post->cover_media['large'] ) ) ? $post->cover_media['large'] : null;

		if ( ! empty( $schema['properties']['has_course_access'] ) && in_array( $context, $schema['properties']['has_course_access']['context'], true ) ) {
			$post->has_course_access   = $this->get_has_course_access( $post );
			$data['has_course_access'] = (bool) $post->has_course_access;
		}

		if ( ! empty( $schema['properties']['offline_disabled'] ) && in_array( $context, $schema['properties']['offline_disabled']['context'], true ) ) {
			$post->offline_disabled   = $this->is_offline_disabled( $post );
			$data['offline_disabled'] = (bool) $post->offline_disabled;
		}

		$post->has_content_access = $this->get_has_content_access( $post );

		if ( ! empty( $schema['properties']['has_content_access'] ) && in_array( $context, $schema['properties']['has_content_access']['context'], true ) ) {
			$data['has_content_access'] = (bool) $post->has_content_access;
		}

		if ( ! empty( $schema['properties']['materials'] ) && in_array( $context, $schema['properties']['materials']['context'], true ) ) {
			$post->materials = $this->get_materials( $post );

			if ( ! $post->has_content_access ) {
				$post->materials = '';
			}

			$data['materials'] = $post->materials;
		}

		if ( ! empty( $schema['properties']['purchasable'] ) && in_array( $context, $schema['properties']['purchasable']['context'], true ) ) {
			$post->purchasable   = $this->is_purchasable( $post );
			$data['purchasable'] = (bool) $post->purchasable;
		}

		if ( ! empty( $schema['properties']['price'] ) && in_array( $context, $schema['properties']['price']['context'], true ) ) {
			$post->price = $this->get_price( $post );

			if ( is_array( $post->price ) ) {
				$data['price'] = array(
					'value'    => $post->price['value'],
					'rendered' => $post->price,
					'code'     => $post->price['code'],
				);
			}
		}

		if ( ! empty( $schema['properties']['hide_content_table'] ) && in_array( $context, $schema['properties']['hide_content_table']['context'], true ) ) {
			$post->hide_content_table   = $this->get_hide_content_table( $post );
			$data['hide_content_table'] = (bool) $post->hide_content_table;
		}

		if ( ! empty( $schema['properties']['progression'] ) && in_array( $context, $schema['properties']['progression']['context'], true ) ) {
			$post->progression   = $this->get_progression( $post );
			$data['progression'] = (int) $post->progression;
		}

		if ( ! empty( $schema['properties']['is_closed'] ) && in_array( $context, $schema['properties']['is_closed']['context'], true ) ) {
			$post->is_closed   = $this->is_closed( $post );
			$data['is_closed'] = (bool) $post->is_closed;
		}

		if ( ! empty( $schema['properties']['can_enroll'] ) && in_array( $context, $schema['properties']['can_enroll']['context'], true ) ) {
			$post->can_enroll   = $this->user_can_enroll( $post );
			$data['can_enroll'] = $post->can_enroll;
		}

		if ( ! empty( $schema['properties']['points'] ) && in_array( $context, $schema['properties']['points']['context'], true ) ) {
			$post->points   = $this->get_points( $post );
			$data['points'] = (int) $post->points;
		}

		if ( ! empty( $schema['properties']['duration'] ) && in_array( $context, $schema['properties']['duration']['context'], true ) ) {
			$data['duration']        = array();
			$data['duration']['min'] = (int) ( is_array( $post->duration ) && isset( $post->duration['min'] ) ) ? $post->duration['min'] : 0;
		}

		if ( ! empty( $schema['properties']['categories'] ) && in_array( $context, $schema['properties']['categories']['context'], true ) ) {
			$post->categories   = $this->get_categories( $post );
			$data['categories'] = $post->categories;
		}

		if ( ! empty( $schema['properties']['tags'] ) && in_array( $context, $schema['properties']['tags']['context'], true ) ) {
			$post->tags   = $this->get_tags( $post );
			$data['tags'] = $post->tags;
		}

		if ( ! empty( $schema['properties']['enrolled_members'] ) && in_array( $context, $schema['properties']['enrolled_members']['context'], true ) ) {
			$course_members           = CoursesMembersRest::instance();
			$total_enrolled           = $course_members->get_access_list( $post );
			$data['enrolled_members'] = (int) count( $total_enrolled );
		}

		if ( ! empty( $schema['properties']['certificate'] ) && in_array( $context, $schema['properties']['certificate']['context'], true ) ) {
			$certificate_id        = bbapp_learndash_get_course_meta_setting( $post->ID, 'certificate' );
			$certificate_available = ( ! empty( $certificate_id ) );
			$certificate_link      = null;
			$certificate_name      = '';

			if ( $certificate_available && $this->is_completed( $post ) ) {
				$certificate_link = learndash_get_course_certificate_link( $post->ID, get_current_user_id() );
				$certificate_link = ! $certificate_link ? null : $certificate_link; // nullify.
				$certificate_name = CertificateRest::get_certificate_name( $post->ID );
			}

			$data['certificate'] = array(
				'available' => $certificate_available,
				'link'      => $certificate_link,
				'filename'  => $certificate_name,
			);
		}

		if ( ! empty( $schema['properties']['module'] ) && in_array( $context, $schema['properties']['module']['context'], true ) ) {
			if ( isset( $post->module ) ) {
				$data['module'] = $post->module;
			} else {
				$data['module'] = array();
			}
		}

		if ( ! empty( $schema['properties']['video'] ) && in_array( $context, $schema['properties']['video']['context'], true ) ) {
			if ( ! $post->has_content_access ) {
				$post->video = '';
			}
			$data['video'] = $post->video;
		}

		if ( ! empty( $schema['properties']['group'] ) && in_array( $context, $schema['properties']['group']['context'], true ) ) {
			$data['group'] = (int) $post->group;
		}

		if ( ! empty( $schema['properties']['forum'] ) && in_array( $context, $schema['properties']['forum']['context'], true ) ) {
			$data['forum'] = (int) $post->forum;
		}

		if ( ! empty( $schema['properties']['lessons'] ) && in_array( $context, $schema['properties']['lessons']['context'], true ) ) {
			$post->lessons   = $this->get_lessons( $post );
			$data['lessons'] = $post->lessons;
		}

		if ( ! empty( $schema['properties']['quizzes'] ) && in_array( $context, $schema['properties']['quizzes']['context'], true ) ) {
			$post->quizzes   = $this->get_quizzes( $post );
			$data['quizzes'] = $post->quizzes;
		}

		if ( ! empty( $schema['properties']['completed'] ) && in_array( $context, $schema['properties']['completed']['context'], true ) ) {
			$post->completed   = $this->is_completed( $post );
			$data['completed'] = (bool) $post->completed;
		}

		if ( ! empty( $schema['properties']['quiz_completed'] ) && in_array( $context, $schema['properties']['quiz_completed']['context'], true ) ) {
			$post->quiz_completed   = $this->is_quiz_completed( $post, $post->ID );
			$data['quiz_completed'] = (bool) $post->quiz_completed;
		}

		if ( ! empty( $schema['properties']['show_start'] ) && in_array( $context, $schema['properties']['show_start']['context'], true ) ) {
			$post->show_start   = $this->is_progress_start( $post );
			$data['show_start'] = (bool) $post->show_start;
		}

		if ( ! empty( $schema['properties']['course_status'] ) && in_array( $context, $schema['properties']['course_status']['context'], true ) ) {
			$post->course_status   = $this->get_course_status( $post );
			$data['course_status'] = $post->course_status;
		}

		if ( ! empty( $schema['properties']['course_completion_page'] ) && in_array( $context, $schema['properties']['course_completion_page']['context'], true ) ) {
			$data['course_completion_page'] = ( function_exists( 'learndash_course_get_completion_url' ) ) ? learndash_course_get_completion_url( $post->ID ) : '';
		}

		if ( ! empty( $schema['properties']['error_message'] ) && in_array( $context, $schema['properties']['error_message']['context'], true ) ) {
			$post->error_message = $this->get_error_message( $post );

			if ( ! empty( $post->error_message ) ) {
				$error_code            = $post->error_message->get_error_code();
				$data['error_message'] = array(
					'code'    => $error_code,
					'message' => $post->error_message->get_error_message(),
					'data'    => $post->error_message->get_error_data( $error_code ),
				);
			} else {
				$data['error_message'] = array();
			}
		}

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filters course response.
		 *
		 * @param WP_REST_Response $response Rest response.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 */
		return apply_filters( 'bbapp_ld_rest_prepare_course', $response, $post, $request );
	}

	/**
	 * Remove excerpt More link as it not need and added dots.
	 *
	 * @return string
	 */
	public function remove_excerpt_more_link_and_add_dots() {
		return '...';
	}

	/**
	 * Prepare items.
	 *
	 * @param array $prepared_args Prepare item parameters.
	 * @param null  $request       Request parameters.
	 *
	 * @return array
	 */
	protected function prepare_items_query( $prepared_args = array(), $request = null ) {
		$query_args = array();

		foreach ( $prepared_args as $key => $value ) {
			/**
			 * Filters the query_vars used in get_items() for the constructed query.
			 *
			 * The dynamic portion of the hook name, `$key`, refers to the query_var key.
			 *
			 * @param string $value The query_var value.
			 *
			 * @since 4.7.0
			 */
			$query_args[ $key ] = apply_filters( "rest_query_var-{$key}", $value ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		}

		$query_args['ignore_sticky_posts'] = true;

		// Map to proper WP_Query orderby param.
		if ( isset( $query_args['orderby'] ) && isset( $request['orderby'] ) ) {
			$orderby_mappings = array(
				'id'            => 'ID',
				'include'       => 'post__in',
				'slug'          => 'post_name',
				'include_slugs' => 'post_name__in',
			);

			if ( isset( $orderby_mappings[ $request['orderby'] ] ) ) {
				$query_args['orderby'] = $orderby_mappings[ $request['orderby'] ];
			}

			if ( is_user_logged_in() ) {
				if ( 'my_progress' === $query_args['orderby'] ) {
					$user_id                  = get_current_user_id();
					$this->my_course_progress = $this->get_courses_progress( $user_id, $query_args['order'] );
					$query_args['order']      = 'desc';
					add_filter( 'posts_clauses', array( $this, 'alter_query_parts_for_my_progress' ), 10, 2 );
				}
			}
		}

		return $query_args;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array $data item object.
	 *
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {
		$links = parent::prepare_links( $data );

		/**
		 * Quiz links
		 */
		if ( isset( $data['quizzes'] ) ) {
			$links['quizzes'] = array();

			foreach ( $data['quizzes'] as $quiz ) {
				$links['quizzes'][] = array(
					'href'       => rest_url( $this->namespace . '/quiz/' . $quiz ),
					'embeddable' => true,
				);
			}
		}

		/**
		 * Lesson link
		 */
		if ( isset( $data['lessons'] ) ) {
			$links['lessons'] = array(
				'href'       => rest_url( $this->namespace ) . '/lessons?course_id=' . $data['id'],
				'embeddable' => false,
			);
		}

		/**
		 * Categories Link
		 */
		if ( ! empty( $data['categories'] ) ) {
			$params = array();

			foreach ( $data['categories'] as $catid ) {
				$params[] = 'include[]=' . $catid;
			}
			$links['categories'] = array(
				'href'       => rest_url( $this->namespace ) . '/course-categories?' . implode( '&', $params ),
				'embeddable' => true,
			);
		}

		/**
		 * Author links.
		 */
		if ( defined( 'BP_PLATFORM_VERSION' ) && function_exists( 'bp_is_active' ) && bp_is_active( 'members' ) ) {
			if ( isset( $data['author'] ) ) {
				$links['members'] = array(
					'href'       => rest_url( '/buddyboss/v1/members/' . $data['author'] ),
					'embeddable' => true,
				);
			}
		}

		return $links;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author'] = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset']            = array(
			'description'       => __( 'Offset the result set by a specific number of items.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
				'my_progress',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']['enum'][] = 'menu_order';

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent IDs.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent ID.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);

		$params['slug']   = array(
			'description'       => __( 'Limit result set to posts with a specific slug.', 'buddyboss-app' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['filter'] = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.', 'buddyboss-app' ),
		);

		$params = $this->prepare_taxonomy_limit_schema( $params );

		$params['mycourses'] = array(
			'description'       => __( 'Limit response to resources which is taken by current user.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['group_id'] = array(
			'description'       => __( 'Limit response to resources that are connected with a group.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'course',
			'type'       => 'object',
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'          => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'single', 'edit', 'embed' ),
						),
					),
				),
				'content'        => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'single', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'single', 'edit' ),
						),
						'short'    => array(
							'description' => __( 'Short content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'single', 'edit' ),
						),
					),
				),
				'date'           => array(
					'description' => __( 'The date the object was published, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
				),
				'date_gmt'       => array(
					'description' => __( 'The date the object was published, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
				),
				'modified'       => array(
					'description' => __( 'The date the object was last modified, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'single', 'edit' ),
					'readonly'    => true,
				),
				'modified_gmt'   => array(
					'description' => __( 'The date the object was last modified, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'single', 'edit' ),
					'readonly'    => true,
				),
				'link'           => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'           => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'single', 'edit' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'         => array(
					'description' => __( 'The ID for the author of the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'single', 'edit' ),
				),
				'excerpt'        => array(
					'description' => __( 'The excerpt for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'single', 'edit', 'embed' ),
						),
					),
				),
				'featured_media' => array(
					'description' => __( 'Feature media object containing thumb and full URL of the image.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
				),
				'cover_media'    => array(
					'description' => __( 'Cover media object containing thumb and full URL of the image.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
				),
				'menu_order'     => array(
					'description' => __( 'The order of the object in relation to other object of its type.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'single', 'edit' ),
				),
			),
		);

		$schema['properties']['categories'] = array(
			'description' => __( 'The terms assigned to the object in the categories taxonomy.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['enrolled_members'] = array(
			'description' => __( 'Total count of enrolled member in course.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['certificate'] = array(
			'description' => __( 'Certificate Details', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'view', 'single', 'edit' ),
			'properties'  => array(
				'properties' => array(
					'available'     => array(
						'description' => __( 'Tells if a certificate is available with this course.', 'buddyboss-app' ),
						'type'        => 'boolean',
						'context'     => array( 'view', 'single', 'edit' ),
					),
					'download_link' => array(
						'description' => __( 'Download link to the certificate if available.', 'buddyboss-app' ),
						'type'        => 'string',
						'context'     => array( 'view', 'single', 'edit' ),
					),
				),
			),
		);

		$schema['properties']['tags'] = array(
			'description' => __( 'The terms assigned to the object in the tags taxonomy.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['module'] = array(
			'description' => __( 'The terms assigned to the object in the module taxonomy.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['featured'] = array(
			'description' => __( 'Whether or not the object should be treated as featured.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['materials'] = array(
			'description' => __( 'Materials for the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['purchasable'] = array(
			'description' => __( 'Whether or not object is purchasable', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['offline_disabled'] = array(
			'description' => __( 'Disable offline support', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['price'] = array(
			'description' => __( 'The price of the object', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'single', 'edit' ),
			'properties'  => array(
				'properties' => array(
					'value'    => array(
						'description' => __( 'value for the object, as it exists in the database.', 'buddyboss-app' ),
						'type'        => 'integer',
						'context'     => array( 'single', 'edit' ),
					),
					'rendered' => array(
						'description' => __( 'Rendered value for object.', 'buddyboss-app' ),
						'type'        => 'string',
						'context'     => array( 'single', 'edit' ),
					),
					'code'     => array(
						'description' => __( 'currency code for object.', 'buddyboss-app' ),
						'type'        => 'string',
						'context'     => array( 'view' ),
					),
				),
			),
		);

		$schema['properties']['video'] = array(
			'description' => __( 'video for the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['group'] = array(
			'description' => __( 'BuddyPress group for the object.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['forum'] = array(
			'description' => __( 'bbpress forum for the object.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['is_closed']  = array(
			'description' => __( 'Whether object is closed or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit' ),
		);
		$schema['properties']['can_enroll'] = array(
			'description' => __( 'Whether object is return user can enrolled.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['has_course_access'] = array(
			'description' => __( 'Whether or not user have the object access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['has_content_access'] = array(
			'description' => __( 'Whether or not user have the object content access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['hide_content_table'] = array(
			'description' => __( 'Whether or not user have the access of object content access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['progression'] = array(
			'description' => __( 'The progression % of user for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['points'] = array(
			'description' => __( 'Point for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['duration'] = array(
			'description' => __( 'The time duration for the object', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'single', 'edit' ),
			'properties'  => array(
				'properties' => array(
					'min' => array(
						'description' => __( 'Minimum required time for this object.', 'buddyboss-app' ),
						'type'        => 'integer',
						'context'     => array( 'single', 'edit' ),
					),
				),
			),
		);

		$schema['properties']['quizzes'] = array(
			'description' => __( 'The quizzes list which has access of the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['lessons'] = array(
			'description' => __( 'The lessons list which has access of the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['completed'] = array(
			'description' => __( 'The object is completed by current user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['quiz_completed'] = array(
			'description' => __( 'The object Quizzes is completed by current user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['show_start'] = array(
			'description' => __( 'Whether or not user has started the process.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['course_status'] = array(
			'description' => __( 'The status of the course for the user.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['course_completion_page'] = array(
			'description' => __( 'The status of the course for the user.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'single' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get the course medias.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return array
	 */
	private function get_feature_media( $post ) {
		$return = array(
			'large' => '',
			'small' => '',
		);
		$large  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'large' );
		$small  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'medium' );

		if ( isset( $large[0] ) ) {
			$return['large'] = $large[0];
		}
		if ( isset( $small[0] ) ) {
			$return['small'] = $small[0];
		}

		return $return;
	}

	/**
	 * Get the course cover media.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return array
	 */
	private function get_cover_media( $post ) {
		$return = array(
			'large' => '',
			'small' => '',
		);

		$course_cover_photo_id = 0;

		/**
		 * BuddyBoss Theme Feature Image Support.
		 */
		if ( class_exists( '\BuddyBossTheme\BuddyBossMultiPostThumbnails' ) ) {
			$course_cover_photo_id = \BuddyBossTheme\BuddyBossMultiPostThumbnails::get_post_thumbnail_id( 'sfwd-courses', 'course-cover-image', $post->ID );
		} elseif ( class_exists( 'BuddyBossApp\Helpers\PostCover' ) ) { // Fallback Feature Image Support.
			$course_cover_photo_id = PostCover::get_post_thumbnail_id(
				'sfwd-courses',
				'course-cover-image',
				$post->ID
			);
		}

		if ( ! empty( $course_cover_photo_id ) ) {
			$large = wp_get_attachment_image_src( $course_cover_photo_id, 'large' );
			$small = wp_get_attachment_image_src( $course_cover_photo_id, 'medium' );

			if ( isset( $large[0] ) ) {
				$return['large'] = $large[0];
			}

			if ( isset( $small[0] ) ) {
				$return['small'] = $small[0];
			}
		}

		return $return;
	}

	/**
	 * Get the course materials.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return mixed
	 */
	private function get_materials( $post ) {
		$course_materials = bbapp_learndash_get_course_meta_setting( $post->ID, 'course_materials' );
		$course_materials = wp_kses_post( wp_specialchars_decode( $course_materials, ENT_QUOTES ) );

		if ( ! empty( $course_materials ) ) {
			$course_materials = do_shortcode( $course_materials );
		};

		return $course_materials;
	}

	/**
	 * If the course purchasable.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return bool
	 */
	private function is_purchasable( $post ) {
		$type  = bbapp_learndash_get_course_meta_setting( $post->ID, 'course_price_type' );
		$price = bbapp_learndash_get_course_meta_setting( $post->ID, 'course_price' );

		switch ( $type ) {
			case 'subscribe':
			case 'paynow':
			case 'closed':
				$is_paid = empty( $price ) ? false : true;
				break;
			default:
				$is_paid = false;
		}

		return $is_paid;
	}

	/**
	 * If the course closed.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return bool
	 */
	private function is_closed( $post ) {
		$type      = bbapp_learndash_get_course_meta_setting( $post->ID, 'course_price_type' );
		$is_closed = false;

		if ( 'closed' === $type ) {
			$is_closed = true;
		}

		return $is_closed;
	}

	/**
	 * User can enrolled or not.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return bool
	 */
	private function user_can_enroll( $post ) {
		$course_pricing = learndash_get_course_price( $post->ID );
		$has_access     = sfwd_lms_has_access( $post->ID, get_current_user_id() );

		if ( is_user_logged_in() && isset( $has_access ) && $has_access ) {
			return false;
		} elseif ( ! is_user_logged_in() && 'open' !== $course_pricing['type'] ) {
			return true;
		} elseif ( ! is_user_logged_in() && 'open' === $course_pricing['type'] ) {
			return false;
		} elseif ( 'open' !== $course_pricing['type'] ) {
			return true;
		}
	}


	/**
	 * Get the course price.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return array
	 */
	private function get_price( $post ) {
		$paypal_settings = LearnDash_Settings_Section::get_section_settings_all( 'LearnDash_Settings_Section_PayPal' );
		$price           = bbapp_learndash_get_course_meta_setting( $post->ID, 'course_price' );
		$price           = filter_var( $price, FILTER_SANITIZE_NUMBER_FLOAT );

		return array(
			'value' => $price,
			'code'  => ! empty( $paypal_settings['paypal_currency'] ) ? $paypal_settings['paypal_currency'] : '',
		);
	}

	/**
	 * Get the course access.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return mixed
	 */
	private function get_has_course_access( $post ) {
		return sfwd_lms_has_access( $post->ID );
	}

	/**
	 * Offline course disable.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return mixed
	 */
	private function is_offline_disabled( $post ) {
		$settings           = Settings::instance()->get_settings();
		$course_downloading = isset( $settings['learndash_course_downloading'] ) ? $settings['learndash_course_downloading'] : 0;

		if ( isset( $course_downloading ) && $course_downloading ) {
			$_is_disable_offline_support = get_post_meta( $post->ID, '_is_disable_offline_support', true );

			if ( 'yes' === $_is_disable_offline_support ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Get course content access.
	 *
	 * @param WP_Post $post    Course post.
	 * @param bool    $message Message.
	 *
	 * @return bool
	 */
	private function get_has_content_access( $post, $message = false ) {
		return bbapp_lms_is_content_access( $post, 'prerequities_completed' ) && bbapp_lms_is_content_access( $post, 'points_access' );
	}

	/**
	 * Get the hidden content table.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return bool
	 */
	private function get_hide_content_table( $post ) {
		if ( sfwd_lms_has_access( $post->ID ) ) {
			return false;
		}

		return 'on' === bbapp_learndash_get_course_meta_setting( $post->ID, 'course_disable_content_table' );
	}

	/**
	 * Get the course progress.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return mixed
	 */
	private function get_progression( $post ) {
		$progress_percentage = 0;

		if ( is_user_logged_in() ) {
			$user_id             = get_current_user_id();
			$progress            = learndash_course_progress(
				array(
					'user_id'   => $user_id,
					'course_id' => $post->ID,
					'array'     => true,
				)
			);
			$progress_percentage = $progress['percentage'];
		}

		return $progress_percentage;
	}

	/**
	 * Get the course points.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return mixed
	 */
	private function get_points( $post ) {
		return bbapp_learndash_get_course_meta_setting( $post->ID, 'course_points' );
	}

	/**
	 * Get the course categories.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return array
	 */
	private function get_categories( $post ) {
		if ( 'yes' === LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Taxonomies', 'ld_course_category' ) ) {
			$terms = get_the_terms( $post, 'ld_course_category' );

			return $terms && ! is_wp_error( $terms ) ? wp_list_pluck( $terms, 'term_id' ) : array();
		}

		return array();
	}

	/**
	 * Get the course tags.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return array
	 */
	private function get_tags( $post ) {
		if ( 'yes' === LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Taxonomies', 'ld_course_tag' ) ) {
			$terms = get_the_terms( $post, 'ld_course_tag' );

			return $terms && ! is_wp_error( $terms ) ? wp_list_pluck( $terms, 'term_id' ) : array();
		}

		return array();
	}

	/**
	 * Get the course quizzes.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return array
	 */
	private function get_quizzes( $post ) {
		$quizes = learndash_get_course_quiz_list( $post );
		$posts  = array();

		foreach ( $quizes as $quiz ) {
			$posts[] = $quiz['post']->ID;
		}

		return $posts;
	}

	/**
	 * Get course lessons.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return array
	 */
	private function get_lessons( $post ) {
		$posts = array();

		/**
		 * This function will get lessons based on course steps which is stored in meta.
		 * Both functions will run based on learndash version.
		 */
		if ( function_exists( 'learndash_course_get_lessons' ) ) {
			$lessons = learndash_course_get_lessons( $post->ID, array( 'per_page' => 0 ) );
		} else {
			$lessons = learndash_get_lesson_list( $post->ID, array( 'num' => - 1 ) );
		}

		if ( ! empty( $lessons ) ) {
			foreach ( $lessons as $lesson ) {
				$posts[] = $lesson->ID;
			}
		}

		$sections_lessons = array();

		if ( function_exists( 'learndash_30_get_course_sections' ) ) {
			$sections = learndash_30_get_course_sections( $post->ID );

			if ( isset( $sections ) && ! empty( $sections ) ) {
				$count = 0;

				foreach ( $posts as $lesson ) {
					if ( isset( $sections[ $lesson ] ) ) {
						$sections_lessons[ ++ $count ]['section_title'] = $sections[ $lesson ]->post_title;
					}

					$sections_lessons[ $count ]['lessons_list'][] = $lesson;
				}
			}

			/**
			 * Support case when first lesson doesn't have any section
			 */
			if ( isset( $sections_lessons[0] ) && ! isset( $sections_lessons[0]['section_title'] ) ) {
				$sections_lessons[0]['section_title'] = '';
			}
		}

		if ( empty( $sections_lessons ) && ! empty( $posts ) ) {
			$sections_lessons[0]['section_title'] = '';
			$sections_lessons[0]['lessons_list']  = $posts;
		}

		return array_values( $sections_lessons );
	}

	/**
	 * If the course is completed.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return mixed
	 */
	private function is_completed( $post ) {
		return learndash_course_completed( null, $post->ID );
	}

	/**
	 * If the course quiz is completed.
	 *
	 * @param WP_Post $post      Course post.
	 * @param int     $course_id Course id.
	 *
	 * @return bool
	 */
	private function is_quiz_completed( $post, $course_id = 0 ) {
		return bbapp_lms_quiz_completed( $post, $course_id );
	}

	/**
	 * Error message.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return array
	 */
	private function get_error_message( $post ) {
		$user_id = get_current_user_id();

		if ( ! bbapp_lms_is_content_access( $post, 'prerequities_completed' ) ) {
			$prerequisite_posts_all = learndash_get_course_prerequisites( $post->ID );
			$course_str             = '';

			if ( ! empty( $prerequisite_posts_all ) ) {
				foreach ( $prerequisite_posts_all as $pre_post_id => $pre_status ) {
					if ( false === $pre_status ) {
						if ( ! empty( $course_str ) ) {
							$course_str .= ', ';
						}

						$course_str .= get_the_title( $pre_post_id );
					}
				}
			}

			return CoursesError::instance()->invalid_course_prerequities( $course_str );
		}

		if ( ! bbapp_lms_is_content_access( $post, 'points_access' ) ) {
			$course_access_points = learndash_get_course_points_access( $post->ID );
			$user_course_points   = learndash_get_user_course_points( $user_id );

			return CoursesError::instance()->invalid_course_access_point( $course_access_points, $user_course_points );
		}

		if ( ! sfwd_lms_has_access( $post->ID, $user_id ) ) {
			return CoursesError::instance()->invalid_course_access();
		}

		return array();
	}

	/**
	 * Get Current user progress for sorting course
	 * Ref: buddyboss-theme/blob/dev/inc/plugins/learndash-helper.php
	 *
	 * @param int    $user_id    User id.
	 * @param string $sort_order Sorting order.
	 *
	 * @return array|bool|mixed|string
	 */
	private function get_courses_progress( $user_id, $sort_order = 'desc' ) {
		$course_completion_percentage = array();
		$course_completion_percentage = wp_cache_get( $user_id, 'ld_courses_progress' );

		if ( ! $course_completion_percentage ) {
			$course_progress = get_user_meta( $user_id, '_sfwd-course_progress', true );

			if ( ! empty( $course_progress ) && is_array( $course_progress ) ) {
				foreach ( $course_progress as $course_id => $coursep ) {
					// We take default progress value as 1 % rather than 0%.
					$course_completion_percentage[ $course_id ] = 1;

					if ( 0 === (int) $coursep['total'] ) {
						continue;
					}

					$course_steps_count     = learndash_get_course_steps_count( $course_id );
					$course_steps_completed = learndash_course_get_completed_steps( $user_id, $course_id, $coursep );
					$completed_on           = get_user_meta( $user_id, 'course_completed_' . $course_id, true );

					if ( ! empty( $completed_on ) ) {
						$coursep['completed'] = $course_steps_count;
						$coursep['total']     = $course_steps_count;
					} else {
						$coursep['total']     = $course_steps_count;
						$coursep['completed'] = $course_steps_completed;

						if ( $coursep['completed'] > $coursep['total'] ) {
							$coursep['completed'] = $coursep['total'];
						}
					}

					// Cannot divide by 0.
					if ( 0 === (int) $coursep['total'] ) {
						$course_completion_percentage[ $course_id ] = 0;
					} else {
						$course_completion_percentage[ $course_id ] = ceil( ( $coursep['completed'] * 100 ) / $coursep['total'] );
					}
				}
			}

			// Avoid running the queries multiple times if user's course progress is empty.
			$course_completion_percentage = ! empty( $course_completion_percentage ) ? $course_completion_percentage : 'empty';

			wp_cache_set( $user_id, $course_completion_percentage, 'ld_courses_progress' );
		}

		$course_completion_percentage = 'empty' !== $course_completion_percentage ? $course_completion_percentage : array();

		if ( ! empty( $course_completion_percentage ) ) {
			// Sort.
			if ( 'asc' === $sort_order ) {
				asort( $course_completion_percentage );
			} else {
				arsort( $course_completion_percentage );
			}
		}

		return $course_completion_percentage;
	}

	/**
	 * Alter Query for support my_progress sorting filter
	 * Ref: buddyboss-theme/blob/dev/inc/plugins/learndash-helper.php
	 *
	 * @param array  $clauses Extra query clause.
	 * @param string $query   Query.
	 *
	 * @return mixed
	 */
	public function alter_query_parts_for_my_progress( $clauses, $query ) {
		remove_filter( 'posts_clauses', array( $this, 'alter_query_parts' ), 10, 2 );

		$my_course_progress = $this->my_course_progress;

		if ( ! empty( $my_course_progress ) ) {
			$clauses['fields'] .= ', CASE ';

			global $wpdb;

			$id_colum_name = $wpdb->posts . '.ID';

			foreach ( $my_course_progress as $course_id => $progress ) {
				$clauses['fields'] .= ' WHEN ' . $id_colum_name . ' = ' . $course_id . ' THEN ' . $progress . ' ';
			}

			$clauses['fields']         .= ' ELSE 0 END AS my_progress ';
			$query->query_vars['order'] = ! empty( $query->query_vars['order'] ) ? $query->query_vars['order'] : 'DESC';
			$clauses['orderby']         = 'my_progress ' . $query->query_vars['order'] . ', ' . $clauses['orderby'];
		}

		return $clauses;
	}

	/**
	 * Current user started this course or not.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return bool
	 */
	private function is_progress_start( $post ) {
		$show_start = true;

		if ( is_user_logged_in() ) {
			$course_progress = get_user_meta( get_current_user_id(), '_sfwd-course_progress', true );

			if ( ( isset( $course_progress ) && ! isset( $course_progress[ $post->ID ] ) ) || 0 === (int) $course_progress[ $post->ID ]['completed'] ) {
				$show_start = true;
			}
		}

		return $show_start;
	}

	/**
	 * Get Course status.
	 *
	 * @param WP_Post $post Course post.
	 *
	 * @return array course status.
	 * @since 1.8.50
	 */
	public function get_course_status( $post ) {

		if ( false === (bool) $post->can_enroll ) {
			return array();
		}

		return bbapp_ld_get_course_status( $post );

	}

	/**
	 * Prepares the collection schema for including and excluding items by LD terms.
	 *
	 * @since 2.0.00
	 *
	 * @param array $query_params Collection schema.
	 * @return array Updated schema.
	 */
	private function prepare_taxonomy_limit_schema( $query_params ) {
		$taxonomies = wp_list_filter( get_object_taxonomies( $this->post_type, 'objects' ), array( 'show_in_rest' => true ) );

		if ( ! $taxonomies ) {
			return $query_params;
		}

		$query_params['tax_relation'] = array(
			'description' => __( 'Limit result set based on relationship between multiple taxonomies.', 'buddyboss-app' ),
			'type'        => 'string',
			'enum'        => array( 'AND', 'OR' ),
		);

		$limit_schema = array(
			'type'  => array( 'object', 'array' ),
			'oneOf' => array(
				array(
					'title'       => __( 'Term ID List', 'buddyboss-app' ),
					'description' => __( 'Match terms with the listed IDs.', 'buddyboss-app' ),
					'type'        => 'array',
					'items'       => array(
						'type' => 'integer',
					),
				),
				array(
					'title'                => __( 'Term ID Taxonomy Query', 'buddyboss-app' ),
					'description'          => __( 'Perform an advanced term query.', 'buddyboss-app' ),
					'type'                 => 'object',
					'properties'           => array(
						'terms' => array(
							'description' => __( 'Term IDs.', 'buddyboss-app' ),
							'type'        => 'array',
							'items'       => array(
								'type' => 'integer',
							),
							'default'     => array(),
						),
					),
					'additionalProperties' => false,
				),
			),
		);

		$include_schema = array_merge(
			array(
				/* translators: %s: Taxonomy name. */
				'description' => __( 'Limit result set to items with specific terms assigned in the LD %s taxonomy.', 'buddyboss-app' ),
			),
			$limit_schema
		);
		// 'operator' is supported only for 'include' queries.
		$include_schema['oneOf'][1]['properties']['operator'] = array(
			'description' => __( 'Whether items must be assigned all or any of the specified terms.', 'buddyboss-app' ),
			'type'        => 'string',
			'enum'        => array( 'AND', 'OR' ),
			'default'     => 'OR',
		);

		$exclude_schema = array_merge(
			array(
				/* translators: %s: Taxonomy name. */
				'description' => __( 'Limit result set to items except those with specific terms assigned in the LD %s taxonomy.', 'buddyboss-app' ),
			),
			$limit_schema
		);

		$supported_taxonomies = array(
			'categories', // Ld course category.
			'tags', // Ld course tags.
		);

		foreach ( $supported_taxonomies as $supported_taxonomy ) {
			$base         = $supported_taxonomy;
			$base_exclude = $base . '_exclude';

			$query_params[ $base ]                = $include_schema;
			$query_params[ $base ]['description'] = sprintf( $query_params[ $base ]['description'], $base );

			$query_params[ $base_exclude ]                = $exclude_schema;
			$query_params[ $base_exclude ]['description'] = sprintf( $query_params[ $base_exclude ]['description'], $base );
		}

		return $query_params;
	}
}