<?php
/**
 * Holds course members rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Course
 */

namespace BuddyBossApp\Api\LearnDash\V1\Course;

use BuddyBossApp\Api\LearnDash\V1\Core\LDUserController;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Course member rest class.
 */
class CoursesMembersRest extends LDUserController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Course post type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'sfwd-courses';

	/**
	 * CoursesMembersRest instance.
	 *
	 * @return CoursesMembersRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		parent::__construct();

		$this->rest_base = 'courses';
	}

	/**
	 * Check if a given request has access to course item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the course `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_course_memebers_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<id>\d+)/members',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_course_members' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve Course Members.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/courses/:id/members Get LearnDash Course Members
	 * @apiName        GetLDCourseMembers
	 * @apiGroup       LD Courses
	 * @apiDescription Retrieve single Courses Members
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=3] Maximum number of items to be returned in result set.
	 */
	public function get_course_members( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course    = get_post( $course_id );

		if ( empty( $course_id ) || $this->post_type !== $course->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		$registered         = $this->get_collection_params();
		$parameter_mappings = array(
			'per_page' => 'per_page',
			'page'     => 'page',
		);

		/**
		 * For each known parameter which is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		$access_list = $this->get_access_list( $course );

		// Items to Skip.
		$item_skip = ( $args['page'] - 1 ) * $args['per_page'];
		$retval    = array();

		if ( ! empty( $access_list ) ) {
			$i = 0;

			foreach ( $access_list as $user_id ) {
				// Skip the entries till item skip is empty.
				if ( ! empty( $item_skip ) ) {
					$item_skip --;
					continue;
				}

				$user = get_userdata( (int) $user_id );

				if ( empty( $user ) || ! $user->exists() ) {
					continue;
				}

				if ( is_multisite() && ! is_user_member_of_blog( $user->ID ) ) {
					continue;
				}

				if ( ! is_wp_error( $user ) ) {
					$user = $this->get_user( $user_id );
					$user = $this->prepare_item_for_response( $user, $request );

					if ( ! empty( $user->data ) ) {
						$retval[] = $user->data;
						$i ++;
					}
				}

				if ( count( $retval ) >= $args['per_page'] ) {
					break;
				}
			}
		}

		$response = rest_ensure_response( $retval );
		$response = bbapp_learners_response_add_total_headers( $response, count( $access_list ), $args['per_page'] );

		return $response;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		return array(
			'context'  => $this->get_context_param( array( 'default' => 'view' ) ),
			'per_page' => array(
				'description'       => __( 'Maximum number of items to be returned in result set.', 'buddyboss-app' ),
				'type'              => 'integer',
				'default'           => 10,
				'minimum'           => 1,
				'maximum'           => 100,
				'sanitize_callback' => 'absint',
				'validate_callback' => 'rest_validate_request_arg',
			),
			'page'     => array(
				'description'       => __( 'Number of pages you want to retrieve items for.', 'buddyboss-app' ),
				'type'              => 'integer',
				'default'           => 1,
				'minimum'           => 1,
				'maximum'           => 100000,
				'sanitize_callback' => 'absint',
				'validate_callback' => 'rest_validate_request_arg',
			),
		);
	}

	/**
	 * Course access list (enrolled user list will return).
	 *
	 * @param \WP_Post $post Course object.
	 *
	 * @return array
	 */
	public function get_access_list( $post ) {
		$access_list = get_transient( 'buddyboss_app_ld_course_enrolled_users_' . $post->ID );

		// If nothing is found, build the object.
		if ( false === $access_list ) {
			$members_arr = learndash_get_users_for_course( $post->ID, array(), false );

			if ( ( $members_arr instanceof \WP_User_Query ) && ( property_exists( $members_arr, 'results' ) ) && ( ! empty( $members_arr->results ) ) ) {
				$access_list = $members_arr->get_results();
				set_transient( 'buddyboss_app_ld_course_enrolled_users_' . $post->ID, $access_list, 3 * HOUR_IN_SECONDS );
			} else {
				$access_list = array();
			}
		}

		return $access_list;
	}
}
