<?php
/**
 * Holds LearnDash course errors messages.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Course
 */

namespace BuddyBossApp\Api\LearnDash\V1\Course;

use BuddyBossApp\RestErrors;
use LearnDash_Custom_Label;
use WP_Error;

/**
 * LearnDash course errors class.
 */
class CoursesError {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return CoursesError
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * When user is not logged in this error message should be used.
	 *
	 * @return WP_Error
	 */
	public function user_not_logged_in() {
		return RestErrors::instance()->user_not_logged_in();
	}

	/**
	 * When Course id not correct.
	 *
	 * @return WP_Error
	 */
	public function invalid_course_id() {
		return new WP_Error( 'learndash_json_course_invalid_id', __( 'Course not found.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * When user don't have course access.
	 *
	 * @return WP_Error
	 */
	public function invalid_course_access() {
		return new WP_Error( 'learndash_json_course_invalid_access', __( 'You do not have access to this course.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user doesn't match required course access point.
	 *
	 * @param int $course_access_points Lesson access point that required for lesson access.
	 * @param int $user_course_points   USer current points.
	 *
	 * @return WP_Error
	 */
	public function invalid_course_access_point( $course_access_points = 0, $user_course_points = 0 ) {
		if ( ! empty( $course_access_points ) || ! empty( $user_course_points ) ) {
			/* translators: %1$s: Course lowercase label, %2$s: Course access points, %3$s: User course points. */
			$message = sprintf( __( 'To take this %1$s you need at least %2$.01f total points. You currently have %3$.01f points.', 'buddyboss-app' ), LearnDash_Custom_Label::label_to_lower( 'course' ), $course_access_points, $user_course_points );

			return new WP_Error( 'learndash_json_course_points_access', $message, array( 'status' => 400 ) );
		}

		return new WP_Error( 'learndash_json_course_points_access', __( 'Course required points do not match.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * When user doesn't match required course pre-requities.
	 *
	 * @param string $course_str Course list that required to access that lesson.
	 *
	 * @return WP_Error
	 */
	public function invalid_course_prerequities( $course_str = '' ) {

		if ( ! empty( $course_str ) ) {
			/* translators: %1$s: Lesson lowercase label, %2$s: Course lowercase label, %3$s: Course string.  */
			$message = sprintf( __( 'To take this %1$s, you need to complete the following %2$s first: %3$s', 'buddyboss-app' ), LearnDash_Custom_Label::label_to_lower( 'lesson' ), LearnDash_Custom_Label::label_to_lower( 'course' ), $course_str );

			return new WP_Error( 'learndash_json_course_prerequities', $message, array( 'status' => 400 ) );
		}

		return new WP_Error( 'learndash_json_course_prerequities', __( 'Course prerequisite is not completed.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * When user need to purchase course.
	 *
	 * @return WP_Error
	 */
	public function course_purchase_required() {
		return new WP_Error( 'learndash_json_course_purchase_required', __( 'You need to purchase this course before you can start.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * When Course zip is not created.
	 *
	 * @return WP_Error
	 */
	public function invalid_course_zip() {
		return new WP_Error( 'learndash_json_course_invalid_zip', __( 'Course Zip file not created.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * When Course Statement action or object is empty.
	 *
	 * @return WP_Error
	 */
	public function invalid_course_statement() {
		return new WP_Error( 'learndash_json_course_invalid_statement', __( 'Course statement action or object is not valid.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * When Course Category invalid.
	 *
	 * @return WP_Error
	 */
	public function invalid_course_catrgory_id() {
		return new WP_Error( 'learndash_json_course_Category_invalid', __( 'Course category not found.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}
}
