<?php
/**
 * Holds Courses action rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Course
 */

namespace BuddyBossApp\Api\LearnDash\V1\Course;

use BuddyBossApp\Api\LearnDash\V1\Lesson\LessonsError;
use BuddyBossApp\Api\LearnDash\V1\Topic\TopicsError;
use WP_Error;
use WP_REST_Response;
use \WP_REST_Server;
use \WP_REST_Request;

/**
 * Courses action rest.
 */
class CoursesActionRest extends CoursesRest {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * CoursesActionRest instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>\d+)/enroll',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'enroll' ),
					'permission_callback' => array( $this, 'get_enroll_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/download',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'download_item' ),
					'permission_callback' => array( $this, 'get_download_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/upload',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'upload_progress' ),
					'permission_callback' => array( $this, 'get_upload_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
			)
		);
	}

	/**
	 * Check if a given request has access to enroll course items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_enroll_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = CoursesError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the course `enroll` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_courses_enroll_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to download course items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_download_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = CoursesError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the course `download`  permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_courses_download_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to upload course items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_upload_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = CoursesError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the course `upload`  permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_courses_upload_permissions_check', $retval, $request );
	}

	/**
	 * Process Course enrollment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/courses/:id/enroll Process LearnDash Course Enrollment
	 * @apiName        PostLDCourseEnroll
	 * @apiGroup       LD Courses
	 * @apiDescription Process Course enrollment
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function enroll( $request ) {

		$retval = false;

		$user_id   = get_current_user_id();
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course    = get_post( $course_id );

		if ( empty( $course_id ) || $this->post_type !== $course->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		if ( ! bbapp_lms_is_content_access( $course, 'points_access' ) ) {
			return CoursesError::instance()->invalid_course_access_point();
		}

		if ( ! bbapp_lms_is_content_access( $course, 'prerequities_completed' ) ) {
			return CoursesError::instance()->invalid_course_prerequities();
		}

		$meta  = get_post_meta( $course_id, '_sfwd-courses', true );
		$price = (float) $meta['sfwd-courses_course_price'];
		if (
		( 'free' === $meta['sfwd-courses_course_price_type'] || 'open' === $meta['sfwd-courses_course_price_type'] // if price type is free or open.
		|| ( 'paynow' === $meta['sfwd-courses_course_price_type'] && empty( $price ) ) ) // or if price type paynow but price is 0.
		) {

			if ( ! sfwd_lms_has_access( $course_id, $user_id ) ) {
				ld_update_course_access( $user_id, $course_id );
			}

			$retval = true;
		} elseif ( sfwd_lms_has_access( $course_id, $user_id ) ) {
			$retval = true; // if course already purchase.
		}

		if ( true === $retval ) {
			return $this->get_item( $request );
		}

		return CoursesError::instance()->course_purchase_required();
	}

	/**
	 * Process Course Download.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/courses/:id/download Process LearnDash Course Download
	 * @apiName        PostLDCourseDownload
	 * @apiGroup       LD Courses
	 * @apiDescription Process Course Download
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function download_item( $request ) {

		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course    = get_post( $course_id );

		if ( empty( $course ) || $this->post_type !== $course->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		// Set course folder path.
		$folder_name = 'course_' . $course_id . '/';

		// Delete course directory if exist.
		$folder_path = bbapp_learners_getpath( $folder_name );
		bbapp_learners_delete_directory( $folder_path );

		// Create course directory.
		$folder_path = bbapp_learners_setpath( $folder_name );

		// Create assests directory for course.
		$assests_path = bbapp_learners_setpath( $folder_name ) . 'courses/';
		if ( ! file_exists( $assests_path ) ) {
			wp_mkdir_p( $assests_path );
		}

		$data = $this->prepare_item_for_response( $course, $request );

		// Download course html and assests.
		$html_attrs = apply_filters( 'bbapp_ld_course_html_attrs', array( 'content', 'materials' ) );

		foreach ( $html_attrs as $attr ) {
			if ( isset( $data->data[ $attr ]['rendered'] ) ) {
				$data->data[ $attr ]['rendered'] = bbapp_learners_download_html( $assests_path, $data->data[ $attr ]['rendered'] );
				$value                           = $data->data[ $attr ]['rendered'];
			} else {
				$data->data[ $attr ] = bbapp_learners_download_html( $assests_path, $data->data[ $attr ] );
				$value               = $data->data[ $attr ];
			}
			$file_name = 'course_' . $attr . '.html';
			bbapp_learners_write_file( $assests_path . $file_name, $value );
		}

		// Create course.json.
		$file_data = wp_json_encode( $data );
		$file_name = 'course_' . $course_id . '.json';
		bbapp_learners_write_file( $folder_path . $file_name, $file_data );

		// Create info.json.
		$info_data = wp_json_encode(
			array(
				'id'           => $course_id,
				'hash_key'     => sha1( $course->post_modified ),
				'modified'     => mysql_to_rfc3339( $course->post_modified ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
				'modified_gmt' => mysql_to_rfc3339( $course->post_modified_gmt ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			)
		);
		$file_name = 'info.json';
		bbapp_learners_write_file( $folder_path . $file_name, $info_data );

		// Create zip file.
		$zip_url = bbapp_learners_zip_create( $folder_path, 'course_' . $course_id );
		if ( empty( $zip_url ) ) {
			return CoursesError::instance()->invalid_course_zip();
		}

		return rest_ensure_response( array( 'zip_url' => $zip_url ) );
	}

	/**
	 * Process Course Statements.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/courses/:id/upload Process LearnDash Course Statements
	 * @apiName        PostLDCourseStatements
	 * @apiGroup       LD Courses
	 * @apiDescription Process Course Statements.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 * @apiParam {Array} [statements] statements data to perform action. action,object,item_id are required.  .
	 */
	public function upload_progress( $request ) {
		$retval    = array();
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$course    = get_post( $course_id );

		if ( empty( $course_id ) || $this->post_type !== $course->post_type ) {
			return CoursesError::instance()->invalid_course_id();
		}

		$temp_request = $_REQUEST; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$statements   = isset( $request['statements'] ) ? $request['statements'] : array();

		if ( ! empty( $statements ) ) {
			foreach ( $statements as $statement ) {
				$action      = isset( $statement['action'] ) ? $statement['action'] : '';
				$object      = isset( $statement['object'] ) ? $statement['object'] : '';
				$object_data = isset( $statement['data'] ) ? $statement['data'] : array();

				if ( empty( $action ) || empty( $object ) ) {
					$retval[] = CoursesError::instance()->invalid_course_statement();
				}

				if ( 'completed' === $action ) {
					switch ( $object ) {
						case 'lesson':
							$lesson_id = isset( $statement['item_id'] ) ? (int) $statement['item_id'] : 0;

							if ( empty( $lesson_id ) ) {
								// error empty action.
								$retval[] = LessonsError::instance()->invalid_lesson_id();
								break;
							}

							$lesson_request = new WP_REST_Request( 'POST', '/buddyboss-app/learndash/v1/lessons/' . $lesson_id . '/complete' );

							$lesson_request->set_url_params( array( 'id' => $lesson_id ) );
							$lesson_request->set_param( 'course_id', $course_id );
							$lesson_request->set_body( $object_data );
							if ( ! empty( $object_data ) ) {
								foreach ( $object_data as $key => $data ) {
									$lesson_request->set_param( $key, $data );
								}
							}

							$_GET      = $lesson_request->get_params();
							$_POST     = $lesson_request->get_params();
							$_REQUEST  = $lesson_request->get_params();
							$temp_file = $_FILES;
							if ( isset( $_FILES[ $lesson_id ] ) ) {
								$_FILES = bbapp_input_clean( wp_unslash( $_FILES[ $lesson_id ] ) ); //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
							} else {
								unset( $_FILES );
							}
							$_FILES = $temp_file;

							$server   = rest_get_server();
							$retval[] = $server->dispatch( $lesson_request );

							break;
						case 'topic':
							$lesson_id = isset( $statement['secondary_item_id'] ) ? (int) $statement['secondary_item_id'] : 0;
							$topic_id  = isset( $statement['item_id'] ) ? (int) $statement['item_id'] : 0;

							if ( empty( $lesson_id ) ) {
								// error empty action.
								$retval[] = LessonsError::instance()->invalid_lesson_id();
								break;
							}

							if ( empty( $topic_id ) ) {
								// error empty action.
								$retval[] = TopicsError::instance()->invalid_topic_id();
								break;
							}

							// Topdo: Change endpoint.
							$topic_request = new WP_REST_Request( 'POST', '/buddyboss-app/learndash/v1/topics/' . $topic_id . '/complete' );

							$topic_request->set_url_params( array( 'id' => $topic_id ) );
							$topic_request->set_param( 'course_id', $course_id );
							$topic_request->set_param( 'lesson_id', $lesson_id );
							$topic_request->set_body( $object_data );
							if ( ! empty( $object_data ) ) {
								foreach ( $object_data as $key => $data ) {
									$topic_request->set_param( $key, $data );
								}
							}

							$_GET      = $topic_request->get_params();
							$_POST     = $topic_request->get_params();
							$_REQUEST  = $topic_request->get_params();
							$temp_file = $_FILES;
							if ( isset( $_FILES[ $topic_id ] ) ) {
								$_FILES = bbapp_input_clean( wp_unslash( $_FILES[ $topic_id ] ) ); //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
							} else {
								unset( $_FILES );
							}
							$_FILES = $temp_file;

							$server   = rest_get_server();
							$retval[] = $server->dispatch( $topic_request );

							break;
					}
				}
				if ( 'incomplete' === $action ) {
					switch ( $object ) {
						case 'lesson':
							$lesson_id = isset( $statement['item_id'] ) ? (int) $statement['item_id'] : 0;

							if ( empty( $lesson_id ) ) {
								// error empty action.
								$retval[] = LessonsError::instance()->invalid_lesson_id();
								break;
							}

							$lesson_request = new WP_REST_Request( 'POST', '/buddyboss-app/learndash/v1/lessons/' . $lesson_id . '/incomplete' );

							$lesson_request->set_url_params( array( 'id' => $lesson_id ) );
							$lesson_request->set_param( 'course_id', $course_id );
							$lesson_request->set_body( $object_data );
							if ( ! empty( $object_data ) ) {
								foreach ( $object_data as $key => $data ) {
									$lesson_request->set_param( $key, $data );
								}
							}

							$_GET      = $lesson_request->get_params();
							$_POST     = $lesson_request->get_params();
							$_REQUEST  = $lesson_request->get_params();
							$temp_file = $_FILES;
							if ( isset( $_FILES[ $lesson_id ] ) ) {
								$_FILES = bbapp_input_clean( wp_unslash( $_FILES[ $lesson_id ] ) ); //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
							} else {
								unset( $_FILES );
							}
							$_FILES = $temp_file;

							$server   = rest_get_server();
							$retval[] = $server->dispatch( $lesson_request );

							break;
						case 'topic':
							$lesson_id = isset( $statement['secondary_item_id'] ) ? (int) $statement['secondary_item_id'] : 0;
							$topic_id  = isset( $statement['item_id'] ) ? (int) $statement['item_id'] : 0;

							if ( empty( $lesson_id ) ) {
								// error empty action.
								$retval[] = LessonsError::instance()->invalid_lesson_id();
								break;
							}

							if ( empty( $topic_id ) ) {
								// error empty action.
								$retval[] = TopicsError::instance()->invalid_topic_id();
								break;
							}

							// Topdo: Change endpoint.
							$topic_request = new WP_REST_Request( 'POST', '/buddyboss-app/learndash/v1/topics/' . $topic_id . '/incomplete' );

							$topic_request->set_url_params( array( 'id' => $topic_id ) );
							$topic_request->set_param( 'course_id', $course_id );
							$topic_request->set_param( 'lesson_id', $lesson_id );
							$topic_request->set_body( $object_data );
							if ( ! empty( $object_data ) ) {
								foreach ( $object_data as $key => $data ) {
									$topic_request->set_param( $key, $data );
								}
							}

							$_GET      = $topic_request->get_params();
							$_POST     = $topic_request->get_params();
							$_REQUEST  = $topic_request->get_params();
							$temp_file = $_FILES;
							if ( isset( $_FILES[ $topic_id ] ) ) {
								$_FILES = bbapp_input_clean( wp_unslash( $_FILES[ $topic_id ] ) ); //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
							} else {
								unset( $_FILES );
							}
							$_FILES = $temp_file;

							$server   = rest_get_server();
							$retval[] = $server->dispatch( $topic_request );

							break;
					}
				}
			}
		}

		$_GET     = $temp_request;
		$_POST    = $temp_request;
		$_REQUEST = $temp_request;

		return $retval;
	}

}
