<?php
/**
 * Holds Categories rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Course\Category
 */

namespace BuddyBossApp\Api\LearnDash\V1\Course\Category;

use BuddyBossApp\Api\LearnDash\V1\Core\LDRestController;
use BuddyBossApp\Api\LearnDash\V1\Course\CoursesError;
use BuddyBossApp\Api\LearnDash\V1\Course\CoursesRest;
use WP_Error;
use WP_Post;
use WP_Query;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use WP_REST_Term_Meta_Fields;

/**
 * Categories rest controller.
 */
class CategoriesRest extends LDRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Course post type.
	 *
	 * @var string $post_type
	 */
	public $post_type = 'sfwd-courses';

	/**
	 * Course category taxonomy.
	 *
	 * @var string $taxonomy
	 */
	protected $taxonomy = 'ld_course_category';

	/**
	 * CategoriesRest instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'course-categories';
		$this->meta      = new WP_REST_Term_Meta_Fields( $this->taxonomy );
		parent::__construct();
	}

	/**
	 * Check if a given request has access to course category items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the course category `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_course_categories_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to course category item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the course category `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_course_category_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve Course Categories.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/course-categories Get LearnDash Course Categories
	 * @apiName        GetLDCourseCategories
	 * @apiGroup       LD Courses
	 * @apiDescription Retrieve Course Categories
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure result set includes specific IDs.
	 * @apiParam {String=asc,desc} [order=asc] Sort result set by given order.
	 * @apiParam {String=date,term_id,name,slug} [orderby=name] Sort result set by given field.
	 * @apiParam {String=asc,desc} [course_order=desc] Sort courses's of result set  by given order.
	 * @apiParam {String=date,id,title,menu_order} [course_orderby=date] Sort courses's of result set by given field.
	 * @apiParam {Number} [courses_limit=5] Maximum number of courses's of item to be returned in result set.
	 */
	public function get_items( $request ) {
		$registered = $this->get_collection_params();

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'exclude'         => 'exclude',
			'include'         => 'include',
			'order'           => 'order',
			'orderby'         => 'orderby',
			'course_order'    => 'course_order',
			'course_orderby'  => 'course_orderby',
			'hide_empty'      => 'hide_empty',
			'per_page'        => 'number',
			'search'          => 'search',
			'slug'            => 'slug',
			'courses_limit'   => 'courses_limit',
			'courses_exclude' => 'courses_exclude',
			'parent'          => 'parent',
			'post'            => 'post',
			'page'            => 'page',
			'mycourses'       => 'mycourses',
		);

		/**
		 * For each known parameter which is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$args = apply_filters( 'bbapp_ld_course_categories_args', $args, $request );

		$args['offset'] = ( $args['page'] - 1 ) * $args['number'];

		if ( isset( $args['mycourses'] ) && $args['mycourses'] ) {
			$mycourse_ids = ld_get_mycourses( get_current_user_id(), array() );
			$args['post'] = $mycourse_ids;
			add_filter( 'terms_clauses', array( $this, 'alter_query_term_for_my_course' ), 999, 3 ); // my courses in term count getting.
		}

		if ( ! empty( $args['post'] ) ) {
			$categories = wp_get_object_terms( $args['post'], $this->taxonomy, $args );

			// Used when calling wp_count_terms() below.
			$args['object_ids'] = $args['post'];
		} else {
			$categories = get_terms( $this->taxonomy, $args );
		}

		foreach ( $categories as $term ) {
			$term->name = wp_specialchars_decode( $term->name ); // Fixed encoded special char.
			$term->meta = $this->meta->get_value( $term->term_id, $request );
			$term->link = get_term_link( $term );
			$term->id   = $term->term_id;
			$term->icon = $this->get_icon_media( $term->term_id );
		}

		$count_args = $args;
		unset( $count_args['post'] );
		unset( $count_args['number'], $count_args['offset'] );
		$total_terms = wp_count_terms( $this->taxonomy, $count_args );

		$response = array();

		foreach ( $categories as $term ) {
			$term->courses = $this->get_courses( $term, $args );

			if ( isset( $term->term_id ) && empty( $term->term_id ) ) {
				$term->count   = $term->courses['count'];
				$term->courses = $term->courses['posts'];
			}
			$data       = $this->prepare_item_for_response( $term, $request );
			$response[] = $this->prepare_response_for_collection( $data );
		}

		$response = rest_ensure_response( $response );

		bbapp_learners_response_add_total_headers( $response, $total_terms, $args['number'] );

		// Add all courses and mycourses count in the header for course category endpoint.
		if ( function_exists( 'learndash_get_courses_count' ) ) {
			$count_args = array();
			if ( isset( $args['post'] ) && is_user_logged_in() ) {
				$count_args['post__in'] = $args['post'];
			}
			$courses_count = learndash_get_courses_count( $count_args );
			$response->header( 'X-WP-TotalCourses', (int) $courses_count );
		}

		/**
		 * Fires after a list of Course categories response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_course_category_items_response', $response, $request );

		return $response;
	}

	/**
	 * Retrieve Course Category.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/course-categories/:id Get LearnDash Course Category
	 * @apiName        GetLDCourseCategory
	 * @apiGroup       LD Courses
	 * @apiDescription Retrieve single Course category
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course category.
	 */
	public function get_item( $request ) {

		$category_id = is_numeric( $request ) ? $request : (int) $request['id'];

		if ( (int) $category_id < 0 ) {
			return CoursesError::instance()->invalid_course_catrgory_id();
		}

		if ( isset( $category_id ) && ! empty( $category_id ) ) {
			$category       = get_term( $category_id, $this->taxonomy );
			$category->meta = $this->meta->get_value( $category->term_id, $request );
		}

		if ( empty( $category ) || $category->taxonomy !== $this->taxonomy ) {
			return CoursesError::instance()->invalid_course_catrgory_id();
		}

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $category, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after an course category response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_course_category_item_response', $response, $request );

		return $response;
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $item    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $item, $request ) {

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$schema  = $this->get_public_item_schema();

		// Base fields for every post.
		$data = array(
			'id'          => $item->term_id,
			'count'       => $item->count,
			'description' => $item->description,
			'link'        => $item->link,
			'name'        => $item->name,
			'slug'        => $item->slug,
		);

		if ( ! empty( $schema['properties']['parent'] ) ) {
			$data['parent'] = (int) $item->parent;
		}

		if ( ! empty( $schema['properties']['meta'] ) ) {
			$data['meta'] = $item->meta;
		}

		if ( ! empty( $schema['properties']['courses'] ) && isset( $item->courses ) ) {
			$data['courses'] = $item->courses;
		}

		/**
		 * Icon.
		 */
		$data['icon']          = array();
		$data['icon']['small'] = ( is_array( $item->icon ) && isset( $item->icon['small'] ) ) ? $item->icon['small'] : null;
		$data['icon']['large'] = ( is_array( $item->icon ) && isset( $item->icon['large'] ) ) ? $item->icon['large'] : null;

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		return apply_filters( 'bbapp_ld_rest_prepare_course_category', $response, $item, $request );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		$params = parent::get_collection_params();

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['order'] = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'asc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['orderby'] = array(
			'description'       => __( 'Sort collection by object attribute.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'name',
			'enum'              => array(
				'date',
				'name',
				'id',
				'slug',
				'term_group',
				'term_id',
				'description',
				'parent',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['course_order']   = array(
			'description'       => __( 'Order sort attribute ascending or descending of courses.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['course_orderby'] = array(
			'description'       => __( 'Sort courses collection by object attribute.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
				'menu_order',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['hide_empty'] = array(
			'description' => __( 'Whether to hide terms not assigned to any items.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'default'     => false,
		);

		$params['slug'] = array(
			'description' => __( 'Limit result set to terms with one or more specific slugs.', 'buddyboss-app' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'string',
			),
		);

		$params['courses_limit'] = array(
			'description'       => __( 'Limit result set to courses populated on each category.', 'buddyboss-app' ),
			'type'              => 'integer',
			'default'           => 5,
			'sanitize_callback' => 'absint',
		);

		$params['parent'] = array(
			'description'       => __( 'Limit result set by courses category parent.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
		);

		$params['courses_exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific course IDs.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['mycourses'] = array(
			'description'       => __( 'Limit response to resources which are taken by the current user.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/schema#',
			'title'      => 'lms-categories',
			'type'       => 'object',
			'properties' => array(
				'id'          => array(
					'description' => __( 'Unique identifier for the term.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'embed', 'edit' ),
					'readonly'    => true,
				),
				'count'       => array(
					'description' => __( 'Number of published posts for the term.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'description' => array(
					'description' => __( 'HTML description of the term.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'link'        => array(
					'description' => __( 'URL of the term.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'embed', 'edit' ),
					'readonly'    => true,
				),
				'name'        => array(
					'description' => __( 'HTML title for the term.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'embed', 'edit' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_text_field',
					),
					'required'    => true,
				),
				'slug'        => array(
					'description' => __( 'An alphanumeric identifier for the term unique to its type.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'embed', 'edit' ),
					'arg_options' => array(
						'sanitize_callback' => array( $this, 'sanitize_slug' ),
					),
				),
				'icon'        => array(
					'description' => __( 'icon object containing thumb and full URL of image.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
			),
		);

		$schema['properties']['parent'] = array(
			'description' => __( 'The parent term ID.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['courses'] = array(
			'description' => __( 'Courses for the term.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['meta'] = array(
			'description' => __( 'Meta for the term.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get category icon.
	 *
	 * @param int $term_id Term Id.
	 *
	 * @return array
	 */
	public function get_icon_media( $term_id = 0 ) {
		$return = array(
			'large' => null,
			'small' => null,
		);

		if ( ! empty( $term_id ) && term_exists( $term_id ) ) {
			$large = bbapp_lms_taxonomy_image_url( $term_id, 'full', true );
			$small = bbapp_lms_taxonomy_image_url( $term_id, 'thumbnail', true );

			if ( isset( $large ) ) {
				$return['large'] = $large;
			}
			if ( isset( $small ) ) {
				$return['small'] = $small;
			}
		}

		return $return;

	}

	/**
	 * Function to get course.
	 *
	 * @param int   $term Taxonomy term.
	 * @param array $args Course Array.
	 *
	 * @since 1.5.2.1
	 * @return bool
	 */
	private function get_courses( $term, $args ) {

		$user_id = get_current_user_id();

		$query_args = array(
			'post_type'              => $this->post_type,
			'fields'                 => 'ids',
			'post__not_in'           => isset( $args['courses_exclude'] ) ? $args['courses_exclude'] : null,
			'order'                  => $args['course_order'],
			'orderby'                => $args['course_orderby'],
			'no_found_rows'          => true,
			'update_post_meta_cache' => false,
			'update_post_term_cache' => false,
		);

		if ( isset( $args['mycourses'] ) && $args['mycourses'] ) {
			$mycourse_ids = ld_get_mycourses( $user_id, array() );

			if ( ! empty( $mycourse_ids ) && ! is_wp_error( $mycourse_ids ) ) {
				$query_args['post__in'] = ! empty( $query_args['post__in'] ) ? array_intersect( $mycourse_ids, $query_args['post__in'] ) : $mycourse_ids;
				/**
				 * If we intersected, but there are no post ids in common,
				 * WP_Query won't return "no posts" for post__in = array()
				 * so we have to fake it a bit.
				 */
				if ( ! $query_args['post__in'] ) {
					$query_args['post__in'] = array( 0 );
				}
			} else {
				$query_args['post__in'] = array( 0 );
			}
		}

		$query_args['posts_per_page'] = $args['courses_limit'];

		if ( isset( $term->term_id ) && empty( $term->term_id ) ) {
			$query_args['tax_query'][] = array(
				'taxonomy' => 'ld_course_category',
				'operator' => 'NOT EXISTS',
			);
			unset( $query_args['no_found_rows'] );
		} else {
			$query_args['tax_query'][] = array(
				'taxonomy'         => 'ld_course_category',
				'field'            => 'term_id',
				'terms'            => $term->term_id,
				'include_children' => false,
			);
		}

		add_filter( 'posts_distinct', array( CoursesRest::instance(), 'bbapp_posts_distinct' ), 10, 2 );
		add_filter( 'posts_join', array( CoursesRest::instance(), 'bbapp_posts_join' ), 10, 2 );
		add_filter( 'posts_where', array( CoursesRest::instance(), 'bbapp_posts_where' ), 10, 1 );

		$posts_query = new WP_Query();
		$posts       = $posts_query->query( $query_args );

		remove_filter( 'posts_where', array( CoursesRest::instance(), 'bbapp_posts_where' ), 10 );
		remove_filter( 'posts_join', array( CoursesRest::instance(), 'bbapp_posts_join' ), 10 );
		remove_filter( 'posts_distinct', array( CoursesRest::instance(), 'bbapp_posts_distinct' ), 10 );

		// Return count for uncategorised so we no need to do extra query to count course without category.
		if ( isset( $term->term_id ) && empty( $term->term_id ) ) {
			$posts = array(
				'posts' => $posts,
				'count' => $posts_query->found_posts,
			);
		}

		return $posts;
	}

	/**
	 * My courses category count mismatch.
	 * Made the distinct query to get correct count.
	 *
	 * @param array $pieces Query parts.
	 * @param array $taxonomies Taxonomy list array.
	 * @param array $args Fields array.
	 *
	 * @return mixed
	 */
	public function alter_query_term_for_my_course( $pieces, $taxonomies, $args ) {
		if ( bbapp_is_rest() ) {
			if ( ! empty( $args['taxonomy'] ) && in_array( 'ld_course_category', $args['taxonomy'], true ) && 'count' === $args['fields'] && 1 === (int) $args['mycourses'] ) {
				$pieces['fields'] = 'COUNT(DISTINCT(t.slug))';
			}
		}

		return $pieces;

	}

}
