<?php
/**
 * Holds certificate rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Certificate
 */

namespace BuddyBossApp\Api\LearnDash\V1\Certificate;

use BuddyBossApp\Api\LearnDash\V1\Core\LDRestController;
use BuddyBossApp\Api\LearnDash\V1\Quiz\QuizError;
use stdClass;
use WP_Error;
use WP_Query;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Certificate rest.
 */
class CertificateRest extends LDRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * CertificateRest instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'certificates';
		parent::__construct();
	}

	/**
	 * Check if a given request has access to Certificate items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the certificate `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_certificates_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to certificate item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->user_not_logged_in();
		}

		/**
		 * Filter the certificate `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_certificate_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}


	/**
	 * Retrieve certificates.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/certificates Get LearnDash Certificates
	 * @apiName        GetLDCertificates
	 * @apiGroup       LD Certificates
	 * @apiDescription Retrieve certificates
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function get_items( $request ) {

		/**
		 * Get Certificates
		 */
		$certificates = $this->get_certificates();

		$retval = array();
		foreach ( $certificates as $quize ) {
			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $quize, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		$response = bbapp_learners_response_add_total_headers( $response, count( $certificates ), count( $certificates ) );

		/**
		 * Fires after a list of certificates response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_certificate_items_response', $response, $request );

		return $response;
	}

	/**
	 * Retrieve certificate.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/certificates/:id Get LearnDash Certificate
	 * @apiName        GetLDCertificate
	 * @apiGroup       LD Certificates
	 * @apiDescription Retrieve single certificate
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function get_item( $request ) {
		return rest_ensure_response( array() );
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param \WP_Post        $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';

		// Base fields for every post.
		$data = array(
			'id'       => $post->ID,
			'title'    => array(
				'raw'      => $post->title,
				'rendered' => get_the_title( $post->ID ),
			),
			'date'     => mysql_to_rfc3339( $post->date ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'link'     => $post->link,
			'filename' => $post->filename,
			'type'     => $post->type,
		);

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		return apply_filters( 'bbapp_ld_rest_prepare_certificate', $response, $post, $request );
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array $data item object.
	 *
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {

		$links = parent::prepare_links( $data );

		// Todo::dips add filter here for course/lesson/topic.

		return $links;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		$params = parent::get_collection_params();

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'ld_certificate',
			'type'       => 'object',

			/*
			 * Base properties for every Post.
			 */
			'properties' => array(
				'id'       => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'    => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'date'     => array(
					'description' => __( 'The date the object was published, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date_gmt' => array(
					'description' => __( 'The date the object was published, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'link'     => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'type'     => array(
					'description' => __( 'type to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'filename' => array(
					'description' => __( 'Filename to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
			),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get current user Certificates.
	 */
	private function get_certificates() {
		$user_id = get_current_user_id();

		/**
		 * Course Certificate
		 */
		$user_courses = ld_get_mycourses( $user_id, array() );
		$certificates = array();
		foreach ( $user_courses as $course_id ) {

			$certificate_link = learndash_get_course_certificate_link( $course_id, $user_id );
			$filename         = self::get_certificate_name( $course_id );

			$date = get_user_meta( $user_id, 'course_completed_' . $course_id, true );

			if ( ! empty( $certificate_link ) ) {
				$certificate           = new stdClass();
				$certificate->ID       = $course_id;
				$certificate->link     = $certificate_link;
				$certificate->title    = get_the_title( $course_id );
				$certificate->filename = $filename;
				$certificate->date     = gmdate( 'Y-m-d h:i:s', $date );
				$certificate->time     = $date;
				$certificate->type     = 'course';
				$certificates[]        = $certificate;
			}
		}

		/**
		 * Quiz Certificate
		 */
		$quizzes = get_user_meta( $user_id, '_sfwd-quizzes', true );
		if ( ! empty( $quizzes ) ) {
			$quiz_ids = wp_list_pluck( $quizzes, 'quiz' );
			if ( ! empty( $quiz_ids ) ) {
				$quiz_total_query_args = array(
					'post_type' => 'sfwd-quiz',
					'fields'    => 'ids',
					'orderby'   => 'title',
					'order'     => 'ASC',
					'nopaging'  => true,
					'post__in'  => $quiz_ids,
				);
				$quiz_query            = new WP_Query( $quiz_total_query_args );
				$quizzes_tmp           = array();
				foreach ( $quiz_query->posts as $post_idx => $quiz_id ) {
					foreach ( $quizzes as $quiz_idx => $quiz_attempt ) {
						if ( $quiz_attempt['quiz'] === $quiz_id ) {
							$quiz_key                 = $quiz_attempt['time'] . '-' . $quiz_attempt['quiz'];
							$quizzes_tmp[ $quiz_key ] = $quiz_attempt;
							unset( $quizzes[ $quiz_idx ] );
						}
					}
				}
				$quizzes = $quizzes_tmp;
				krsort( $quizzes );
				if ( ! empty( $quizzes ) ) {
					foreach ( $quizzes as $quizdata ) {
						if ( ! in_array( $quizdata['quiz'], wp_list_pluck( $certificates, 'ID' ), true ) ) {
							$quiz_settings    = learndash_get_setting( $quizdata['quiz'] );
							$certificate_data = learndash_certificate_details( $quizdata['quiz'], $user_id );
							if ( ! empty( $certificate_data['certificateLink'] ) && $certificate_data['certificate_threshold'] <= $quizdata['percentage'] / 100 ) {
								$filename              = self::get_certificate_name( $quizdata['quiz'] );
								$certificate           = new stdClass();
								$certificate->ID       = $quizdata['quiz'];
								$certificate->link     = $certificate_data['certificateLink'];
								$certificate->title    = get_the_title( $quizdata['quiz'] );
								$certificate->filename = $filename;
								$certificate->date     = gmdate( 'Y-m-d h:i:s', $quizdata['time'] );
								$certificate->time     = $quizdata['time'];
								$certificate->type     = 'quiz';
								$certificates[]        = $certificate;
							}
						}
					}
				}
			}
		}

		usort(
			$certificates,
			function ( $a, $b ) {
				return strcmp( $b->time, $a->time );
			}
		);

		return $certificates;
	}

	/**
	 * Course certificate name.
	 *
	 * @param integer $post_id course | quiz id.
	 *
	 * @return string
	 */
	public static function get_certificate_name( $post_id ) {
		$filename       = 'Certificate.pdf';
		$course_title   = get_the_title( $post_id );
		$certificate_id = learndash_get_setting( $post_id, 'certificate' );
		if ( ! empty( $certificate_id ) ) {
			$certificate = get_post( $certificate_id );
			$filename    = sanitize_file_name( $course_title ) . '-' . sanitize_file_name( $certificate->post_title ) . '.pdf';
		}

		return $filename;
	}
}
