<?php
/**
 * Holds assignments rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Assignments
 */

namespace BuddyBossApp\Api\LearnDash\V1\Assignments;

use BuddyBossApp\Api\LearnDash\V1\Core\LDRestController;
use WP_Error;
use WP_Post;
use WP_Query;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Assignment rest.
 */
class AssignmentsRest extends LDRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Assignment post type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'sfwd-assignment';

	/**
	 * AssignmentsRest instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'assignments';
		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/(?P<assignment_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				array(
					'methods'             => WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::DELETABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve Assignments.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/assignments/{{lesson_id|topic_id}}?type=lesson Get LearnDash Assignments
	 * @apiName        GetLDAssignments
	 * @apiGroup       LD Assignments
	 * @apiDescription Retrieve Assignments
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure result set includes specific IDs.
	 * @apiParam {String} [after]  Limit results to those published after a given ISO8601 compliant date.
	 * @apiParam {String} [before] Limit results to those published before a given ISO8601 compliant date.
	 * @apiParam {Array} [author] Limit results to those assigned to specific authors.
	 * @apiParam {Array} [author_exclude] Ensure results to excludes those assigned to specific authors.
	 * @apiParam {String=asc,desc} [order=desc] Sort result set by given order.
	 * @apiParam {String=date,id,title,menu_order} [orderby=date] Sort result set by given field.
	 * @apiParam {Array} [parent] Limit results to those assigned to specific parent.
	 */
	public function get_items( $request ) {
		$user_id = get_current_user_id();

		$id        = $request->get_param( 'id' );
		$type      = $request->get_param( 'type' );
		$course_id = $request->get_param( 'course_id' );

		if ( learndash_get_post_type_slug( 'lesson' ) !== get_post_type( $id ) && learndash_get_post_type_slug( 'topic' ) !== get_post_type( $id ) ) {
			return rest_ensure_response( AssignmentsError::instance()->invalid_provided_id() );
		}

		$lesson_id = ( 'lesson' === $type ) ? $id : 0;
		$topic_id  = ( 'topic' === $type ) ? $id : 0;

		$registered = $this->get_collection_params();

		/**
		 * Filter the the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$request = apply_filters( 'bbapp_ld_get_assignments_request', $request );

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'author'         => 'author__in',
			'author_exclude' => 'author__not_in',
			'exclude'        => 'post__not_in',
			'include'        => 'post__in',
			'offset'         => 'offset',
			'order'          => 'order',
			'orderby'        => 'orderby',
			'page'           => 'paged',
			'parent'         => 'post_parent__in',
			'parent_exclude' => 'post_parent__not_in',
			'search'         => 's',
			'slug'           => 'post_name__in',
			'status'         => 'post_status',
			'per_page'       => 'posts_per_page',
		);

		/**
		 * For each known parameter which is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		// Check for & assign any parameters which require special handling or setting.
		$args['date_query'] = array();

		// Set before into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['before'], $request['before'] ) ) {
			$args['date_query'][0]['before'] = $request['before'];
		}

		// Set after into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['after'], $request['after'] ) ) {
			$args['date_query'][0]['after'] = $request['after'];
		}

		if ( ! empty( $lesson_id ) ) {
			$args['meta_query'] = $this->bbapp_learndash_get_user_assignments_meta_query( $lesson_id, $course_id );
		}
		if ( ! empty( $topic_id ) ) {
			$args['meta_query'] = $this->bbapp_learndash_get_user_assignments_meta_query( $topic_id, $course_id );
		}
		if ( ! empty( $user_id ) ) {
			$args['author'] = $user_id;
		}
		$args = $this->prepare_items_query( $args, $request );

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$args = apply_filters( 'bbapp_ld_get_assignments_args', $args, $request );

		$args['post_type'] = $this->post_type;

		$posts_query                = new WP_Query();
		$assignments['posts']       = $posts_query->query( $args );
		$assignments['total_posts'] = $posts_query->found_posts;

		/**
		 * Fires list of Assignment is fetched via Query.
		 *
		 * @param array            $assignments Fetched assignments.
		 * @param WP_REST_Response $args        Query arguments.
		 * @param WP_REST_Request  $request     The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		$assignments = apply_filters( 'bbapp_ld_get_assignments', $assignments, $args, $request );

		$retval = array();

		$assignment_stats = learndash_get_assignment_progress( $assignments['posts'] );

		foreach ( $assignments['posts'] as $assignment ) {
			if ( ! $this->check_read_permission( $assignment ) ) {
				continue;
			}
			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $assignment, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		$response = bbapp_learners_response_add_total_headers( $response, $assignments['total_posts'], $args['posts_per_page'] );

		// Assignments header.
		$response->header( 'LD-Total-Assignments', (int) $assignment_stats['total'] );
		$response->header( 'LD-Approved-Assignments', (int) $assignment_stats['complete'] );

		/**
		 * Fires after a list of assignmenta response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_assignments_items_response', $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to assignment items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the assignment `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_assignments_items_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve Assignment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/assignments/{{lesson_id|topic_id}}/{{assignment_id}} Get LearnDash Assignment
	 * @apiName        GetLDAssignment
	 * @apiGroup       LD Assignments
	 * @apiDescription Retrieve single Assignments
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Assignment.
	 */
	public function get_item( $request ) {

		$id            = $request->get_param( 'id' );
		$assignment_id = $request->get_param( 'assignment_id' );

		if ( learndash_get_post_type_slug( 'lesson' ) !== get_post_type( $id ) && learndash_get_post_type_slug( 'topic' ) !== get_post_type( $id ) ) {
			return AssignmentsError::instance()->invalid_provided_id();
		}

		$assignment = get_post( $assignment_id );

		if ( empty( $assignment ) || $this->post_type !== $assignment->post_type ) {
			return AssignmentsError::instance()->invalid_assignment_id();
		}

		/**
		 * Fire after Assignment is fetched via Query.
		 *
		 * @param array           $assignment    Fetched assignment.
		 * @param WP_REST_Request $assignment_id assignment id.
		 *
		 * @since 0.1.0
		 */
		$assignment = apply_filters( 'bbapp_ld_get_assignment', $assignment, $assignment_id );

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $assignment, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after an assignment respose is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_assignment_item_response', $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to assignment item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the assignment `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_assignments_item_permissions_check', $retval, $request );
	}

	/**
	 * Delete a Assignment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error
	 * @since          0.1.0
	 *
	 * @api            {DELETE} /wp-json/buddyboss-app/learndash/v1/assignments/:id Delete LearnDash Assignment
	 * @apiName        DeleteLDAssignment
	 * @apiGroup       Learndash
	 * @apiDescription Trash OR Delete a assignment.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the assignment.
	 */
	public function delete_item( $request ) {

		$id            = $request->get_param( 'id' );
		$assignment_id = $request->get_param( 'assignment_id' );

		if ( learndash_get_post_type_slug( 'lesson' ) !== get_post_type( $id ) && learndash_get_post_type_slug( 'topic' ) !== get_post_type( $id ) ) {
			return AssignmentsError::instance()->invalid_provided_id();
		}

		$assignment = get_post( $assignment_id );
		$user_id    = get_current_user_id();

		if ( empty( $assignment ) || $this->post_type !== $assignment->post_type ) {
			return AssignmentsError::instance()->invalid_assignment_id();
		}

		if ( ( (int) $assignment->post_author !== (int) $user_id ) && ( ! learndash_is_admin_user( $user_id ) ) && ( ! learndash_is_group_leader_of_user( $user_id, $assignment->post_author ) ) ) {
			return AssignmentsError::instance()->invalid_user_access();
		}

		if ( false === $this->get_can_delete( $assignment ) ) {
			return AssignmentsError::instance()->cannot_delete_assignment();
		}

		$previous = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $assignment, $request )
		);

		$course_id = get_post_meta( $assignment->ID, 'course_id', true );
		if ( empty( $course_id ) ) {
			$course_id = learndash_get_course_id( $assignment->ID );
		}
		$course_step_id = get_post_meta( $assignment->ID, 'lesson_id', true );

		learndash_process_mark_incomplete( $user_id, $course_id, $course_step_id );

		/**
		 * Filters whether to force delete the assignment or not.
		 *
		 * @param boolean $force_delete  Whether to force delete assignment or not.
		 * @param int     $assignment_id Assignment ID.
		 * @param WP_POST $assignment    Assignment post object.
		 */
		$success = wp_delete_post( $assignment->ID, apply_filters( 'learndash_assignment_force_delete', true, $assignment->ID, $assignment ) );

		// Build the response.
		$response = new WP_REST_Response();
		$response->set_data(
			array(
				'deleted'  => ( ! empty( $success ) && ! is_wp_error( $success ) ? true : $success ),
				'previous' => $previous,
			)
		);

		/**
		 * Fires after a assignment is deleted via the REST API.
		 *
		 * @param array            $assignment Fetched Assignment.
		 * @param WP_REST_Response $response   The response data.
		 * @param WP_REST_Request  $request    The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_ld_assignment_item_response', $assignment, $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to delete a assignment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error|bool
	 * @since 0.1.0
	 */
	public function delete_item_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = AssignmentsError::instance()->user_not_logged_in();
		}

		if ( true === $retval ) {
			$retval = $this->get_item_permissions_check( $request );
		}

		/**
		 * Filter the assignment `delete_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_assignments_delete_item_permissions_check', $retval, $request );
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$GLOBALS['post'] = $post; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		setup_postdata( $post );
		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$schema  = $this->get_public_item_schema();

		// Base fields for every post.
		$data = array(
			'id'           => $post->ID,
			'title'        => array(
				'raw'      => $post->post_title,
				'rendered' => get_the_title( $post->ID ),
			),
			'content'      => array(
				'raw'      => bbapp_learners_fix_relative_urls_protocol( $post->post_content ),
				'rendered' => bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_content', $post->post_content ) ),
			),
			'date'         => mysql_to_rfc3339( $post->post_date ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'date_gmt'     => mysql_to_rfc3339( $post->post_date_gmt ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'modified'     => mysql_to_rfc3339( $post->post_modified ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'modified_gmt' => mysql_to_rfc3339( $post->post_modified_gmt ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'link'         => get_permalink( $post->ID ),
			'slug'         => $post->post_name,
			'author'       => (int) $post->post_author,
			'excerpt'      => array(
				'raw'      => bbapp_learners_fix_relative_urls_protocol( $post->post_excerpt ),
				'rendered' => bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_excerpt', $post->post_excerpt ) ),
			),
			'menu_order'   => (int) $post->menu_order,
		);

		if ( ! empty( $schema['properties']['can_delete'] ) && in_array( $context, $schema['properties']['can_delete']['context'], true ) ) {
			$data['can_delete'] = $this->get_can_delete( $post );
		}
		if ( ! empty( $schema['properties']['file_link'] ) && in_array( $context, $schema['properties']['file_link']['context'], true ) ) {
			$data['file_link'] = $this->get_assignment_file_link( $post );
		}
		if ( ! empty( $schema['properties']['can_comment'] ) && in_array( $context, $schema['properties']['can_comment']['context'], true ) ) {
			$data['can_comment'] = $this->get_assignment_can_comment( $post );
		}
		if ( ! empty( $schema['properties']['comments_count'] ) && in_array( $context, $schema['properties']['comments_count']['context'], true ) ) {
			$data['comments_count'] = $this->get_comments_count( $post );
		}
		if ( ! empty( $schema['properties']['assignment_status'] ) && in_array( $context, $schema['properties']['assignment_status']['context'], true ) ) {
			$data['assignment_status'] = $this->get_assignment_status( $post );
		}

		$associated_assignment_ids = $this->get_associated_assignment_ids( $post );
		if ( ! empty( $schema['properties']['course'] ) && in_array( $context, $schema['properties']['course']['context'], true ) && ! empty( $associated_assignment_ids['course_id'] ) ) {
			$data['course'] = (int) $associated_assignment_ids['course_id'];
		}
		if ( ! empty( $schema['properties']['lesson'] ) && in_array( $context, $schema['properties']['lesson']['context'], true ) && ! empty( $associated_assignment_ids['lesson_id'] ) ) {
			$data['lesson'] = (int) $associated_assignment_ids['lesson_id'];
		}
		if ( ! empty( $schema['properties']['topic'] ) && in_array( $context, $schema['properties']['topic']['context'], true ) && ! empty( $associated_assignment_ids['topic_id'] ) ) {
			$data['topic'] = (int) $associated_assignment_ids['topic_id'];
		}

		if ( ! empty( $schema['properties']['error_message'] ) && in_array( $context, $schema['properties']['error_message']['context'], true ) ) {
			$post->error_message = $this->get_error_message( $post );
			if ( ! empty( $post->error_message ) ) {
				$error_code            = $post->error_message->get_error_code();
				$data['error_message'] = array(
					'code'    => $error_code,
					'message' => $post->error_message->get_error_message(),
					'data'    => $post->error_message->get_error_data( $error_code ),

				);
			} else {
				$data['error_message'] = array();
			}
		}

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		return apply_filters( 'bbapp_ld_rest_prepare_assignment', $response, $post, $request );
	}

	/**
	 * Prepare items.
	 *
	 * @param array $prepared_args query parameters.
	 * @param null  $request Request parameters.
	 *
	 * @return array
	 */
	protected function prepare_items_query( $prepared_args = array(), $request = null ) {
		$query_args = array();

		foreach ( $prepared_args as $key => $value ) {
			/**
			 * Filters the query_vars used in get_items() for the constructed query.
			 *
			 * The dynamic portion of the hook name, `$key`, refers to the query_var key.
			 *
			 * @param string $value The query_var value.
			 *
			 * @since 4.7.0
			 */
			$query_args[ $key ] = apply_filters( "rest_query_var-{$key}", $value ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		}

		$query_args['ignore_sticky_posts'] = true;

		// Map to proper WP_Query orderby param.
		if ( isset( $query_args['orderby'] ) && isset( $request['orderby'] ) ) {
			$orderby_mappings = array(
				'id'            => 'ID',
				'include'       => 'post__in',
				'slug'          => 'post_name',
				'include_slugs' => 'post_name__in',
			);

			if ( isset( $orderby_mappings[ $request['orderby'] ] ) ) {
				$query_args['orderby'] = $orderby_mappings[ $request['orderby'] ];
			}
		}

		return $query_args;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array $data item object.
	 *
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {

		$links = parent::prepare_links( $data );

		/**
		 * Course
		 */
		if ( ! empty( $data['course'] ) ) {
			$links['course'] = array(
				'href'       => rest_url( $this->namespace . '/courses/' . $data['course'] ),
				'embeddable' => false,
			);
		}

		/**
		 * Lesson
		 */
		if ( ! empty( $data['lesson'] ) ) {
			$links['lesson'] = array(
				'href'       => rest_url( $this->namespace ) . '/lessons/' . $data['lesson'] . '?course_id=' . $data['course'],
				'embeddable' => false,
			);
		}

		/**
		 * Topic
		 */
		if ( ! empty( $data['topic'] ) ) {
			$rest_url = rest_url( $this->namespace ) . '/topics/' . $data['topic'] . '?course_id=' . $data['course'];
			if ( ! empty( $data['lesson'] ) ) {
				$rest_url .= '&lesson_id=' . $data['lesson'];
			}
			$links['topic'] = array(
				'href'       => $rest_url,
				'embeddable' => false,
			);
		}

		return $links;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		$params = parent::get_collection_params();

		$params['id'] = array(
			'description' => __( 'A unique numeric ID for the lesson or topic.', 'buddyboss-app' ),
			'type'        => 'integer',
			'required'    => true,
		);

		$params['type'] = array(
			'description'       => __( 'Which type assignment fetching.', 'buddyboss-app' ),
			'type'              => 'string',
			'required'          => true,
			'default'           => 'lesson',
			'enum'              => array( 'lesson', 'topic' ),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['course_id'] = array(
			'description'       => __( 'Assignment course id.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'required'          => true,
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author'] = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset']            = array(
			'description'       => __( 'Offset the result set by a specific number of items.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']['enum'][] = 'menu_order';

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent IDs.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent ID.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);

		$params['slug']   = array(
			'description'       => __( 'Limit result set to posts with a specific slug.', 'buddyboss-app' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['filter'] = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.', 'buddyboss-app' ),
		);

		return $params;
	}

	/**
	 * Edit some arguments for the endpoint's CREATABLE, EDITABLE and DELETABLE methods.
	 *
	 * @param string $method Optional. HTTP method of the request.
	 *
	 * @return array Endpoint arguments.
	 * @since 0.1.0
	 */
	public function get_endpoint_args_for_item_schema( $method = WP_REST_Server::CREATABLE ) {
		$args = WP_REST_Controller::get_endpoint_args_for_item_schema( $method );
		$key  = 'create_item';

		if ( WP_REST_Server::DELETABLE === $method ) {
			$key = 'delete_item';

			$args = array(
				'id' => array(
					'description' => __( 'A unique numeric ID for the reply.', 'buddyboss-app' ),
					'type'        => 'integer',
					'required'    => true,
				),
			);
		}

		/**
		 * Filters the method query arguments.
		 *
		 * @param array  $args   Query arguments.
		 * @param string $method HTTP method of the request.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( "bbapp_rest_assignment_{$key}_query_arguments", $args, $method );
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'assignment',
			'type'       => 'object',
			'properties' => array(
				'id'           => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'        => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content'      => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'date'         => array(
					'description' => __( 'The date the object was published, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date_gmt'     => array(
					'description' => __( 'The date the object was published, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'modified'     => array(
					'description' => __( 'The date the object was last modified, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'modified_gmt' => array(
					'description' => __( 'The date the object was last modified, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'link'         => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'         => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'       => array(
					'description' => __( 'The ID for the author of the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'excerpt'      => array(
					'description' => __( 'The excerpt for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
			),
		);

		$schema['properties']['can_delete'] = array(
			'description' => __( 'User can delete his/her assignment.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['file_link'] = array(
			'description' => __( 'Assignment file link.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['can_comment'] = array(
			'description' => __( 'User can create comment.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['comments_count'] = array(
			'description' => __( 'how much comment exists in this assignment.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['video'] = array(
			'description' => __( 'video for the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['assignment_status'] = array(
			'description' => __( 'Assignment status with necessary data.', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'view', 'edit' ),
			'properties'  => array(
				'properties' => array(
					'label' => array(
						'description' => __( 'Label of status.', 'buddyboss-app' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'point' => array(
						'description' => __( 'Minimum required time for this object.', 'buddyboss-app' ),
						'type'        => 'object',
						'context'     => array( 'view', 'edit' ),
						'properties'  => array(
							'properties' => array(
								'current'     => array(
									'description' => __( 'Current points.', 'buddyboss-app' ),
									'type'        => 'integer',
									'context'     => array( 'view', 'edit' ),
								),
								'max'         => array(
									'description' => __( 'Maximum points.', 'buddyboss-app' ),
									'type'        => 'integer',
									'context'     => array( 'view', 'edit' ),
								),
								'point_label' => array(
									'description' => __( 'Point label.', 'buddyboss-app' ),
									'type'        => 'string',
									'context'     => array( 'view', 'edit' ),
								),
							),
						),
					),
				),
			),
		);

		$schema['properties']['course'] = array(
			'description' => __( 'Assignment associated course id.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['lesson'] = array(
			'description' => __( 'Assignment associated lesson id.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['topic'] = array(
			'description' => __( 'Assignment associated topic id.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * If user has course accesss.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return mixed
	 */
	public function get_has_course_access( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post->ID );

		return sfwd_lms_has_access( $course_id );
	}

	/**
	 * Current user can delete assignment or not.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return bool
	 */
	private function get_can_delete( $post ) {
		$can_delete = false;
		if ( is_user_logged_in() ) {
			$user_id       = get_current_user_id();
			$post_settings = learndash_get_setting( $post->lesson_id );
			$can_delete    = ( ! (bool) learndash_is_assignment_approved_by_meta( $post->ID ) ) && ( isset( $post_settings['lesson_assignment_deletion_enabled'] ) && 'on' === $post_settings['lesson_assignment_deletion_enabled'] && (int) $post->post_author === $user_id ) || ( learndash_is_admin_user( $user_id ) ) || ( learndash_is_group_leader_of_user( $user_id, $post->post_author ) );
		}

		return $can_delete;
	}

	/**
	 * Get assignment file link.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return bool
	 */
	private function get_assignment_file_link( $post ) {
		return get_post_meta( $post->ID, 'file_link', true );
	}

	/**
	 * User can comment or not..
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return bool
	 */
	private function get_assignment_can_comment( $post ) {
		$can_comment = false;
		if ( is_user_logged_in() ) {
			$can_comment = post_type_supports( $this->post_type, 'comments' ) && apply_filters( 'comments_open', $post->comment_status, $post->ID );
		}

		return $can_comment;
	}

	/**
	 * Get assignment comment count.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return bool
	 */
	private function get_comments_count( $post ) {
		return (int) get_comments_number( $post->ID );
	}

	/**
	 * Get assignment status.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return array
	 */
	private function get_assignment_status( $post ) {
		$status            = array(
			'label'  => '',
			'status' => '',
			'point'  => array(),
		);
		$assignment_points = learndash_get_points_awarded_array( $post->ID );
		if ( ! learndash_is_assignment_approved_by_meta( $post->ID ) && ! $assignment_points ) {

			$status['label']  = __( 'Waiting Review', 'buddyboss-app' );
			$status['status'] = 'waiting_review';
			$status['point']  = array();

		} elseif ( $assignment_points || learndash_is_assignment_approved_by_meta( $post->ID ) ) {
			$status['label']  = __( 'Approved', 'buddyboss-app' );
			$status['status'] = 'approved';
			if ( $assignment_points ) {
				$status['point'] = array(
					'current'     => (int) $assignment_points['current'],
					'max'         => (int) $assignment_points['max'],
					'point_label' => sprintf( '%1$s/%2$s', __( 'Points Awarded', 'buddyboss-app' ), $assignment_points['current'], $assignment_points['max'] ),
				);
			}
		}

		return $status;
	}

	/**
	 * Get assignment assignment course,lesson and topic id.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @return array
	 */
	private function get_associated_assignment_ids( $post ) {
		$associated_assignment_ids = array(
			'course_id' => 0,
			'lesson_id' => 0,
			'topic_id'  => 0,
		);

		$course_id = get_post_meta( $post->ID, 'course_id', true );
		$lesson_id = get_post_meta( $post->ID, 'lesson_id', true );

		$associated_assignment_ids['course_id'] = $course_id;
		$associated_assignment_ids['lesson_id'] = $lesson_id;
		if ( ! empty( $lesson_id ) ) {
			if ( learndash_get_post_type_slug( 'topic' ) === get_post_type( $lesson_id ) ) {
				$topic_id                              = absint( $lesson_id );
				$associated_assignment_ids['topic_id'] = $topic_id;
				$lesson_id                             = get_post_meta( $topic_id, 'lesson_id', true );
				if ( ( empty( $lesson_id ) ) && ( ! empty( $course_id ) ) ) {
					$associated_assignment_ids['lesson_id'] = learndash_course_get_single_parent_step( $course_id, $topic_id );
				}
			}
		}

		return $associated_assignment_ids;
	}

	/**
	 * Enable comment on assignment.
	 *
	 * @return bool
	 */
	public function enable_assignment_comment() {
		$comment_status = \LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Assignments_CPT', 'comment_status' );

		return ( 'yes' === $comment_status );
	}

	/**
	 * Get error message.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @return array
	 */
	public function get_error_message( $post ) {
		if ( ! learndash_is_assignment_approved_by_meta( $post->ID ) ) {
			return AssignmentsError::instance()->awaiting_for_approval_assignment();
		}

		return array();
	}

	/**
	 * Gets a list of user's assignments.
	 *
	 * @param int $post_id   Lesson ID|Topic ID.
	 * @param int $course_id Optional. Course ID. Default 0.
	 *
	 * @return array Array of post objects or post IDs.
	 * @since 2.1.0
	 */
	protected function bbapp_learndash_get_user_assignments_meta_query( $post_id, $course_id = 0 ) {
		if ( empty( $course_id ) ) {
			$course_id = function_exists( 'learndash_get_course_id' ) ? learndash_get_course_id( $post_id ) : 0;
		}

		$opt = array(
			'relation' => 'AND',
			array(
				'key'     => 'lesson_id',
				'value'   => $post_id,
				'compare' => '=',
			),
			array(
				'key'     => 'course_id',
				'value'   => $course_id,
				'compare' => '=',
			),
		);

		return $opt;
	}
}
