<?php
/**
 * Holds assignments error rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Assignments
 */

namespace BuddyBossApp\Api\LearnDash\V1\Assignments;

use BuddyBossApp\RestErrors;
use WP_Error;

/**
 * Assignment error.
 */
class AssignmentsError {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return AssignmentsError
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * When user is not logged in this error message should be used.
	 *
	 * @return WP_Error
	 */
	public function user_not_logged_in() {
		return RestErrors::instance()->user_not_logged_in();
	}

	/**
	 * When Assignment id not correct.
	 *
	 * @return WP_Error
	 */
	public function awaiting_for_approval_assignment() {
		return new WP_Error( 'learndash_json_assignment_awaiting_for_approval', __( 'You have an assignment awaiting approval.', 'buddyboss-app' ), array( 'status' => 202 ) );
	}

	/**
	 * When Assignment id not correct.
	 *
	 * @return WP_Error
	 */
	public function invalid_assignment_id() {
		return new WP_Error( 'learndash_json_assignment_invalid_id', __( 'Assignment not found.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * When Assignment file is empty.
	 *
	 * @return WP_Error
	 */
	public function assignment_empty_file() {
		return new WP_Error( 'learndash_json_assignment_file_empty', __( 'Assignment file shouldn’t be empty.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When Assignment id not correct.
	 *
	 * @return WP_Error
	 */
	public function invalid_id() {
		return new WP_Error( 'learndash_json_invalid_id', __( 'Lesson/Topic not found.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * Assignment file upload permission.
	 *
	 * @return WP_Error
	 */
	public function invalid_permission() {
		return new WP_Error( 'learndash_json_assignment_upload_permission', __( 'Lesson/Topic does not allow uploading an assignment.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * Assignment file upload permission.
	 *
	 * @since 1.7.2
	 * @return WP_Error
	 */
	public function invalid_upload_limit() {
		return new WP_Error( 'learndash_json_assignment_upload_limit', __( 'You have reached your maximum limit of uploading assignments', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user don't have lesson/topic access.
	 *
	 * @return WP_Error
	 */
	public function invalid_access() {
		return new WP_Error( 'learndash_json_invalid_access', __( 'You do not have access to this lesson/topic.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When user don't have lesson/topic access.
	 *
	 * @return WP_Error
	 */
	public function assignment_authorization_required() {
		return new WP_Error( 'bbapp_rest_authorization_required', __( 'Sorry, you are not allowed to delete this assignment.', 'buddyboss-app' ), array( 'status' => rest_authorization_required_code() ) );
	}

	/**
	 * When user don't have lesson/topic access.
	 *
	 * @return WP_Error
	 */
	public function invalid_user_access() {
		return new WP_Error( 'learndash_json_invalid_user_access', __( 'You do not have access to delete this assignment.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * Can not delete assignment.
	 *
	 * @return WP_Error
	 */
	public function cannot_delete_assignment() {
		return new WP_Error( 'learndash_json_can_not_delete_assignment', __( 'Cannot delete the assignment.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * Can not delete assignment.
	 *
	 * @return WP_Error
	 */
	public function enable_comment_on_assignment_setting() {
		return new WP_Error( 'learndash_json_enable_comment_on_assignment_setting', __( 'Enable comment on assignment setting.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * Lesson or Topic is is invalid.
	 *
	 * @return WP_Error
	 */
	public function invalid_provided_id() {
		return new WP_Error( 'learndash_json_invalid_provided_id', __( 'Invalid provided lesson or topic ID.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When assignment directory is not writable.
	 *
	 * @since 2.3.20
	 *
	 * @return WP_Error
	 */
	public function assignment_directory_is_not_writeable() {
		return new WP_Error( 'learndash_json_assignment_directory_is_not_writeable', __( 'Unable to write to directory. Is this directory writable by the server?', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * When assignment file move error.
	 *
	 * @param string $temporary_file Temporary file.
	 * @param string $destination_file Destination file.
	 *
	 * @since 2.3.20
	 * @return WP_Error
	 */
	public function get_file_move_error( $temporary_file, $destination_file ) {
		return new WP_Error(
			'learndash_json_assignment_file_move_error',
			sprintf(
			// translators: %1$s is the temporary file, %2$s is the destination file.
				esc_html__( 'Error: The file %1$s could not be moved to %2$s.', 'buddyboss-app' ),
				esc_html( $temporary_file ),
				esc_html( $destination_file )
			),
			[ 'status' => 400 ]
		);
	}

	/**
	 * Upload directory access issue.
	 *
	 * @since 2.3.20
	 * @return WP_Error
	 */
	public function get_upload_dir_access_error() {
		return new WP_Error( 'learndash_json_assignment_upload_dir_access_error', __( 'Unable to write to UPLOADS directory. Is this directory writable by the server?', 'buddyboss-app' ), array( 'status' => 400 ) );
	}
}
