<?php
/**
 * Holds Assignment comment rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Assignments
 */

namespace BuddyBossApp\Api\LearnDash\V1\Assignments;

use BuddyBossApp\Api\LearnDash\V1\Core\LDRestController;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Assignment comment rest.
 */
class AssignmentsCommentRest extends LDRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Assignment post type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'sfwd-assignment';

	/**
	 * AssignmentsCommentRest instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'assignments';
		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/comments/(?P<assignment_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/comments/(?P<assignment_id>[\d]+)/(?P<comment_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::EDITABLE ),
				),
				array(
					'methods'             => WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::DELETABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to assignment items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {

		$retval = true;

		/**
		 * Assignments setting.
		 */
		if ( true === $retval ) {
			$is_enable = AssignmentsRest::instance()->enable_assignment_comment();
			$retval    = ( true === $is_enable ) ? true : AssignmentsError::instance()->enable_comment_on_assignment_setting();
		}

		/**
		 * Filter the assignment `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_assignments_comments_items_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve Assignments Comments.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/learndash/v1/assignments/{{lesson_id|topic_id}}/comments/{{assignment_id}} Get LearnDash Assignments Comments
	 * @apiName        GetLDAssignmentsComments
	 * @apiGroup       LD AssignmentsComments
	 * @apiDescription Retrieve Assignments Comments
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure result set includes specific IDs.
	 * @apiParam {String} [after]  Limit results to those published after a given ISO8601 compliant date.
	 * @apiParam {String} [before] Limit results to those published before a given ISO8601 compliant date.
	 * @apiParam {Array} [author] Limit results to those assigned to specific authors.
	 * @apiParam {Array} [author_exclude] Ensure results to excludes those assigned to specific authors.
	 * @apiParam {String=asc,desc} [order=desc] Sort result set by given order.
	 * @apiParam {String=date,id,title,menu_order} [orderby=date] Sort result set by given field.
	 * @apiParam {Array} [parent] Limit results to those assigned to specific parent.
	 */
	public function get_items( $request ) {

		$id            = $request->get_param( 'id' );
		$assignment_id = $request->get_param( 'assignment_id' );

		if ( learndash_get_post_type_slug( 'lesson' ) !== get_post_type( $id ) && learndash_get_post_type_slug( 'topic' ) !== get_post_type( $id ) ) {
			return AssignmentsError::instance()->invalid_provided_id();
		}

		if ( learndash_get_post_type_slug( 'assignment' ) !== get_post_type( $assignment_id ) ) {
			return AssignmentsError::instance()->invalid_assignment_id();
		}

		$request->set_param( 'post', $assignment_id );

		return $this->get_route_response( $request, '/wp/v2/comments' );
	}

	/**
	 * Checks if a given request has access to create a comment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return true|WP_Error True if the request has access to create items, error object otherwise.
	 * @since 4.7.0
	 */
	public function create_item_permissions_check( $request ) {
		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = AssignmentsError::instance()->user_not_logged_in();
		}

		/**
		 * Assignments setting.
		 */
		if ( true === $retval ) {
			$is_enable = AssignmentsRest::instance()->enable_assignment_comment();
			$retval    = ( true === $is_enable ) ? true : AssignmentsError::instance()->enable_comment_on_assignment_setting();
		}

		/**
		 * Filter the Assignment `create_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_assignment_create_item_permissions_check', $retval, $request );
	}

	/**
	 * Creates a comment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error Response object on success, or error object on failure.
	 * @since 4.7.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/assignments/{{lesson_id|topic_id}}/comment/{{assignment_id}}/{{comment_id}} Create LearnDash Assignment comment.
	 * @apiName        CreateLDAssignmentComment
	 * @apiGroup       LD AssignmentsComment
	 * @apiDescription Retrieve Assignments Comment
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Assignment.
	 */
	public function create_item( $request ) {
		$id            = $request->get_param( 'id' );
		$assignment_id = $request->get_param( 'assignment_id' );
		$content       = $request->get_param( 'content' );

		if ( learndash_get_post_type_slug( 'lesson' ) !== get_post_type( $id ) && learndash_get_post_type_slug( 'topic' ) !== get_post_type( $id ) ) {
			return AssignmentsError::instance()->invalid_provided_id();
		}

		if ( learndash_get_post_type_slug( 'assignment' ) !== get_post_type( $assignment_id ) ) {
			return AssignmentsError::instance()->invalid_assignment_id();
		}

		$request->set_param( 'id', 0 ); // Lesson|Topic id override.
		$request->set_param( 'post', $assignment_id );
		$request->set_param( 'content', $content );
		$request->set_method( $request->get_method() );

		return $this->get_route_response( $request, '/wp/v2/comments' );
	}

	/**
	 * Retrieve Assignment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            /wp-json/buddyboss-app/learndash/v1/assignments/{{lesson_id|topic_id}}/comment/{{assignment_id}}/{{comment_id}} Get LearnDash Assignment single comment.
	 * @apiName        GetLDAssignmentComment
	 * @apiGroup       LD AssignmentsComment
	 * @apiDescription Retrieve single Assignments Comment
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Assignment.
	 */
	public function get_item( $request ) {

		$id            = $request->get_param( 'id' );
		$assignment_id = $request->get_param( 'assignment_id' );
		$comment_id    = $request->get_param( 'comment_id' );

		if ( learndash_get_post_type_slug( 'lesson' ) !== get_post_type( $id ) && learndash_get_post_type_slug( 'topic' ) !== get_post_type( $id ) ) {
			return AssignmentsError::instance()->invalid_provided_id();
		}

		$request->set_param( 'post', $assignment_id );
		$request->set_param( 'id', $comment_id ); // override lesson|topic id to comment id.

		return $this->get_route_response( $request, '/wp/v2/comments/' . $comment_id );
	}

	/**
	 * Check if a given request has access to assignment item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {

		$retval = true;

		/**
		 * Assignments setting.
		 */
		if ( true === $retval ) {
			$is_enable = AssignmentsRest::instance()->enable_assignment_comment();
			$retval    = ( true === $is_enable ) ? true : AssignmentsError::instance()->enable_comment_on_assignment_setting();
		}

		/**
		 * Filter the assignment `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_assignments_item_permissions_check', $retval, $request );
	}

	/**
	 * Checks if a given REST request has access to update a comment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return true|WP_Error True if the request has access to update the item, error object otherwise.
	 * @since 4.7.0
	 */
	public function update_item_permissions_check( $request ) {
		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = AssignmentsError::instance()->user_not_logged_in();
		}

		/**
		 * Assignments setting.
		 */
		if ( true === $retval ) {
			$is_enable = AssignmentsRest::instance()->enable_assignment_comment();
			$retval    = ( true === $is_enable ) ? true : AssignmentsError::instance()->enable_comment_on_assignment_setting();
		}

		/**
		 * Filter the Assignment `update` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_assignment_update_permissions_check', $retval, $request );
	}

	/**
	 * Updates a comment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error Response object on success, or error object on failure.
	 * @since 4.7.0
	 *
	 * @api            /wp-json/buddyboss-app/learndash/v1/assignments/{{lesson_id|topic_id}}/comment/{{assignment_id}}/{{comment_id}} Update LearnDash Assignment comment.
	 * @apiName        UpdateLDAssignmentComment
	 * @apiGroup       LD AssignmentsComment
	 * @apiDescription Retrieve single Assignments Comment
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Assignment.
	 */
	public function update_item( $request ) {
		$id            = $request->get_param( 'id' );
		$assignment_id = $request->get_param( 'assignment_id' );
		$comment_id    = $request->get_param( 'comment_id' );
		$content       = $request->get_param( 'content' );

		if ( learndash_get_post_type_slug( 'lesson' ) !== get_post_type( $id ) && learndash_get_post_type_slug( 'topic' ) !== get_post_type( $id ) ) {
			return AssignmentsError::instance()->invalid_provided_id();
		}

		$request->set_param( 'post', $assignment_id );
		$request->set_param( 'id', $comment_id ); // override lesson|topic id to comment id.
		$request->set_param( 'content', $content );
		$request->set_method( $request->get_method() );

		return $this->get_route_response( $request, '/wp/v2/comments/' . $comment_id );
	}

	/**
	 * Delete a Assignment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error
	 * @since          0.1.0
	 *
	 * @api            {DELETE} /wp-json/buddyboss-app/learndash/v1/assignments/{{lesson_id|topic_id}}/comment/{{assignment_id}}/{{comment_id}} Delete LearnDash Assignment Comment
	 * @apiName        DeleteLDAssignmentComment
	 * @apiGroup       Learndash
	 * @apiDescription Trash OR Delete a assignment comment..
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the assignment.
	 */
	public function delete_item( $request ) {
		$id            = $request->get_param( 'id' );
		$assignment_id = $request->get_param( 'assignment_id' );
		$comment_id    = $request->get_param( 'comment_id' );

		if ( learndash_get_post_type_slug( 'lesson' ) !== get_post_type( $id ) && learndash_get_post_type_slug( 'topic' ) !== get_post_type( $id ) ) {
			return AssignmentsError::instance()->invalid_provided_id();
		}

		$request->set_param( 'post', $assignment_id );
		$request->set_param( 'id', $comment_id ); // override lesson|topic id to comment id.
		$request->set_method( $request->get_method() );

		return $this->get_route_response( $request, '/wp/v2/comments/' . $comment_id );
	}

	/**
	 * Check if a given request has access to delete a assignment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error|bool
	 * @since 0.1.0
	 */
	public function delete_item_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = AssignmentsError::instance()->user_not_logged_in();
		}

		if ( true === $retval ) {
			$retval = $this->get_item_permissions_check( $request );
		}

		/**
		 * Assignments setting.
		 */
		if ( true === $retval ) {
			$is_enable = AssignmentsRest::instance()->enable_assignment_comment();
			$retval    = ( true === $is_enable ) ? true : AssignmentsError::instance()->enable_comment_on_assignment_setting();
		}

		/**
		 * Filter the assignment `delete_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_assignments_delete_item_permissions_check', $retval, $request );
	}

	/**
	 * Retrieves the query params for collections.
	 *
	 * @return array Comments collection parameters.
	 * @since 4.7.0
	 */
	public function get_collection_params() {
		$query_params = parent::get_collection_params();

		$query_params['context']['default'] = 'view';

		$params['id'] = array(
			'description' => __( 'A unique numeric ID for the lesson or topic.', 'buddyboss-app' ),
			'type'        => 'integer',
			'required'    => true,
		);

		$params['assignment_id'] = array(
			'description' => __( 'A unique numeric ID for the assignment.', 'buddyboss-app' ),
			'type'        => 'integer',
			'required'    => true,
		);

		$query_params['after'] = array(
			'description' => __( 'Limit response to comments published after a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'        => 'string',
			'format'      => 'date-time',
		);

		$query_params['author'] = array(
			'description' => __( 'Limit result set to comments assigned to specific user IDs. Requires authorization.', 'buddyboss-app' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
		);

		$query_params['author_exclude'] = array(
			'description' => __( 'Ensure result set excludes comments assigned to specific user IDs. Requires authorization.', 'buddyboss-app' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
		);

		$query_params['author_email'] = array(
			'default'     => null,
			'description' => __( 'Limit result set to that from a specific author email. Requires authorization.', 'buddyboss-app' ),
			'format'      => 'email',
			'type'        => 'string',
		);

		$query_params['before'] = array(
			'description' => __( 'Limit response to comments published before a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'        => 'string',
			'format'      => 'date-time',
		);

		$query_params['exclude'] = array(
			'description' => __( 'Ensure result set excludes specific IDs.', 'buddyboss-app' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
			'default'     => array(),
		);

		$query_params['include'] = array(
			'description' => __( 'Limit result set to specific IDs.', 'buddyboss-app' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
			'default'     => array(),
		);

		$query_params['offset'] = array(
			'description' => __( 'Offset the result set by a specific number of items.', 'buddyboss-app' ),
			'type'        => 'integer',
		);

		$query_params['order'] = array(
			'description' => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'type'        => 'string',
			'default'     => 'desc',
			'enum'        => array(
				'asc',
				'desc',
			),
		);

		$query_params['orderby'] = array(
			'description' => __( 'Sort collection by object attribute.', 'buddyboss-app' ),
			'type'        => 'string',
			'default'     => 'date_gmt',
			'enum'        => array(
				'date',
				'date_gmt',
				'id',
				'include',
				'post',
				'parent',
				'type',
			),
		);

		$query_params['parent'] = array(
			'default'     => array(),
			'description' => __( 'Limit result set to comments of specific parent IDs.', 'buddyboss-app' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
		);

		$query_params['parent_exclude'] = array(
			'default'     => array(),
			'description' => __( 'Ensure result set excludes specific parent IDs.', 'buddyboss-app' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
		);

		$query_params['post'] = array(
			'default'     => array(),
			'description' => __( 'Limit result set to comments assigned to specific post IDs.', 'buddyboss-app' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
		);

		$query_params['status'] = array(
			'default'           => 'approve',
			'description'       => __( 'Limit result set to comments assigned a specific status. Requires authorization.', 'buddyboss-app' ),
			'sanitize_callback' => 'sanitize_key',
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$query_params['type'] = array(
			'default'           => 'comment',
			'description'       => __( 'Limit result set to comments assigned a specific type. Requires authorization.', 'buddyboss-app' ),
			'sanitize_callback' => 'sanitize_key',
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$query_params['password'] = array(
			'description' => __( 'The password for the post if it is password protected.', 'buddyboss-app' ),
			'type'        => 'string',
		);

		/**
		 * Filters REST API collection parameters for the comments controller.
		 *
		 * This filter registers the collection parameter, but does not map the
		 * collection parameter to an internal WP_Comment_Query parameter. Use the
		 * `rest_comment_query` filter to set WP_Comment_Query parameters.
		 *
		 * @param array $query_params JSON Schema-formatted collection parameters.
		 *
		 * @since 4.7.0
		 */
		return apply_filters( 'rest_comment_collection_params', $query_params );
	}

	/**
	 * Edit some arguments for the endpoint's CREATABLE, EDITABLE and DELETABLE methods.
	 *
	 * @param string $method Optional. HTTP method of the request.
	 *
	 * @return array Endpoint arguments.
	 * @since 0.1.0
	 */
	public function get_endpoint_args_for_item_schema( $method = WP_REST_Server::CREATABLE ) {
		$args = WP_REST_Controller::get_endpoint_args_for_item_schema( $method );
		$key  = 'create_item';

		if ( WP_REST_Server::DELETABLE === $method ) {
			$key = 'delete_item';

			$args = array(
				'id' => array(
					'description' => __( 'A unique numeric ID for the reply.', 'buddyboss-app' ),
					'type'        => 'integer',
					'required'    => true,
				),
			);
		}

		/**
		 * Filters the method query arguments.
		 *
		 * @param array  $args   Query arguments.
		 * @param string $method HTTP method of the request.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( "bbapp_rest_assignment_{$key}_query_arguments", $args, $method );
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'assignment-comments',
			'type'       => 'object',
			'properties' => array(
				'id'                => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'author'            => array(
					'description' => __( 'The ID of the user object, if author was a user.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'author_email'      => array(
					'description' => __( 'Email address for the object author.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'email',
					'context'     => array( 'edit' ),
					'arg_options' => array(
						'sanitize_callback' => array( $this, 'check_comment_author_email' ),
						'validate_callback' => null, // Skip built-in validation of 'email'.
					),
				),
				'author_ip'         => array(
					'description' => __( 'IP address for the object author.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'ip',
					'context'     => array( 'edit' ),
				),
				'author_name'       => array(
					'description' => __( 'Display name for the object author.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
				'author_url'        => array(
					'description' => __( 'URL for the object author.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'author_user_agent' => array(
					'description' => __( 'User agent for the object author.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'edit' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
				'content'           => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'arg_options' => array(
						'sanitize_callback' => null,
						// Note: sanitization implemented in self::prepare_item_for_database().
						'validate_callback' => null,
						// Note: validation implemented in self::prepare_item_for_database().
					),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
							'readonly'    => true,
						),
					),
				),
				'date'              => array(
					'description' => __( "The date the object was published, in the site's timezone.", 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date_gmt'          => array(
					'description' => __( 'The date the object was published, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit' ),
				),
				'link'              => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'parent'            => array(
					'description' => __( 'The ID for the parent of the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'default'     => 0,
				),
				'post'              => array(
					'description' => __( 'The ID of the associated post object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'default'     => 0,
				),
				'status'            => array(
					'description' => __( 'State of the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_key',
					),
				),
				'type'              => array(
					'description' => __( 'Type of Comment for the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
			),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Handle the request and return a valid response for this endpoint.
	 *
	 * @param \WP_REST_Request $request   Request object.
	 * @param string           $route_url Rest route URL.
	 *
	 * @return mixed
	 */
	protected function get_route_response( $request, $route_url ) {
		global $wp_rest_server;
		/**
		 * Redirect to topic endpoint.
		 */
		$request_curl = new WP_REST_Request( $request->get_method(), $route_url );
		$request_curl->set_headers( $request->get_headers() );
		$request_curl->set_body( $request->get_body() );
		if ( ! empty( $request->get_headers() ) ) {
			foreach ( $request->get_headers() as $key => $data ) {
				$request_curl->set_header( $key, $data );
			}
		}

		if ( ! empty( $request->get_params() ) ) {
			foreach ( $request->get_params() as $key => $data ) {
				$request_curl->set_param( $key, $data );
			}
		}

		$response = $this->dispatch( $request_curl );

		if ( isset( $response->headers ) ) {
			$wp_rest_server->send_headers( $response->headers );
		}

		return rest_ensure_response( $response );

	}

	/**
	 * Dispatch the request item.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {

		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()['_embed'] ) ) {
			$query_params['_embed'] = $request->get_query_params()['_embed'];
		}

		$request->set_query_params( $query_params );

		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}
}
