<?php
/**
 * Holds Assignment action rest functionality.
 *
 * @package BuddyBossApp\Api\LearnDash\V1\Assignments
 */

namespace BuddyBossApp\Api\LearnDash\V1\Assignments;

use BuddyBossApp\Api\LearnDash\V1\Lesson\LessonsError;
use BuddyBossApp\Api\LearnDash\V1\Topic\TopicsError;
use BuddyBossApp\Helpers\BBAPP_File;
use WP_Error;
use WP_REST_Request;
use WP_REST_Server;

/**
 * Assignment action rest.
 */
class AssignmentsActionRest extends AssignmentsRest {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * AssignmentsActionRest instance.
	 *
	 * @since 0.1.0
	 *
	 * @return AssignmentsActionRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/upload',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'upload' ),
					'permission_callback' => array( $this, 'get_upload_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to upload assignment item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_upload_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = AssignmentsError::instance()->user_not_logged_in();
		}

		if ( is_user_logged_in() ) {
			$user_id       = get_current_user_id();
			$post_settings = learndash_get_setting( $request->get_param( 'id' ) );
			if ( isset( $post_settings['assignment_upload_limit_count'] ) ) {
				$assignment_upload_limit_count = intval( $post_settings['assignment_upload_limit_count'] );
				if ( $assignment_upload_limit_count > 0 ) {
					$assignments = learndash_get_user_assignments( $request->get_param( 'id' ), $user_id );
					if ( ! empty( $assignments ) && count( $assignments ) >= $assignment_upload_limit_count ) {
						$retval = AssignmentsError::instance()->invalid_upload_limit();
					}
				}
			}
		}

		/**
		 * Filter the Assignment `upload` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_assignment_upload_permissions_check', $retval, $request );
	}

	/**
	 * Upload Assignment file.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_REST_Response|WP_Error|boolean
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/learndash/v1/assignments/:id/upload Upload LearnDash Assignment file.
	 * @apiName        LDAssignmentUpload
	 * @apiGroup       LD Assignments
	 * @apiDescription Upload Assignments file. This endpoint requires request to be sent in "multipart/form-data" format.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Assignment.
	 * @apiParam {File} [ attachment ] A file to upload for Assignment file.
	 */
	public function upload( $request ) {
		$user_id   = get_current_user_id();
		$lesson_id = 0;
		$topic_id  = 0;
		$post      = null;

		$id = $request->get_param( 'id' );

		if ( learndash_get_post_type_slug( 'lesson' ) !== get_post_type( $id ) && learndash_get_post_type_slug( 'topic' ) !== get_post_type( $id ) ) {
			return AssignmentsError::instance()->invalid_provided_id();
		}

		if ( learndash_get_post_type_slug( 'lesson' ) === get_post_type( $id ) ) {
			$lesson_id = $id;
			$post      = get_post( $lesson_id );
		}

		if ( learndash_get_post_type_slug( 'topic' ) === get_post_type( $id ) ) {
			$topic_id  = $id;
			$lesson_id = get_post_meta( $topic_id, 'lesson_id', true );
			$post      = get_post( $topic_id );
		}

		if ( empty( $post ) ) {
			return AssignmentsError::instance()->invalid_id();
		}

		if ( ! $this->get_lesson_topic_assignment_upload( $post ) ) {
			return AssignmentsError::instance()->invalid_permission();
		}

		if ( empty( $_FILES ) ) {
			return AssignmentsError::instance()->assignment_empty_file();
		}

		if ( ! sfwd_lms_has_access( $lesson_id, $user_id ) ) {
			return AssignmentsError::instance()->invalid_access();
		}

		$created_assignment_id = $this->upload_attachment_file( $lesson_id, $topic_id );

		if ( is_wp_error( $created_assignment_id ) ) {
			return $created_assignment_id;
		} elseif ( is_numeric( $created_assignment_id ) ) {
			$request['assignment_id'] = $created_assignment_id;

			return $this->get_item( $request );
		}

		return false;
	}

	/**
	 * Upload assignment attachment file.
	 *
	 * @param int $lesson_id Lesson Id.
	 * @param int $topic_id Toppic Id.
	 *
	 * @return bool|int|void|WP_Error
	 */
	public function upload_attachment_file( $lesson_id, $topic_id = 0 ) {
		$assignment_id = 0;
		$post_id = ! empty( $topic_id ) ? $topic_id : $lesson_id;

		if ( ! empty( $_FILES['attachment'] ) ) {
			if ( 0 == $_FILES['attachment']['error'] ) {
				$file_tmp = $_FILES['attachment']['tmp_name'];

				// clean filename.
				$filename = learndash_clean_filename( $_FILES['attachment']['name'] );

				// extract extension.
				if ( ! function_exists( 'wp_get_current_user' ) ) {
					include ABSPATH . 'wp-includes/pluggable.php';
				}

				// Before this function we have already validated the file extension/type via the function learndash_check_upload
				$file_uploaded_name = pathinfo( basename( $filename ), PATHINFO_FILENAME );
				$file_ext           = pathinfo( basename( $filename ), PATHINFO_EXTENSION );

				$upload_dir      = wp_upload_dir();
				$upload_dir_base = str_replace( '\\', DIRECTORY_SEPARATOR, $upload_dir['basedir'] );
				$upload_dir_path = $upload_dir_base . DIRECTORY_SEPARATOR . 'learndash' . DIRECTORY_SEPARATOR . 'assignments';

				if ( ! file_exists( $upload_dir_path ) ) {
					if ( is_writable( dirname( $upload_dir_path ) ) ) {
						wp_mkdir_p( $upload_dir_path );
					} else {
						return AssignmentsError::instance()->get_upload_dir_access_error();
					}
				}

				$file_time = microtime( true ) * 100;
				$filename  = sprintf( 'assignment_%d_%d_%s_%s.%s', $post_id, $file_time, $file_uploaded_name, uniqid(), $file_ext );

				/**
				 * Filters the assignment upload filename.
				 *
				 * @since 3.2.0
				 *
				 * @param string $filename   File name.
				 * @param int    $post_id    Post ID.
				 * @param float  $file_time  Unix timestamp.
				 * @param string $file_title Title of the file.
				 * @param string $file_ext   File extension.
				 */
				$filename = apply_filters(
					'learndash_assignment_upload_filename',
					$filename,
					$post_id,
					$file_time,
					$file_uploaded_name,
					$file_ext
				);

				/**
				 * Check if the filename already exist in the directory and rename the
				 * file if necessary
				 */
				$i = 0;

				$file_current_name = pathinfo( basename( $filename ), PATHINFO_FILENAME );
				$file_ext          = pathinfo( basename( $filename ), PATHINFO_EXTENSION );

				while ( file_exists( $upload_dir_path . '/' . $filename ) ) {
					$i++;
					$filename = $file_current_name . '_' . $i . '.' . $file_ext;
				}

				$file_dest = $upload_dir_path . '/' . $filename;

				/**
				 * Check write permissions
				 */
				if ( ! is_writeable( $upload_dir_path ) ) {
					return AssignmentsError::instance()->assignment_directory_is_not_writeable();
				}

				/**
				 * Save temporary file to uploads dir
				 */
				if ( ! @move_uploaded_file( $file_tmp, $file_dest ) ) { // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged -- Better not to touch it for now.
					return AssignmentsError::instance()->get_file_move_error( $file_tmp, $file_dest );
				}

				$assignment_id = bbapp_lms_upload_assignment_init( $post_id, $filename, $file_uploaded_name );
			}
		} else {
			if ( ! empty( $topic_id ) ) {
				$waiting_approval = $this->get_assignment_waiting_approval( $topic_id );

				if ( $waiting_approval ) {
					return TopicsError::instance()->topic_assignment_not_approved( $waiting_approval );
				}

				return TopicsError::instance()->topic_assignment_not_completed();
			} else {
				$waiting_approval = $this->get_assignment_waiting_approval( $lesson_id );

				if ( $waiting_approval ) {
					return LessonsError::instance()->lesson_assignment_not_approved( $waiting_approval );
				}

				return LessonsError::instance()->lesson_assignment_not_completed();
			}
		}

		return $assignment_id;
	}

	/**
	 * Check If Assignment exists or not.
	 *
	 * @param WP_Post $post    Post object.
	 *
	 * @return bool
	 */
	protected function get_lesson_topic_assignment_upload( $post ) {
		return function_exists( 'learndash_lesson_hasassignments' ) ? learndash_lesson_hasassignments( $post ) : lesson_hasassignments( $post );
	}

	/**
	 * Get Assignment waiting for approval.
	 *
	 * @param int $post_id Post ID.
	 *
	 * @since 2.1.00
	 * @return int|mixed
	 */
	public function get_assignment_waiting_approval( $post_id ) {
		$ret_val = 0;

		if ( is_user_logged_in() ) {
			$assignments      = learndash_get_user_assignments( $post_id, get_current_user_id() );
			$assignments      = ( ! empty( $assignments ) ) ? $assignments : array();
			$assignment_stats = learndash_get_assignment_progress( $assignments );
			$ret_val          = $assignment_stats['total'] - $assignment_stats['complete'];
		}

		return $ret_val;
	}
}
