<?php
/**
 * Holds LearnDash helping functions.
 *
 * @package BuddyBossApp\Api\Learner\LearnDash\QuizComplete
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\Api\Learner\LearnDash\QuizComplete;

if ( ! function_exists( 'bbapp_lms_is_content_access' ) ) {
	/**
	 * Content access.
	 *
	 * @param WP_Post $obj_post   Post data.
	 * @param string  $field_name Field name.
	 *
	 * @return bool
	 */
	function bbapp_lms_is_content_access( $obj_post, $field_name ) {
		$course_id = bbapp_learndash_get_course_id( $obj_post->ID, true );
		$user_id   = get_current_user_id();
		$logged_in = ! empty( $user_id );

		if ( $logged_in ) {
			if ( learndash_is_admin_user( $user_id ) ) {
				$can_bypass_course_limit = LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Section_General_Admin_User', 'bypass_course_limits_admin_users' );

				if ( 'yes' === $can_bypass_course_limit ) {
					$can_bypass_course_limit = true;
				} else {
					$can_bypass_course_limit = false;
				}
			} elseif ( learndash_is_group_leader_user( $user_id ) ) {
				$can_bypass_course_limit = LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Section_Groups_Group_Leader_User', 'bypass_course_limits' );

				if ( 'yes' === $can_bypass_course_limit ) {
					$can_bypass_course_limit = true;
				} else {
					$can_bypass_course_limit = false;
				}
			} else {
				$can_bypass_course_limit = false;
			}

			// For logged in users to allow an override filter.
			$can_bypass_course_limit = apply_filters( 'learndash_prerequities_bypass', $can_bypass_course_limit, $user_id, $course_id, $obj_post );
		} else {
			$can_bypass_course_limit = true;
		}

		if ( 'prerequities_completed' === $field_name ) {
			return ( ! $logged_in ) || learndash_is_course_prerequities_completed( $course_id ) || $can_bypass_course_limit;
		} elseif ( 'points_access' === $field_name ) {
			return ( ! $logged_in ) || learndash_check_user_course_points_access( $course_id, $user_id ) || $can_bypass_course_limit;
		}

		if ( 'previous_lesson_completed' === $field_name || 'previous_topic_completed' === $field_name ) {
			$lesson_progression_enabled = bbapp_learndash_get_course_meta_setting( $course_id, 'course_disable_lesson_progression' );
			$previous_completed         = true;

			if ( empty( $lesson_progression_enabled ) ) {
				if ( ( learndash_is_admin_user( $user_id ) || learndash_is_group_leader_user( $user_id ) ) && ( $can_bypass_course_limit ) ) {
					$previous_completed = true;
				} else {
					if ( 'previous_lesson_completed' === $field_name ) {
						if ( 'sfwd-topic' === $obj_post->post_type ) {
							$lesson_id   = learndash_get_setting( $obj_post->ID, 'lesson' );
							$lesson_post = get_post( $lesson_id );
						} else {
							$lesson_post = $obj_post;
						}

						if ( ! empty( $lesson_post ) ) {
							$previous_completed = apply_filters( 'learndash_previous_step_completed', is_previous_complete( $lesson_post ), $lesson_post->ID, $user_id );
						}
					} elseif ( 'previous_topic_completed' === $field_name ) {
						$previous_completed = apply_filters( 'learndash_previous_step_completed', is_previous_complete( $obj_post ), $obj_post->ID, $user_id );
					}
				}
			}

			return (bool) $previous_completed;
		}

		return false;
	}
}

if ( ! function_exists( 'bbapp_lms_lesson_access_from' ) ) {
	/**
	 * Function to get lesson/topic/quiz access or not.
	 *
	 * @param int|WP_Post $obj_post    Post object.
	 * @param bool        $bool_return Bool return flag.
	 *
	 * @since 1.7.3
	 *
	 * @return bool|int
	 */
	function bbapp_lms_lesson_access_from( $obj_post, $bool_return = true ) {
		$course_id                        = bbapp_learndash_get_course_id( $obj_post->ID );
		$user_id                          = get_current_user_id();
		$bypass_course_limits_admin_users = false;

		if ( learndash_is_admin_user( $user_id ) ) {
			$bypass_course_limits_admin_users = LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Section_General_Admin_User', 'bypass_course_limits_admin_users' );

			if ( 'yes' === $bypass_course_limits_admin_users ) {
				$bypass_course_limits_admin_users = true;
			} else {
				$bypass_course_limits_admin_users = false;
			}
		}

		$lesson_access_from = ld_lesson_access_from( $obj_post->ID, $user_id, $course_id );

		if ( ( ! empty( $lesson_access_from ) ) && ( ! $bypass_course_limits_admin_users ) ) {
			if ( ! $bool_return ) {
				return $lesson_access_from;
			}

			return false;
		}

		return true;
	}
}

if ( ! function_exists( 'bbapp_lesson_visible_after' ) ) {
	/**
	 * Function to get lesson/topic/quiz visible after or not.
	 *
	 * @param WP_Post $post Post object.
	 *
	 * @since 1.7.3
	 *
	 * @return false|int
	 */
	function bbapp_lesson_visible_after( $post = null ) {
		if ( ! is_a( $post, 'WP_Post' ) ) {
			$post_id = get_the_ID();
			if ( ! empty( $post_id ) ) {
				$post = get_post( $post_id );
				if ( ! is_a( $post, 'WP_Post' ) ) {
					return false;
				}
			}
		}

		if ( ! in_array( $post->post_type, learndash_get_post_types(), true ) ) {
			return false;
		}

		if ( is_user_logged_in() ) {
			$user_id = get_current_user_id();
		} else {
			return false;
		}

		$bypass_course_limits_admin_users = learndash_can_user_bypass( $user_id, 'learndash_course_lesson_not_available', $post->ID, $post );

		// For logged in users to allow an override filter.
		/** This filter is documented in includes/course/ld-course-progress.php */
		if ( apply_filters( 'learndash_prerequities_bypass', $bypass_course_limits_admin_users, $user_id, $post->ID, $post ) ) {
			return false;
		}

		$course_id = learndash_get_course_id( $post );
		if ( empty( $course_id ) ) {
			return false;
		}

		$lesson_access_from = learndash_course_step_available_date( $post->ID, $course_id, $user_id, true );
		if ( ! empty( $lesson_access_from ) ) {
			return $lesson_access_from;
		}

		return false;
	}
}

if ( ! function_exists( 'bbapp_lms_next_lesson_id' ) ) {
	/**
	 * Get next lesson id.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @return array|int|mixed|string
	 */
	function bbapp_lms_next_lesson_id( $post ) {
		if ( empty( $post->ID ) ) {
			return '';
		}

		$next = bbapp_lms_next_post_id( $post );

		if ( ! empty( $next ) ) {
			$next_id = $next;
		} else {
			if ( 'sfwd-topic' === $post->post_type ) {
				$next_id = learndash_get_setting( $post, 'lesson' );
			} else {
				$next_id = bbapp_learndash_get_course_id( $post );
			}
		}

		if ( ! empty( $next_id ) ) {
			return $next_id;
		} else {
			return '';
		}
	}
}

if ( ! function_exists( 'bbapp_lms_next_post_id' ) ) {
	/**
	 * Get next post id.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @return string
	 */
	function bbapp_lms_next_post_id( $post ) {
		if ( empty( $post->ID ) ) {
			return '';
		}

		if ( 'sfwd-lessons' === $post->post_type ) {
			$course_id = bbapp_learndash_get_course_id( $post );
			$posts     = learndash_get_lesson_list( $course_id );
		} elseif ( 'sfwd-topic' === $post->post_type ) {
			$lesson_id = learndash_get_setting( $post, 'lesson' );
			$posts     = learndash_get_topic_list( $lesson_id );
		} else {
			return '';
		}

		foreach ( $posts as $k => $p ) {
			if ( $p instanceof WP_Post ) {
				if ( (int) $p->ID === $post->ID ) {
					$found_at = $k;
					break;
				}
			}
		}

		if ( isset( $found_at ) && ! empty( $posts[ $found_at + 1 ] ) ) {
			return $posts[ $found_at + 1 ]->ID;
		}

		return '';
	}
}

if ( ! function_exists( 'bbapp_lms_quiz_completed' ) ) {
	/**
	 * If the quiz is completed.
	 *
	 * @param WP_Post $post      Post data.
	 * @param int     $course_id Course id.
	 *
	 * @return bool
	 */
	function bbapp_lms_quiz_completed( $post, $course_id = 0 ) {
		if ( ! is_user_logged_in() ) {
			return false;
		}

		$course_id = bbapp_learndash_get_course_id( $post->ID );
		$quizzes   = learndash_get_lesson_quiz_list( $post );
		$quizids   = array();

		if ( ! empty( $quizzes ) ) {
			foreach ( $quizzes as $quiz ) {
				$quizids[ $quiz['post']->ID ] = $quiz['post']->ID;
			}
		}

		if ( ! empty( $quizids ) ) {
			$all_quizzes_completed = ! learndash_is_quiz_notcomplete( null, $quizids, false, $course_id );
		} else {
			$all_quizzes_completed = true;
		}

		return $all_quizzes_completed;
	}
}

if ( ! function_exists( 'bbapp_lms_media_upload_dir' ) ) {
	/**
	 * Media upload path.
	 *
	 * @param array $param Parameters.
	 *
	 * @return mixed
	 */
	function bbapp_lms_media_upload_dir( $param ) {
		$param['path'] = $param['basedir'] . '/assignments';
		$param['url']  = $param['baseurl'] . '/assignments';

		return $param;
	}
}

/**
 * Ref_Files: sfwd-lms/includes/ld-assignment-uploads.php
 * Ref_Funcation: learndash_upload_assignment_init
 */
if ( ! function_exists( 'bbapp_lms_upload_assignment_init' ) ) {
	/**
	 * Upload assignment.
	 *
	 * @param int    $post_id Post id.
	 * @param string $fname   File name.
	 * @param string $file_uploaded_name File uploaded name. Added in @since 2.3.20
	 *
	 * @return false|int|WP_Error
	 */
	function bbapp_lms_upload_assignment_init( $post_id, $fname, $file_uploaded_name ) {
		if ( ! function_exists( 'wp_get_current_user' ) ) {
			include ABSPATH . 'wp-includes/pluggable.php';
		}

		$new_assignmnt_meta = array();
		$current_user       = wp_get_current_user();
		$username           = $current_user->user_login;
		$dispname           = $current_user->display_name;
		$url_link_arr       = wp_upload_dir();
		$url_link           = $url_link_arr['baseurl'];
		$dir_link           = $url_link_arr['basedir'];
		$file_path          = $dir_link . DIRECTORY_SEPARATOR . 'learndash' . DIRECTORY_SEPARATOR . 'assignments' . DIRECTORY_SEPARATOR;
		$url_path           = $url_link . DIRECTORY_SEPARATOR . 'learndash' . DIRECTORY_SEPARATOR . 'assignments' . DIRECTORY_SEPARATOR . $fname;

		if ( file_exists( $file_path . $fname ) ) {
			$dest = $url_path;
		} else {
			return false;
		}

		update_post_meta( $post_id, 'sfwd_lessons-assignment', $new_assignmnt_meta );

		$post            = get_post( $post_id );
		$course_id       = bbapp_learndash_get_course_id( $post->ID );
		$assignment_meta = array(
			'file_name'    => $fname,
			'file_link'    => $dest,
			'user_name'    => $username,
			'disp_name'    => $dispname,
			'file_path'    => rawurlencode( $file_path . $fname ),
			'user_id'      => $current_user->ID,
			'lesson_id'    => $post->ID,
			'course_id'    => $course_id,
			'lesson_title' => $post->post_title,
			'lesson_type'  => $post->post_type,
		);

		if ( defined( 'LEARNDASH_VERSION' ) ) {
			$assignment_meta['learndash_version'] = LEARNDASH_VERSION;
		}

		$points_enabled  = learndash_get_setting( $post, 'lesson_assignment_points_enabled' );

		if ( 'on' === $points_enabled ) {
			$assignment_meta['points'] = 'Pending';
		}

		$assignment         = array(
			'post_title'   => sprintf(
			// translators: placeholder: assignment file title.
				__( 'Assignment %s', 'learndash' ),
				sanitize_text_field( $file_uploaded_name )
			),
			'post_type'    => 'sfwd-assignment',
			'post_status'  => 'publish',
			'post_content' => "<a href='" . $dest . "' target='_blank'>" . $fname . '</a>',
			'post_author'  => $current_user->ID,
		);
		$assignment_post_id = wp_insert_post( $assignment );
		$auto_approve       = learndash_get_setting( $post, 'auto_approve_assignment' );

		if ( $assignment_post_id ) {
			foreach ( $assignment_meta as $key => $value ) {
				update_post_meta( $assignment_post_id, $key, $value );
			}

			/**
			 * Run action hook after assignment is uploaded
			 *
			 * @param int   $assignment_post_id           Newly created assignment post ID which the
			 *                                            assignment is uploaded to
			 * @param array $assignment_meta              Assignment meta data
			 *
			 * @since 2.2
			 */
			do_action( 'learndash_assignment_uploaded', $assignment_post_id, $assignment_meta );
		}

		if ( ! empty( $auto_approve ) ) {
			learndash_approve_assignment( $current_user->ID, $post_id, $assignment_post_id );

			// Assign full points if auto approve & points are enabled.
			if ( 'on' === $points_enabled ) {
				$points = learndash_get_setting( $post, 'lesson_assignment_points_amount' );
				update_post_meta( $assignment_post_id, 'points', intval( $points ) );
			}
		}

		return $assignment_post_id;
	}
}

/**
 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/view/WpProQuiz_View_FrontQuiz.php
 * Ref_Funcation: fetchCloze
 */
if ( ! function_exists( 'bbapp_lms_fetchCloze' ) ) {
	/**
	 * Fetch Fill in blank (cloze) question data.
	 *
	 * @param string $answer_text Answer text.
	 *
	 * @return array
	 */
	function bbapp_lms_fetchCloze( $answer_text ) { //phpcs:ignore WordPress.NamingConventions.ValidFunctionName.FunctionNameInvalid
		_deprecated_function( __FUNCTION__, '1.7.90', 'bbapp_lms_fetch_cloze' );

		return bbapp_lms_fetch_cloze( $answer_text );
	}
}

/**
 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/view/WpProQuiz_View_FrontQuiz.php
 * Ref_Funcation: fetchCloze
 */
if ( ! function_exists( 'bbapp_lms_fetch_cloze' ) ) {
	/**
	 * Fetch Fill in blank (cloze) question data.
	 *
	 * @param string $answer_text Answer text.
	 *
	 * @return array
	 */
	function bbapp_lms_fetch_cloze( $answer_text ) {
		preg_match_all( '#\{(.*?)(?:\|(\d+))?(?:[\s]+)?\}#im', $answer_text, $matches, PREG_SET_ORDER );

		$data = array();

		foreach ( $matches as $k => $v ) {
			$text            = $v[1];
			$points          = ! empty( $v[2] ) ? (int) $v[2] : 1;
			$row_text        = array();
			$multi_text_data = array();
			$len             = array();

			if ( preg_match_all( '#\[(.*?)\]#im', $text, $multi_text_matches ) ) {
				foreach ( $multi_text_matches[1] as $multi_text ) {
					if ( function_exists( 'mb_strtolower' ) ) {
						$x = mb_strtolower( trim( html_entity_decode( $multi_text, ENT_QUOTES ) ) );
					} else {
						$x = strtolower( trim( html_entity_decode( $multi_text, ENT_QUOTES ) ) );
					}

					$len[]             = strlen( $x );
					$multi_text_data[] = $x;
					$row_text[]        = $multi_text;
				}
			} else {
				if ( function_exists( 'mb_strtolower' ) ) {
					$x = mb_strtolower( trim( html_entity_decode( $text, ENT_QUOTES ) ) );
				} else {
					$x = strtolower( trim( html_entity_decode( $text, ENT_QUOTES ) ) );
				}

				$len[]             = strlen( $x );
				$multi_text_data[] = $x;
				$row_text[]        = $text;
			}

			$a  = '<span class="wpProQuiz_cloze"><input data-wordlen="' . max( $len ) . '" type="text" value=""> ';
			$a .= '<span class="wpProQuiz_clozeCorrect" style="display: none;"></span></span>';

			$data['correct'][] = $multi_text_data;
			$data['points'][]  = $points;
			$data['data'][]    = $a;
		}

		$data['replace'] = preg_replace( '#\{(.*?)(?:\|(\d+))?(?:[\s]+)?\}#im', '@@wpProQuizCloze@@', $answer_text );

		return $data;
	}
}

/**
 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/view/WpProQuiz_View_FrontQuiz.php
 * Ref_Funcation: fetchAssessment
 */
if ( ! function_exists( 'bbapp_lms_fetchAssessment' ) ) {
	/**
	 * Fetch assessments.
	 *
	 * @param string $answer_text Answer text.
	 *
	 * @return array
	 */
	function bbapp_lms_fetchAssessment( $answer_text ) { //phpcs:ignore WordPress.NamingConventions.ValidFunctionName.FunctionNameInvalid
		_deprecated_function( __FUNCTION__, '1.7.90', 'bbapp_lms_fetch_assessment' );

		return bbapp_lms_fetch_assessment( $answer_text );
	}
}

/**
 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/view/WpProQuiz_View_FrontQuiz.php
 * Ref_Funcation: fetchAssessment
 */
if ( ! function_exists( 'bbapp_lms_fetch_assessment' ) ) {
	/**
	 * Fetch assessments.
	 *
	 * @param string $answer_text Answer text.
	 *
	 * @return array
	 */
	function bbapp_lms_fetch_assessment( $answer_text ) {
		preg_match_all( '#\{(.*?)\}#im', $answer_text, $matches );

		$data                  = array();
		$data['assessmetTemp'] = array();

		for ( $i = 0, $ci = count( $matches[1] ); $i < $ci; $i ++ ) {
			$match = $matches[1][ $i ];

			preg_match_all( '#\[([^\|\]]+)(?:\|(\d+))?\]#im', $match, $ms );

			for ( $j = 0, $cj = count( $ms[1] ); $j < $cj; $j ++ ) {
				$v                             = $ms[1][ $j ];
				$data['assessmetTemp'][ $i ][] = array(
					'text' => $v,
					'val'  => $j + 1,
				);
			}
		}

		$data['replace'] = preg_replace( '#\{(.*?)\}#im', '@@wpProQuizAssessment@@', $answer_text );

		return $data;
	}
}

/**
 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/controller/WpProQuiz_Controller_Admin.php
 * Ref_Funcation: completedQuiz
 */
if ( ! function_exists( 'bbapp_lms_completedQuiz' ) ) {
	/**
	 * If the quiz is completed.
	 *
	 * @return array|bool
	 */
	function bbapp_lms_completedQuiz() { //phpcs:ignore WordPress.NamingConventions.ValidFunctionName.FunctionNameInvalid
		_deprecated_function( __FUNCTION__, '1.7.90', 'bbapp_lms_completed_quiz' );

		return bbapp_lms_completed_quiz();
	}
}

/**
 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/controller/WpProQuiz_Controller_Admin.php
 * Ref_Funcation: completedQuiz
 */
if ( ! function_exists( 'bbapp_lms_completed_quiz' ) ) {
	/**
	 * If the quiz is completed.
	 *
	 * @return array|bool
	 */
	function bbapp_lms_completed_quiz() {
		// First we unpack the $_POST['results'] string.
		if ( ( ! empty( $_POST['results'] ) ) && ( is_string( $_POST['results'] ) ) ) {
			$_POST['results'] = json_decode( wp_unslash( $_POST['results'] ), true ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}

		if ( is_user_logged_in() ) {
			$user_id = get_current_user_id();
		} else {
			$user_id = 0;
		}

		if ( isset( $_POST['quizId'] ) ) {
			$id = ( ! empty( $_POST['quizId'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['quizId'] ) ) : 0; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		} else {
			$id = 0;
		}

		if ( isset( $_POST['quiz'] ) ) {
			$quiz_post_id = ( ! empty( $_POST['quiz'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['quiz'] ) ) : 0; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		} else {
			$quiz_post_id = 0;
		}

		// LD 2.4.3 - Change in logic. Instead of accepting the values for points, correct etc from JS we now pass the 'results' array on the complete quiz
		// AJAX action. This now let's us verify the points, correct answers etc. as each have a unique nonce.
		$total_awarded_points = 0;
		$total_correct        = 0;

		// If the results is not present then abort.
		if ( ! isset( $_POST['results'] ) ) {
			return array(
				'text'  => __( 'An error has occurred.', 'buddyboss-app' ),
				'clear' => true,
			);
		}

		// Loop over the 'results' items. We verify and tally the points+correct counts as well as the student response 'data'. When we get to the 'comp' results element
		// we set the award points and correct as well as determine the total possible points.
		// @TODO Need to test how this works with variable question quizzes.
		foreach ( $_POST['results'] as $r_idx => $result ) { //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.CSRF.NonceVerification, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
			if ( 'comp' === $r_idx ) {
				$_POST['results'][ $r_idx ]['points']           = intval( $total_awarded_points );
				$_POST['results'][ $r_idx ]['correctQuestions'] = intval( $total_correct );

				continue;
			}

			$points_array = array(
				'points'         => function_exists( 'learndash_format_course_points' ) ? learndash_format_course_points( $result['points'] ) : intval( $result['points'] ),
				'correct'        => intval( $result['correct'] ),
				'possiblePoints' => function_exists( 'learndash_format_course_points' ) ? learndash_format_course_points( $result['possiblePoints'] ) : intval( $result['possiblePoints'] ),
			);

			if ( false === $points_array['correct'] ) {
				$points_array['correct'] = 0;
			} elseif ( true === $points_array['correct'] ) {
				$points_array['correct'] = 1;
			}

			$points_str = maybe_serialize( $points_array );

			if ( ! wp_verify_nonce( $result['p_nonce'], 'ld_quiz_pnonce' . $user_id . '_' . $id . '_' . $quiz_post_id . '_' . $r_idx . '_' . $points_str ) ) {
				$_POST['results'][ $r_idx ]['points']         = 0;
				$_POST['results'][ $r_idx ]['correct']        = 0;
				$_POST['results'][ $r_idx ]['possiblePoints'] = 0;
			}

			$total_awarded_points += isset( $_POST['results'][ $r_idx ]['points'] ) ? intval( $_POST['results'][ $r_idx ]['points'] ) : 0;
			$total_correct        += isset( $_POST['results'][ $r_idx ]['correct'] ) ? intval( $_POST['results'][ $r_idx ]['correct'] ) : 0;
			$response_str          = maybe_serialize( $result['data'] );

			if ( ! wp_verify_nonce( $result['a_nonce'], 'ld_quiz_anonce' . $user_id . '_' . $id . '_' . $quiz_post_id . '_' . $r_idx . '_' . $response_str ) ) {
				$_POST['results'][ $r_idx ]['data'] = array();
			}
		}

		$quiz = new QuizComplete();

		return $quiz->completedQuiz();
	}
}

if ( ! function_exists( 'bbapp_lms_taxonomy_image_url' ) ) {
	/**
	 * Get taxonomy imag url.
	 *
	 * @param int    $term_id            Term id.
	 * @param string $size               Image size.
	 * @param string $return_placeholder Placeholder.
	 */
	function bbapp_lms_taxonomy_image_url( $term_id = null, $size = 'full', $return_placeholder = false ) {
		if ( function_exists( 'bbapp_taxonomy_custom_icon_url' ) ) {
			return bbapp_taxonomy_custom_icon_url( $term_id, $size, $return_placeholder );
		}
	}
}

if ( ! function_exists( 'bbapp_learndash_get_course_id' ) ) {
	/**
	 * Get course id.
	 *
	 * @param int  $id        Id.
	 * @param bool $bypass_cb Bypass callback.
	 *
	 * @return int|mixed|string
	 */
	function bbapp_learndash_get_course_id( $id = null, $bypass_cb = false ) {
		global $wpdb;

		if ( ! empty( $_GET['course_id'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			return intval( bbapp_input_clean( wp_unslash( $_GET['course_id'] ) ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		}

		// Somewhat a kludge. Here we try ans assume the course_id being handled.
		if ( 'yes' === LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'shared_steps' ) && false === $bypass_cb ) {
			$course_get     = ( ! empty( $_GET['course'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['course'] ) ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$course_id_post = ( ! empty( $_POST['course_id'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['course_id'] ) ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Missing
			$post_get       = ( ! empty( $_GET['post'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['post'] ) ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( ! empty( $course_get ) ) {
				return intval( $course_get );
			} elseif ( ! empty( $course_id_post ) ) {
				return intval( $course_id_post );
			} elseif ( ! empty( $post_get ) ) {
				if ( 'sfwd-courses' === get_post_type( intval( $post_get ) ) ) {
					return intval( $post_get );
				}
			} else {
				// Get first course id.
				if ( is_object( $id ) && $id->ID ) {
					$id = $id->ID;
				}

				if ( is_numeric( $id ) ) {
					$course_id = $wpdb->get_var( $wpdb->prepare( "SELECT `meta_value` FROM $wpdb->postmeta WHERE `post_id` = %d AND `meta_key`", $id ) . "LIKE '%ld_course_%' ORDER BY `meta_id` DESC LIMIT 1" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

					if ( ! empty( $course_id ) ) {
						return $course_id;
					}
				}
			}
		}

		if ( is_object( $id ) && $id->ID ) {
			$p  = $id;
			$id = $p->ID;
		} elseif ( is_numeric( $id ) ) {
			$p = get_post( $id );
		}

		if ( empty( $p->ID ) ) {
			return 0;
		}

		if ( 'sfwd-courses' === $p->post_type ) {
			return $p->ID;
		}

		return get_post_meta( $id, 'course_id', true );
	}
}

if ( ! function_exists( 'bbapp_learndash_get_lesson_id' ) ) {
	/**
	 * Get lesson id.
	 *
	 * @param int $post_id   Post id.
	 * @param int $course_id Course id.
	 *
	 * @return int|mixed
	 */
	function bbapp_learndash_get_lesson_id( $post_id = null, $course_id = null ) {
		if ( 'yes' === LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'shared_steps' ) ) {
			if ( empty( $course_id ) ) {
				$course_id = bbapp_learndash_get_course_id( $post_id );
			}

			if ( ! empty( $course_id ) ) {
				return learndash_course_get_single_parent_step( $course_id, $post_id );
			}
		}

		return get_post_meta( $post_id, 'lesson_id', true );
	}
}

if ( ! function_exists( 'bbapp_learndash_fix_html' ) ) {
	/**
	 * Function unclosed fixes the provided html.
	 * Dom feature is not exists it will trim html and provide plain text.
	 *
	 * @param string $html Html.
	 *
	 * @return string
	 */
	function bbapp_learndash_fix_html( $html ) {
		if ( ! class_exists( 'DOMDocument' ) ) {
			return wp_strip_all_tags( $html );
		}

		$doc = new \DOMDocument();

		$doc->loadHTML( '<?xml encoding=\"utf-8\" ?>' . $html );

		$body = $doc->documentElement->lastChild; //phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		$html = '';

		foreach ( $body->childNodes as $node ) { //phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			$html .= $doc->saveHTML( $node );
		}

		return $html;
	}
}

if ( ! function_exists( 'bbapp_learndash_get_course_meta_setting' ) ) {
	/**
	 * Deprecated support for get_course_meta_setting
	 *
	 * @param int    $course_id   Course id.
	 * @param string $setting_key Settings key.
	 */
	function bbapp_learndash_get_course_meta_setting( $course_id, $setting_key ) {
		return learndash_get_course_meta_setting( $course_id, $setting_key );
	}
}

if ( ! function_exists( 'is_rest' ) ) {
	/**
	 * This will check request is rest or not.
	 *
	 * @return bool
	 */
	function is_rest() {
		return ( str_contains( $_SERVER['REQUEST_URI'], '/wp-json' ) ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.InputNotValidated, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
	}
}

if ( ! function_exists( 'bbapp_learndash_remove_filters_for_anonymous_class' ) ) {
	/**
	 * This function is Created to remove hook of WordPress using classNme instead of class object.
	 *
	 * @param string $tag    Tag name.
	 * @param string $class  Class name.
	 * @param string $method Method name.
	 */
	function bbapp_learndash_remove_filters_for_anonymous_class( $tag, $class, $method ) {
		$filters = $GLOBALS['wp_filter'][ $tag ];

		if ( empty( $filters ) ) {
			return;
		}

		foreach ( $filters as $priority => $filter ) {
			foreach ( $filter as $identifier => $function ) {
				if ( is_array( $function ) && is_a( $function['function'][0], $class ) && $method === $function['function'][1] ) {
					remove_filter( $tag, array( $function['function'][0], $method ), $priority );
				}
			}
		}
	}
}

if ( ! function_exists( 'bbapp_learndash_comments_support_enabled' ) ) {
	/**
	 * Check if comments support is enabled for the post type.
	 *
	 * @param null|string $post_type Post type.
	 *
	 * @since 1.8.40
	 *
	 * @return bool
	 */
	function bbapp_learndash_comments_support_enabled( $post_type = null ) {
		$focus_mode = LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Theme_LD30', 'focus_mode_enabled' );

		if ( is_user_logged_in() && 'yes' === $focus_mode && comments_open() ) {
			return true;
		}

		$post_type_comments = learndash_post_type_supports_comments( $post_type );
		if ( true === $post_type_comments && comments_open() ) {
			return true;
		}

		return false;
	}
}

if ( ! function_exists( 'bbapp_ld_get_course_status' ) ) {

	/**
	 * Get Course status.
	 *
	 * @param WP_Post|int $course_post Course post.
	 * @since 1.8.50
	 *
	 * @return array course status.
	 */
	function bbapp_ld_get_course_status( $course_post ) {
		$course_status = array();

		// If LearnDash is not active or version is less than 4.7.0, return empty array.
		if ( ! bbapp_is_learndash_enabled() ) {
			return $course_status;
		}

		if ( is_numeric( $course_post ) ) {
			$course_post = get_post( $course_post );
		}

		if ( is_wp_error( $course_post ) || empty( $course_post ) ) {
			return $course_status;
		}

		$ld_product = \LearnDash\Core\Models\Product::find( $course_post->ID );

		if ( empty( $ld_product ) ) {
			return $course_status;
		}

		$course_status['seats_available']    = $ld_product->get_seats_available();
		$course_status['is_seats_available'] = 0 !== $course_status['seats_available'];
		$course_status['text']               = '';

		// If seats are available, show the button.
		if ( false === $course_status['is_seats_available'] ) {
			$course_status['btn_text'] = sprintf(
				// translators: placeholder: Course label.
				esc_html_x( '%s is full', 'placeholder: Course label', 'buddyboss-app' ),
				\LearnDash_Custom_Label::get_label( 'course' )
			);
		}

		if ( ! empty( $course_status['seats_available'] ) ) {
			$course_status['text'] = sprintf(
				// translators: placeholder: number of places remaining.
				_nx(
					'(%s place remaining)',
					'(%s places remaining)',
					$course_status['seats_available'],
					'placeholder: number of places remaining',
					'buddyboss-app'
				),
				number_format_i18n( $course_status['seats_available'] )
			);
		}

		if ( $ld_product->has_ended() ) {
			$course_status['tooltip'] = sprintf(
				// translators: placeholder: course.
				esc_attr_x( 'This %s has ended', 'placeholder: course', 'buddyboss-app' ),
				esc_html( learndash_get_custom_label_lower( 'course' ) )
			);
			$course_status['is_ended'] = true;
			$course_status['text']     = esc_html__( 'Ended', 'buddyboss-app' );
			$course_status['btn_text'] = sprintf(
				// translators: placeholder: Course label.
				esc_html_x( '%s ended', 'placeholder: Course label', 'buddyboss-app' ),
				\LearnDash_Custom_Label::get_label( 'course' )
			);
		} elseif ( ! $ld_product->has_started() ) {

			if ( ! $ld_product->can_be_purchased() ) {
				$course_status['tooltip'] = sprintf(
					// translators: placeholder: course, course start date.
					esc_attr_x( 'This %1$s starts on %2$s', 'placeholder: course, course start date', 'buddyboss-app' ),
					esc_html( learndash_get_custom_label_lower( 'course' ) ),
					esc_html( learndash_adjust_date_time_display( $ld_product->get_start_date() ) )
				);
			} else {
				$course_status['tooltip'] = sprintf(
					// translators: placeholder: course, course start date.
					esc_attr_x( 'It is a pre-order. Enroll in this %1$s to get access after %2$s', 'placeholder: course', 'buddyboss-app' ),
					esc_html( learndash_get_custom_label_lower( 'course' ) ),
					esc_html( learndash_adjust_date_time_display( $ld_product->get_start_date() ) )
				);
			}

			$course_status['is_pre_order'] = true;
			$course_status['text']         = sprintf(
				// translators: placeholder: course pre-order.
				esc_attr_x( 'Pre-order %s', 'placeholder: course pre-order', 'buddyboss-app' ),
				esc_html( $course_status['text'] )
			);
		} else {

			if ( $ld_product->can_be_purchased() ) {
				$course_status['tooltip'] = sprintf(
				// translators: placeholder: course.
					esc_attr_x( 'Enroll in this %s to get access', 'placeholder: course', 'buddyboss-app' ),
					esc_html( learndash_get_custom_label_lower( 'course' ) )
				);
			} else {
				$course_status['tooltip'] = sprintf(
				// translators: placeholder: course.
					esc_attr_x( 'This %s is not available', 'placeholder: course', 'buddyboss-app' ),
					esc_html( learndash_get_custom_label_lower( 'course' ) )
				);
			}
			$course_status['text'] = sprintf(
				// translators: placeholder: course not enrolled.
				esc_attr_x( 'Not Enrolled %s', 'placeholder: course not enrolled', 'buddyboss-app' ),
				esc_html( $course_status['text'] )
			);
		}

		// If the course is pre-ordered, show the button.
		if ( $ld_product->is_pre_ordered() ) {
			$course_status['btn_text'] = sprintf(
				// translators: placeholder: Course label.
				esc_html_x( '%s pre-ordered', 'placeholder: Course label', 'buddyboss-app' ),
				\LearnDash_Custom_Label::get_label( 'course' )
			);
		}

		return $course_status;
	}
}
