<?php
/**
 * Holds Learndash common functionality.
 *
 * @package BuddyBossApp\Api\LearnDash
 */

namespace BuddyBossApp\Api\LearnDash;

use LD_QuizPro;
use WpProQuiz_Model_QuestionMapper;
use WpProQuiz_Model_QuizMapper;

/**
 * Learndash common controller.
 */
class LearndashCommon {
	/**
	 * LearndashCommon constructor.
	 */
	public function __construct() {
		$this->hooks();
	}

	/**
	 * Define all Hooks.
	 */
	public function hooks() {
		add_action( 'learndash_post_args', array( $this, 'rest_support_learndash_taxonomy' ), 99 );
		add_action( 'bbapp_ld_taxonomies_ld_course_category_args', array( $this, 'default_term_support_learndash_taxonomy' ), 10, 2 );
		add_filter( 'bbapp_rest_search_tab', array( $this, 'add_search_tabs' ) );
		add_filter( 'bbapp_settings_rest_response', array( $this, 'add_settings' ) );

		if ( is_rest() ) {
			/**
			 * Removing `wp_pro_quiz_completed_quiz` which is added in sfwd-lms/includes/quiz/ld-quiz-pro.php.
			 * As in above function ```echo json_encode( $results ); exit();``` added by core plugin but that stop our rest execution.
			 * For fixing above issue we removed core LD plugin hook and added custom hook here so we can removed ```echo json_encode( $results ); exit();``` code.
			 */
			bbapp_learndash_remove_filters_for_anonymous_class( 'wp_pro_quiz_completed_quiz', 'LD_QuizPro', 'wp_pro_quiz_completed' );
			add_filter( 'wp_pro_quiz_completed_quiz', array( $this, 'wp_pro_quiz_completed' ) );
		}

		add_action( 'learndash_update_course_access', array( $this, 'learndash_update_course_access_callback' ), 10, 4 );
	}

	/**
	 * Add Rest Support for learndash taxonomy withing post.
	 *
	 * @param array $post_args Post object.
	 *
	 * @return mixed
	 * @since 1.0.0
	 */
	public function rest_support_learndash_taxonomy( $post_args ) {

		foreach ( $post_args as $post_type => $cpt_args ) {
			if ( ! empty( $cpt_args['taxonomies'] ) ) {
				foreach ( $cpt_args['taxonomies'] as $taxonomy => $tax_args ) {
					if ( is_array( $tax_args ) ) {
						$tax_args = array_merge( $tax_args, array( 'show_in_rest' => true ) );
						$post_args[ $post_type ]['taxonomies'][ $taxonomy ] = apply_filters( 'bbapp_ld_taxonomies_' . $taxonomy . '_args', $tax_args, $post_type );
					}
				}
			}
		}

		return $post_args;
	}

	/**
	 * Add Default Term Support for learndash taxonomy withing post.
	 *
	 * @param array  $tax_args Taxonomy object.
	 * @param string $post_type Post type.
	 *
	 * @return mixed
	 * @since 1.0.0
	 */
	public function default_term_support_learndash_taxonomy( $tax_args, $post_type ) {
		if ( ! isset( $tax_args['default_term'] ) ) {
			$tax_args['default_term'] = __( 'Uncategorized', 'buddyboss-app' );
		}

		return $tax_args;
	}

	/**
	 * Add a require search tab into the app.
	 *
	 * @param array $tabs Tabs.
	 *
	 * @return array
	 */
	public function add_search_tabs( $tabs ) {
		$ld_tabs = array(
			array(
				'id'    => 'courses',
				'title' => __( 'Courses ', 'buddyboss-app' ),
			),
			array(
				'id'    => 'lessons',
				'title' => __( 'Lessons', 'buddyboss-app' ),
			),
		);

		return array_merge( $tabs, $ld_tabs );
	}

	/**
	 * Get learndash settings option.
	 *
	 * @param array $response Settings array.
	 *
	 * @return mixed
	 */
	public function add_settings( $response ) {
		global $sfwd_lms;
		$response['learndash'] = array(
			'labels'                            => array(
				'course'                        => learndash_get_custom_label( 'course' ),
				'courses'                       => learndash_get_custom_label( 'courses' ),
				'lesson'                        => learndash_get_custom_label( 'lesson' ),
				'lessons'                       => learndash_get_custom_label( 'lessons' ),
				'topic'                         => learndash_get_custom_label( 'topic' ),
				'topics'                        => learndash_get_custom_label( 'topics' ),
				'quiz'                          => learndash_get_custom_label( 'quiz' ),
				'quizzes'                       => learndash_get_custom_label( 'quizzes' ),
				'question'                      => learndash_get_custom_label( 'question' ),
				'questions'                     => learndash_get_custom_label( 'questions' ),
				'button_take_this_course'       => learndash_get_custom_label( 'button_take_this_course' ),
				'button_mark_complete'          => learndash_get_custom_label( 'button_mark_complete' ),
				'button_click_here_to_continue' => learndash_get_custom_label( 'button_click_here_to_continue' ),
			),
			'permalinks'                        => array(
				'courses' => $this->learndash_settings_permalinks( 'courses', 'courses' ),
				'lessons' => $this->learndash_settings_permalinks( 'lessons', 'lessons' ),
				'topics'  => $this->learndash_settings_permalinks( 'topics', 'topic' ),
				'quizzes' => $this->learndash_settings_permalinks( 'quizzes', 'quizzes' ),
				'groups'  => $this->learndash_settings_permalinks( 'groups', 'groups' ),
			),
			'is_course_mark_incomplete_enabled' => empty( \LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Management_Display', 'course_mark_incomplete_enabled', false ) ) ? false : true,
		);

		// Course Taxonomies.
		$courses_taxonomies = $sfwd_lms->get_post_args_section( 'sfwd-courses', 'taxonomies' );
		if ( ( isset( $courses_taxonomies['ld_course_category'] ) ) && ( true === $courses_taxonomies['ld_course_category']['public'] ) ) {
			$response['learndash']['permalinks']['ld_course_category'] = $this->learndash_settings_permalinks_taxonomies( 'ld_course_category', 'course-category' );
		}
		if ( ( isset( $courses_taxonomies['ld_course_tag'] ) ) && ( true === $courses_taxonomies['ld_course_tag']['public'] ) ) {
			$response['learndash']['permalinks']['ld_course_tag'] = $this->learndash_settings_permalinks_taxonomies( 'ld_course_tag', 'course-tag' );
		}
		// Lesson Taxonomies.
		$lessons_taxonomies = $sfwd_lms->get_post_args_section( 'sfwd-lessons', 'taxonomies' );
		if ( ( isset( $lessons_taxonomies['ld_lesson_category'] ) ) && ( true === $lessons_taxonomies['ld_lesson_category']['public'] ) ) {
			$response['learndash']['permalinks']['ld_lesson_category'] = $this->learndash_settings_permalinks_taxonomies( 'ld_lesson_category', 'lesson-category' );
		}
		if ( ( isset( $lessons_taxonomies['ld_lesson_tag'] ) ) && ( true === $lessons_taxonomies['ld_lesson_tag']['public'] ) ) {
			$response['learndash']['permalinks']['ld_lesson_tag'] = $this->learndash_settings_permalinks_taxonomies( 'ld_lesson_tag', 'lesson-tag' );
		}
		// Topic Taxonomies.
		$topics_taxonomies = $sfwd_lms->get_post_args_section( 'sfwd-topic', 'taxonomies' );
		if ( ( isset( $topics_taxonomies['ld_topic_category'] ) ) && ( true === $topics_taxonomies['ld_topic_category']['public'] ) ) {
			$response['learndash']['permalinks']['ld_topic_category'] = $this->learndash_settings_permalinks_taxonomies( 'ld_topic_category', 'topic-category' );
		}
		if ( ( isset( $topics_taxonomies['ld_topic_tag'] ) ) && ( true === $topics_taxonomies['ld_topic_tag']['public'] ) ) {
			$response['learndash']['permalinks']['ld_topic_tag'] = $this->learndash_settings_permalinks_taxonomies( 'ld_topic_tag', 'topic-tag' );
		}
		// Quiz Taxonomies.
		$quizzes_taxonomies = $sfwd_lms->get_post_args_section( 'sfwd-quiz', 'taxonomies' );
		if ( ( isset( $quizzes_taxonomies['ld_quiz_category'] ) ) && ( true === $quizzes_taxonomies['ld_quiz_category']['public'] ) ) {
			$response['learndash']['permalinks']['ld_quiz_category'] = $this->learndash_settings_permalinks_taxonomies( 'ld_quiz_category', 'quiz-category' );
		}
		if ( ( isset( $quizzes_taxonomies['ld_quiz_tag'] ) ) && ( true === $quizzes_taxonomies['ld_quiz_tag']['public'] ) ) {
			$response['learndash']['permalinks']['ld_quiz_tag'] = $this->learndash_settings_permalinks_taxonomies( 'ld_quiz_tag', 'quiz-tag' );
		}
		// Group Taxonomies.
		$groups_taxonomies = $sfwd_lms->get_post_args_section( 'groups', 'taxonomies' );
		if ( ( isset( $groups_taxonomies['ld_group_category'] ) ) && ( true === $groups_taxonomies['ld_group_category']['public'] ) ) {
			$response['learndash']['permalinks']['ld_group_category'] = $this->learndash_settings_permalinks_taxonomies( 'ld_group_category', 'group-category' );
		}
		if ( ( isset( $groups_taxonomies['ld_group_tag'] ) ) && ( true === $groups_taxonomies['ld_group_tag']['public'] ) ) {
			$response['learndash']['permalinks']['ld_group_tag'] = $this->learndash_settings_permalinks_taxonomies( 'ld_group_tag', 'group-tag' );
		}

		$response['learndash']['permalinks']['ld_group_tag'] = $this->learndash_settings_permalinks_taxonomies( 'ld_group_tag', 'group-tag' );
		$response['learndash']['permalinks']['my-courses']   = apply_filters( 'bp_learndash_profile_courses_slug', 'my-courses' );
		$response['learndash']['permalinks']['certificates'] = apply_filters( 'bp_learndash_profile_certificates_slug', 'certificates' );
		return $response;
	}

	/**
	 * Replica of `wp_pro_quiz_completed` in sfwd-lms/includes/quiz/ld-quiz-pro.php
	 * To remove ```echo json_encode( $results );```
	 *
	 * @param int $statistic_ref_id Post Id.
	 *
	 * @return false|string|void
	 */
	public function wp_pro_quiz_completed( $statistic_ref_id = 0 ) {

		$results      = array();
		$quiz_pro_id  = isset( $_POST['quizId'] ) ? absint( $_POST['quizId'] ) : null;
		$quiz_post_id = isset( $_POST['quiz'] ) ? absint( $_POST['quiz'] ) : null;
		$score        = isset( $_POST['results']['comp']['correctQuestions'] ) ? bbapp_input_clean( wp_unslash( $_POST['results']['comp']['correctQuestions'] ) ) : null; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$points       = isset( $_POST['results']['comp']['points'] ) ? absint( $_POST['results']['comp']['points'] ) : null;
		$result       = isset( $_POST['results']['comp']['result'] ) ? bbapp_input_clean( wp_unslash( $_POST['results']['comp']['result'] ) ) : null; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$timespent    = isset( $_POST['timespent'] ) ? floatval( $_POST['timespent'] ) : null;

		if ( ( is_null( $quiz_post_id ) ) || ( is_null( $quiz_pro_id ) ) || ( is_null( $points ) ) ) {
			return wp_json_encode( $results );
		}

		$course_id = ( ( isset( $_POST['course_id'] ) ) && ( intval( $_POST['course_id'] ) > 0 ) ) ? intval( $_POST['course_id'] ) : learndash_get_course_id( $quiz_pro_id ); //phpcs:ignore WordPress.Security.NonceVerification.Missing
		$lesson_id = ( ( isset( $_POST['lesson_id'] ) ) && ( intval( $_POST['lesson_id'] ) > 0 ) ) ? intval( $_POST['lesson_id'] ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Missing
		$topic_id  = ( ( isset( $_POST['topic_id'] ) ) && ( intval( $_POST['topic_id'] ) > 0 ) ) ? intval( $_POST['topic_id'] ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( is_user_logged_in() ) {
			$user_id = get_current_user_id();
		} else {
			$user_id = 0;
		}

		$quiz_mapper = new WpProQuiz_Model_QuizMapper();
		$quiz_pro    = $quiz_mapper->fetch( $quiz_pro_id );
		if ( ( ! $quiz_pro ) || ( ! is_a( $quiz_pro, 'WpProQuiz_Model_Quiz' ) ) ) {
			return wp_json_encode( $results );
		}
		$quiz_pro->setPostId( $quiz_post_id );

		$question_mapper = new WpProQuiz_Model_QuestionMapper();
		$questions       = $question_mapper->fetchAll( $quiz_pro );
		if ( is_array( $questions ) ) {
			$questions_count = count( $questions );
		}

		// Check if these set of questions has questions that need to be graded.
		$has_graded = LD_QuizPro::quiz_has_graded_question( $questions );

		// Store the id's of the graded question to be saved in usermeta.
		$graded = array();
		foreach ( $_POST['results'] as $question_id => $individual_result ) { //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if ( 'comp' === $question_id ) {
				continue;
			}

			if ( isset( $individual_result['graded_id'] ) && ! empty( $individual_result['graded_id'] ) ) {
				$graded[ $question_id ] = array(
					'post_id'        => intval( $individual_result['graded_id'] ),
					'status'         => esc_html( $individual_result['graded_status'] ),
					'points_awarded' => intval( $individual_result['points'] ),
				);
			}
		}

		if ( empty( $result ) ) {
			$total_points = 0;

			// Rewrote logic here to only count points for the questions shown to the user.
			// For example I might have a Quiz showing only 5 of 10 questions. In the above code
			// the points counted inlcude ALL 10 questions. Not correct.
			// Insead we do the logic below and only process the 5 shown questions.
			foreach ( $_POST['results'] as $question_id => $q_result ) { //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				if ( 'comp' === $question_id ) {
					continue;
				}

				if ( ( isset( $q_result['possiblePoints'] ) ) && ( ! empty( $q_result['possiblePoints'] ) ) ) {
					$total_points += intval( $q_result['possiblePoints'] );
				}
			}
		} else {
			$total_points = round( $points * 100 / $result );
		}

		$questions_shown_count = count( $_POST['results'] ) - 1;

		if ( ( isset( $_POST['quiz_nonce'] ) ) && ( isset( $_POST['quizId'] ) ) && ( isset( $_POST['quizId'] ) ) && ( ! empty( $user_id ) ) ) {
			if ( ! wp_verify_nonce( $_POST['quiz_nonce'], 'sfwd-quiz-nonce-' . absint( $_POST['quiz'] ) . '-' . absint( $_POST['quizId'] ) . '-' . $user_id ) ) { //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				return;
			}
		} elseif ( empty( $_SERVER['HTTP_REFERER'] ) ) {
			return;
		}

		$user_quiz_meta = get_user_meta( $user_id, '_sfwd-quizzes', true );
		$user_quiz_meta = maybe_unserialize( $user_quiz_meta );

		if ( ! is_array( $user_quiz_meta ) ) {
			$user_quiz_meta = array();
		}

		$quiz_post_settings = learndash_get_setting( $quiz_post_id );
		if ( ! is_array( $quiz_post_settings ) ) {
			$quiz_post_settings = array();
		}
		if ( ! isset( $quiz_post_settings['passingpercentage'] ) ) {
			$quiz_post_settings['passingpercentage'] = 0;
		}

		$passingpercentage = absint( $quiz_post_settings['passingpercentage'] );
		$pass              = ( $result >= $passingpercentage ) ? 1 : 0;

		$quizdata = array(
			'quiz'                => $quiz_post_id,
			'score'               => $score,
			'count'               => $questions_count,
			'question_show_count' => $questions_shown_count,
			'pass'                => $pass,
			'rank'                => '-',
			'time'                => time(),
			'pro_quizid'          => $quiz_pro_id,
			'course'              => $course_id,
			'lesson'              => $lesson_id,
			'topic'               => $topic_id,
			'points'              => absint( $points ),
			'total_points'        => absint( $total_points ),
			'percentage'          => $result,
			'timespent'           => $timespent,
			'has_graded'          => ( $has_graded ) ? true : false,
			'statistic_ref_id'    => absint( $statistic_ref_id ),
		);

		// On the timestamps below we divide against 1000 because they were generated via JavaScript which uses milliseconds.
		if ( isset( $_POST['results']['comp']['quizStartTimestamp'] ) ) {
			$quizdata['started'] = intval( $_POST['results']['comp']['quizStartTimestamp'] / 1000 );
		}
		if ( isset( $_POST['results']['comp']['quizEndTimestamp'] ) ) {
			$quizdata['completed'] = intval( $_POST['results']['comp']['quizEndTimestamp'] / 1000 );
		}

		if ( $graded ) {
			$quizdata['graded'] = $graded;
		}

		$user_quiz_meta[] = $quizdata;

		update_user_meta( $user_id, '_sfwd-quizzes', $user_quiz_meta );

		if ( ! empty( $course_id ) ) {
			$quizdata['course'] = get_post( $course_id );
		} else {
			$quizdata['course'] = 0;
		}

		if ( ! empty( $lesson_id ) ) {
			$quizdata['lesson'] = get_post( $lesson_id );
		} else {
			$quizdata['lesson'] = 0;
		}

		if ( ! empty( $topic_id ) ) {
			$quizdata['topic'] = get_post( $topic_id );
		} else {
			$quizdata['topic'] = 0;
		}

		$quizdata['questions'] = $questions;

		/**
		 * Does the action 'learndash_quiz_submitted'
		 *
		 * @param array  $quizdata
		 * @param object $current_user
		 *
		 * @since 3.0
		 */
		do_action( 'learndash_quiz_submitted', $quizdata, get_user_by( 'id', $user_id ) );

		/**
		 * Does the action 'learndash_quiz_completed'
		 *
		 * @param array  $quizdata
		 * @param object $current_user
		 *
		 * @since 2.1.0
		 */

		/**
		 * Changed in 2.6.0. If the quiz has essay type questions that are not
		 * auto-graded we don't send out the 'learndash_quiz_completed' action.
		 */
		$send_quiz_completed = true;
		if ( ( isset( $quizdata['has_graded'] ) ) && ( true === $quizdata['has_graded'] ) ) {
			if ( ( isset( $quizdata['graded'] ) ) && ( ! empty( $quizdata['graded'] ) ) ) {
				foreach ( $quizdata['graded'] as $grade_item ) {
					if ( ( isset( $grade_item['status'] ) ) && ( 'graded' !== $grade_item['status'] ) ) {
						$send_quiz_completed = false;
					}
				}
			}
		}

		if ( true === $send_quiz_completed ) {
			if ( ! empty( $courseid ) ) {
				learndash_process_mark_complete( $user_id, $quiz_post_id, false, $courseid );
			}

			do_action( 'learndash_quiz_completed', $quizdata, get_user_by( 'id', $user_id ) );
		} elseif ( defined( 'LEARNDASH_QUIZ_ESSAY_SUBMIT_COMPLETED' ) && LEARNDASH_QUIZ_ESSAY_SUBMIT_COMPLETED === true ) {
			do_action( 'learndash_quiz_completed', $quizdata, get_user_by( 'id', $user_id ) );
		}
	}

	/**
	 * Learndash permalink settings.
	 *
	 * @param string $key Post type slug.
	 * @param string $default Post type.
	 *
	 * @return mixed|string
	 */
	public function learndash_settings_permalinks( $key, $default = '' ) {
		$ld_permalink_options = get_option( 'learndash_settings_permalinks', array() );

		return isset( $ld_permalink_options[ $key ] ) ? $ld_permalink_options[ $key ] : $default;
	}

	/**
	 * Learndash taxonomy permalink settings.
	 *
	 * @param string $key Post taxonomy slug.
	 * @param string $default Post taxonomy slug.
	 *
	 * @return mixed|string
	 */
	public function learndash_settings_permalinks_taxonomies( $key, $default = '' ) {
		$ld_settings_permalinks_taxonomies = get_option( 'learndash_settings_permalinks_taxonomies', array() );

		return isset( $ld_settings_permalinks_taxonomies[ $key ] ) ? $ld_settings_permalinks_taxonomies[ $key ] : $default;
	}

	/**
	 * Clear transient when user enroll from front side.
	 *
	 * @param int     $user_id User ID.
	 * @param int     $course_id  Course ID.
	 * @param string  $course_access_list A comma-separated list of user IDs used for the course_access_list field.
	 * @param boolean $remove Whether to remove course access from the user.
	 */
	public function learndash_update_course_access_callback( $user_id, $course_id, $course_access_list, $remove ) {
		bbapp_delete_transients( 'buddyboss_app_ld_course_enrolled_users_' . $course_id );
	}
}
