<?php
/**
 * Holds product rest API functionality.
 *
 * @package BuddyBossApp\Api\InAppPurchases\V1\Products
 */

namespace BuddyBossApp\Api\InAppPurchases\V1\Products;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\InAppPurchases\Controller;
use WP_Error as WP_Error;
use WP_REST_Server;

/**
 * Product rest API class.
 */
final class ProductsRest {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * API namespace.
	 *
	 * @var string $namespace
	 */
	public $namespace = 'buddyboss-app/iap/v1';

	/**
	 * ProductsRest constructor.
	 */
	private function __construct() {
		// ... leave empty, see ProductsRest below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return object
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$classname      = __CLASS__;
			self::$instance = new $classname();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Magic methods for old methods.
	 *
	 * @param string             $name      Method name.
	 * @param string|mixed|array $arguments Method arguments.
	 *
	 * @since 1.7.5
	 *
	 * @return mixed|void
	 */
	public static function __callStatic( $name, $arguments ) {
		// Compatibility Support for CamelCase.
		$compatibility_func = array(
			'getBuddyBossAppProduct' => 'get_buddyboss_app_product',
		);

		foreach ( $compatibility_func as $old => $new ) {
			if ( $name === $old ) {
				_deprecated_function( esc_html( $old ), esc_html( bbapp()->plugin_version ), esc_html( $new ) );

				return call_user_func_array( array( self::instance(), $new ), $arguments );
			}
		}
	}

	/**
	 * Define all actions and filters here
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register endpoints related to this class
	 */
	public function register_routes() {
		$iap_integrations   = Controller::instance()->integrations;
		$integrations_types = wp_list_pluck( $iap_integrations, 'type' );

		// BuddyBossApp-IAP Products.
		register_rest_route(
			$this->namespace,
			'/products',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_bbapp_products' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'order'   => array(
							'description'       => esc_html__( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
							'default'           => 'desc',
							'type'              => 'string',
							'enum'              => array( 'asc', 'desc' ),
							'sanitize_callback' => 'sanitize_key',
							'validate_callback' => 'rest_validate_request_arg',
						),
						'orderby' => array(
							'description'       => esc_html__( 'Order by a specific parameter (default: id).', 'buddyboss-app' ),
							'default'           => 'id',
							'type'              => 'string',
							'sanitize_callback' => 'sanitize_key',
							'validate_callback' => 'rest_validate_request_arg',
						),
					),
				),
			)
		);

		// BuddyBossApp-IAP Product.
		register_rest_route(
			$this->namespace,
			'/products/(?P<id>\d+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_buddyboss_app_product' ),
					'permission_callback' => '__return_true',
					'args'                => array(),
				),
			)
		);

		// BuddyBossApp Product by item which is connected directly/indirectly with BuddyBossApp Product for all Integration.
		register_rest_route(
			$this->namespace,
			'/products/integration/(?P<item_id>\d+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_integrated_products_by_item' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'order'   => array(
							'description'       => esc_html__( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
							'default'           => 'desc',
							'type'              => 'string',
							'enum'              => array( 'asc', 'desc' ),
							'sanitize_callback' => 'sanitize_key',
							'validate_callback' => 'rest_validate_request_arg',
						),
						'orderby' => array(
							'description'       => esc_html__( 'Order by a specific parameter (default: id).', 'buddyboss-app' ),
							'default'           => 'id',
							'type'              => 'string',
							'sanitize_callback' => 'sanitize_key',
							'validate_callback' => 'rest_validate_request_arg',
						),
					),
				),
			)
		);

		// BuddyBossApp Product by items which is connected directly/indirectly with BuddyBossApp Product for given Integration.
		register_rest_route(
			$this->namespace,
			'/products/(?P<integration_type>\S+)/(?P<item_id>\d+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_integrated_products_by_integration_items' ),
					'args'                => array(
						'integration_type' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => esc_html__( 'Integration Type', 'buddyboss-app' ),
							'enum'        => $integrations_types,
						),
						'order'            => array(
							'description'       => esc_html__( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
							'default'           => 'desc',
							'type'              => 'string',
							'enum'              => array( 'asc', 'desc' ),
							'sanitize_callback' => 'sanitize_key',
							'validate_callback' => 'rest_validate_request_arg',
						),
						'orderby'          => array(
							'description'       => esc_html__( 'Order by a specific parameter (default: id).', 'buddyboss-app' ),
							'default'           => 'id',
							'type'              => 'string',
							'sanitize_callback' => 'sanitize_key',
							'validate_callback' => 'rest_validate_request_arg',
						),
					),
					'permission_callback' => '__return_true',
				),
			)
		);

		// BuddyBossApp Product by Integration for InApp-Purchase.
		register_rest_route(
			$this->namespace,
			'/products/(?P<integration_type>\S+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_bbapp_products_by_integration' ),
					'args'                => array(
						'integration_type' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => esc_html__( 'Integration Type', 'buddyboss-app' ),
							'enum'        => $integrations_types,
						),
						'order'            => array(
							'description'       => esc_html__( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
							'default'           => 'desc',
							'type'              => 'string',
							'enum'              => array( 'asc', 'desc' ),
							'sanitize_callback' => 'sanitize_key',
							'validate_callback' => 'rest_validate_request_arg',
						),
						'orderby'          => array(
							'description'       => esc_html__( 'Order by a specific parameter (default: id).', 'buddyboss-app' ),
							'default'           => 'id',
							'type'              => 'string',
							'sanitize_callback' => 'sanitize_key',
							'validate_callback' => 'rest_validate_request_arg',
						),
					),
					'permission_callback' => '__return_true',
				),
			)
		);
	}

	/**
	 * Get bbapp products.
	 *
	 * @param object $request Request data.
	 *
	 * @return WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/iap/v1/products Products
	 * @apiName        GetBuddyBossAppIAPProducts
	 * @apiGroup       In-App Purchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription BuddyBossApp product(s) for In-App Purchase purpose
	 * @apiUse         apidocForGetIapProducts
	 */
	public function get_bbapp_products( $request ) {
		$default_args = array(
			'orderby' => 'id',
			'order'   => 'desc',
		);
		$args         = array_merge( $default_args, $request->get_params() );
		$include      = ! empty( $request->get_param( 'include' ) ) ? $request->get_param( 'include' ) : array();

		if ( is_array( $include ) ) {
			// NOTE : Removing empty elements below.
			$include = array_filter( $include );
			if ( ! empty( $include ) ) {
				$args['include'] = $include;
			}
		}

		if ( isset( $request['global_subscription'] ) ) {
			$args['global_subscription'] = (bool) $request['global_subscription'];
		}

		$products = Helpers::get_products( $args );

		if ( is_wp_error( $products ) ) {
			return new WP_Error( 'error_while_retrieving', __( 'Error encountered while retrieving product by Id.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		return rest_ensure_response( $products );
	}

	/**
	 * Get BB App product.
	 *
	 * @param object $request Request data.
	 *
	 * @return WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/iap/v1/products/:id Product
	 * @apiName        GetBuddyBossAppIAPProduct
	 * @apiGroup       In-App Purchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription BuddyBossApp product for In-App Purchase purpose
	 * @apiUse         apidocForGetIapProduct
	 */
	public function get_buddyboss_app_product( $request ) {
		$product = Helpers::get_product_by_id( $request['id'] );

		if ( is_wp_error( $product ) ) {
			return new WP_Error( 'error_while_retrieving', __( 'Error encountered while retrieving product by Id.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		return rest_ensure_response( $product );
	}

	/**
	 * Get products by integration.
	 *
	 * @param object $request Request data.
	 *
	 * @return WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/iap/v1/products/:integration_type Products by integration
	 * @apiName        GetIAPProductsByIntegration
	 * @apiGroup       In-App Purchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get BuddyBossApp product(s) by integration type for In-App Purchase
	 * @apiUse         apidocForGetIapProductBYIntegration
	 */
	public function get_bbapp_products_by_integration( $request ) {
		$integration_type    = $request->get_param( 'integration_type' );
		$default_args        = array(
			'orderby' => 'id',
			'order'   => 'desc',
		);
		$args                = array_merge( $default_args, $request->get_params() );
		$integrated_products = Helpers::search_integration( null, $integration_type, $args );

		if ( is_wp_error( $integrated_products ) ) {
			return new WP_Error( 'error_while_searching', __( 'Error encountered while searching integration.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		return rest_ensure_response( $integrated_products );
	}

	/**
	 * Get integrated products by integration time.
	 *
	 * @param object $request Request data.
	 *
	 * @return WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/iap/v1/products/:integration_type/:item_id Integrated products
	 * @apiName        GetIAPProductsByIntegrationItems
	 * @apiGroup       In-App Purchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get BuddyBossApp product(s) where particular integration item of given integration_type is attached/linked for InAppPurchase.
	 * @apiUse         apidocForGetIAPProductsByIntegrationItems
	 */
	public function get_integrated_products_by_integration_items( $request ) {
		$integration_type    = $request->get_param( 'integration_type' );
		$item_id             = $request->get_param( 'item_id' );
		$default_args        = array(
			'orderby' => 'id',
			'order'   => 'desc',
		);
		$integrated_products = Helpers::search_integration( $item_id, $integration_type, $default_args );

		if ( is_wp_error( $integrated_products ) ) {
			return new WP_Error( 'error_while_searching', __( 'Error encountered while searching integration.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		return rest_ensure_response( $integrated_products );
	}

	/**
	 * This callback strictly used for learndash courses.
	 *
	 * @param object $request Request data.
	 *
	 * @return WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/iap/v1/products/integration/:item_id Integrated products
	 * @apiName        GetIAPProductsByItem
	 * @apiGroup       In-App Purchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get BuddyBossApp product(s) where particular item is attached/linked for In-App Purchase.
	 * @apiUse         apidocForGetIAPProductsByItem
	 * @apiDeprecated  This API is deprecated
	 * @deprecated
	 */
	public function get_integrated_products_by_item( $request ) {

		$item_id             = $request->get_param( 'item_id' );
		$integrated_products = Helpers::search_integration( $item_id, '' ); // LearnDash group now working when we pass integration type.

		return rest_ensure_response( $integrated_products );
	}
}
