<?php
/**
 * Order rest version 1
 *
 * @package BuddyBossApp\Api\InAppPurchases\V1\Orders
 */

namespace BuddyBossApp\Api\InAppPurchases\V1\Orders;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\InAppPurchases\IntegrationAbstract;
use BuddyBossApp\InAppPurchases\Orders;
use BuddyBossApp\InAppPurchases\StoreAbstract;
use WP_Error as WP_Error;
use WP_REST_Server;

/**
 * Class OrdersRest
 *
 * @package BuddyBossApp\Api\InAppPurchases\V1\Orders
 */
final class OrdersRest {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * Rest namespace.
	 *
	 * @var string $namespace
	 */
	public $namespace = 'buddyboss-app/iap/v1';

	/**
	 * Order-rest constructor.
	 */
	private function __construct() {
		// ... leave empty, see OrdersRest below.
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return OrdersRest
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * All hooks here
	 *
	 * @return void
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register all api endpoints for this class here
	 *
	 * @return void
	 */
	public function register_routes() {
		// Create Order Endpoint.
		register_rest_route(
			$this->namespace,
			'/order',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'rest_get_orders' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'device_platform' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => esc_html__( 'InAppPurchase Platform(also known as IAP Type)', 'buddyboss-app' ),
							'enum'        => array( 'ios', 'android' ),

						),
						'status'          => array(
							'required'    => true,
							'type'        => 'string',
							'default'     => 'all',
							'description' => esc_html__( 'Order status', 'buddyboss-app' ),
							'enum'        => array( 'all', 'active' ),
						),
					),
				),
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'rest_create_order' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'iap_receipt_token' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => esc_html__( 'InAppPurchase Token (receipt token)', 'buddyboss-app' ),
						),
						'device_platform'   => array(
							'required'    => true,
							'type'        => 'string',
							'description' => esc_html__( 'InAppPurchase Platform(also known as IAP Type)', 'buddyboss-app' ),
							'enum'        => array( 'ios', 'android' ),

						),
						'bbapp_product_id'  => array(
							'required'    => true,
							'type'        => 'string',
							'description' => esc_html__( 'Integration Id(also known as iap_product_id) or Store product id.', 'buddyboss-app' ),
						),
						'test_mode'         => array(
							'type'        => 'integer',
							'description' => esc_html__( 'Order is for testing or real order.', 'buddyboss-app' ),
						),
					),
				),
			)
		);

		// Refresh Status Order.
		register_rest_route(
			$this->namespace,
			'/refresh_order/(?P<order_id>\d+)',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'rest_refresh_order' ),
					'args'                => array(
						'order_id' => array(
							'required'    => true,
							'type'        => 'integer',
							'description' => esc_html__( 'InAppPurchase - Purchase ID.', 'buddyboss-app' ),
						),
					),
					'permission_callback' => '__return_true',
				),
			)
		);
	}

	/**
	 * Get orders.
	 *
	 * @param object $request Request data.
	 *
	 * @return WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/iap/v1/orders Orders
	 * @apiName        GetIAPOrders
	 * @apiGroup       In-App Purchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get In-App Purchase orders
	 * @apiUse         apidocForGetOrders
	 */
	public function rest_get_orders( $request ) {
		// Prevent unauthorised access.
		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'not_logged_in', esc_html__( 'Please login to view products', 'buddyboss-app' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$device_platform = $request->get_param( 'device_platform' );
		$status          = $request->get_param( 'status' );
		$order_args      = array(
			'user_id'         => get_current_user_id(),
			'device_platform' => $device_platform,
		);

		if ( 'active' === $status ) {
			$order_args['order_status'] = array(
				'subscribed',
				'completed',
			);
		} elseif ( 'expired' === $status ) {
			$order_args['order_status'] = array(
				'expired',
				'retrying',
			);
		}

		$orders     = array();
		$orders_obj = Orders::instance()->get_orders( $order_args );

		if ( ! empty( $orders_obj ) ) {
			$orders = $orders_obj;

			foreach ( $orders as $key => $order ) {
				if ( isset( $order->integration_types ) ) {
					if ( is_serialized( $order->integration_types ) ) {
						$integration_types = maybe_unserialize( $order->integration_types );

						if ( is_array( $integration_types ) ) {
							// NOTE : Overridden values.
							$order->integration_types = array_values( $integration_types );

							if ( isset( $order->item_ids ) ) {
								if ( is_serialized( $order->item_ids ) ) {
									$item_ids = maybe_unserialize( $order->item_ids );

									foreach ( $integration_types as $integration_type ) {
										if ( bbapp()->is_network_activated() ) {
											$order->item_ids = array( $integration_type => maybe_unserialize( $item_ids[ get_current_blog_id() ][ $integration_type ] ) );
										} else {
											$order->item_ids = array( $integration_type => maybe_unserialize( $item_ids[ $integration_type ] ) );
										}
									}
								}
							}
						}
					}
				}

				$order->transaction_id          = Orders::instance()->get_meta( $order->id, '_transaction_id', true );
				$order->original_transaction_id = Orders::instance()->get_meta( $order->id, '_parent_transaction_id', true );
				$order->transaction_date_ms     = Orders::instance()->get_meta( $order->id, '_transaction_date_ms', true );
			}
		}

		return rest_ensure_response( $orders );
	}

	/**
	 * Create order.
	 *
	 * @param object $request Request object.
	 *
	 * @return array|WP_Error
	 * @api            {POST} /wp-json/buddyboss-app/iap/v1/order Create order
	 * @apiName        CreateIAPOrder
	 * @apiGroup       In-App Purchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Create order for In-App Purchases
	 * @apiParam {String} iap_receipt_token InAppPurchase Token (receipt token)
	 * @apiParam {String=ios,android} device_platform InAppPurchase Platform (also known as InAppPurchase type)
	 * @apiParam {Number} bbapp_product_id BuddyBossApp InAppPurchase Product Id (also known as Integration Id, nothing related to store_product_id)
	 * <a href="#api-InAppPurchases-GetIAPProducts">Gettable at endpoint</a>
	 */
	public function rest_create_order( $request ) {
		// Prevent unauthorised access.
		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'not_logged_in', __( 'Please login to create order.', 'buddyboss-app' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$iap_receipt_token = $request->get_param( 'iap_receipt_token' );
		$device_platform   = $request->get_param( 'device_platform' );
		$bbapp_product_id  = (int) $request->get_param( 'bbapp_product_id' );
		$test_mode         = (int) $request->get_param( 'test_mode' );
		$test_mode         = ( ! empty( $test_mode ) && 1 === $test_mode );
		$user_id           = 0;

		if ( is_user_logged_in() ) {
			$user_id = get_current_user_id();
		}

		$blog_id = 1;

		if ( is_multisite() ) {
			$blog_id = get_current_blog_id();
		}

		// some validations.
		if ( ! isset( bbapp_iap()->iap[ $device_platform ] ) ) {
			/* translators: %s: Platform name. */
			return new WP_Error( 'invalid_device_platform', sprintf( esc_html__( 'InAppPurchase for Platform(%s) you requested is not available currently.', 'buddyboss-app' ), esc_html( $device_platform ) ), array( 'status' => 404 ) );
		}

		if ( empty( $bbapp_product_id ) ) {
			return new WP_Error( 'iap_product_id_missing', esc_html__( "InAppPurchase bbapp_product_id shouldn't be empty.", 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( empty( $iap_receipt_token ) ) {
			return new WP_Error( 'iap_receipt_token_missing', esc_html__( "InAppPurchase Receipt token shouldn't be empty.", 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		// Get current user login data and update order data and meta data.
		$userdata = get_userdata( $user_id );

		/**
		 * Store abstract data.
		 *
		 * @var $iap StoreAbstract
		 */
		$iap        = bbapp_iap()->iap[ $device_platform ];
		$order_args = array(
			'bbapp_product_id'  => $bbapp_product_id,
			'device_platform'   => $device_platform,
			'blog_id'           => $blog_id,
			'iap_receipt_token' => $iap_receipt_token,
			'user_id'           => $user_id,
			'user_email'        => $userdata->user_email,
			'test_mode'         => $test_mode,
		);
		$_order     = Orders::instance()->creating_order( $order_args, $iap );

		if ( is_wp_error( $_order ) ) {
			return $_order;
		}

		/**
		 * Only when use is only we need to complete the order in same time.
		 */
		if ( ! in_array( $_order->order_status, array( 'subscribed', 'completed' ), true ) ) {
			$order_args = array(
				'order_id'          => $_order->id,
				'iap_receipt_token' => $iap_receipt_token,
				'user_id'           => $user_id,
			);

			$_order = Orders::instance()->completing_order( $order_args );
		}
		if ( is_wp_error( $_order ) ) {
			return $_order;
		}

		return $this->rest_order_status_response( $_order );
	}

	/**
	 * Get the refresh order.
	 *
	 * @param object $request Request data.
	 *
	 * @return array|WP_Error
	 * @api            {PUT | PATCH} /wp-json/buddyboss-app/iap/v1/refresh_order/:order_id Refresh order
	 * @apiName        RefreshIAPOrder
	 * @apiGroup       In-App Purchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Fix the Order Permission & Status.(To provide user access if not)
	 * @apiParam {String} order_id In-App Purchase - Purchase ID
	 */
	public function rest_refresh_order( $request ) {
		// Prevent unauthorised access.
		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'not_logged_in', __( 'Please login to create order.', 'buddyboss-app' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$order_id = (int) $request->get_param( 'order_id' );
		$order    = Orders::instance()->get_by_id( $order_id );

		if ( empty( $order ) ) {
			return new WP_Error( 'order_not_found', esc_html__( 'Purchase not found.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		if ( ! isset( bbapp_iap()->iap[ $order->device_platform ] ) ) {
			return new WP_Error( 'invalid_device_platform', esc_html__( 'The IAP type you requested is currently not available.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		$integration_types = maybe_unserialize( $order->integration_types );
		$updated_counter   = 0;

		if ( in_array( $order->order_status, array( 'subscribed', 'completed' ), true ) ) {
			foreach ( $integration_types as $integration_type ) {
				if ( isset( bbapp_iap()->integration[ $integration_type ] ) ) {
					/**
					 * Integration data.
					 *
					 * @var $integration IntegrationAbstract
					 */
					$integration = bbapp_iap()->integration[ $integration_type ];

					// @todo : We need to pass itemId as 1st param in below
					// @Ayush asked not to worry about this endpoint currently
					if ( $integration->is_purchase_available( false, $order->id, $order->device_platform ) ) {
						// Make sure user has access.
						$integration->_on_order_completed( $order );
						$updated_counter ++;
					}
				} else {
					Orders::instance()->add_history( $order_id, 'warning', esc_html__( 'Requested purchase integration not found. The related plugin might not be activated.', 'buddyboss-app' ) );
				}
			}
		}

		$response            = $this->rest_order_status_response( $order );
		$response['updated'] = true;

		if ( count( $integration_types ) !== $updated_counter ) {
			$response['updated'] = false;
		}

		return $response;
	}

	/**
	 * Provides status of order.
	 *
	 * @param object $order Order data.
	 *
	 * @return array
	 */
	public function rest_order_status_response( $order ) {

		$response = array(
			'order_id'         => $order->id,
			'status'           => $order->order_status,
			'date_created_gmt' => mysql_to_rfc3339( $order->date_created ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			'date_updated_gmt' => mysql_to_rfc3339( $order->date_updated ), // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
		);

		return $response;
	}
}
