<?php
/**
 * Directory Block
 *
 * @package BuddyBossApp\Api\DirectoryBlock
 */

namespace BuddyBossApp\Api\DirectoryBlock;

use WP_Error;
use WP_Query;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Directory Block
 */
class Directory_Block {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	public static $instance;

	/**
	 * The namespace of this controller's route.
	 *
	 * @var string
	 */
	protected $namespace;

	/**
	 * The base of this controller's route.
	 *
	 * @var string
	 */
	protected $rest_base;

	/**
	 * Directory block constructor.
	 */
	public function __construct() {
		$this->namespace = 'buddyboss-app/v1';
		$this->rest_base = 'directory-block';
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return Directory_Block
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 *
	 * @since 2.3.40
	 */
	public function load() {
		$this->hooks();
	}

	/**
	 * Registers hooks or filters.
	 *
	 * @since 2.3.40
	 * @return void
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );
	}

	/**
	 * Register the REST API routes.
	 *
	 * @since 2.3.40
	 * @return void
	 */
	public function register_rest_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => 'GET',
					'callback'            => array(
						$this,
						'get_items',
					),
					'permission_callback' => array( $this, 'process_api_request' ),
				),
				array(
					'methods'             => 'POST',
					'callback'            => array(
						$this,
						'fetch_settings',
					),
					'permission_callback' => array( $this, 'process_api_request' ),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/filter',
			array(
				'methods'             => 'GET',
				'callback'            => array(
					$this,
					'get_filter',
				),
				'permission_callback' => array( $this, 'process_api_request' ),
			),
		);
	}

	/**
	 * Get items.
	 *
	 * @param \WP_REST_Request $request Request object.
	 *
	 * @since 2.3.40
	 * @return mixed
	 */
	public function get_items( $request ) {
		$args = array(
			'post_type'     => $request->get_param( 'post_type' ),
			'thumbnail'     => $request->get_param( 'thumbnail' ),
			'search'        => $request->get_param( 'search' ),
			'show_avatar'   => $request->get_param( 'show_avatar' ),
			'show_comment'  => $request->get_param( 'show_comment' ),
			'show_bookmark' => $request->get_param( 'show_bookmark' ),
			'page'          => $request->get_param( 'page' ),
			'per_page'      => $request->get_param( 'per_page' ),
			'order'         => $request->get_param( 'order' ),
			'order_by'      => $request->get_param( 'order_by' ),
			'is_preview'    => $request->get_param( 'is_preview' ),
		);

		$cpts = $this->get_public_rest_post_types();

		if (
			empty( $args['post_type'] ) ||
			! isset( $cpts[ $args['post_type'] ] )
		) {
			return new WP_Error(
				'bbapp_empty_block_type',
				__( 'Invalid post type.', 'buddyboss-app' ),
				array(
					'status' => 400,
				)
			);
		}

		foreach ( $request->get_params() as $key => $value ) {
			if ( 0 === strpos( $key, 'meta_' ) ) { // Check if the parameter starts with 'meta_'
				$args[ $key ] = $value;
			}
		}

		if ( empty( $args['page'] ) ) {
			$args['page'] = 1;
		}

		if ( empty( $args['per_page'] ) ) {
			$args['per_page'] = 10;
		}

		$query_args = array(
			'post_type'      => $args['post_type'],
			'paged'          => $args['page'],
			'posts_per_page' => $args['per_page'],
		);

		if ( ! empty( $args['search'] ) ) {
			$query_args['s'] = $args['search'];
		}

		if ( ! empty( $args['order'] ) ) {
			$query_args['order'] = $args['order'];
		}

		if ( ! empty( $args['order_by'] ) ) {
			$valid_orderby = array( 'ID', 'author', 'title', 'name', 'date', 'modified', 'parent', 'rand', 'comment_count', 'content' );

			// Check if order_by starts with 'meta_' (handles any number after meta_)
			if ( 0 === strpos( $args['order_by'], 'meta_' ) && is_numeric( substr( $args['order_by'], 5 ) ) ) {
				$meta_value = $args[ $args['order_by'] ];
				
				// Check if the meta value is a special field that's supported by the post type
				$supported_fields = $this->post_type_supports( $args['post_type'] );
				
				if ( in_array( $meta_value, $supported_fields, true ) ) {
					// Handle special supported fields appropriately
					if ( 'author name' === $meta_value ) {
						// Order by author display name instead of author ID
						$order = !empty($args['order']) ? $args['order'] : 'ASC';
						
						// Create a named function we can reference for removal
						$author_name_filter = function($clauses) use ($order) {
							global $wpdb;
							$clauses['join'] .= " LEFT JOIN {$wpdb->users} ON {$wpdb->posts}.post_author = {$wpdb->users}.ID ";
							$clauses['orderby'] = "{$wpdb->users}.display_name " . ($order === 'DESC' ? 'DESC' : 'ASC');
							return $clauses;
						};
						
						// Add the filter
						add_filter('posts_clauses', $author_name_filter);
						
						// Remove the filter after the query is run
						add_action('posts_selection', function() use ($author_name_filter) {
							remove_filter('posts_clauses', $author_name_filter);
						});
					} elseif ( 'title' === $meta_value ) {
						$query_args['orderby'] = 'title';
					} elseif ( 'content' === $meta_value ) {
						$query_args['orderby'] = 'content';
					} else {
						$query_args['orderby'] = $meta_value;
					}
				} else {
					// Regular meta field
					$query_args['orderby'] = in_array( $meta_value, $valid_orderby ) ? $meta_value : 'meta_value';
					
					if ( $query_args['orderby'] === 'meta_value' ) {
						$query_args['meta_key'] = $meta_value;
					}
				}
			} elseif ( in_array( $args['order_by'], $valid_orderby ) ) {
				$query_args['orderby'] = $args['order_by'];
			}
		}

		if ( ! empty( $request['filter'] ) ) {
			foreach ( $request['filter'] as $filter ) {
				if ( taxonomy_exists( $filter['type'] ) ) {
					$query_args['tax_query'][] = array(
						'taxonomy' => $filter['type'],
						'field'    => 'slug',
						'terms'    => $filter['value'],
					);
				} else {
					$query_args[ $filter['type'] ] = $filter['value'];
				}
			}
		}

		$query_args = apply_filters( 'bbapp_block_query_args', $query_args, $request );

		$retval = array();

		// the query.
		$the_query = new WP_Query( $query_args );
		$total     = $the_query->found_posts;
		if ( $the_query->have_posts() ) :
			while ( $the_query->have_posts() ) :
				$the_query->the_post();

				$data = array(
					'id'        => get_the_ID(),
					'thumbnail' => '',
					'avatar'    => array(
						'full'  => '',
						'thumb' => '',
						'name'  => '',
						'link'  => '',
						'id'    => 0,
					),
					'comments'  => '',
					'link'      => get_permalink( get_the_ID() ),
				);

				$post_author = get_post_field( 'post_author', get_the_ID() );

				// Loop through all $args keys and dynamically handle meta fields
				foreach ( $args as $key => $m_key ) {
					if ( 0 === strpos( $key, 'meta_' ) && ! empty( $m_key ) ) { // Check if the key starts with 'meta_'
						$supported_fields = $this->post_type_supports( $args['post_type'] );
						if ( in_array( $m_key, $supported_fields, true ) ) {
							if ( 'title' === $m_key ) {
								$data[ $key ] = array(
									'type'  => 'string',
									'value' => get_the_title( get_the_ID() ),
								);
							} elseif ( 'excerpt' === $m_key ) {
								$excerpt = get_the_excerpt( get_the_ID() );
								if ( $args['is_preview'] ) {
									// Get the excerpt, strip all tags, and remove URLs before shortening
									$clean_excerpt = strip_tags( $excerpt );
									$clean_excerpt = preg_replace( '/\bhttps?:\/\/[^\s()<>]+(?:\([\w\d]+\)|([^[:punct:]\s]|\/))/i', '', $clean_excerpt );
									$excerpt = wp_trim_words( $clean_excerpt, 6, '…' );
								}	
								$data[ $key ] = array(
									'type'  => 'string',
									'value' => $excerpt,
								);
							} elseif ( 'content' === $m_key ) {
								$content = get_the_content( get_the_ID() );
								if ( $args['is_preview'] ) {
									// Get the content, strip all tags, and remove URLs before shortening
									$clean_content = strip_tags( $content );
									$clean_content = preg_replace( '/\bhttps?:\/\/[^\s()<>]+(?:\([\w\d]+\)|([^[:punct:]\s]|\/))/i', '', $clean_content );
									$content = wp_trim_words( $clean_content, 4, '…' );
								}
								
								$data[ $key ] = array(
									'type'  => 'string',
									'value' => $content,
								);
							} elseif ( 'author name' === $m_key ) {
								$data[ $key ] = array(
									'type'  => 'string',
									'value' => get_the_author_meta( 'display_name', $post_author ),
								);
							}
						} else {
							// Fetch meta value and determine its type
							$meta_value   = get_post_meta( get_the_ID(), $m_key, true );
							$data[ $key ] = array(
								'type'  => $this->get_meta_value_type( $meta_value ),
								'value' => $meta_value,
							);
						}
					}
				}

				if ( ! empty( $args['thumbnail'] ) ) {
					$data['thumbnail'] = get_the_post_thumbnail_url( get_the_ID(), 'large' );
				}

				if ( ! empty( $args['show_avatar'] ) ) {
					$data['avatar']['full']  = get_avatar_url( $post_author, apply_filters( 'bbapp_avatar_full_url_args', array( 'size' => 300 ) ) );
					$data['avatar']['thumb'] = get_avatar_url( $post_author, apply_filters( 'bbapp_avatar_thumb_url_args', array( 'size' => 150 ) ) );
					$data['avatar']['name']  = get_the_author_meta( 'display_name', $post_author );
					$data['avatar']['link']  = get_the_author_meta( 'url', $post_author );
					$data['avatar']['id']    = (int) $post_author;
				}

				if ( ! empty( $args['show_comment'] ) ) {
					$data['comments'] = get_comments_number( get_the_ID() );
				}

				if ( ! empty( $args['show_bookmark'] ) ) {
					$settings = \BuddyBossApp\Admin\Settings::instance()->get_global_settings();
					$key      = $args['post_type'] . '_bookmark_enable';
					if ( isset( $settings[ $key ] ) && true === (bool) $settings[ $key ] ) {
						$bookmark_obj = array(
							'bookmark_id'   => 0,
							'is_bookmarked' => false,
							'bookmark_date' => 0,
						);

						if ( is_user_logged_in() ) {
							$bookmarks = bb_bookmarks_get_bookmark_by_item( 'post', get_the_ID() );
							if ( ! is_wp_error( $bookmarks ) && ! empty( $bookmarks[0] ) ) {
								$bookmark_obj['bookmark_id']   = $bookmarks[0]->id;
								$bookmark_obj['is_bookmarked'] = true;
								$bookmark_obj['bookmark_date'] = mysql_to_rfc3339( $bookmarks[0]->date_recorded ); // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
							}
						}

						$data['bb_bookmark'] = $bookmark_obj;
					}
				}

				$retval[] = $data;

			endwhile;
			wp_reset_postdata();
		endif;

		$response = rest_ensure_response( $retval );

		$total_items = (int) $total;
		$max_pages   = ceil( $total_items / (int) $args['per_page'] );

		$response->header( 'X-WP-Total', $total_items );
		$response->header( 'X-WP-TotalPages', (int) $max_pages );

		return $response;
	}

	/**
	 * Helper function to determine the type of a value.
	 *
	 * @param mixed $value The value to check.
	 *
	 * @since 2.3.40
	 * @return string The type of the value.
	 *
	 */
	private function get_meta_value_type( $value ) {
		if ( is_bool( $value ) ) {
			return 'bool';
		} elseif ( is_int( $value ) ) {
			return 'integer';
		} elseif ( is_float( $value ) ) {
			return 'float';
		} elseif ( is_array( $value ) ) {
			return 'array';
		} elseif ( $this->is_date_format( $value ) ) {
			return 'date';
		} else {
			return 'string';
		}
	}

	/**
	 * Helper function to check if a value is a valid date.
	 *
	 * @param string $value The value to check.
	 *
	 * @since 2.3.40
	 * @return bool True if the value is a date, false otherwise.
	 *
	 */
	private function is_date_format( $value ) {
		if ( is_string( $value ) ) {
			// Check if the string can be parsed as a valid date
			$timestamp = strtotime( $value );

			return false !== $timestamp;
		}

		return false;
	}

	/**
	 * Fetch settings.
	 *
	 * @param \WP_REST_Request $request Request object.
	 *
	 * @since 2.3.40
	 * @return mixed
	 */
	public function fetch_settings( $request ) {
		$type = $request->get_param( 'post_type' );
		if ( empty( $type ) ) {
			return new WP_Error(
				'bbapp_empty_block_type',
				__( 'Please, enter post type for fetch the settings.', 'buddyboss-app' ),
				array(
					'status' => 400,
				)
			);
		}

		$cpts = $this->get_public_rest_post_types();

		if ( ! isset( $cpts[ $type ] ) ) {
			return new WP_Error(
				'bbapp_invalid_block_type',
				__( 'Please, enter valid post type for fetch the settings.', 'buddyboss-app' ),
				array(
					'status' => 400,
				)
			);
		}

		$cpt = $cpts[ $type ];

		$retval = array(
			'name'              => $cpt->name,
			'label'             => $cpt->label,
			'thumbnail'         => post_type_supports( $cpt->name, 'thumbnail' ),
			'meta'              => $this->meta_keys( $cpt->name ),
			'directory_options' => $this->directory_options( $cpt->name ),
			'order'             => array(
				'ASC'  => __( 'Ascending (order from lowest to highest values)', 'buddyboss-app' ),
				'DESC' => __( 'Descending (order from highest to lowest values)', 'buddyboss-app' ),
			),
			'order_by'          => array(
				'ID'            => __( 'Order by post id', 'buddyboss-app' ),
				'author'        => __( 'Order by author', 'buddyboss-app' ),
				'title'         => __( 'Order by title', 'buddyboss-app' ),
				'name'          => __( 'Order by post name (post slug)', 'buddyboss-app' ),
				'date'          => __( 'Order by date', 'buddyboss-app' ),
				'modified'      => __( 'Order by last modified date', 'buddyboss-app' ),
				'parent'        => __( 'Order by post/page parent id', 'buddyboss-app' ),
				'rand'          => __( 'Random order', 'buddyboss-app' ),
				'comment_count' => __( 'Order by number of comments', 'buddyboss-app' ),
				'meta_1'        => __( 'Meta 1', 'buddyboss-app' ),
				'meta_2'        => __( 'Meta 2', 'buddyboss-app' ),
				'meta_3'        => __( 'Meta 3', 'buddyboss-app' ),
				'meta_4'        => __( 'Meta 4', 'buddyboss-app' ),
			),
		);

		return rest_ensure_response( $retval );
	}

	/**
	 * Get filter.
	 *
	 * @param \WP_REST_Request $request Request object.
	 *
	 * @since 2.3.40
	 * @return mixed
	 */
	public function get_filter( $request ) {
		$taxonomies = $request->get_param( 'taxonomies' );
		if ( ! is_array( $taxonomies ) ) {
			$taxonomies = explode( ',', $taxonomies );
		}
		$filter_list = array();

		foreach ( $taxonomies as $tax ) {
			if ( taxonomy_exists( $tax ) ) {
				$terms = get_terms(
					array(
						'taxonomy'   => $tax,
						'hide_empty' => true,
					)
				);

				if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
					$filter_list[ $tax ] = array_values( $terms );
				}
			} else {
				$filter_list[ $tax ] = array();
			}
		}

		/**
		 * Filter the filter list.
		 *
		 * @param array $filter_list Filter options.
		 * @param array $taxonomies  Taxonomies.
		 *
		 * @since 2.3.40
		 */
		$filter_list = apply_filters( 'bbapp_block_filter_list', $filter_list, $taxonomies );

		return rest_ensure_response( $filter_list );
	}

	/**
	 * Process API request.
	 *
	 * @param \WP_REST_Request $request Request object.
	 *
	 * @since 2.3.40
	 * @return bool
	 */
	public function process_api_request( $request ) {
		$method = $request->get_method();

		if ( 'POST' === $method && ! is_user_logged_in() ) {
			return false;
		}

		return true;
	}

	/**
	 * Get all public post types.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function get_public_rest_post_types() {
		static $post_types;

		if ( ! empty( $post_types ) ) {
			return $post_types;
		}

		// Get all registered post types
		$post_types = get_post_types(
			array(
				'public' => true,    // Only public post types.
			),
			'objects'        // Return as objects for more details.
		);
		
		// Allow custom exclusion of post types via filter
		$excluded_post_types = apply_filters( 'bbapp_excluded_post_types', array(
				'attachment',
				'sfwd-certificates',
				'sfwd-transactions',
				'sfwd-essays',
				'sfwd-assignment',
				'memberpressproduct',
				'memberpressgroup'
			) 
		);
		
		// Remove any excluded post types (typically custom post types that should be hidden)
		foreach ( $excluded_post_types as $excluded_type ) {
			if ( isset( $post_types[$excluded_type] ) ) {
				unset( $post_types[$excluded_type] );
			}
		}

		return $post_types;
	}

	/**
	 * Get all public post types.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function post_type_supports( $post_type ) {
		$post_data = array();
		if ( post_type_supports( $post_type, 'title' ) ) {
			$post_data[] = __( 'title', 'buddyboss-app' );
		}
		if ( post_type_supports( $post_type, 'excerpt' ) ) {
			$post_data[] = __( 'excerpt', 'buddyboss-app' );
		}
		if ( post_type_supports( $post_type, 'editor' ) ) {
			$post_data[] = __( 'content', 'buddyboss-app' );
		}
		if ( post_type_supports( $post_type, 'author' ) ) {
			$post_data[] = __( 'author name', 'buddyboss-app' );
		}

		return $post_data;
	}

	/**
	 * Get all meta keys.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function meta_keys( $post_type ) {
		$defaults_to_remove = array( '_edit_last', '_edit_lock' );
		$post_meta_names    = array();

		$args = array(
			'post_type'      => $post_type,
			'posts_per_page' => 20,
		);

		$the_query = new WP_Query( $args );
		if ( $the_query->have_posts() ) {
			while ( $the_query->have_posts() ) {
				$the_query->the_post();
				$meta_array = get_post_meta( get_the_ID() );
				
				$meta_array = array_filter($meta_array, function ($key) {
					return $key[0] !== '_';
				}, ARRAY_FILTER_USE_KEY);
				foreach ( $meta_array as $key => $value ) {

					$value_data = get_post_meta( get_the_ID(), $key, true );

					if ( ! is_string( $value_data ) || json_decode( $value_data ) ) {
						continue;
					}

					if ( in_array( $key, $post_meta_names, true ) ) {
						continue;
					}
				
					array_push( $post_meta_names, $key );
				}
				// Allow for ACF Fields to be added to the list.
				if ( function_exists( 'acf_get_field_groups' ) ) {
					$acf_groups = acf_get_field_groups( array( 'post_id' => get_the_ID() ) );
					foreach ( $acf_groups as $acf_group ) {
						$acf_group_fields = acf_get_fields( $acf_group['key'] );
						foreach ( $acf_group_fields as $acf_group_field ) {
							$var            = '_' . $acf_group_field['name'];
							$acf_field_meta = get_post_meta( get_the_ID(), $var, true );
							if (
								! in_array(
									$acf_group_field['name'],
									$post_meta_names,
									true
								) &&
								is_string( $acf_field_meta )
							) {
								$defaults_to_remove[] = $var;
								array_push( $post_meta_names, $acf_group_field['name'] );
							} else {
								$defaults_to_remove[] = $var;
							}
						}
					}
				}
			}
		}

		wp_reset_postdata();

		// Remove the default WP key names + ACF _keys.
		$post_meta_names = array_diff( $post_meta_names, $defaults_to_remove );

		$post_data       = $this->post_type_supports( $post_type );
		$post_meta_names = array_merge( $post_data, array_unique( $post_meta_names ) );

		return apply_filters( 'bbapp_block_metas', $post_meta_names, $post_type );
	}

	/**
	 * Get all directory options.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function directory_options( $post_type ) {
		$retval = array();

		if ( is_post_type_viewable( $post_type ) ) {
			$retval['search_bar'] = __( 'Show Search Bar', 'buddyboss-app' );
		}

		$retval['show_avatar'] = __( 'Show Avatar', 'buddyboss-app' );

		if ( post_type_supports( $post_type, 'comments' ) ) {
			$retval['show_comment'] = __( 'Show Comments', 'buddyboss-app' );
		}

		$settings = \BuddyBossApp\Admin\Settings::instance()->get_global_settings();
		$key      = $post_type . '_bookmark_enable';
		if ( isset( $settings[ $key ] ) && true === (bool) $settings[ $key ] ) {
			$retval['show_bookmark'] = __( 'Show Bookmarks', 'buddyboss-app' );
		}

		$retval = array_merge( $retval, $this->get_filter_options( $post_type ) );

		return $retval;
	}

	/**
	 * Get all filter options.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function get_filter_options( $post_type ) {
		$retval              = array();
		$show_filter_options = array();

		$retval['show_filter']         = __( 'Show Filter', 'buddyboss-app' );
		$retval['show_filter_options'] = array();

		$taxonomies = get_object_taxonomies( $post_type, 'objects' );
		if ( ! empty( $taxonomies ) ) {
			foreach ( $taxonomies as $tax ) {
				$show_filter_options[] = array(
					'name'  => $tax->name,
					'label' => $tax->label,
				);
			}
		}

		/**
		 * Filter the filter options.
		 *
		 * @param array  $show_filter_options Filter options.
		 * @param string $post_type           Post type.
		 *
		 * @since 2.3.40
		 */
		$show_filter_options = apply_filters( 'bbapp_block_filter_options', $show_filter_options, $post_type );

		if ( empty( $show_filter_options ) ) {
			unset( $retval['show_filter'] );
			unset( $retval['show_filter_options'] );
		} else {
			$retval['show_filter_options'] = $show_filter_options;
		}

		return $retval;
	}
}
