<?php
/**
 * Class DeepLink Rest API V1.
 *
 * @package BuddyBossApp\DeepLinking\Api\V1
 */

namespace BuddyBossApp\Api\DeepLinking\V1;

/**
 * Class RestAPI
 *
 * @package BuddyBossApp\DeepLinking\Api\V1
 */
class RestAPI {

	/**
	 * Class object.
	 *
	 * @var object
	 */
	private static $instance;

	/**
	 * API version.
	 *
	 * @var string
	 */
	private static $version = 'v1';

	/**
	 * API Namespace.
	 *
	 * @var string
	 */
	private static $namespace;

	/**
	 * URL data.
	 *
	 * @var array
	 */
	private static $url_data = array();

	/**
	 * RestAPI constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return RestAPI
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			self::$namespace = 'buddyboss-app/core/' . self::$version;

			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Load deep linking types
	 */
	public function load() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register all required route for Deep Linking..
	 */
	public function register_routes() {
		register_rest_route(
			self::$namespace,
			'/url-details',
			array(
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'url_details' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'url' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'URL', 'buddyboss-app' ),
						),
					),
				),
			)
		);

		register_rest_route(
			self::$namespace,
			'/bulk-url-details',
			array(
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'bulk_url_details' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'urls' => array(
							'required'    => true,
							'type'        => 'array',
							'items'       => array(
								'type' => 'string',
							),
							'description' => __( 'URLs', 'buddyboss-app' ),
						),
					),
				),
			)
		);
	}

	/**
	 * Url detail callback.
	 *
	 * @param \WP_REST_Request $request Rest request params.
	 *
	 * @return array|mixed|void|\WP_Error|\WP_REST_Response
	 *
	 * @api            {POST} /wp-json/buddyboss-app/core/v1/url-details URL details
	 * @apiName        UpdateDeepLinkingUrlDetails
	 * @apiGroup       Deep Linking
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Get deep-linking URL Details. This endpoint requires request to be sent in "multipart/form-data" format.
	 * @apiParam       url URL Details. (Must be valid url)
	 */
	public function url_details( $request ) {
		$url = $request->get_param( 'url' );
		$url = bbapp_untrailingslashit( $url );

		return $this->parse_url( $url );
	}

	/**
	 * Bulk URL detail callback.
	 *
	 * @param \WP_REST_Request $request Rest request params.
	 *
	 * @return array|mixed|void|\WP_Error|\WP_REST_Response
	 *
	 * @api            {POST} /wp-json/buddyboss-app/core/v1/bulk-url-details  bulk URL details
	 * @apiName        UpdateDeepLinkingBulkUrlDetails
	 * @apiGroup       Deep Linking
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Get deep-linking bulk URL Details. This endpoint requires request to be sent in "multipart/form-data" format.
	 * @apiParam       url URL Details. (Must be valid url)
	 * @apiError {Object}  Response object
	 */
	public function bulk_url_details( $request ) {
		// Debug log the incoming request.

		$urls          = $request->get_param( 'urls' );
		$bulk_response = array();

		foreach ( $urls as $url ) {
			$url = bbapp_untrailingslashit( $url );

			// Standard processing for other URLs.
			$response = $this->parse_url( $url );
			if ( ! is_wp_error( $response ) || ! $response ) {
				$data = $this->prepare_response_for_collection( $response );

				$bulk_response[] = $data;
			}
		}

		return rest_ensure_response( $bulk_response );
	}

	/**
	 * Prepare the response for collection.
	 *
	 * @param \WP_REST_Response $response Rest response.
	 *
	 * @return array
	 */
	public function prepare_response_for_collection( $response ) {
		$data   = (array) $response->get_data();
		$server = rest_get_server();
		$links  = $server->get_compact_response_links( $response );

		// If no links, just add _links to the existing data.
		if ( ! empty( $links ) ) {
			$data['_links'] = $links;
		}

		return $data;
	}

	/**
	 * Parse Url.
	 *
	 * @param string $url URL to deeplink.
	 *
	 * @return array|mixed|void|\WP_Error
	 */
	private function parse_url( $url ) {
		$url = trim( $url ); // remove extra WhiteSpaces.

		// If URL not provided.
		if ( empty( $url ) ) {
			return new \WP_Error( 'url_not_provided', __( 'URL not provided.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		// Check if URL valid or not.
		if ( ! preg_match( "/\b(?:(?:https?|ftp):\/\/|www\.)[-a-z0-9+&@#\/%?=~_|!:,.;]*[-a-z0-9+&@#\/%=~_|]/i", $url ) ) {
			return new \WP_Error( 'url_not_valid', __( 'URL is not valid.', 'buddyboss-app' ), array( 'status' => 403 ) );
		}

		/**
		 * If it's a External URL just Output Directly.
		 */
		if ( $this->is_url_external( $url ) ) {
			/**
			 * Internally TypeAbstract Class Uses this filter to apply changes into API Response.
			 *
			 * @param array  $response item object.
			 * @param string $url      URL to deeplink.
			 * @param string $version  API version.
			 *
			 * @since 2.0.70
			 */
			$response = apply_filters( 'bbapp_deeplinking_details', null, $url, self::$version );

			if ( ! isset( $response ) ) {
				$response = array(
					'action'    => 'inapp',
					'namespace' => 'external_url',
					'url'       => $url,
				);
			}

			return $this->prepare_response( $response );
		}

		/**
		 * Internally TypeAbstract Class Uses this filter to apply changes into API Response.
		 */
		$response = apply_filters( 'bbapp_deeplinking_details', null, $url, self::$version );

		if ( ! isset( $response ) ) {
			$response = array(
				'action'    => 'inapp',
				'namespace' => 'url',
				'url'       => $url,
			);
		}

		return $this->prepare_response( $response );
	}

	/**
	 * Prepare Response.
	 *
	 * @param array $data item object.
	 *
	 * @return mixed|\WP_REST_Response
	 */
	public function prepare_response( $data ) {
		$response = $data;
		if ( ! empty( $data['_link_action'] ) || ! empty( $data['_link'] ) ) {
			unset( $response['_link_action'] );
			unset( $response['_link'] );
			unset( $response['_sub_link_action'] );
		}

		$response = rest_ensure_response( $response );

		if ( ! empty( $data['_link_action'] ) || ! empty( $data['_link'] ) ) {
			$response->add_links( $this->prepare_links( $data ) );
		}

		return $response;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array $data item object.
	 *
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {
		$links = array();

		if ( ! empty( $data['_link'] ) ) {
			$links = $data['_link'];
		} elseif ( ! empty( $data['item_id'] ) ) {
			switch ( $data['_link_action'] ) {
				case 'wp_user':
					$item_id = isset( $data['user_id'] ) ? $data['user_id'] : $data['item_id'];
					$links   = array(
						'user' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/wp/v2/users/' . $item_id ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'buddypress_group':
					$links = array(
						'group' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss/v1/groups/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'buddypress_member':
					$item_id = isset( $data['user_id'] ) ? $data['user_id'] : $data['item_id'];
					$links   = array(
						'member' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss/v1/members/' . $item_id ) ),
								'embeddable' => true,
							),
						),
					);
					if ( ! empty( $data['_sub_link_action'] ) && 'buddypress_messages' === $data['_sub_link_action'] && ! empty( $data['sub_tab'] ) ) {
						if ( ! empty( $data['thread_recipients'] ) ) {
							$links['recipients'] = array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss/v1/members' ) . '?include=' . implode( ',', $data['thread_recipients'] ) ),
								'embeddable' => true,
							);
						}
						$links['messages'] = array(
							'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss/v1/messages/' . $data['sub_tab'] ) ),
							'embeddable' => true,
						);
					}
					break;
				case 'buddypress_activity':
					$links = array(
						'activity' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss/v1/activity/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'bbpress_forum':
					$links = array(
						'forum' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss/v1/forums/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'bbpress_topic':
					$links = array(
						'topic' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss/v1/topics/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'bbpress_reply':
					$links = array(
						'reply' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss/v1/reply/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'ld_course':
					$links = array(
						'course' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss-app/learndash/v1/courses/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'ld_lesson':
					$links = array(
						'lesson' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss-app/learndash/v1/lessons/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'ld_topic':
					$links = array(
						'topic' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss-app/learndash/v1/topics/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'ld_quiz':
					$links = array(
						'quiz' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss-app/learndash/v1/quiz/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'ld_group':
					$links = array(
						'group' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/ldlms/v1/groups/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'app_page':
					$links = array(
						'app_page' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/wp/v2/app_pages/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'core_tutor_assignmentss':
					$links = array(
						'tutor_assignments' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss-app/tutor/v1/assignments/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'lesson':
					$links = array(
						'tutor_lesson' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss-app/tutor/v1/lessons/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case 'tutor_quiz':
					$links = array(
						'tutor_quiz' => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/buddyboss-app/tutor/v1/quiz/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
				case $data['_link_action']:
					$link_key = str_replace( 'core_', '', $data['_link_action'] );
					$links    = array(
						$link_key => array(
							array(
								'href'       => bbapp_untrailingslashit( rest_url( '/wp/v2/' . $link_key . '/' . $data['item_id'] ) ),
								'embeddable' => true,
							),
						),
					);
					break;
			}
		}

		return apply_filters( 'bbapp_deeplinking_links', $links, $data, self::$version );
	}

	/**
	 * Check whether requested URL internal/External.
	 *
	 * @param string $url URL to check its external or not.
	 *
	 * @return bool
	 */
	public function is_url_external( $url ) {
		$url_host      = str_replace( 'www.', '', wp_parse_url( $url, PHP_URL_HOST ) );
		$home_url_host = str_replace( 'www.', '', wp_parse_url( home_url(), PHP_URL_HOST ) );

		if ( $url_host && $url_host !== $home_url_host ) {
			return true;
		}

		return false;
	}
}
