<?php
/**
 * Class APP Rest Signup.
 *
 * @package BuddyBossApp\Api\Core
 */

namespace BuddyBossApp\Api\Core\V1\SignUp;

use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Class SignUpRest
 *
 * @package BuddyBossApp\Api\Core\V1\SignUp
 */
class SignUpRest extends WP_REST_Controller {

	/**
	 * API namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'buddyboss-app/v1';

	/**
	 * API restbase.
	 *
	 * @var string
	 */
	protected $rest_base = 'signup';

	/**
	 * Class instance.
	 *
	 * @var object.
	 */
	protected static $instance;

	/**
	 * SignUpRest constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return mixed|object
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Class hooks.
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register the component routes.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/form',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'signup_form_items' ),
					'permission_callback' => array( $this, 'signup_form_items_permissions_check' ),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/verify',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'signup_verify' ),
					'permission_callback' => array( $this, 'signup_verify_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve Signup Form Fields.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error
	 *
	 * @api            {GET} /wp-json/buddyboss-app/v1/signup/form Signup Form
	 * @apiName        GetBBSignupFormFields
	 * @apiGroup       Signups
	 * @apiDescription Retrieve Signup Form Fields.
	 * @apiVersion     1.0.0
	 * @apiPermission  WithoutLoggedInUser
	 */
	public function signup_form_items( $request ) {
		// Registration form type.
		$settings = \BuddyBossApp\Auth\Auth::instance()->is_app_registration_form();

		if ( defined( 'BP_PLATFORM_VERSION' ) && 'buddyboss_registration' === $settings && ! bbapp_is_lite_app() ) {
			/**
			 * Redirect to topic endpoint.
			 */
			$request_curl = new WP_REST_Request( 'GET', '/buddyboss/v1/signup/form' );

			if ( ! empty( $request['_embed'] ) ) {
				$request_curl->set_param( '_embed', $request['_embed'] );
			}

			$response = $this->dispatch( $request_curl );

			return rest_ensure_response( $response );
		}

		$fields = array(
			array(
				'id'          => 'username',
				'label'       => __( 'User Name', 'buddyboss-app' ),
				'description' => '',
				'type'        => 'text',
				'required'    => true,
				'options'     => '',
				'member_type' => '',
			),
			array(
				'id'          => 'email',
				'label'       => __( 'Email', 'buddyboss-app' ),
				'description' => '',
				'type'        => 'email',
				'required'    => true,
				'options'     => '',
				'member_type' => '',
			),
			array(
				'id'          => 'password',
				'label'       => __( 'Password', 'buddyboss-app' ),
				'description' => '',
				'type'        => 'password',
				'required'    => true,
				'options'     => '',
				'member_type' => '',
			),

		);

		$response = rest_ensure_response( $fields );

		/**
		 * Fires after a list of signup fields is fetched via the REST API.
		 *
		 * @param array            $fields   Fetched Form fields.
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 */
		do_action( 'bbapp_rest_signup_form_items', $fields, $response, $request );

		return $response;
	}

	/**
	 * Checks if a given request has access to view the signup form fields.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function signup_form_items_permissions_check( $request ) {
		$retval = true;

		if ( is_user_logged_in() ) {
			$retval = new WP_Error( 'bbapp_rest_authorization_required', __( 'Sorry, you are not able to view the register form fields.', 'buddyboss-app' ), array( 'status' => rest_authorization_required_code() ) );
		}

		/**
		 * Filter the signup `signup_form_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_signup_form_items_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve signups.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error
	 *
	 * @api            {GET} /wp-json/buddyboss-app/v1/signup Signups
	 * @apiName        GetBBSignups
	 * @apiGroup       Signups
	 * @apiDescription Retrieve signups
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Array} [include] Ensure result set includes specific IDs.
	 * @apiParam {String=asc,desc} [order] Order sort attribute ascending or descending.
	 * @apiParam {String} [orderby=signup_id] Order by a specific parameter.
	 * @apiParam {String} [user_login] Specific user login to return.
	 * @apiParam {Number} [number] Total number of signups to return.
	 * @apiParam {Number} [offset] 'Offset the result set by a specific number of items.
	 */
	public function get_items( $request ) {
		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			/**
			 * Redirect to topic endpoint.
			 */
			$request_curl = new WP_REST_Request( 'GET', '/buddyboss/v1/signup' );

			$request_curl->set_body( $request );

			if ( ! empty( $request->get_params() ) ) {
				foreach ( $request->get_params() as $key => $data ) {
					$request_curl->set_param( $key, $data );
				}
			}

			$response = $this->dispatch( $request_curl );

			return rest_ensure_response( $response );
		}

		return false;
	}

	/**
	 * Check if a given request has access to signup items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = new WP_Error( 'bbapp_rest_authorization_required', __( 'Sorry, you need to be logged in to perform this action.', 'buddyboss-app' ), array( 'status' => rest_authorization_required_code() ) );
		}

		/**
		 * Filter the signup `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_signup_get_items_permissions_check', $retval, $request );
	}

	/**
	 * Create signup.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_REST_Response | WP_Error
	 *
	 * @api            {POST} /wp-json/buddyboss-app/v1/signup Create signup
	 * @apiName        CreateBBSignups
	 * @apiGroup       Signups
	 * @apiDescription Create signup
	 * @apiVersion     1.0.0
	 * @apiPermission  WithoutLoggedInUser
	 * @apiParam {String} signup_email New user email address.
	 * @apiParam {String} [signup_email_confirm] New user confirm email address.
	 * @apiParam {String} signup_password New user account password.
	 * @apiParam {String} [signup_password_confirm] New user confirm account password.
	 */
	public function create_item( $request ) {
		// Registration form type.
		$settings = \BuddyBossApp\Auth\Auth::instance()->is_app_registration_form();

		if ( defined( 'BP_PLATFORM_VERSION' ) && 'buddyboss_registration' === $settings && ! bbapp_is_lite_app() ) {
			/**
			 * Redirect to topic endpoint.
			 */
			$request_curl = new WP_REST_Request( 'POST', '/buddyboss/v1/signup' );
			$request_curl->set_body( $request );

			if ( ! empty( $request->get_params() ) ) {
				foreach ( $request->get_params() as $key => $data ) {
					$request_curl->set_param( $key, $data );
				}
			}

			$response = $this->dispatch( $request_curl );

			return rest_ensure_response( $response );
		}

		$request_curl = new WP_REST_Request( 'POST', '/buddyboss-app/auth/v1/register' );

		$request_curl->set_body( $request );

		if ( ! empty( $request->get_params() ) ) {
			foreach ( $request->get_params() as $key => $data ) {
				$request_curl->set_param( $key, $data );
			}
		}

		$response = $this->dispatch( $request_curl );

		return rest_ensure_response( $response );
	}

	/**
	 * Checks if a given request has access to create a signup.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function create_item_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the signup `create_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_signup_create_item_permissions_check', $retval, $request );
	}

	/**
	 * Verify Signup with same error validation to signup endpoint without creating user.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return boolean | WP_Error
	 *
	 * @api            {POST} /wp-json/buddyboss-app/v1/signup/verify Create signup verify
	 * @apiName        CreateBBSignupsVerify
	 * @apiGroup       SignupsVerify
	 * @apiDescription Create signup verify
	 * @apiVersion     1.0.0
	 * @apiPermission  WithoutLoggedInUser
	 * @apiParam {String} signup_email New user email address.
	 * @apiParam {String} [signup_email_confirm] New user confirm email address.
	 * @apiParam {String} signup_password New user account password.
	 * @apiParam {String} [signup_password_confirm] New user confirm account password.
	 */
	public function signup_verify( $request ) {
		// Registration form type.
		$settings = \BuddyBossApp\Auth\Auth::instance()->is_app_registration_form();

		if ( defined( 'BP_PLATFORM_VERSION' ) && 'buddyboss_registration' === $settings ) {
			return rest_ensure_response( $this->bbapp_platform_signup_verify( $request ) );
		}

		return rest_ensure_response( $this->bbapp_signup_verify( $request ) );
	}

	/**
	 * Checks if a given request has access to create a signup verify.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function signup_verify_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the signup `signup_verify` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_signup_verify_permissions_check', $retval, $request );
	}

	/**
	 * Get the query params for collections.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params                       = parent::get_collection_params();
		$params['context']['default'] = 'view';

		unset( $params['page'], $params['per_page'], $params['search'] );

		$params['number'] = array(
			'description'       => __( 'Total number of signups to return.', 'buddyboss-app' ),
			'default'           => 1,
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['offset'] = array(
			'description'       => __( 'Offset the result set by a specific number of items.', 'buddyboss-app' ),
			'default'           => 0,
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['include'] = array(
			'description'       => __( 'Ensure result set includes specific IDs.', 'buddyboss-app' ),
			'default'           => array(),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['order'] = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'default'           => 'desc',
			'type'              => 'string',
			'enum'              => array( 'asc', 'desc' ),
			'sanitize_callback' => 'sanitize_key',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['orderby'] = array(
			'description'       => __( 'Order by a specific parameter (default: signup_id).', 'buddyboss-app' ),
			'default'           => 'signup_id',
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_key',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['user_login'] = array(
			'description'       => __( 'Specific user login to return.', 'buddyboss-app' ),
			'default'           => '',
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_key',
			'validate_callback' => 'rest_validate_request_arg',
		);

		/**
		 * Filters the collection query params.
		 *
		 * @param array $params Query params.
		 */
		return apply_filters( 'bbapp_rest_signup_collection_params', $params );
	}

	/**
	 * Dispatch the request item.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {
		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()['_embed'] ) ) {
			$query_params['_embed'] = $request->get_query_params()['_embed'];
		}

		$request->set_query_params( $query_params );

		$server = rest_get_server();

		$response = $server->dispatch( $request );

		return $response;
	}

	/**
	 * Without Platform verify signup form validation.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return bool[]|mixed|void|WP_Error
	 */
	public function bbapp_signup_verify( $request ) {
		/**
		 * Check if current WordPress is multisite & endpoint is not being accessed from mainsite.
		 * On that case we will throw the error because we don't support authentication on subsites.
		 */
		if ( is_multisite() && ! is_main_site() ) {
			return new Wp_Error( 'rest_bbapp_register_subsite_not_supported', __( 'Registration is only available on main network site.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		/**
		 * Throw the error if user registration are disabled on admin settings.
		 */
		if ( ! get_option( 'users_can_register' ) ) {
			return new Wp_Error( 'rest_bbapp_register_is_not_allowed', __( 'User registration is currently not allowed.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		/**
		 * Filters user rest registration inputs.
		 *
		 * @param WP_REST_Request Rest request.
		 */
		$request = apply_filters( 'bbapp_auth_rest_user_registration_inputs', $request );

		/**
		 * Validate Incoming Inputs.
		 */
		if ( ! $request->get_param( 'username' ) || '' === $request->get_param( 'username' ) ) {
			return new Wp_Error( 'rest_bbapp_register_username_req', __( 'A valid username param is required.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		if ( ! $request->get_param( 'email' ) || '' === $request->get_param( 'email' ) ) {
			return new Wp_Error( 'rest_bbapp_register_email_req', __( 'A valid email param is required.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		if ( ! $request->get_param( 'password' ) || '' === $request->get_param( 'password' ) ) {
			return new Wp_Error( 'rest_bbapp_register_password_req', __( 'A valid password param is required.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		/**
		 * Validate if UserName or Email Exists or not.
		 */
		if ( username_exists( $request->get_param( 'username' ) ) ) {
			return new Wp_Error( 'rest_bbapp_register_username_exists', __( 'Given username already exists.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		if ( email_exists( $request->get_param( 'email' ) ) ) {
			return new Wp_Error( 'rest_bbapp_register_email_exists', __( 'Given email already exists.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		/**
		 * Pre Registration Check Hook.
		 * This filter can used to extra custom validation or for aborting the registration process.
		 *
		 * @param bool $pre_check Auth rest user registration pre check.
		 * @param WP_REST_Request $request Rest request.
		 */
		$pre_check = apply_filters( 'bbapp_auth_rest_user_registration_pre_check', true, $request );

		if ( is_wp_error( $pre_check ) ) {
			return $pre_check;
		}

		return array( 'verify' => true );
	}

	/**
	 * With Platform verify signup form validation.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return bool[]|mixed|void|WP_Error
	 */
	public function bbapp_platform_signup_verify( $request ) {
		$bp = buddypress();
		$request->set_param( 'context', 'edit' );

		/**
		 * Redirect to topic endpoint.
		 */
		$request_curl = new WP_REST_Request( 'GET', '/buddyboss/v1/signup/form' );

		if ( ! empty( $request['_embed'] ) ) {
			$request_curl->set_param( '_embed', $request['_embed'] );
		}

		$response        = $this->dispatch( $request_curl );
		$form_fields     = ( isset( $response->data ) ) ? $response->data : array();
		$form_fields_all = $form_fields;
		$param           = $request->get_params();
		$posted_data     = array();
		$date_fields     = array();

		if ( ! empty( $form_fields_all ) ) {
			$form_fields_with_type = array_column( $form_fields_all, 'type', 'id' );
			$form_fields           = array_column( $form_fields_all, 'id' );

			if ( in_array( 'datebox', $form_fields_with_type, true ) ) {
				$key           = array_search( 'datebox', $form_fields_with_type, true );
				$form_fields[] = $key;
				$date_fields[] = $key;
				$param[ $key ] = '';
				$form_fields[] = $key . '_day';
				$form_fields[] = $key . '_month';
				$form_fields[] = $key . '_year';
			}

			$form_fields = array_flip( $form_fields );
			$posted_data = array_intersect_key( $param, $form_fields );
		}

		if ( empty( $posted_data ) ) {
			return new WP_Error( 'bp_rest_signup_cannot_create', __( 'Cannot create new signup.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		// verification for phpcs.
		wp_verify_nonce( wp_create_nonce( 'rest_signup' ), 'rest_signup' );

		$_POST      = array();
		$_POST      = $posted_data;
		$user_name  = ( function_exists( 'bp_get_signup_username_value' ) ? bp_get_signup_username_value() : ( isset( $_POST['signup_username'] ) ? sanitize_text_field( wp_unslash( $_POST['signup_username'] ) ) : '' ) );
		$user_email = (
		function_exists( 'bp_get_signup_email_value' ) ? bp_get_signup_email_value() : ( isset( $_POST['signup_email'] ) ? sanitize_text_field( wp_unslash( $_POST['signup_email'] ) ) : '' ) );

		// Check the base account details for problems.
		$account_details = bp_core_validate_user_signup( $user_name, $user_email );
		$email_opt       = function_exists( 'bp_register_confirm_email' ) && true === bp_register_confirm_email() ? true : false;
		$password_opt    = function_exists( 'bp_register_confirm_password' ) ? bp_register_confirm_password() : true;

		// If there are errors with account details, set them for display.
		if ( ! empty( $account_details['errors']->errors['user_name'] ) ) {
			$bp->signup->errors['signup_username'] = $account_details['errors']->errors['user_name'][0];
		}

		if ( ! empty( $account_details['errors']->errors['user_email'] ) ) {
			$bp->signup->errors['signup_email'] = $account_details['errors']->errors['user_email'][0];
		}

		// Check that both password fields are filled in.
		if ( isset( $_POST['signup_password'] ) && empty( $_POST['signup_password'] ) ) {
			$bp->signup->errors['signup_password'] = __( 'Please make sure to enter your password.', 'buddyboss-app' );
		}

		// if email opt enabled.
		if ( true === $email_opt ) {
			// Check that both password fields are filled in.
			if ( empty( $_POST['signup_email'] ) || empty( $_POST['signup_email_confirm'] ) ) {
				$bp->signup->errors['signup_email'] = __( 'Please make sure to enter your email twice.', 'buddyboss-app' );
			}

			// Check that the passwords match.
			if ( ( ! empty( $_POST['signup_email'] ) && ! empty( $_POST['signup_email_confirm'] ) ) && $_POST['signup_email'] !== $_POST['signup_email_confirm'] ) {
				$bp->signup->errors['signup_email'] = __( 'The emails entered do not match.', 'buddyboss-app' );
			}
		}

		// if password opt enabled.
		if ( true === $password_opt ) {
			// Check that both password fields are filled in.
			if ( empty( $_POST['signup_password'] ) || empty( $_POST['signup_password_confirm'] ) ) {
				$bp->signup->errors['signup_password'] = __( 'Please make sure to enter your password twice.', 'buddyboss-app' );
			}

			// Check that the passwords match.
			if ( ( ! empty( $_POST['signup_password'] ) && ! empty( $_POST['signup_password_confirm'] ) ) && $_POST['signup_password'] !== $_POST['signup_password_confirm'] ) {
				$bp->signup->errors['signup_password'] = __( 'The passwords entered do not match.', 'buddyboss-app' );
			}
		}

		// Adding error message for the legal agreement checkbox.
		if ( function_exists( 'bb_register_legal_agreement' ) && true === bb_register_legal_agreement() && empty( $_POST['legal_agreement'] ) ) {
			$bp->signup->errors['legal_agreement'] = __( 'This is a required field.', 'buddyboss-app' );
		}

		$bp->signup->username = $user_name;
		$bp->signup->email    = $user_email;

		// Now we've checked account details, we can check profile information.
		if ( bp_is_active( 'xprofile' ) ) {

			$xprofile_fields = array_filter(
				$posted_data,
				function ( $v, $k ) {
					return strpos( $k, 'field_' ) === 0 && empty( strpos( $k, '_day' ) ) && empty( strpos( $k, '_month' ) ) && empty( strpos( $k, '_year' ) );
				},
				ARRAY_FILTER_USE_BOTH
			);

			$all_fields           = array_column( $form_fields_all, null, 'id' );
			$profile_fields       = array_intersect_key( $all_fields, $xprofile_fields );
			$fields_with_type     = array_column( $form_fields_all, 'type', 'id' );
			$fields_member_types  = array_filter( array_column( $profile_fields, 'member_type', 'id' ) );
			$member_type_field_id = array_search( 'membertypes', $fields_with_type, true );
			$all_member_types     = ! empty( $member_type_field_id ) && isset( $profile_fields[ $member_type_field_id ] ) ? array_column( $profile_fields[ $member_type_field_id ]['options'], 'key', 'id' ) : array();
			$selected_member_type = ( ! empty( $member_type_field_id ) && isset( $posted_data[ $member_type_field_id ] ) ? $posted_data[ $member_type_field_id ] : '' );
			$selected_member_type = ( ! empty( $selected_member_type ) && isset( $all_member_types[ $selected_member_type ] ) ) ? $all_member_types[ $selected_member_type ] : $selected_member_type;
			$profile_field_ids    = array();

			// Make sure hidden field is passed and populated.
			if ( isset( $xprofile_fields ) && ! empty( $xprofile_fields ) ) {

				// Loop through the posted fields formatting any datebox values then validate the field.
				foreach ( (array) $xprofile_fields as $field => $value ) {
					$field_type = ( ! empty( $fields_member_types ) && isset( $fields_member_types[ $field ] ) ? array_filter(
						$fields_member_types[ $field ],
						function ( $v ) {
							return 'null' !== $v;
						}
					) : array() );

					if ( ! empty( $field_type ) && ! empty( $selected_member_type ) && ! in_array( $selected_member_type, $field_type, true ) ) {
						unset( $_POST[ $field ] );
						continue;
					}

					$field_id            = str_replace( 'field_', '', $field );
					$profile_field_ids[] = $field_id;

					if ( ! empty( $date_fields ) && in_array( $field, $date_fields, true ) ) {
						unset( $_POST[ $field ] );
					}

					bp_xprofile_maybe_format_datebox_post_data( $field_id );

					// Trim post fields.
					if ( isset( $_POST[ 'field_' . $field_id ] ) ) {
						if ( is_array( $_POST[ 'field_' . $field_id ] ) ) {
							$_POST[ 'field_' . $field_id ] = array_map( 'trim', $_POST[ 'field_' . $field_id ] ); // phpcs:ignore
						} else {
							$_POST[ 'field_' . $field_id ] = trim( $_POST[ 'field_' . $field_id ] ); // phpcs:ignore
						}
					}

					if ( ! empty( $date_fields ) && in_array( $field, $date_fields, true ) && ! isset( $_POST[ 'field_' . $field_id ] ) ) {
						$_POST[ 'field_' . $field_id ] = '';
					}

					// Create errors for required fields without values.
					if ( xprofile_check_is_required_field( $field_id ) && empty( $_POST[ 'field_' . $field_id ] ) && ! bp_current_user_can( 'bp_moderate' ) ) {
						$bp->signup->errors[ 'field_' . $field_id ] = __( 'This is a required field.', 'buddyboss-app' );
					} else {
						// Validate xprofile.
						$message = ( function_exists( 'xprofile_validate_field' ) ? xprofile_validate_field( $field_id, $_POST[ 'field_' . $field_id ], '' ) : '' ); // phpcs:ignore

						if ( isset( $_POST[ 'field_' . $field_id ] ) && ! empty( $message ) ) {
							$bp->signup->errors[ 'field_' . $field_id ] = $message;
						}
					}
				}
			}
		}

		// Finally, let's check the blog details, if the user wants a blog and blog creation is enabled.
		if ( isset( $_POST['signup_with_blog'] ) && ! empty( $_POST['signup_with_blog'] ) ) {
			$active_signup = bp_core_get_root_option( 'registration' );

			if ( 'blog' === $active_signup || 'all' === $active_signup ) {
				$blog_details = bp_core_validate_blog_signup( ( isset( $_POST['signup_blog_url'] ) ? $_POST['signup_blog_url'] : '' ), ( isset( $_POST['signup_blog_title'] ) ? $_POST['signup_blog_title'] : '' ) ); // phpcs:ignore

				// If there are errors with blog details, set them for display.
				if ( ! empty( $blog_details['errors']->errors['blogname'] ) ) {
					$bp->signup->errors['signup_blog_url'] = $blog_details['errors']->errors['blogname'][0];
				}

				if ( ! empty( $blog_details['errors']->errors['blog_title'] ) ) {
					$bp->signup->errors['signup_blog_title'] = $blog_details['errors']->errors['blog_title'][0];
				}
			}
		}

		if ( ! empty( $bp->signup->errors ) ) {
			if ( function_exists( 'bp_xprofile_nickname_field_id' ) && isset( $bp->signup->errors['signup_username'] ) ) {
				if ( ! isset( $bp->signup->errors[ 'field_' . bp_xprofile_nickname_field_id() ] ) ) {
					$bp->signup->errors[ 'field_' . bp_xprofile_nickname_field_id() ] = $bp->signup->errors['signup_username'];
				}

				unset( $bp->signup->errors['signup_username'] );
			}

			return new WP_Error( 'bp_rest_register_errors', $bp->signup->errors, array( 'status' => 400 ) );
		}

		// No errors! Let's register those deets.
		$active_signup = bp_core_get_root_option( 'registration' );

		if ( 'none' === $active_signup ) {
			return new WP_Error( 'bp_rest_signup_disabled', __( 'Sorry, you are not authorized to perform this action.', 'buddyboss-app' ), array( 'status' => rest_authorization_required_code() ) );
		}

		return array( 'verify' => true );
	}
}
