<?php
/**
 * Class APP Rest API.
 *
 * @package BuddyBossApp\Api\Core
 */

namespace BuddyBossApp\Api\Core\V1;

use BuddyBossApp\Api\Core\V1\AccountSettings\AccountSettingsRest;
use BuddyBossApp\Api\Core\V1\AccountSettingsOptions\AccountSettingsOptionsRest;
use BuddyBossApp\Api\Core\V1\Core\CoreRest;
use BuddyBossApp\Api\Core\V1\Notification\NotificationRest;
use BuddyBossApp\Api\Core\V1\SignUp\SignUpRest;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Class RestAPI
 *
 * @package BuddyBossApp\Api\Core\V1
 */
class RestAPI {

	/**
	 * Class instance.
	 *
	 * @var object
	 */
	protected static $instance;

	/**
	 * RestAPI constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return mixed|object
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Class hooks.
	 */
	public function hooks() {
		add_action( 'init', array( $this, 'init' ) );
		add_action( 'rest_api_init', array( $this, 'api_init' ) );
		$this->load_apis();
	}

	/**
	 * Init Hooks.
	 */
	public function init() {
		add_filter( 'rest_prepare_user', array( $this, 'rest_prepare_user' ), 10, 3 );
	}

	/**
	 * Load different api classes.
	 */
	public function load_apis() {
		AccountSettingsRest::instance();
		AccountSettingsOptionsRest::instance();
		SignUpRest::instance();
		NotificationRest::instance();
		// Core Rest API.
		CoreRest::instance();
	}

	/**
	 * Rest api init.
	 */
	public function api_init() {
		// Remove tag from audio shortcode.
		add_filter(
			'wp_audio_shortcode_library',
			function () {
				return '';
			}
		);
	}

	/**
	 * Prepare user avatar.
	 *
	 * @param WP_REST_Response $response The response data.
	 * @param object           $user     WP_User object.
	 * @param WP_REST_Request  $request  Request used to generate the response.
	 *
	 * @return mixed
	 */
	public function rest_prepare_user( $response, $user, $request ) {

		if ( ! defined( 'BP_PLATFORM_VERSION' ) ) {

			/**
			 * We add member single request on users API on member_data object.
			 */

			$response->data['avatar'] = array(
				'full'  => html_entity_decode( get_avatar_url( $user->ID ) ),
				'thumb' => html_entity_decode( get_avatar_url( $user->ID ) ),
			);
		}

		return $response;

	}

}
