<?php
/**
 * Class APP Rest Notification.
 *
 * @package BuddyBossApp\Api\Core
 */

namespace BuddyBossApp\Api\Core\V1\Notification;

use BuddyBossApp\Notification\Notification;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Class NotificationRest
 *
 * @package BuddyBossApp\Api\Core\V1\Notification
 */
class NotificationRest extends WP_REST_Controller {

	/**
	 * API namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'buddyboss-app/v1';

	/**
	 * API Rest base.
	 *
	 * @var string
	 */
	protected $rest_base = 'notifications';

	/**
	 * Class instance.
	 *
	 * @var object
	 */
	protected static $instance;

	/**
	 * Allow batch.
	 *
	 * @var true[] $allow_batch
	 */
	protected $allow_batch = array( 'v1' => true );

	/**
	 * NotificationRest constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return mixed|object
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Class hooks.
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'allow_batch' => $this->allow_batch,
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema'      => array( $this, 'get_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				'args'   => array(
					'id' => array(
						'description' => __( 'A unique numeric ID for the notification.', 'buddyboss-app' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param(
							array(
								'default' => 'view',
							)
						),
					),
				),
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::EDITABLE ),
				),
				array(
					'methods'             => WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/bulk/read',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_read' ),
					'permission_callback' => array( $this, 'update_read_permissions_check' ),
					'args'                => array(
						'context'  => $this->get_context_param(),
						'page'     => array(
							'description'       => __( 'Current page of the collection.' ),
							'type'              => 'integer',
							'default'           => 1,
							'sanitize_callback' => 'absint',
							'validate_callback' => 'rest_validate_request_arg',
							'minimum'           => 1,
						),
						'per_page' => array(
							'description'       => __( 'Maximum number of items to be returned in result set.' ),
							'type'              => 'integer',
							'default'           => 10,
							'minimum'           => 1,
							'maximum'           => 100,
							'sanitize_callback' => 'absint',
							'validate_callback' => 'rest_validate_request_arg',
						),
					),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve notifications.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_REST_Response | WP_Error
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/v1/notifications Notifications
	 * @apiName        GetBBNotifications
	 * @apiGroup       Notifications
	 * @apiDescription Retrieve notifications
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page=1] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String=id,date_notified,item_id,secondary_item_id,component_name,component_action} [order_by=id] Name of the field to order according to.
	 * @apiParam {String=ASC,DESC} [sort_order=ASC] Order sort attribute ascending or descending.
	 * @apiParam {String} [component_name]  Limit result set to notifications associated with a specific component.
	 * @apiParam {String} [component_action]  Limit result set to notifications associated with a specific component's action name.
	 * @apiParam {Number} [user_id] Limit result set to notifications addressed to a specific user.
	 * @apiParam {Number} [item_id] Limit result set to notifications associated with a specific item ID.
	 * @apiParam {Number} [secondary_item_id] Limit result set to notifications associated with a specific secondary item ID.
	 * @apiParam {Boolean} [is_new=true] Limit result set to items from specific states.
	 */
	public function get_items( $request ) {
		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			return $this->get_route_response( $request, '/buddyboss/v1/notifications' );
		}

		return $this->get_route_response( $request, '/buddyboss-app/core/v2/notifications' );

	}

	/**
	 * Check if a given request has access to notification items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = new WP_Error(
				'bbapp_rest_authorization_required',
				__( 'Sorry, you are not allowed to see notifications.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		/**
		 * Filter the notifications `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_rest_notifications_get_items_permissions_check', $retval, $request );
	}

	/**
	 * Create a notification.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return false
	 * @since          0.1.0
	 *
	 * @api            {POST} /wp-json/buddyboss-app/v1/notifications Create Notification
	 * @apiName        CreateBBNotifications
	 * @apiGroup       Notifications
	 * @apiDescription Create a notifications
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [user_id] A unique numeric ID for the notification.
	 * @apiParam {Number|String} [item_id] The ID of the item associated with the notification.
	 * @apiParam {Number|String} [secondary_item_id] The ID of the secondary item associated with the notification.
	 * @apiParam {String} [component] The name of the component associated with the notification.
	 * @apiParam {String} [action] The name of the component action associated with the notification.
	 * @apiParam {String} [date] The date the notification was sent/created.
	 * @apiParam {Number} [is_new] Whether the notification is new or not.
	 */
	public function create_item( $request ) {

		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			return $this->get_route_response( $request, '/buddyboss/v1/notifications' );
		}

		return $this->get_route_response( $request, '/buddyboss-app/core/v2/notifications' );
	}

	/**
	 * Checks if a given request has access to create a notification.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error|bool
	 * @since 0.1.0
	 */
	public function create_item_permissions_check( $request ) {
		$retval = $this->get_items_permissions_check( $request );

		/**
		 * Filter the notifications `create_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_rest_notifications_create_item_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve a notification.
	 *
	 * @param WP_REST_Request|int $request Full details about the request.
	 *
	 * @return \WP_REST_Response | WP_Error
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/v1/notifications/:id Notification
	 * @apiName        GetBBNotification
	 * @apiGroup       Notifications
	 * @apiDescription Retrieve a notification
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the notification.
	 */
	public function get_item( $request ) {
		$notification_id = is_numeric( $request ) ? $request : (int) $request['id'];
		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			return $this->get_route_response( $request, "/buddyboss/v1/notifications/{$notification_id}" );
		}

		return $this->get_route_response( $request, "/buddyboss-app/core/v2/notifications/{$notification_id}" );
	}

	/**
	 * Check if a given request has access to get information about a specific notification.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = new WP_Error(
				'bbapp_rest_authorization_required',
				__( 'Sorry, you are not allowed to see the notification.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		/**
		 * Filter the notifications `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_rest_notifications_get_item_permissions_check', $retval, $request );
	}

	/**
	 * Update a notification.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_REST_Response | WP_Error
	 * @since          0.1.0
	 *
	 * @api            {PATCH} /wp-json/buddyboss-app/v1/notifications/:id Update Notification
	 * @apiName        UpdateBBNotification
	 * @apiGroup       Notifications
	 * @apiDescription Update notification
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the notification.
	 * @apiParam {Number} [is_new=0] Whether it's a new notification or not.
	 */
	public function update_item( $request ) {
		$notification_id = is_numeric( $request ) ? $request : (int) $request['id'];
		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			return $this->get_route_response( $request, "/buddyboss/v1/notifications/{$notification_id}" );
		}

		return $this->get_route_response( $request, "/buddyboss-app/core/v2/notifications/{$notification_id}" );
	}

	/**
	 * Check if a given request has access to update a notification.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error|bool
	 * @since 0.1.0
	 */
	public function update_item_permissions_check( $request ) {
		$retval = $this->get_item_permissions_check( $request );

		/**
		 * Filter the notifications `update_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_rest_notifications_update_item_permissions_check', $retval, $request );
	}

	/**
	 * Delete a notification.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_REST_Response | WP_Error
	 * @since          0.1.0
	 *
	 * @api            {DELETE} /wp-json/buddyboss-app/v1/notifications/:id Delete Notification
	 * @apiName        DeleteBBNotification
	 * @apiGroup       Notifications
	 * @apiDescription Delete notification
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the notification.
	 */
	public function delete_item( $request ) {
		$notification_id = is_numeric( $request ) ? $request : (int) $request['id'];
		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			return $this->get_route_response( $request, "/buddyboss/v1/notifications/{$notification_id}" );
		}

		return $this->get_route_response( $request, "/buddyboss-app/core/v2/notifications/{$notification_id}" );
	}

	/**
	 * Bulk read notifications.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          1.8.90
	 *
	 * @return \WP_REST_Response | WP_Error
	 * @api            {DELETE} /wp-json/buddyboss-app/v1/notifications/bulk/read Bulk notification read
	 * @apiName        BulkBBNotificationRead
	 * @apiGroup       Notifications
	 * @apiDescription Bulk notification read
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function update_read( $request ) {
		$user_id = get_current_user_id();

		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			return $this->get_route_response( $request, '/buddyboss/v1/notifications/bulk/read' );
		}

		$page     = ! empty( $request->get_param( 'page' ) ) ? (int) $request->get_param( 'page' ) : 1;
		$per_page = ! empty( $request->get_param( 'per_page' ) ) ? (int) $request->get_param( 'per_page' ) : 10;

		$notifications = Notification::instance()->get_inbuilt_notifications(
			array(
				'per_page' => $per_page * $page,
				'user_id'  => $user_id,
				'page'     => 1,
				'is_new'   => true,
			)
		);

		if ( ! empty( $notifications ) ) {
			foreach ( $notifications as $notification ) {
				Notification::instance()->mark_notification_as_read_by_id( $user_id, $notification->id );
			}
		}

		// Get another unread notification.
		$request->set_param( 'user_id', $user_id );
		$request->set_param( 'page', 1 );
		$request->set_param( 'per_page', $per_page );
		$request->set_param( 'is_new', true );

		$response = $this->get_items( $request );

		/**
		 * Fire after user read notifications has been updated via the REST API.
		 *
		 * @param WP_REST_Request  $request  The request sent to the API.
		 * @param WP_REST_Response $response The response data.
		 *
		 * @since 1.8.90
		 */
		do_action( 'bbapp_rest_notifications_update_read', $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to notification update read.
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 *
	 * @return \WP_Error|bool
	 */
	public function update_read_permissions_check( $request ) {
		$retval = new \WP_Error(
			'rest_not_logged_in',
			__( 'Sorry, you are not allowed to read notifications.', 'buddyboss-app' ),
			array(
				'status' => rest_authorization_required_code(),
			)
		);

		if ( is_user_logged_in() ) {
			$retval = true;
		}

		/**
		 * Filter the notifications `update_read` permissions check.
		 *
		 * @param bool|\WP_Error   $retval  Returned value.
		 * @param \WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_notifications_update_read_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to delete a notification.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function delete_item_permissions_check( $request ) {
		$retval = $this->get_item_permissions_check( $request );

		/**
		 * Filter the notifications `delete_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_rest_notifications_delete_item_permissions_check', $retval, $request );
	}

	/**
	 * Get the query params for the notifications collections.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params                       = parent::get_collection_params();
		$params['context']['default'] = 'view';

		// Remove the search argument.
		unset( $params['search'] );

		$params['order_by'] = array(
			'description'       => __( 'Name of the field to order according to.', 'buddyboss-app' ),
			'default'           => 'id',
			'type'              => 'string',
			'enum'              => array(
				'id',
				'date_notified',
				'item_id',
				'secondary_item_id',
				'component_name',
				'component_action',
			),
			'sanitize_callback' => 'sanitize_key',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['sort_order'] = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'default'           => 'ASC',
			'type'              => 'string',
			'enum'              => array( 'ASC', 'DESC' ),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['component_name'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific component', 'buddyboss-app' ),
			'default'           => '',
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['component_action'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific component\'s action name.', 'buddyboss-app' ),
			'default'           => '',
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['user_id'] = array(
			'description'       => __( 'Limit result set to notifications addressed to a specific user.', 'buddyboss-app' ),
			'default'           => get_current_user_id(),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['item_id'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific item ID.', 'buddyboss-app' ),
			'default'           => 0,
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['secondary_item_id'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific secondary item ID.', 'buddyboss-app' ),
			'default'           => 0,
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['is_new'] = array(
			'description'       => __( 'Limit result set to items from specific states.', 'buddyboss-app' ),
			'default'           => true,
			'type'              => 'boolean',
			'sanitize_callback' => 'rest_sanitize_boolean',
			'validate_callback' => 'rest_validate_request_arg',
		);

		/**
		 * Filters the collection query params.
		 *
		 * @param array $params Query params.
		 */
		return apply_filters( 'bbapp_rest_notifications_collection_params', $params );
	}

	/**
	 * Select the item schema arguments needed for the EDITABLE method.
	 *
	 * @param string $method Optional. HTTP method of the request.
	 *
	 * @return array Endpoint arguments.
	 */
	public function get_endpoint_args_for_item_schema( $method = WP_REST_Server::CREATABLE ) {
		$args = WP_REST_Controller::get_endpoint_args_for_item_schema( $method );
		$key  = 'get_item';

		if ( WP_REST_Server::EDITABLE === $method ) {
			$key = 'update_item';

			// Only switching the is_new property can be achieved.
			$args                      = array_intersect_key( $args, array( 'is_new' => true ) );
			$args['is_new']['default'] = 0;
		} elseif ( WP_REST_Server::CREATABLE === $method ) {
			$key = 'create_item';
		} elseif ( WP_REST_Server::DELETABLE === $method ) {
			$key = 'delete_item';
		}

		/**
		 * Filters the method query arguments.
		 *
		 * @param array  $args   Query arguments.
		 * @param string $method HTTP method of the request.
		 */
		return apply_filters( "bbapp_rest_notifications_{$key}_query_arguments", $args, $method );
	}

	/**
	 * Get the notification schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'bbapp_notifications',
			'type'       => 'object',
			'properties' => array(
				'id'                => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'A unique numeric ID for the notification.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'integer',
				),
				'user_id'           => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The ID of the user the notification is addressed to.', 'buddyboss-app' ),
					'type'        => 'integer',
					'default'     => get_current_user_id(),
				),
				'item_id'           => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The ID of the item associated with the notification.', 'buddyboss-app' ),
					'type'        => 'integer',
				),
				'secondary_item_id' => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The ID of the secondary item associated with the notification.', 'buddyboss-app' ),
					'type'        => 'integer',
				),
				'component'         => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The name of the BuddyPress component the notification relates to.', 'buddyboss-app' ),
					'type'        => 'string',
				),
				'action'            => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The name of the component\'s action the notification is about.', 'buddyboss-app' ),
					'type'        => 'string',
				),
				'date'              => array(
					'description' => __( 'The date the notification was created, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'embed', 'view', 'edit' ),
				),
				'is_new'            => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Whether it\'s a new notification or not.', 'buddyboss-app' ),
					'type'        => 'integer',
					'default'     => 1,
				),
				'object'            => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The name of the notification object.', 'buddyboss-app' ),
					'type'        => 'string',
				),
				'object_id'         => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The ID of the notification object.', 'buddyboss-app' ),
					'type'        => 'integer',
				),
				'description'       => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Description of notification.', 'buddyboss-app' ),
					'type'        => 'object',
					'readonly'    => true,
					'properties'  => array(
						'rendered' => array(
							'description' => __( 'HTML description for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'embed', 'view', 'edit' ),
							'readonly'    => true,
						),
					),
				),
				'link_url'          => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Link URL for the notification.', 'buddyboss-app' ),
					'type'        => 'string',
				),
				'rest_actions'      => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Rest Actions which perform accept/reject based on the status.', 'buddyboss-app' ),
					'type'        => 'object',
				),
			),
		);

		if ( function_exists( 'bp_core_fetch_avatar' ) || function_exists( 'get_avatar_url' ) ) {
			$avatar_properties = array();

			$avatar_properties['full'] = array(
				'description' => __( 'Avatar URL with full image size.', 'buddyboss-app' ),
				'type'        => 'string',
				'format'      => 'uri',
				'context'     => array( 'embed', 'view', 'edit' ),
			);

			$avatar_properties['thumb'] = array(
				'description' => __( 'Avatar URL with thumb image size.', 'buddyboss-app' ),
				'type'        => 'string',
				'format'      => 'uri',
				'context'     => array( 'embed', 'view', 'edit' ),
			);

			$schema['properties']['avatar_urls'] = array(
				'description' => __( 'Avatar URLs for the notification.', 'buddyboss-app' ),
				'type'        => 'object',
				'context'     => array( 'embed', 'view', 'edit' ),
				'readonly'    => true,
				'properties'  => $avatar_properties,
			);
		}

		/**
		 * Filters the notifications schema.
		 *
		 * @param array $schema The endpoint schema.
		 */
		return apply_filters( 'bbapp_rest_notification_schema', $this->add_additional_fields_schema( $schema ) );
	}

	/**
	 * Handle the request and return a valid response for this endpoint.
	 *
	 * @param \WP_REST_Request $request   Request object.
	 * @param string           $route_url Rest route URL.
	 *
	 * @return mixed
	 */
	protected function get_route_response( $request, $route_url ) {
		global $wp_rest_server;
		/**
		 * Redirect to topic endpoint.
		 */
		$request_curl = new WP_REST_Request( $request->get_method(), $route_url );
		$request_curl->set_headers( $request->get_headers() );
		// As per set_body function, we need pass string here. $request is as an object so need to encode here.
		$request_curl->set_body( $request->get_body() );
		if ( ! empty( $request->get_headers() ) ) {
			foreach ( $request->get_headers() as $key => $data ) {
				$request_curl->set_header( $key, $data );
			}
		}

		if ( ! empty( $request->get_params() ) ) {
			foreach ( $request->get_params() as $key => $data ) {
				$request_curl->set_param( $key, $data );
			}
		}

		$response = $this->dispatch( $request_curl );

		if ( isset( $response->headers ) ) {
			$wp_rest_server->send_headers( $response->headers );
		}

		return rest_ensure_response( $response );

	}

	/**
	 * Dispatch the request item.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {

		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()['_embed'] ) ) {
			$query_params['_embed'] = $request->get_query_params()['_embed'];
		}

		$request->set_query_params( $query_params );

		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}
}
