<?php
/**
 * Holds core rest functionality.
 *
 * @package BuddyBossApp\Api\Core\V1\Core
 */

namespace BuddyBossApp\Api\Core\V1\Core;

use BuddyBossApp\AppMenu;
use BuddyBossApp\AppSettings;
use BuddyBossApp\Auth\Common;
use BuddyBossApp\ClientCommon;
use BuddyBossApp\Helpers\BBAPP_File;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Menu;
use BuddyBossApp\RestErrors;
use WP_Error;
use WP_REST_Request;

/**
 * Class CoreRest
 *
 * @package BuddyBossApp\Api\Core\V1\Core
 */
class CoreRest {

	/**
	 * API namespace.
	 *
	 * @var string $namespace
	 */
	protected $namespace = 'buddyboss-app/core/v1';

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Allow batch.
	 *
	 * @var true[] $allow_batch
	 */
	protected $allow_batch = array( 'v1' => true );

	/**
	 * CoreRest constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 */
	public function hooks() {
		// Other Rest Endpoints.
		add_action( 'rest_api_init', array( $this, 'register_common_fields' ), 99 );
		add_filter( 'init', array( $this, 'dispatch_headers' ), 8 );
		add_filter( 'rest_pre_dispatch', array( $this, 'register_from_header' ), 99, 3 );
		add_filter( 'rest_pre_dispatch', array( $this, 'rest_set_cookie' ), 10, 3 );
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
		add_filter(
			'bb_exclude_endpoints_from_restriction',
			array(
				$this,
				'exclude_endpoints_from_restriction',
			),
			10,
			2
		);
		add_filter( 'rest_post_disprepare_header_cache', array( $this, 'add_maintenance_mode_header' ) );
	}

	/**
	 * Register WordPress endpoint routes.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/menu',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'menu' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'allow_more' => array(
							'type'              => 'boolean',
							'default'           => true,
							'enum'              => array( true, false ),
							'validate_callback' => 'rest_validate_request_arg',
						),
					),
				),
				'allow_batch' => $this->allow_batch,
			)
		);

		register_rest_route(
			$this->namespace,
			'/menu/(?P<app_id>[a-zA-Z0-9-]+)',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'menu' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'allow_more' => array(
							'type'              => 'boolean',
							'default'           => true,
							'enum'              => array( true, false ),
							'validate_callback' => 'rest_validate_request_arg',
						),
					),
				),
				'allow_batch' => $this->allow_batch,
			)
		);

		register_rest_route(
			$this->namespace,
			'/device_token',
			array(
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( $this, 'update_device_token' ),
					'args'                => array(),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/app_details',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( $this, 'app_details' ),
				),
				'allow_batch' => $this->allow_batch,
			)
		);

		register_rest_route(
			$this->namespace,
			'/settings(?:/(?P<app_id>[a-zA-Z0-9-]+))?',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( AppSettings::instance(), 'rest_response' ),
					'args'                => array(
						'app_id' => array(
							'required'          => false,
							'validate_callback' => function ( $param ) {
								return empty( $param ) || preg_match( '/^[a-zA-Z0-9-]+$/', $param );
							},
							'description'       => __( 'Optional app ID for specific app settings', 'buddyboss-app' ),
						),
					),
				),
				'allow_batch' => $this->allow_batch,
			)
		);

		register_rest_route(
			$this->namespace,
			'/profile/details',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( $this, 'get_profile_details' ),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/search/details',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( $this, 'get_search_details' ),
				),
			)
		);

		/**
		 * This endpoint is only for Health check
		 */
		register_rest_route(
			$this->namespace,
			'/cache_test',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( $this, 'get_cache_test' ),
				),
			)
		);

		/**
		 * This endpoint is feedback endpoint.
		 */
		register_rest_route(
			$this->namespace,
			'/feedback',
			array(
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'send_feedback' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'message' => array(
							'required'          => true,
							'type'              => 'string',
							'validate_callback' => 'rest_validate_request_arg',
						),
						'fields'  => array(
							'description' => __( 'Feedback fields.', 'buddyboss-app' ),
							'default'     => array(),
						),
						'type'    => array(
							'description'       => __( 'Feedback Type.', 'buddyboss-app' ),
							'type'              => 'string',
							'default'           => 'bug',
							'enum'              => array( 'bug', 'feedback' ),
							'validate_callback' => 'rest_validate_request_arg',
						),
					),
				),
			)
		);

		if ( ! defined( 'BP_PLATFORM_VERSION' ) ) {
			/**
			 * This endpoint is delete account endpoint.
			 */
			register_rest_route(
				$this->namespace,
				'/delete-account',
				array(
					array(
						'methods'             => \WP_REST_Server::EDITABLE,
						'callback'            => array( $this, 'send_delete_account' ),
						'permission_callback' => '__return_true',
						'args'                => array(
							'username'  => array(
								'required'          => true,
								'description' => __( 'Username.', 'buddyboss-app' ),
								'default'     => '',
							),
							'user_email'  => array(
								'required'          => true,
								'description' => __( 'User Email.', 'buddyboss-app' ),
								'default'     => '',
							),
						),
					),
				)
			);
		}
	}

	/**
	 * Get app menus.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return mixed
	 * @api            {GET} /wp-json/buddyboss-app/core/v1/menu/:app_id App Menu
	 * @apiName        CoreMenuSettings
	 * @apiGroup       Core Settings
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Core menu settings
	 * @apiParam {String} app_id App Id
	 * @apiUse         apidocForCoreMenuSettingsV1
	 */
	public function menu( $request ) {
		$app_id = $request->get_header( 'app_id' );

		if ( ! empty( $app_id ) ) {
			$app_id = $request->get_header( 'appid' );
		}

		if ( ! empty( $app_id ) ) {
			$request->set_param( 'app_id', $app_id );
		}

		return Menu::instance()->rest_response( $request );
	}

	/**
	 * Update device token.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return WP_Error
	 * @api            {PATCH} /wp-json/buddyboss-app/core/v1/device_token Update device token
	 * @apiName        UpdateDeviceToken
	 * @apiGroup       Core Settings
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Update device token
	 * @apiParam {String} [app_id=false] App Id
	 * @apiParam {String} [unique_id=false] Used interchangeably with app_id, one is required
	 */
	public function update_device_token( $request ) {

		// Prevent unauthorised access.
		if ( ! is_user_logged_in() ) {
			return new \WP_Error( 'not_logged_in', __( 'Sorry, please login to perform the action.', 'buddyboss-app' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$user_id      = get_current_user_id();
		$device_token = $request->get_param( 'device_token' );
		$device_id    = $request->get_param( 'device_id' );
		$device_model = $request->get_param( 'device_model' );

		$result = bbapp_notifications()->register_device_for_user(
			$user_id,
			$device_token,
			false,
			array(
				'device_id'    => $device_id,
				'device_model' => $device_model,
			)
		);

		if ( $result ) {
			return rest_ensure_response(
				array(
					'success' => true,
					'message' => __( 'Token updated successfully.', 'buddyboss-app' ),
				)
			);
		}

		return new \WP_Error( 'error_updating', __( 'Error while updating device token.', 'buddyboss-app' ), array( 'status' => 500 ) );
	}

	/**
	 * Get profile details.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return array|WP_Error|\WP_HTTP_Response|\WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/core/v1/profile/details Profile tabs details
	 * @apiName        GetAProfileDetails
	 * @apiGroup       Core Settings
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Get a profile details
	 * @apiError {Object}  Response object
	 */
	public function get_profile_details( $request ) {
		/**
		 * Get Navigation item for Profile page
		 */
		$tabs = apply_filters( 'bbapp_rest_profile_tab', array() );

		/**
		 * Make sure tabs are array.
		 */
		$_tabs = array();

		foreach ( $tabs as $tab ) {
			$_tabs[] = $tab;
		}

		return rest_ensure_response(
			array(
				'tabs' => $_tabs,
			)
		);
	}

	/**
	 * Get search details.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return array|WP_Error|\WP_HTTP_Response|\WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/core/v1/search/details Search tabs details
	 * @apiName        GetASearchDetails
	 * @apiGroup       Core Settings
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Get a search details
	 * @apiError {Object}  Response object
	 */
	public function get_search_details( $request ) {
		// Get Navigation item for Profile page.
		$tabs = apply_filters( 'bbapp_rest_search_tab', array() );

		return rest_ensure_response(
			array(
				'tabs' => $tabs,
			)
		);
	}

	/**
	 * App details.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return mixed|\WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/core/v1/app_details App details
	 * @apiName        GetAppDetails
	 * @apiGroup       Core Settings
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription This endpoint returns the signature information from bbapp server.
	 * @apiUse         apidocForGetAppDetailsV1
	 */
	public function app_details( $request ) {
		$details = ManageApp::instance()->get_app_sign_details();

		if ( ! $details ) {
			return new \WP_Error( 'error_getting_details', __( 'Error while getting details, try again later.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		return rest_ensure_response( $details );
	}

	/**
	 * Get cache test.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return array
	 */
	public function get_cache_test( $request ) {
		return array(
			'timestamp' => time(),
		);
	}

	/**
	 * Feedback api callback.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return array|bool|WP_Error
	 * @api           {POST} /wp-json/buddyboss-app/core/v1/feedback
	 * @apiGroup      Core Settings
	 * @apiVersion    1.0.0
	 * @apiPermission LoggedInUser
	 */
	public function send_feedback( $request ) {
		if ( ! is_user_logged_in() ) {
			return RestErrors::instance()->user_not_logged_in();
		}

		$message = isset( $request['message'] ) ? sanitize_textarea_field( $request['message'] ) : '';
		$fields  = isset( $request['fields'] ) ? $request['fields'] : array();
		$type    = isset( $request['type'] ) ? sanitize_text_field( $request['type'] ) : 'bug';

		if ( empty( $message ) ) {
			return new WP_Error( 'rest_bbapp_message_req', __( 'A message param is required.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$sanitize_fields = array(
			'Message' => $message,
			'App ID'  => ManageApp::instance()->get_app_id(),
		);

		if ( isset( $fields ) && ! empty( $fields ) ) {
			// Get the fields with sanitize.
			foreach ( $fields as $key => $field ) {
				$sanitize_fields[ $key ] = sanitize_text_field( $field );
			}
		}

		if ( ! is_email( $sanitize_fields['Email'] ) ) {
			return new WP_Error( 'rest_bbapp_email_invalid', __( 'Email provided is invalid.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$email_fields = array(
			'Message'         => array(
				'label' => _x( 'Message', 'app feedback email', 'buddyboss-app' ),
			),
			'Email'           => array(
				'label' => _x( 'Email', 'app feedback email', 'buddyboss-app' ),
			),
			'App ID'          => array(
				'label' => _x( 'App ID', 'app feedback email', 'buddyboss-app' ),
			),
			'Platform'        => array(
				'label' => _x( 'Platform', 'app feedback email', 'buddyboss-app' ),
			),
			'App Version'     => array(
				'label' => _x( 'App Version', 'app feedback email', 'buddyboss-app' ),
			),
			'iOS Version'     => array(
				'label' => _x( 'iOS Version', 'app feedback email', 'buddyboss-app' ),
			),
			'Android Version' => array(
				'label' => _x( 'Android Version', 'app feedback email', 'buddyboss-app' ),
			),
			'Device Model'    => array(
				'label' => _x( 'Device Model', 'app feedback email', 'buddyboss-app' ),
			),
		);

		// If network mode so we pass the current site name as blog name.
		if ( is_multisite() ) {
			$blogname = $GLOBALS['current_site']->site_name;
		} else {
			$blogname = wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES );
		}

		$email_setting_key         = 'support_email.email';
		$email_subject_setting_key = 'support_email.subject';

		// Create mail title.
		if ( 'bug' === $type ) {
			$email_setting_key         = 'bugs_email.email';
			$email_subject_setting_key = 'bugs_email.subject';
		}

		$reply_to = $sanitize_fields['Email'];
		// Mail header.
		$headers[] = 'Content-Type: text/html; charset="' . get_option( 'blog_charset' ) . "\"\n";
		$headers[] = "Reply-To: {$reply_to} <{$reply_to}>";
		// Get the feedback email from setting.
		$email   = AppSettings::instance()->get_setting_value( $email_setting_key );
		$subject = AppSettings::instance()->get_setting_value( $email_subject_setting_key );

		if ( empty( $subject ) ) {
			if ( 'bug' === $type ) {
				/* translators: %s: BLog name. */
				$subject = sprintf( __( '[%s] App Bug', 'buddyboss-app' ), $blogname );
			} else {
				/* translators: %s: BLog name. */
				$subject = sprintf( __( '[%s] App Feedback', 'buddyboss-app' ), $blogname );
			}
		}

		if ( empty( $email ) ) {
			$email = get_option( 'admin_email' );
		}

		// Create the content html with data.
		$app_message_content = '<table>';

		foreach ( $email_fields as $key => $field ) {
			if ( ! isset( $sanitize_fields[ $key ] ) ) {
				continue;
			}

			$value                = $sanitize_fields[ $key ];
			$app_message_content .= "<tr><th style='text-align: left;'>{$field["label"]}:</th><td>{$sanitize_fields[ $key ]}</td></tr>";
		}

		$app_message_content .= '</table>';
		// Get the image file from $_FILES.
		$files = $request->get_file_params();

		if ( ! empty( $files ) ) {
			// File allowed ext.
			$allowed_types       = array( 'image', 'video' );
			$file_type_extension = explode( '/', $files['file']['type'] );

			if ( in_array( $file_type_extension[0], $allowed_types, true ) ) {
				$dir = $this->bbapp_feedback_dir();
				// File upload on selected directory.
				move_uploaded_file( $files['file']['tmp_name'], $dir . basename( $files['file']['name'] ) );
				$feedback_file = $dir . basename( $files['file']['name'] );
				// Send email with feedback|bug file.
				$send = wp_mail( $email, wp_specialchars_decode( $subject ), $app_message_content, $headers, $feedback_file );

				if ( file_exists( $feedback_file ) ) {
					wp_delete_file( $feedback_file );
				}

				if ( ! $send ) {
					return new \WP_Error(
						'couldnt_send_mail',
						__( 'The email could not be sent. Possible reason: your host may have disabled the mail() function.', 'buddyboss-app' ),
						array(
							'status' =>
								500,
						)
					);
				}
			} else {
				return new \WP_Error(
					'invalid_uploaded_file',
					__( "Uploaded files are invalid. Please upload ('png' ,'jpg', 'mp4') files.", 'buddyboss-app' ),
					array(
						'status' =>
							500,
					)
				);
			}
		} else {
			if ( ! wp_mail( $email, wp_specialchars_decode( $subject ), $app_message_content, $headers ) ) {
				return new \WP_Error(
					'couldnt_send_mail',
					__( 'The email could not be sent. Possible reason: your host may have disabled the mail() function.', 'buddyboss-app' ),
					array(
						'status' =>
							500,
					)
				);
			}
		}

		return array( 'success' => true );
	}

	/**
	 * Delete Account api callback.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return array|bool|WP_Error
	 * @api           {POST} /wp-json/buddyboss-app/core/v1/delete-account
	 * @apiGroup      Core Settings
	 * @apiVersion    1.0.0
	 * @apiPermission LoggedInUser
	 */
	public function send_delete_account( $request ) {
		if ( ! is_user_logged_in() ) {
			return RestErrors::instance()->user_not_logged_in();
		}

		$username   = isset( $request['username'] ) ? sanitize_text_field( $request['username'] ) : '';
		$user_email = isset( $request['user_email'] ) ? sanitize_text_field( $request['user_email'] ) : '';

		if ( empty( $username ) || empty( $user_email ) ) {
			return new WP_Error( 'rest_bbapp_username_email_req', __( 'A username and user email param is required.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		// If network mode so we pass the current site name as blog name.
		if ( is_multisite() ) {
			$blogname = $GLOBALS['current_site']->site_name;
		} else {
			$blogname = wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES );
		}

		$reply_to = $user_email;
		// Mail header.
		$headers[] = 'Content-Type: text/html; charset="' . get_option( 'blog_charset' ) . "\"\n";
		$headers[] = "Reply-To: {$reply_to} <{$reply_to}>";
		// Get the feedback email from setting.
		$admin_email = AppSettings::instance()->get_setting_value( 'delete_account.admin_email' );
		$subject     = AppSettings::instance()->get_setting_value( 'delete_account.subject' );

		if ( empty( $subject ) ) {
			/* translators: %s: BLog name. */
			$subject = sprintf( __( '[%s] Account Deletion Request', 'buddyboss-app' ), $blogname );
		}

		if ( empty( $admin_email ) ) {
			$admin_email = get_option( 'admin_email' );
		}

		// Create the content html with data.
		$app_message_content = esc_html__( 'A member of your community has requested to delete their account:', 'buddyboss-app' );
		$app_message_content .= '<br><br>';
		$app_message_content .= esc_html__( 'Username: ', 'buddyboss-app' );
		$app_message_content .= '<strong>' . $username . '</strong><br>';
		$app_message_content .= esc_html__( 'User Email: ', 'buddyboss-app' );
		$app_message_content .= '<strong>' . $user_email . '</strong><br>';

		if ( ! wp_mail( $admin_email, wp_specialchars_decode( $subject ), $app_message_content, $headers ) ) {
			return new \WP_Error(
				"couldn't_send_mail",
				__( 'The email could not be sent. Possible reason: your host may have disabled the mail() function.', 'buddyboss-app' ),
				array(
					'status' =>
						500,
				)
			);
		}

		return array( 'success' => true );
	}

	/**
	 * Returns the directory of custom icons.
	 *
	 * @return string
	 */
	public function bbapp_feedback_dir() {
		$upload_dir = wp_upload_dir();
		$dir        = $upload_dir['basedir'] . '/bbapp-feedback/';

		if ( ! file_exists( $dir ) ) {
			BBAPP_File::create_dir( $dir );
		}

		return $dir;
	}

	/**
	 * Register rest fields.
	 */
	public function register_common_fields() {
		register_rest_field(
			'user',
			'push_notification_settings',
			array(
				'get_callback'    => array( $this, 'get_push_notification_settings' ),
				'update_callback' => array( $this, 'update_push_notification_settings' ),
				'schema'          => null,
			)
		);

		register_rest_field(
			'user',
			'nicename',
			array(
				'get_callback'    => array( $this, 'get_nicename' ),
				'update_callback' => null,
				'schema'          => null,
			)
		);

		register_rest_field(
			'user',
			'user_email',
			array(
				'get_callback'    => array( $this, 'get_user_email' ),
				'update_callback' => null,
				'schema'          => null,
			)
		);
	}

	/**
	 * Dispatch headers.
	 */
	public function dispatch_headers() {
		$request_uri_server = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( str_contains( $request_uri_server, '/wp-json' ) ) {
			// Register maintenance mode header.
			( ManageApp::instance()->get_app_setting( 'app_maintenance_mode' ) ) ? header( 'X-BuddyBoss-App-Maintenance-Mode: yes' ) : header( 'X-BuddyBoss-App-Maintenance-Mode: no' );

			// Check if the user access token is set.
			if ( ! empty( Common::instance()->get_access_token() ) ) {
				// Register logged in header.
				( is_user_logged_in() ) ? header( 'bbapp-logged-in: yes' ) : header( 'bbapp-logged-in: no' );
			}
		}

		/**
		 * Fire any additional headers.
		 */
		do_action( 'bbapp_dispatch_headers' );
	}

	/**
	 * Register from headers.
	 *
	 * @param array $value Value.
	 *
	 * @return mixed
	 */
	public function register_from_header( $value ) {
		global $bbapp_var;

		$common                        = ClientCommon::instance();
		$bbapp_var['app_id']           = $common->capture_header( 'appid' );
		$bbapp_var['access_token']     = $common->capture_header( 'accesstoken' );
		$bbapp_var['app_version']      = $common->capture_header( 'appversion' );
		$bbapp_var['app_code_version'] = $common->capture_header( 'appcodeversion' );
		$bbapp_var['app_platform']     = $common->capture_header( 'appplatform' );

		return $value;
	}

	/**
	 * Get push notification settings.
	 *
	 * @param object          $object     Object data.
	 * @param string          $field_name Field name.
	 * @param WP_REST_Request $request    Rest request.
	 *
	 * @return array
	 */
	public function get_push_notification_settings( $object, $field_name, $request ) {
		$notification = get_user_meta( $object['id'], 'push_notification_settings', true );
		$notification = bbapp_notifications()->get_push_notification_settings_defaults( $notification );

		return $notification;
	}

	/**
	 * Update push notification settings.
	 *
	 * @param array  $value      Push notification values.
	 * @param object $object     Object data.
	 * @param string $field_name Field name.
	 *
	 * @return bool|int
	 */
	public function update_push_notification_settings( $value, $object, $field_name ) {
		return update_user_meta( $object->ID, 'push_notification_settings', $value );
	}

	/**
	 * Get user nicename.
	 *
	 * @param array           $object     Object data.
	 * @param string          $field_name Field name.
	 * @param WP_REST_Request $request    Rest request.
	 *
	 * @return string
	 */
	public function get_nicename( $object, $field_name, $request ) {
		$p = get_userdata( $object['id'] );

		return $p->user_nicename;
	}

	/**
	 * Get user email.
	 *
	 * @param array            $object     user data object.
	 * @param string           $field_name Firld name.
	 * @param \WP_REST_Request $request    WP rest request object.
	 *
	 * @since 1.7.1
	 * @return string
	 */
	public function get_user_email( $object, $field_name, $request ) {
		$user_email = '';

		if ( is_user_logged_in() ) {
			$user_data  = get_userdata( $object['id'] );
			$user_email = $user_data->user_email;
		}

		return $user_email;
	}

	/**
	 * Set rest api cookies.
	 *
	 * @param string          $result  Coockie result.
	 * @param array           $server  Server data.
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return mixed
	 */
	public function rest_set_cookie( $result, $server, $request ) {
		$cookies = $request->get_header( 'cookie' );

		if ( ! empty( $cookies ) ) {
			$cookies_parts = explode( ';', $cookies );
			$rest_cookies  = array();

			foreach ( $cookies_parts as $cookie_parts ) {
				$cookie_part = explode( '=', $cookie_parts );

				if ( empty( $cookie_part[0] ) ) {
					continue;
				}

				$rest_cookies[ trim( $cookie_part[0] ) ] = $this->utf8_rawurldecode( $cookie_part[1] );
			}

			$rest_cookies = apply_filters( 'bbapp_set_cookies', $rest_cookies, $request->get_route() );

			foreach ( $rest_cookies as $rest_cookie_key => $rest_cookie_value ) {
				$_COOKIE[ $rest_cookie_key ] = $rest_cookie_value;
			}
		}

		return $result;
	}

	/**
	 * Rest cookie to convert in correct format.
	 *
	 * @param string $raw_url_encoded Encoded URL.
	 *
	 * @return false|string
	 */
	protected function utf8_rawurldecode( $raw_url_encoded ) {
		$enc = mb_convert_encoding( rawurldecode( $raw_url_encoded ), 'ISO-8859-1', 'UTF-8' ); // Decode UTF-8.

		if ( mb_convert_encoding( $enc, 'UTF-8', 'ISO-8859-1' ) === $enc ) { // Encode UTF-8.
			return rawurldecode( $raw_url_encoded );
		} else {
			return mb_convert_encoding( rawurldecode( $raw_url_encoded ), 'UTF-8', 'ISO-8859-1' ); // Encode UTF-8.
		}
	}

	/**
	 * Unlock endpoints when the APP is in restricted mode.
	 *
	 * @since 1.3.7
	 *
	 * @param array  $endpoints        The list of unlock endpoints.
	 * @param string $current_endpoint Get current endpoint.
	 *
	 * @return array
	 */
	public function exclude_endpoints_from_restriction( $endpoints, $current_endpoint ) {
		// Get current route.
		$routes = $this->unlock_endpoints();

		if ( isset( $routes ) && is_array( $routes ) ) {
			foreach ( $routes as $unlock_path ) {
				$match = preg_match( '@^' . $unlock_path . '$@i', $current_endpoint, $matches );

				if ( ! empty( $match ) ) {
					$endpoints[] = $current_endpoint;
				}
			}
		}

		return $endpoints;
	}

	/**
	 * Add maintenance mode header to disallow headers list for cache.
	 *
	 * @param array $disallow_headers Disallowed headers.
	 *
	 * @since 1.8.50
	 *
	 * @return mixed
	 */
	public function add_maintenance_mode_header( $disallow_headers ) {
		$disallow_headers[] = 'X-BuddyBoss-App-Maintenance-Mode';

		return $disallow_headers;
	}

	/**
	 * Lists of unlock endpoints.
	 *
	 * @since 1.3.7
	 *
	 * @return array
	 */
	public function unlock_endpoints() {
		$unlock_endpoints = array(
			'/buddyboss-app/core/v1/settings/(?P<app_id>[a-zA-Z0-9-]+)',
			'/buddyboss-app/core/v1/settings',
			'/buddyboss/v1/settings',
			'/buddyboss-app/core/v1/app_details',
			'/buddyboss-app/core/v1/menu/(?P<app_id>[a-zA-Z0-9-]+)',
			'/buddyboss-app/v1/signup/form',
			'/buddyboss/v1/signup/form',
			'/buddyboss-app/v1/signup/verify',
			'/buddyboss-app/auth/v1/verify',
			'/buddyboss-app/auth/v2/verify',
			'/buddyboss-app/v1/signup',
			'/buddyboss/v1/signup',
			'/buddyboss-app/auth/v1/jwt/token',
			'/buddyboss-app/auth/v2/jwt/login',
			'/buddyboss-app/auth/v2/jwt/access-token',
			'/buddyboss-app/core/v1/device_token',
			'/buddyboss-app/auth/v1/forgot',
			'/buddyboss-app/auth/v2/jwt/logout',
			'/buddyboss-app/v1/batch',
			'/buddyboss-app/auth/v2/app-device',
		);

		// When user is logged-in.
		if ( is_user_logged_in() ) {
			$unlock_endpoints[] = '/wp/v2/users/(?P<id>[\d]+)';
			$unlock_endpoints[] = '/buddyboss/v1/members/(?P<id>[\d]+)';
		}

		// If IAP is enabled.
		if ( bbapp_is_active( 'iap' ) ) {
			$unlock_endpoints[] = '/buddyboss-app/iap/v1/products';
			$unlock_endpoints[] = '/buddyboss-app/iap/v1/order';
			$unlock_endpoints[] = '/buddyboss-app/iap/v2/order';
		}

		// Set route for registration TOS and PP if platform is active.
		if ( function_exists( 'bp_core_get_directory_page_ids' ) ) {
			$bp_pages = bp_core_get_directory_page_ids();
			$terms    = isset( $bp_pages['terms'] ) ? $bp_pages['terms'] : '';
			$privacy  = isset( $bp_pages['privacy'] ) ? $bp_pages['privacy'] : '';

			if ( ! empty( $terms ) ) {
				$unlock_endpoints[] = '/wp/v2/pages/' . $terms;
			}

			if ( ! empty( $privacy ) ) {
				$unlock_endpoints[] = '/wp/v2/pages/' . $privacy;
			}
		}

		/**
		 * Unlock endpoints when APP is in restricted mode.
		 *
		 * @since 1.3.7
		 *
		 * @param array $unlock_endpoints Endpoint lists.
		 */
		return apply_filters( 'bbapp_exclude_endpoints_from_restriction', $unlock_endpoints );
	}
}
