<?php
/**
 * Class APP Rest Account Settings.
 *
 * @package BuddyBossApp\Api\Core
 */

namespace BuddyBossApp\Api\Core\V1\AccountSettings;

use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Class AccountSettingsRest
 *
 * @package BuddyBossApp\Api\Core\V1\AccountSettings
 */
class AccountSettingsRest extends WP_REST_Controller {

	/**
	 * API namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'buddyboss-app/v1';

	/**
	 * Rest base.
	 *
	 * @var string
	 */
	protected $rest_base = 'account-settings';

	/**
	 * Class instance.
	 *
	 * @var object
	 */
	protected static $instance;

	/**
	 * AccountSettingsRest constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Class hooks.
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}


	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);
	}


	/**
	 * Retrieve Account Settings.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error
	 * @since 0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss-app/v1/account-settings Account Settings
	 * @apiName        GetBBAccountSettings
	 * @apiGroup       Account Settings
	 * @apiDescription Retrieve account settings tabs.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function get_items( $request ) {

		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			/**
			 * Redirect to topic endpoint.
			 */
			$request_curl = new WP_REST_Request( 'GET', '/buddyboss/v1/account-settings' );
			if ( ! empty( $request['_embed'] ) ) {
				$request_curl->set_param( '_embed', $request['_embed'] );
			}
			$response = $this->dispatch( $request_curl );

			return rest_ensure_response( $response );
		}

		$retval = array();

		$navs = array(
			(object) array(
				'name'     => __( 'Login Information', 'buddyboss-app' ),
				'slug'     => 'general',
				'position' => 10,
				'link'     => '',
			),
			(object) array(
				'name'     => __( 'Export Data', 'buddyboss-app' ),
				'slug'     => 'export',
				'position' => 20,
				'link'     => '',
			),
		);

		if ( ! empty( $navs ) ) {
			foreach ( $navs as $nav ) {
				$retval[] = $this->prepare_response_for_collection(
					$this->prepare_item_for_response( $nav, $request )
				);
			}
		}

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after account settings are fetched via the REST API.
		 *
		 * @param array            $navs     Fetched Navigations.
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_rest_account_settings_get_items', $navs, $response, $request );

		return $response;
	}

	/**
	 * Prepare Rest response.
	 *
	 * @param object          $nav     Navigation object.
	 * @param WP_REST_Request $request Request used to generate the response.
	 *
	 * @return mixed|void|WP_Error|WP_REST_Response
	 */
	public function prepare_item_for_response( $nav, $request ) {
		$data = array(
			'name'     => $nav->name,
			'slug'     => $nav->slug,
			'position' => $nav->position,
			'link'     => $nav->link,
		);

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		// @todo add prepare_links
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data['slug'] ) );

		/**
		 * Filter a notification value returned from the API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  Request used to generate the response.
		 * @param object           $nav      Navigation object.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_rest_account_setting_prepare_value', $response, $request, $nav );
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param string $nav_slug Navigation slug.
	 * @return array Links for the given group.
	 */
	protected function prepare_links( $nav_slug ) {
		$base  = '/' . $this->namespace . '/' . $this->rest_base;
		$links = array(
			'options' => array(
				'embeddable' => true,
				'href'       => rest_url( trailingslashit( $base ) . $nav_slug ),
			),
		);

		return $links;
	}

	/**
	 * Check if a given request has access to account settings.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = new WP_Error(
				'bbapp_rest_authorization_required',
				__( 'Sorry, you are not allowed to see the account settings.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		/**
		 * Filter the account settings `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_rest_account_settings_get_items_permissions_check', $retval, $request );
	}


	/**
	 * Dispatch the request item.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {

		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()['_embed'] ) ) {
			$query_params['_embed'] = $request->get_query_params()['_embed'];
		}

		$request->set_query_params( $query_params );

		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}
}
