<?php
/**
 * Class APP Rest Account Settings Options.
 *
 * @package BuddyBossApp\Api\Core
 */

namespace BuddyBossApp\Api\Core\V1\AccountSettingsOptions;

use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Class AccountSettingsOptionsRest
 *
 * @package BuddyBossApp\Api\Core\V1\AccountSettingsOptions
 */
class AccountSettingsOptionsRest extends WP_REST_Controller {

	/**
	 * API namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'buddyboss-app/v1';

	/**
	 * Rest base.
	 *
	 * @var string
	 */
	protected $rest_base = 'account-settings/(?P<nav>[\w-]+)';

	/**
	 * Class instance.
	 *
	 * @var object
	 */
	protected static $instance;

	/**
	 * AccountSettingsOptionsRest constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return mixed|object
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Class hooks.
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				'args'   => array(
					'nav' => array(
						'description'       => __( 'Navigation item slug.', 'buddyboss-app' ),
						'type'              => 'string',
						'required'          => true,
						'sanitize_callback' => 'sanitize_key',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::EDITABLE ),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve Account Settings options.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error
	 * @since          0.1.0
	 *
	 * @api            {GET} /wp-json/buddyboss/v1/account-settings/:nav Get Settings Options
	 * @apiName        GetBBAccountSettingsOptions
	 * @apiGroup       Account Settings
	 * @apiDescription Retrieve account setting options based on navigation tab.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {String=general,notifications,profile,invites,export,delete-account} nav Navigation item slug.
	 */
	public function get_item( $request ) {
		$nav = $request->get_param( 'nav' );

		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			/**
			 * Redirect to topic endpoint.
			 */
			$request_curl = new WP_REST_Request( 'GET', "/buddyboss/v1/account-settings/{$nav}" );

			if ( ! empty( $request['_embed'] ) ) {
				$request_curl->set_param( '_embed', $request['_embed'] );
			}

			$response = $this->dispatch( $request_curl );

			return rest_ensure_response( $response );
		}

		switch ( $nav ) {
			case 'general':
				$fields = $this->get_general_fields();
				break;
			case 'export':
				$fields = $this->get_export_fields();
				break;
		}

		if ( empty( $fields ) ) {
			return new WP_Error(
				'bbapp_rest_invalid_setting_nav',
				__( 'Sorry, you are not allowed to see the account settings options.', 'buddyboss-app' ),
				array(
					'status' => 400,
				)
			);
		}

		$retval = array();

		if ( ! empty( $fields ) ) {
			foreach ( $fields as $field ) {
				$retval[] = $this->prepare_response_for_collection(
					$this->prepare_item_for_response( $field, $request )
				);
			}
		}

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after account setting options are fetched via the REST API.
		 *
		 * @param array            $fields   Fetched Fields.
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_rest_account_settings_options_get_item', $fields, $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to account settings options.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = new WP_Error(
				'bbapp_rest_authorization_required',
				__( 'Sorry, you are not allowed to see the account settings options.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		$nav = $request->get_param( 'nav' );

		if ( true === $retval && empty( $nav ) ) {
			return new WP_Error(
				'bbapp_rest_invalid_setting_nav',
				__( 'Sorry, you are not allowed to see the account settings options.', 'buddyboss-app' ),
				array(
					'status' => 400,
				)
			);
		}

		/**
		 * Filter the account settings options `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_rest_account_settings_options_get_item_permissions_check', $retval, $request );
	}


	/**
	 * Update Account Settings options.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error | WP_REST_Response
	 * @since          0.1.0
	 *
	 * @api            {PATCH} /wp-json/buddyboss/v1/account-settings/:nav Update Settings Options
	 * @apiName        UpdateBBAccountSettingsOptions
	 * @apiGroup       Account Settings
	 * @apiDescription Update account setting options based on navigation tab.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {String=general,notifications,profile,invites,export,delete-account} nav Navigation item slug.
	 * @apiParam {Array} fields The list of fields to update with name and value of the field.
	 */
	public function update_item( $request ) {
		$nav         = $request->get_param( 'nav' );
		$fields      = array();
		$updated     = array();
		$post_fields = $request->get_param( 'fields' );

		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			/**
			 * Redirect to topic endpoint.
			 */
			$request_curl = new WP_REST_Request( 'POST', "/buddyboss/v1/account-settings/{$nav}" );

			$request_curl->set_param( 'fields', $post_fields );

			if ( ! empty( $request['_embed'] ) ) {
				$request_curl->set_param( '_embed', $request['_embed'] );
			}

			$response = $this->dispatch( $request_curl );

			return rest_ensure_response( $response );
		}

		switch ( $nav ) {
			case 'general':
				$updated = $this->update_general_fields( $request );
				$fields  = $this->get_general_fields();
				break;
			case 'export':
				$updated = $this->update_export_fields( $request );
				$fields  = $this->get_export_fields();
				break;
		}

		$fields_update = $this->update_additional_fields_for_object( $nav, $request );
		if ( is_wp_error( $fields_update ) ) {
			return $fields_update;
		}

		$data = array();

		if ( ! empty( $fields ) ) {
			foreach ( $fields as $field ) {
				$data[] = $this->prepare_response_for_collection(
					$this->prepare_item_for_response( $field, $request )
				);
			}
		}

		$retval   = array(
			'error'   => ( isset( $updated['error'] ) ? $updated['error'] : false ),
			'success' => ( empty( $updated['error'] ) ? __( 'Your settings has been successfully updated.', 'buddyboss-app' ) : false ),
			'notices' => ( isset( $updated['notice'] ) ? $updated['notice'] : false ),
			'data'    => $data,
		);
		$response = rest_ensure_response( $retval );

		/**
		 * Fires after account setting options are updated via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		do_action( 'bbapp_rest_account_settings_options_update_item', $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to update account settings options.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 * @since 0.1.0
	 */
	public function update_item_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = new WP_Error(
				'bbapp_rest_authorization_required',
				__( 'Sorry, you are not allowed to see the account settings options.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		$nav = $request->get_param( 'nav' );

		if ( true === $retval && empty( $nav ) ) {
			return new WP_Error(
				'bbapp_rest_invalid_setting_nav',
				__( 'Sorry, you are not allowed to update the account settings options.', 'buddyboss-app' ),
				array(
					'status' => 400,
				)
			);
		}

		/**
		 * Filter the account settings options `update_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_rest_account_settings_options_update_item_permissions_check', $retval, $request );
	}

	/**
	 * Get Fields for the General "Login Information".
	 * - From: 'members/single/settings/general'
	 *
	 * @return array|mixed|void
	 */
	public function get_general_fields() {
		$fields = array();

		if ( ! is_super_admin() ) {
			$fields[] = array(
				'name'        => 'current_password',
				'label'       => __( 'Current Password <span>(required to update email or change current password)</span>', 'buddyboss-app' ),
				'field'       => 'password',
				'value'       => '',
				'placeholder' => __( 'Enter password', 'buddyboss-app' ),
				'options'     => array(),
				'group_label' => '',
			);
		}

		$fields[] = array(
			'name'        => 'account_email',
			'label'       => __( 'Account Email', 'buddyboss-app' ),
			'field'       => 'email',
			'value'       => esc_attr( get_the_author_meta( 'email', get_current_user_id() ) ),
			'placeholder' => __( 'Enter email', 'buddyboss-app' ),
			'options'     => array(),
			'group_label' => '',
		);

		$fields[] = array(
			'name'        => 'pass1',
			'label'       => __( 'Add Your New Password', 'buddyboss-app' ),
			'field'       => 'password',
			'value'       => '',
			'placeholder' => __( 'Enter password', 'buddyboss-app' ),
			'options'     => array(),
			'group_label' => '',
		);

		$fields[] = array(
			'name'        => 'pass2',
			'label'       => __( 'Repeat Your New Password', 'buddyboss-app' ),
			'field'       => 'password',
			'value'       => '',
			'placeholder' => __( 'Enter password', 'buddyboss-app' ),
			'options'     => array(),
			'group_label' => '',
		);

		/**
		 * Filters rest account general settings.
		 *
		 * @param array $fields Fields to export.
		 */
		$fields = apply_filters( 'bbapp_rest_account_settings_general', $fields );

		return $fields;
	}

	/**
	 * Get Fields for the Export "Export Data".
	 * - From: 'members/single/settings/export-data'
	 *
	 * @return array|mixed|void
	 */
	public function get_export_fields() {
		$fields = array(
			array(
				'name'        => 'member-data-export-submit',
				'label'       => __( 'Request Data Export', 'buddyboss-app' ),
				'field'       => 'button',
				'value'       => '',
				'options'     => array(),
				'group_label' => '',
			),
		);

		/**
		 * Filters rest account settings export.
		 *
		 * @param array $fields Fields to export.
		 */
		$fields = apply_filters( 'bbapp_rest_account_settings_export', $fields );

		return $fields;
	}

	/**
	 * Update general fields.
	 * - from bbapp-settings\actions\general.php.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return array|bool|void
	 */
	public function update_general_fields( $request ) {
		$post_fields   = $request->get_param( 'fields' );
		$user_id       = get_current_user_id();
		$update_user   = get_userdata( $user_id );
		$email_error   = false;
		$pass_error    = false;
		$pass_changed  = false; // true if the user changes their password .
		$email_changed = false; // true if the user changes their email.
		$feedback      = array(); // array of strings for feedback.

		if ( ( is_super_admin() ) || ( isset( $post_fields['current_password'] ) && ! empty( $post_fields['current_password'] ) && wp_check_password( $post_fields['current_password'], $update_user->data->user_pass, $user_id ) ) ) {

			/* Email Change Attempt ******************************************/
			if ( ! empty( $post_fields['account_email'] ) ) {
				// What is missing from the profile page vs signup.
				// let's double check the goodies.
				$user_email     = sanitize_email( esc_html( trim( $post_fields['account_email'] ) ) );
				$old_user_email = $update_user->data->user_email;

				// User is changing email address.
				if ( $old_user_email !== $user_email ) {
					// Run some tests on the email address.
					$email_checks = $this->bbapp_core_validate_email_address( $user_email );

					if ( true !== $email_checks ) {
						if ( isset( $email_checks['invalid'] ) ) {
							$email_error = 'invalid';
						}

						if ( isset( $email_checks['domain_banned'] ) || isset( $email_checks['domain_not_allowed'] ) ) {
							$email_error = 'blocked';
						}

						if ( isset( $email_checks['in_use'] ) ) {
							$email_error = 'taken';
						}
					}

					// Store a hash to enable email validation.
					if ( false === $email_error ) {
						$email_changed = wp_update_user(
							array(
								'ID'         => $user_id,
								'user_email' => trim( $user_email ),
							)
						);

						// We mark that the change has taken place so as to ensure a.
						// success message, even though verification is still required.
						$post_fields['account_email'] = $update_user->user_email;
						$email_changed                = ! is_wp_error( $email_changed );
					}
					// No change.
				} else {
					$email_error = false;
				}
				// Email address cannot be empty.
			} else {
				$email_error = 'empty';
			}

			/* Password Change Attempt ***************************************/
			if ( ! empty( $post_fields['pass1'] ) && ! empty( $post_fields['pass2'] ) ) {
				if ( ( $post_fields['pass1'] === $post_fields['pass2'] ) && ! strpos( ' ' . wp_unslash( $post_fields['pass1'] ), '\\' ) ) {
					// Password change attempt is successful.
					if ( ( ! empty( $post_fields['current_password'] ) && $post_fields['current_password'] !== $post_fields['pass1'] ) || is_super_admin() ) {
						$update_user->user_pass = $post_fields['pass1'];
						$pass_changed           = true;
						// The new password is the same as the current password.
					} else {
						$pass_error = 'same';
					}
					// Password change attempt was unsuccessful.
				} else {
					$pass_error = 'mismatch';
				}

				// Both password fields were empty.
			} elseif ( empty( $post_fields['pass1'] ) && empty( $post_fields['pass2'] ) ) {
				$pass_error = false;

				// One of the password boxes was left empty.
			} elseif ( ( empty( $post_fields['pass1'] ) && ! empty( $post_fields['pass2'] ) ) || ( ! empty( $post_fields['pass1'] ) && empty( $post_fields['pass2'] ) ) ) {
				$pass_error = 'empty';
			}

			// The structure of the $update_user object changed in WP 3.3, but wp_update_user() still expects the old format.
			if ( isset( $update_user->data ) && is_object( $update_user->data ) ) {
				$update_user = $update_user->data;
				$update_user = get_object_vars( $update_user );

				// Unset the password field to prevent it from emptying out the user's user_pass field in the database.
				// @see wp_update_user().
				if ( false === $pass_changed ) {
					unset( $update_user['user_pass'] );
				}
			}

			// Clear cached data, so that the changed settings take effect on the current page load.
			clean_user_cache( bp_displayed_user_id() );

			if ( ( false === $email_error ) && ( false === $pass_error ) && ( wp_update_user( $update_user ) ) ) {
				$update_user = get_userdata( $user_id );
			}
			// Password Error.
		} else {
			$pass_error = 'invalid';
		}

		// Email feedback.
		switch ( $email_error ) {
			case 'invalid':
				$feedback['email_invalid'] = __( 'That email address is invalid. Check the formatting and try again.', 'buddyboss-app' );
				break;
			case 'blocked':
				$feedback['email_blocked'] = __( 'That email address is currently unavailable for use.', 'buddyboss-app' );
				break;
			case 'taken':
				$feedback['email_taken'] = __( 'That email address is already taken.', 'buddyboss-app' );
				break;
			case 'empty':
				$feedback['email_empty'] = __( 'Email address cannot be empty.', 'buddyboss-app' );
				break;
			case false:
				// No change.
				break;
		}

		// Password feedback.
		switch ( $pass_error ) {
			case 'invalid':
				$feedback['pass_error'] = __( 'Your current password is invalid.', 'buddyboss-app' );
				break;
			case 'mismatch':
				$feedback['pass_mismatch'] = __( 'The new password fields did not match.', 'buddyboss-app' );
				break;
			case 'empty':
				$feedback['pass_empty'] = __( 'One of the password fields was empty.', 'buddyboss-app' );
				break;
			case 'same':
				$feedback['pass_same'] = __( 'The new password must be different from the current password.', 'buddyboss-app' );
				break;
			case false:
				// No change.
				break;
		}

		// Some kind of errors occurred.
		if ( ( ( false === $email_error ) || ( false === $pass_error ) ) && ( ( true !== $pass_changed ) && ( true !== $email_changed ) ) ) {
			$feedback['nochange'] = __( 'No changes were made to your account.', 'buddyboss-app' );
		}

		$notice = $this->bbapp_rest_settings_pending_email_notice();

		if ( empty( $feedback ) ) {
			return array(
				'error'  => false,
				'notice' => $notice,
			);
		} else {
			return array(
				'error'  => $feedback,
				'notice' => $notice,
			);
		}

		return false;
	}

	/**
	 * Add the 'pending email change' message to the settings page.
	 * -- from: bbapp_settings_pending_email_notice().
	 *
	 * @return void|string
	 */
	protected function bbapp_rest_settings_pending_email_notice() {
		$pending_email = get_user_meta( get_current_user_id(), 'pending_email_change', true );

		if ( empty( $pending_email['newemail'] ) ) {
			return;
		}

		$user = get_user_by( 'ID', get_current_user_id() );
		if ( empty( $user ) || is_wp_error( $user ) ) {
			return;
		}

		if ( $user->user_email === $pending_email['newemail'] ) {
			return;
		}

		return sprintf(
		/* translators: 1: New email. 2: Current email. */
			__( 'There is a pending change of your email address to %1$s. Check your email (%2$s) for the verification link.', 'buddyboss-app' ),
			'<strong>' . esc_html( $pending_email['newemail'] ) . '</strong>',
			'<strong>' . esc_html( $user->display_name ) . '</strong>'
		);
	}

	/**
	 * Update Export fields "Export Data".
	 * - from bbapp_settings_action_export().
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return array|bool|void
	 */
	public function update_export_fields( $request ) {
		$post_fields = $request->get_param( 'fields' );
		$error       = false;
		$notice      = false;

		if ( isset( $post_fields['member-data-export-submit'] ) ) {
			$user_id    = get_current_user_id();
			$user       = get_userdata( $user_id );
			$request_id = wp_create_user_request( $user->data->user_email, 'export_personal_data' );

			if ( is_wp_error( $request_id ) ) {
				$error = $request_id->get_error_message();
			} elseif ( ! $request_id ) {
				$error = __( 'Unable to initiate the data export request.', 'buddyboss-app' );
			}

			if ( empty( $error ) ) {
				wp_send_user_request( $request_id );
				$notice = __( 'Please check your email to confirm the data export request.', 'buddyboss-app' );
			}
		}

		return array(
			'error'  => $error,
			'notice' => $notice,
		);
	}

	/**
	 * Prepares account settings data for return as an object.
	 *
	 * @param object          $field   Field object.
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since 0.1.0
	 */
	public function prepare_item_for_response( $field, $request ) {
		$data    = array(
			'name'        => ( isset( $field['name'] ) && ! empty( $field['name'] ) ? $field['name'] : '' ),
			'label'       => ( isset( $field['label'] ) && ! empty( $field['label'] ) ? $field['label'] : '' ),
			'type'        => ( isset( $field['field'] ) && ! empty( $field['field'] ) ? $field['field'] : '' ),
			'value'       => ( isset( $field['value'] ) && ! empty( $field['value'] ) ? $field['value'] : '' ),
			'placeholder' => ( isset( $field['placeholder'] ) && ! empty( $field['placeholder'] ) ? $field['placeholder'] : '' ),
			'options'     => ( isset( $field['options'] ) && ! empty( $field['options'] ) ? $field['options'] : array() ),
			'headline'    => ( isset( $field['group_label'] ) && ! empty( $field['group_label'] ) ? $field['group_label'] : '' ),
		);
		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		// @todo add prepare_links
		$response = rest_ensure_response( $data );

		/**
		 * Filter a notification value returned from the API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  Request used to generate the response.
		 * @param object           $field    Field object.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_rest_account_setting_prepare_value', $response, $request, $field );
	}

	/**
	 * Edit some properties for the CREATABLE & EDITABLE methods.
	 *
	 * @param string $method Optional. HTTP method of the request.
	 *
	 * @return array Endpoint arguments.
	 * @since 0.1.0
	 */
	public function get_endpoint_args_for_item_schema( $method = WP_REST_Server::CREATABLE ) {
		$args = WP_REST_Controller::get_endpoint_args_for_item_schema( $method );
		$key  = 'get_item';

		if ( WP_REST_Server::EDITABLE === $method ) {
			$key  = 'update_item';
			$args = array(
				'nav'    => array(
					'description'       => __( 'Navigation item slug.', 'buddyboss-app' ),
					'type'              => 'string',
					'required'          => true,
					'sanitize_callback' => 'sanitize_key',
					'validate_callback' => 'rest_validate_request_arg',
				),
				'fields' => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'The list of fields Objects to update with name and value of the field.', 'buddyboss-app' ),
					'type'        => 'object',
					'required'    => true,
				),
			);
		}

		/**
		 * Filters the method query arguments.
		 *
		 * @param array  $args   Query arguments.
		 * @param string $method HTTP method of the request.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( "bbapp_rest_update_accounts_{$key}_query_arguments", $args, $method );
	}


	/**
	 * Get the Account Settings schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'bbapp_account_settings_options',
			'type'       => 'object',
			'properties' => array(
				'name'     => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'A unique name for the field.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'string',
				),
				'label'    => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Label of the field.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'string',
				),
				'type'     => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The type the field.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'string',
				),
				'value'    => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The saved value for the field.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'string',
				),
				'options'  => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Available options for the field.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'object',
				),
				'headline' => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Headline text for the field.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'string',
				),
			),
		);

		/**
		 * Filters the Account Settings schema.
		 *
		 * @param array $schema The endpoint schema.
		 */
		return apply_filters( 'bbapp_rest_account_settings_options_schema', $this->add_additional_fields_schema( $schema ) );
	}

	/**
	 * Get the query params for the Account Settings collections.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_collection_params() {
		$params                       = parent::get_collection_params();
		$params['context']['default'] = 'view';

		// Remove the search argument.
		unset( $params['search'] );
		unset( $params['page'] );
		unset( $params['per_page'] );

		/**
		 * Filters the collection query params.
		 *
		 * @param array $params Query params.
		 */
		return apply_filters( 'bbapp_rest_account_settings_options_collection_params', $params );
	}

	/**
	 * Check that an email address is valid for use.
	 *
	 * Performs the following checks:
	 *   - Is the email address well-formed?
	 *   - Is the email address already used?
	 *   - If there's an email domain blacklist, is the current domain on it?
	 *   - If there's an email domain whitelest, is the current domain on it?
	 *
	 * @param string $user_email The email being checked.
	 *
	 * @return bool|array True if the address passes all checks; otherwise an array
	 *                    of error codes.
	 */
	public function bbapp_core_validate_email_address( $user_email ) {
		$errors     = array();
		$user_email = sanitize_email( $user_email );

		// Is the email well-formed?
		if ( ! is_email( $user_email ) ) {
			$errors['invalid'] = 1;
		}

		// Is the email on the Banned Email Domains list?
		// Note: This check only works on Multisite.
		if ( function_exists( 'is_email_address_unsafe' ) && is_email_address_unsafe( $user_email ) ) {
			$errors['domain_banned'] = 1;
		}

		// Is the email on the Limited Email Domains list?
		// Note: This check only works on Multisite.
		$limited_email_domains = get_site_option( 'limited_email_domains' );

		if ( is_array( $limited_email_domains ) && ! empty( $limited_email_domains ) ) {
			$emaildomain = substr( $user_email, 1 + strpos( $user_email, '@' ) );

			if ( ! in_array( $emaildomain, $limited_email_domains, true ) ) {
				$errors['domain_not_allowed'] = 1;
			}
		}

		// Is the email alreday in use?
		if ( email_exists( $user_email ) ) {
			$errors['in_use'] = 1;
		}

		$retval = ! empty( $errors ) ? $errors : true;

		return $retval;
	}

	/**
	 * Dispatch the request item.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {
		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()['_embed'] ) ) {
			$query_params['_embed'] = $request->get_query_params()['_embed'];
		}

		$request->set_query_params( $query_params );

		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}
}
